/////////////////////////////////////////////////////////////////////////////
// Windows Frotz
// Frotz application class
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"

#include "FrotzApp.h"

#include "FrotzGfx.h"
#include "FrotzSound.h"
#include "FrotzWnd.h"
#include "FrotzFrameWnd.h"


/// New stuff

extern "C"
{
#include "blorblow.h"
#include ".\frotzapp.h"

int main(int argc, char* argv[]);
void reset_memory(void);
void replay_close(void);
void set_header_extension (int entry, zword val);
}

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif



BEGIN_MESSAGE_MAP(FrotzApp, CWinApp)
	//{{AFX_MSG_MAP(FrotzApp)
	ON_COMMAND(ID_FILE_NEW, OnFileNew)
	ON_COMMAND(ID_FILE_OPEN, OnFileOpen)
	ON_COMMAND(ID_FILE_SAVE, OnFileSave)
	ON_UPDATE_COMMAND_UI(ID_FILE_OPEN, OnUpdateFileOpen)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE, OnUpdateFileSave)
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_VIEW_OPTIONS, OnViewOptions)
	ON_COMMAND(ID_VIEW_SCROLLBACK, OnViewScrollback)
	ON_COMMAND(ID_APP_ABOUT, OnAppAbout)
	ON_COMMAND(ID_FILE_STOP, OnFileStop)
	ON_UPDATE_COMMAND_UI(ID_FILE_STOP, OnUpdateFileStop)
	ON_COMMAND(ID_LINKS_ARCHIVE, OnLinksArchive)
	ON_COMMAND(ID_LINKS_ZCODE, OnLinksZcode)
	ON_COMMAND(ID_LINKS_BAF, OnLinksBaf)
	ON_COMMAND(ID_LINKS_IFCOMP, OnLinksIFComp)
	ON_COMMAND(ID_LINKS_ARTSHOW, OnLinksArtShow)
	ON_COMMAND(ID_LINKS_INFORM, OnLinksInform)
	ON_COMMAND(ID_LINKS_FAQ, OnLinksFaq)
    ON_COMMAND(ID_APP_SPEECH, OnAppSpeech)
    ON_COMMAND(ID_WORDLIST, OnWordlist)
    ON_COMMAND(ID_REPEAT, OnRepeat)
    ON_COMMAND(ID_SCRATCH, OnScratch)
    ON_COMMAND(ID_VOICES, OnVoices)
    ON_COMMAND(ID_ADD, OnAdd)
    ON_COMMAND(ID_UPDATE, OnUpdate)
    ON_COMMAND(IDWEBHELP,OnWebHelp)

END_MESSAGE_MAP()

// The single application instance
FrotzApp theApp;

// The single output window instance
FrotzWnd* theWnd = NULL;

// Constructor
FrotzApp::FrotzApp()
{
    myTTS = CifTTS::GetSingletonPtr();
//    IpVoice = NULL;
    DebugStr("DBGVIEWCLEAR");
	Initialize();
	FrotzGfx::SetGamma(2.2);
	EnableHtmlHelp();

	m_register = true;
	m_toolBar = true;
	m_statusBar = true;
	m_notifyFull = true;
	m_fastScroll = false;
	m_morePrompts = true;
	m_leftMargin = 0;
	m_rightMargin = 0;

	// Standard Z-Machine colours
	m_colours[0]  = RGB5ToTrue(0x0000); // black
	m_colours[1]  = RGB5ToTrue(0x001D); // red
	m_colours[2]  = RGB5ToTrue(0x0340); // green
	m_colours[3]  = RGB5ToTrue(0x03BD); // yellow
	m_colours[4]  = RGB5ToTrue(0x59A0); // blue
	m_colours[5]  = RGB5ToTrue(0x7C1F); // magenta
	m_colours[6]  = RGB5ToTrue(0x77A0); // cyan
	m_colours[7]  = RGB5ToTrue(0x7FFF); // white
	m_colours[8]  = RGB5ToTrue(0x5AD6); // light grey
	m_colours[9]  = RGB5ToTrue(0x4631); // medium grey
	m_colours[10] = RGB5ToTrue(0x2D6B); // dark grey
    
}
BOOL FrotzApp::InitializeSAPI()
{

    myTTS->InitializeSAPI(m_voiceToken.GetBuffer(),m_Rate,m_Volume);
    
    if(m_bDialogSpeak)
        myTTS->addFlag(IF_DIALOG_SPEECH);
   
    return TRUE;
}

void FrotzApp::Speak(const WCHAR *szText , int flags = SPF_ASYNC)
{
    if(szText==NULL) return;

    if(myTTS->checkFlags(IF_GLOBAL_SPEECH))
    {     
        myTTS->Speak(szText,flags);
    }
}


void FrotzApp::SpeakA(const char *szText, int flags = SPF_ASYNC)
{
    if(myTTS->checkFlags(IF_GLOBAL_SPEECH))
    {   
        myTTS->Speak(szText,flags);
    }

    return;
}

// Initialize the application
void FrotzApp::Initialize(void)
{
	m_exitPause = false;
	m_blorbFile = NULL;
	m_blorbMap = NULL;

	for (int i = 0; i < NON_STD_COLS; i++)
		m_nonStdColours[i] = 0xFFFFFFFF;
	m_nonStdIndex = 0;

	if (theWnd != NULL)
		theWnd->Initialize();

	m_scrollback.SetSize(0,8192);
	m_startTime = CTime::GetCurrentTime();    
}

// Run the interpreter core
FrotzApp::ExitStatus FrotzApp::RunInterpreter(void)
{
	try
	{
		// Start the interpreter
		main(__argc,__argv);
		AfxGetMainWnd()->DestroyWindow();
	}
	catch (FrotzApp::AbortFrotz&)
	{
		return Aborted;
	}
	catch (FrotzApp::ExitFrotz&)
	{
		return Exited;
	}
	catch (FrotzApp::RestartFrotz&)
	{
		return Restarting;
	}
	return Exited;
}

// Read in settings
void FrotzApp::ReadSettings(void)
{
	h_interpreter_number = GetProfileInt("Interpreter","Number",INTERP_AMIGA);
	err_report_mode = GetProfileInt("Interpreter","Error Reporting",ERR_REPORT_ONCE);
	option_ignore_errors = GetProfileInt("Interpreter","Ignore Errors",0);
	option_expand_abbreviations = GetProfileInt("Interpreter","Expand Abbreviations",0);
	m_tandy = GetProfileInt("Interpreter","Tandy Bit",0) ? true : false;
	m_quetzal = GetProfileInt("Interpreter","Quetzal Format",1) ? true : false;

	m_filename = GetProfileString("Files","Initial Game","");
	m_register = GetProfileInt("Files","Register File Types",1) ? true : false;

	m_wndSize.left = GetProfileInt("Window","Left",0);
	m_wndSize.top = GetProfileInt("Window","Top",0);
	m_wndSize.right = GetProfileInt("Window","Right",0);
	m_wndSize.bottom = GetProfileInt("Window","Bottom",0);
	m_wndState = GetProfileInt("Window","State",SW_SHOWNORMAL);

	m_toolBar = GetProfileInt("Window","Toolbar",1) ? true : false;
	m_statusBar = GetProfileInt("Window","Status Bar",1) ? true : false;
	m_notifyFull = GetProfileInt("Window","Notify Full Screen",1) ? true : false;

    // Speech
    m_speech = GetProfileInt("TTS","Enable TTS",1) ? true : false;
    m_bDialogSpeak = GetProfileInt("TTS","Speak Dialog",1) ? true : false;
    m_voiceName = GetProfileString("TTS","VoiceName","");
    m_voiceToken = GetProfileString("TTS","VoiceToken","");
    m_Rate = GetProfileInt("TTS","Rate",0);
    m_Volume = GetProfileInt("TTS","Volume",100);


	m_propFontName = GetProfileString("Display","Proportional Font Name",
		"Times New Roman");
	m_fixedFontName = GetProfileString("Display","Fixed Font Name",
		"Courier New");
	m_fontSize = GetProfileInt("Display","Font Size",10);
	m_v6scale = GetProfileInt("Display","Infocom V6 Scaling",2);
	m_defaultFore = Make5BitColour(
		GetProfileInt("Display","Foreground",RGB(0xFF,0xFF,0xFF)));
	m_defaultBack = Make5BitColour(
		GetProfileInt("Display","Background",RGB(0x00,0x00,0x80)));
	m_fastScroll = GetProfileInt("Display","Fast Scrolling",0) ? true : false;
	m_morePrompts = GetProfileInt("Display","Show More Prompts",1) ? true : false;
	m_leftMargin = GetProfileInt("Display","Left Margin",0);
	m_rightMargin = GetProfileInt("Display","Right Margin",0);

	// Are the settings from the latest version?
	int version = GetProfileInt("Version","Number",0);
	if (version < 103)
	{
		int w = ::GetSystemMetrics(SM_CXSCREEN);
		int h = ::GetSystemMetrics(SM_CYSCREEN);
		if ((w > 800) && (h > 600))
		{
			m_fontSize = 14;
			m_v6scale = 3;
		}
		else
		{
			m_fontSize = 10;
			m_v6scale = 2;
		}

		// Ignore errors by default
		err_report_mode = ERR_REPORT_NEVER;
	}
}

// Write out settings
void FrotzApp::WriteSettings(void)
{
	// The current version number
	WriteProfileInt("Version","Number",104);

	WriteProfileInt("Interpreter","Number",h_interpreter_number);
	WriteProfileInt("Interpreter","Error Reporting",err_report_mode);
	WriteProfileInt("Interpreter","Ignore Errors",option_ignore_errors);
	WriteProfileInt("Interpreter","Expand Abbreviations",option_expand_abbreviations);
	WriteProfileInt("Interpreter","Tandy Bit",m_tandy ? 1 : 0);
	WriteProfileInt("Interpreter","Quetzal Format",m_quetzal ? 1 : 0);

	WriteProfileString("Files","Initial Game",m_filename);
	WriteProfileInt("Files","Register File Types",m_register ? 1 : 0);

	WriteProfileInt("Window","Left",m_wndSize.left);
	WriteProfileInt("Window","Top",m_wndSize.top);
	WriteProfileInt("Window","Right",m_wndSize.right);
	WriteProfileInt("Window","Bottom",m_wndSize.bottom);
	WriteProfileInt("Window","State",m_wndState);

	WriteProfileInt("Window","Toolbar",m_toolBar ? 1 : 0);
	WriteProfileInt("Window","Status Bar",m_statusBar ? 1 : 0);
	WriteProfileInt("Window","Notify Full Screen",m_notifyFull ? 1 : 0);

    // TTS
    WriteProfileInt("TTS","Enable TTS",m_speech ? 1 : 0);
    WriteProfileInt("TTS","Speak Dialog",m_bDialogSpeak ? 1 : 0);
    WriteProfileString("TTS","VoiceName",m_voiceName);
    WriteProfileString("TTS","VoiceToken",m_voiceToken);
    WriteProfileInt("TTS","Rate",m_Rate);
    WriteProfileInt("TTS","Volume",m_Volume);

	WriteProfileString("Display","Proportional Font Name",m_propFontName);
	WriteProfileString("Display","Fixed Font Name",m_fixedFontName);
	WriteProfileInt("Display","Font Size",m_fontSize);
	WriteProfileInt("Display","Infocom V6 Scaling",m_v6scale);
	WriteProfileInt("Display","Foreground",m_defaultFore);
	WriteProfileInt("Display","Background",m_defaultBack);
	WriteProfileInt("Display","Fast Scrolling",m_fastScroll);
	WriteProfileInt("Display","Show More Prompts",m_morePrompts);
	WriteProfileInt("Display","Left Margin",m_leftMargin);
	WriteProfileInt("Display","Right Margin",m_rightMargin);
}

// Get the display settings
void FrotzApp::GetDisplaySettings(CString& propName, CString& fixedName, int& size)
{
	propName = m_propFontName;
	fixedName = m_fixedFontName;
	size = m_fontSize;
}

/// Added from MFC source to allow dynamic linking
CWnd::PFNNOTIFYWINEVENT CWnd::m_pfnNotifyWinEvent = NULL;

/// Added from MFC source to allow dynamic linking
BOOL AFXAPI _AfxSetRegKey(LPCTSTR lpszKey, LPCTSTR lpszValue, LPCTSTR lpszValueName = NULL)
{
	if (lpszValueName == NULL)
	{
		if (::RegSetValue(HKEY_CLASSES_ROOT, lpszKey, REG_SZ,
			  lpszValue, lstrlen(lpszValue) * sizeof(TCHAR)) != ERROR_SUCCESS)
		{
			TRACE(traceAppMsg, 0, _T("Warning: registration database update failed for key '%s'.\n"),
				lpszKey);
			return FALSE;
		}
		return TRUE;
	}
	else
	{
		HKEY hKey;

		if(::RegCreateKey(HKEY_CLASSES_ROOT, lpszKey, &hKey) == ERROR_SUCCESS)
		{
			LONG lResult = ::RegSetValueEx(hKey, lpszValueName, 0, REG_SZ,
				(CONST BYTE*)lpszValue, (lstrlen(lpszValue) + 1) * sizeof(TCHAR));

			if(::RegCloseKey(hKey) == ERROR_SUCCESS && lResult == ERROR_SUCCESS)
				return TRUE;
		}
		TRACE(traceAppMsg, 0, _T("Warning: registration database update failed for key '%s'.\n"), lpszKey);
		return FALSE;
	}
}

/// Added from MFC source to allow dynamic linking
void AFXAPI AfxGetModuleShortFileName(HINSTANCE hInst, CString& strShortName)
{
    TCHAR szLongPathName[_MAX_PATH];
    ::GetModuleFileName(hInst, szLongPathName, _MAX_PATH);
    if (::GetShortPathName(szLongPathName,
        strShortName.GetBuffer(_MAX_PATH), _MAX_PATH) == 0)
    {
        // rare failure case (especially on not-so-modern file systems)
        strShortName = szLongPathName;
    }
    strShortName.ReleaseBuffer();
}

// Register file types
void FrotzApp::RegisterFileTypes(void)
{
	// Check that file registration hasn't been disabled
	if (m_register == false)
		return;

	// Get the path to the executable
	CString path;
	AfxGetModuleShortFileName(AfxGetInstanceHandle(),path);

	// Register each Z-code file extension
	for (int i = 1; i <= 8; i++)
	{
		CString extension, type, key, value;

		extension.Format(".z%d",i);
		type.Format("ZMachine.V%d",i);
		if (_AfxSetRegKey(extension,type))
		{
			// Set up a name for the file type
			value.Format("Z-code V%d Adventure",i);
			_AfxSetRegKey(type,value);

			// Set up an icon
			key.Format("%s\\DefaultIcon",type);
			value.Format("%s,%d",path,i+1);
			_AfxSetRegKey(key,value);

			// Set up an open command
			key.Format("%s\\shell\\open\\command",type);
			value.Format("%s \"%%1\"",path);
			_AfxSetRegKey(key,value);
		}
	}

	// Notify the shell that associations have changed
	::SHChangeNotify(SHCNE_ASSOCCHANGED,SHCNF_IDLIST,0,0);
}

// Load an international version of resources
void FrotzApp::LoadInternationalResources(void)
{
	const char* resDllName = NULL;
	switch (PRIMARYLANGID(::GetUserDefaultLangID()))
	{
	case LANG_FRENCH:
		resDllName = "FrotzFranais.dll";
		break;
	case LANG_GERMAN:
		resDllName = "FrotzDeutsch.dll";
		break;
	case LANG_ITALIAN:
		resDllName = "FrotzItaliano.dll";
		break;
	case LANG_SPANISH:
		resDllName = "FrotzEspaol.dll";
		break;
	}

	if (resDllName != NULL)
	{
		HINSTANCE dll = ::LoadLibrary(resDllName);
		if (dll != NULL)
			AfxSetResourceHandle(dll);
	}
}

void FrotzApp::Browse( std::string &sFolderName, std::string &sDef, const char * szName  )
{

} 

void FrotzApp::setMode(int iMode)
{
    myTTS->SetMode(iMode);


    int len = m_speechBuffer.GetSize();
    

    if(iMode == 1)
    {
        

        if( len <=0 ) 
        {
            return;
        }

        std::wstring wStr = m_speechBuffer.GetData();
        wStr = wStr.substr(0,len);

        if(wStr[wStr.length()-1] == L'>' )
        {
            wStr = wStr.substr(0,wStr.length()-1);
        }
        
        myTTS->AddToBuffer(wStr);
        m_speechBuffer.SetSize(0,0);
        
        myTTS->SpeakParagraph(wStr.c_str(), SPF_ASYNC);
    }
}
int FrotzApp::getMode( void )
{
    return myTTS->GetMode();
}

void FrotzApp::SetHistory( int iHist )
{
    myTTS->SetHistory(iHist);
}
/// TTS enhancement to allow input to be heard
void FrotzApp::MonitorInput(const unsigned short* buffer)
{
    myTTS->MonitorInput(buffer);

}
// Open a file dialog to prompt the user for a game
bool FrotzApp::PromptForGame(bool initial)
{
   
	if (initial)
	{ 
		if (__argc > 1)
		{
			m_filename = __argv[1];
			return true;
		}
	}

    setMode(0);
    /// TTS enhancements
    /// Visually impared people need the game titles read aloud
    /// we match titles to CRCs on the files
    if(myTTS->checkFlags(IF_GLOBAL_SPEECH))
    {
        myTTS->SetRoot("");

        WString sText;
        myTTS->getList( sText,CifTTS::if_games,".blb,.dat,.zip,.z1,.z2,.z3,.z4,.z5,.z6,.z7,.z8",".sav" );

        if(!sText.empty())
        {
            std::string sTmp = static_cast<std::string>(sText);
            
            m_filename = sTmp.c_str();

            //TODO poss prob
            String sCRC;
            if(myTTS->GetCRC(sTmp,sCRC))
            {
                //DebugStr("Load game: %s %s",sTmp.c_str(),sCRC.c_str());
                myTTS->SetGame(sCRC);
            }
           
            return true;
        }

        GameFileDialog dialog(m_filename,AfxGetMainWnd());
        if (dialog.DoModal() == IDOK)
        {
            // ok add this game to the list
            m_filename = dialog.GetPathName();

            WString sText2;
            String sName = m_filename.GetBuffer(0);
            myTTS->getInput(sText2,"What is the title of this new game.");
            String sTitle= sText2;

            if(myTTS->addNewGame(sName,sTitle) == false)
            {
                Speak(L"Game could not be added to supported list. Hit enter to Continue.",SPF_ASYNC|SPF_PURGEBEFORESPEAK);
                ::MessageBox(GetActiveWindow(),"Game could not be added to supported list. Hit enter to Continue.","Error",MB_OK);

            }
            String sCRC;
            if(myTTS->GetCRC(sName,sCRC))
            {
                myTTS->SetGame(sCRC);
            }

            return true;
        }
    }
    else
    {
        /// Otherwise let them use the normal dialog
        GameFileDialog dialog(m_filename,AfxGetMainWnd());
        if (dialog.DoModal() == IDOK)
        {
            m_filename = dialog.GetPathName();
            return true;
        }
    }


	return false;
}

// Get the filename of the game to run
LPCTSTR FrotzApp::GetGameFileName(void)
{
	return m_filename;
}

// Create the main window
void FrotzApp::CreateMainWindow(void)
{
	FrotzFrameWnd* wnd = (FrotzFrameWnd*)m_pMainWnd;
	if (wnd == NULL)
	{
		wnd = new FrotzFrameWnd;
		if (wnd->Create(m_toolBar,m_statusBar) == false)
			throw AbortFrotz();
		m_pMainWnd = wnd;
	}


	// Reset the menus
	wnd->ResetMenus();

	// Set the graphics scaling
	if (IsInfocomV6() || (story_id == BEYOND_ZORK))
		m_gfxScale = m_v6scale;
	else
		m_gfxScale = 1;

	// Get the Frotz output window
	theWnd = wnd->GetClientWnd();

	// Resize the window. For Infocom's V6 games, a fixed
	// window size is always used
	theWnd->SetAllowResize(false);
	if (IsInfocomV6())
	{
		CRect screen;
		::SystemParametersInfo(SPI_GETWORKAREA,0,(LPRECT)screen,0);

		// Resize the window large enough that the non-client
		// area can be measured
		if (wnd->IsWindowVisible() == FALSE)
			wnd->MoveWindow(CRect(0,0,640,400),TRUE);

		CRect clientSize, windowSize;
		theWnd->GetClientRect(clientSize);
		wnd->GetWindowRect(windowSize);

		int borderX = windowSize.Width() - clientSize.Width();
		int borderY = windowSize.Height() - clientSize.Height();
		int width = (320*m_gfxScale)+borderX+m_leftMargin+m_rightMargin;
		int height = (200*m_gfxScale)+borderY;

		WINDOWPLACEMENT place;
		::ZeroMemory(&place,sizeof(WINDOWPLACEMENT));
		place.length = sizeof(WINDOWPLACEMENT);
		wnd->GetWindowPlacement(&place);
		if (m_wndSize.Width() > 0)
			place.rcNormalPosition = m_wndSize;

		int x = place.rcNormalPosition.left;
		int y = place.rcNormalPosition.top;
		if (x+width > screen.Width())
			x = screen.Width()-width;
		if (y+height > screen.Height())
			y = screen.Height()-height;
		if (x < 0)
			x = 0;
		if (y < 0)
			y = 0;

		place.showCmd = SW_SHOWNORMAL;
		place.rcNormalPosition.left = x;
		place.rcNormalPosition.top = y;
		place.rcNormalPosition.right = x+width;
		place.rcNormalPosition.bottom = y+height;
		wnd->SetWindowPlacement(&place);
	}
	else
	{
		WINDOWPLACEMENT place;
		::ZeroMemory(&place,sizeof(WINDOWPLACEMENT));
		place.length = sizeof(WINDOWPLACEMENT);

		wnd->GetWindowPlacement(&place);
		place.showCmd = m_wndState;
		if (m_wndSize.Width() > 0)
			place.rcNormalPosition = m_wndSize;
		wnd->SetWindowPlacement(&place);
	}
	theWnd->SetAllowResize(true);

	wnd->UpdateWindow();
	::SetCursor(::LoadCursor(NULL,IDC_ARROW));
}

// Store the size of the main window
void FrotzApp::StoreWindowSize(void)
{
	if (IsInfocomV6() == false)
	{
		if (m_pMainWnd != NULL)
		{
			WINDOWPLACEMENT place;
			::ZeroMemory(&place,sizeof(WINDOWPLACEMENT));
			place.length = sizeof(WINDOWPLACEMENT);
			m_pMainWnd->GetWindowPlacement(&place);

			m_wndSize = place.rcNormalPosition;
			m_wndState = place.showCmd;
		}
	}
}

// Store the state of the control bars
void FrotzApp::StoreBarState(bool toolbar, bool statusbar)
{
	m_toolBar = toolbar;
	m_statusBar = statusbar;
}

// Process messages
void FrotzApp::MessagePump(void)
{
	MSG msg;
	if (::PeekMessage(&msg,NULL,0,0,PM_NOREMOVE))
	{ 
		while (::PeekMessage(&msg,NULL,0,0,PM_NOREMOVE))
		{
			if (PumpMessage() == FALSE)
				throw ExitFrotz();
		}
	}
	else
	{
		LONG lIdle = 0;
		BOOL bIdle = TRUE;
		while (AfxGetMainWnd() && bIdle)
			bIdle = CWinApp::OnIdle(lIdle++);
		::WaitMessage();
	}

	if (AfxGetMainWnd() == NULL)
		throw ExitFrotz();
}

// Get the elapsed time for this game
CTimeSpan FrotzApp::GetElapsedTime(void)
{
	return CTime::GetCurrentTime() - m_startTime;
}

// Check if a character is printable
bool FrotzApp::IsValidChar(unsigned short c)
{
	if (c >= ZC_ASCII_MIN && c <= ZC_ASCII_MAX)
		return true;
	if (c >= ZC_LATIN1_MIN && c <= ZC_LATIN1_MAX)
		return true;
	if (c >= 0x100)
		return true;
	return false;
}

// Get a default colour
COLORREF FrotzApp::GetDefaultColour(bool fore)
{
	if (IsInfocomV6())
		return GetColour(fore ? WHITE_COLOUR : BLACK_COLOUR);
	return fore ? m_defaultFore : m_defaultBack;
}

// Get a colour
COLORREF FrotzApp::GetColour(int colour)
{
	// Standard colours
	if ((colour >= BLACK_COLOUR) && (colour <= DARKGREY_COLOUR))
		return m_colours[colour-BLACK_COLOUR];

	// Default colours
	if (colour == 16)
		return m_defaultFore;
	if (colour == 17)
		return m_defaultBack;

	// Non standard colours
	if ((colour >= 18) && (colour < 256))
	{
		if (m_nonStdColours[colour-18] != 0xFFFFFFFF)
			return m_nonStdColours[colour-18];
	}
	return m_colours[0];
}

// Get an index for a non-standard colour
int FrotzApp::GetColourIndex(COLORREF colour)
{
	// Is this a standard colour?
	for (int i = 0; i < 11; i++)
	{
		if (m_colours[i] == colour)
			return i+BLACK_COLOUR;
	}

	// Is this a default colour?
	if (m_defaultFore == colour)
		return 16;
	if (m_defaultBack == colour)
		return 17;

	// Is this colour already in the table?
	for (i = 0; i < NON_STD_COLS; i++)
	{
		if (m_nonStdColours[i] == colour)
			return i+18;
	}

	m_nonStdColours[m_nonStdIndex] = colour;
	int index = m_nonStdIndex+18;

	m_nonStdIndex++;
	if (m_nonStdIndex > NON_STD_COLS)
		m_nonStdIndex = 0;
	return index;
}

// Adjust a foreground colour for the given style
COLORREF FrotzApp::AdjustForeColour(COLORREF colour, int style)
{
	if (h_version != V6)
	{
		if (style & BOLDFACE_STYLE)
		{
			int r = GetRValue(colour)+0x30;
			if (r > 0xFF)
				r = 0xFF;
			int g = GetGValue(colour)+0x30;
			if (g > 0xFF)
				g = 0xFF;
			int b = GetBValue(colour)+0x30;
			if (b > 0xFF)
				b = 0xFF;
			return RGB(r,g,b);
		}
	}
	return colour;
}

// Convert from 5-bit RGB to a true colour
COLORREF FrotzApp::RGB5ToTrue(unsigned short five)
{
	int r = five&0x001F;
	int g = (five&0x03E0)>>5;
	int b = (five&0x7C00)>>10;
	return RGB((r<<3)|(r>>2),(g<<3)|(g>>2),(b<<3)|(b>>2));
}

// Convert from a true colour to 5-bit RGB
unsigned short FrotzApp::TrueToRGB5(COLORREF colour)
{
	int r = GetRValue(colour)>>3;
	int g = GetGValue(colour)>>3;
	int b = GetBValue(colour)>>3;
	return r|(b<<5)|(g<<10);
}

// Rescale colour components so that the colour can be expressed
// exactly in 5-bit RGB
COLORREF FrotzApp::Make5BitColour(COLORREF colour)
{
	int r = GetRValue(colour)>>3;
	int g = GetGValue(colour)>>3;
	int b = GetBValue(colour)>>3;
	return RGB((r<<3)|(r>>2),(g<<3)|(g>>2),(b<<3)|(b>>2));
}

// Get whether to pause on exit
bool FrotzApp::GetExitPause(void)
{
	return m_exitPause;
}

// Set whether to pause on exit
void FrotzApp::SetExitPause(bool pause)
{
	m_exitPause = pause;
}

// Get whether to use the Quetzal save format
bool FrotzApp::GetUseQuetzal(void)
{
	return m_quetzal;
}

// Set whether to use the Quetzal save format
void FrotzApp::SetUseQuetzal(bool quetzal)
{
	m_quetzal = quetzal;
}

// Get whether to notify when entering full screen mode
bool FrotzApp::GetNotifyFullScreen(void)
{
	return m_notifyFull;
}

// Set whether to notify when entering full screen mode
void FrotzApp::SetNotifyFullScreen(bool notify)
{
	m_notifyFull = notify;
}

// Get whether to use fast or slow scrolling
bool FrotzApp::GetFastScrolling(void)
{
	return m_fastScroll;
}

// Get whether to show [More] prompts
bool FrotzApp::GetShowMorePrompts(void)
{
	return m_morePrompts;
}

// Get the left margin size
int FrotzApp::GetLeftMargin(void)
{
	return m_leftMargin;
}

// Get the right margin size
int FrotzApp::GetRightMargin(void)
{
	return m_rightMargin;
}

// Set whether line input is in progress
void FrotzApp::SetLineInput(bool input)
{
	m_lineInput = input;
}

// Set up the Blorb resource file
void FrotzApp::SetBlorbFile(void)
{
	if (m_blorbMap == NULL)
	{
		// Create the filename from the story filename
		CString blorbFileName(story_name);
		{
			int ext = blorbFileName.ReverseFind('.');
			if (ext > 0)
			{
				blorbFileName = blorbFileName.Left(ext);
				blorbFileName += ".blb";

				m_blorbFile = fopen(blorbFileName,"rb");
				if (m_blorbFile != NULL)
				{
					if (bb_create_map(m_blorbFile,&m_blorbMap) != bb_err_None)
						CloseBlorbFile();
				}
			}
		}
	}

	// Attempt to load palette information
	if (m_blorbMap != NULL)
		FrotzGfx::LoadPaletteInfo(m_blorbMap);
}

// Set up the Blorb resource file, given a file handle
void FrotzApp::SetBlorbZCode(FILE* file, long* size)
{
	if (m_blorbMap == NULL)
	{
		if (bb_create_map(file,&m_blorbMap) == bb_err_None)
		{
			// Look for an executable chunk
			bb_result_t result;
			if (bb_load_resource(m_blorbMap,bb_method_FilePos,&result,bb_ID_Exec,0) == bb_err_None)
			{
				unsigned int id = m_blorbMap->chunks[result.chunknum].type;
				if (id == bb_make_id('Z','C','O','D'))
				{
					// If this is a Z-code game, set the file pointer and return
					fseek(file,result.data.startpos,SEEK_SET);
					*size = result.length;
					return;
				}
				else if (id == bb_make_id('G','L','U','L'))
				{
					// Tell the user to use Windows Glulxe instead
					::MessageBox(AfxGetMainWnd()->GetSafeHwnd(),
						CResString(IDS_BLORB_GLULX),CResString(IDS_FATAL),MB_ICONERROR|MB_OK);
					throw AbortFrotz();
				}
			}

			// Tell the user that there was no game in the Blorb file
			::MessageBox(AfxGetMainWnd()->GetSafeHwnd(),
				CResString(IDS_BLORB_NOEXEC),CResString(IDS_FATAL),MB_ICONERROR|MB_OK);
			throw AbortFrotz();
		}

		// This is not a Blorb file, so go back to the start
		fseek(file,0,SEEK_SET);
		*size = -1;
	}
}

// Close the Blorb resource file
void FrotzApp::CloseBlorbFile(void)
{
	if (m_blorbMap)
		bb_destroy_map(m_blorbMap);
	m_blorbMap = NULL;
	if (m_blorbFile)
		fclose(m_blorbFile);
	m_blorbFile = NULL;
}

// Returns true if a Blorb resource file has been loaded
bool FrotzApp::GotBlorbFile(void)
{
	return (m_blorbMap != NULL);
}

// Returns the Blorb map
bb_map_t* FrotzApp::GetBlorbMap(void)
{
	return m_blorbMap;
}

// If true, running one of Infocom's V6 games
bool FrotzApp::IsInfocomV6(void)
{
	switch (story_id)
	{
	case ARTHUR:
	case JOURNEY:
	case SHOGUN:
	case ZORK_ZERO:
		return true;
	}
	return false;
}

// If true, the Tandy header flag should be set
bool FrotzApp::IsTandyBitSet(void)
{
	return m_tandy;
}

// Get the scaling for graphics
int FrotzApp::GetGfxScaling(void)
{
	return m_gfxScale;
}

void FrotzApp::AddSpacechar( unsigned short c = L' ')
{
    m_speechBuffer.Add(c);
}

// Add a character to the scrollback buffer
void FrotzApp::ScrollbackChar(unsigned short c)
{
	m_scrollback.Add(c);

	// TTS
    if(c == 0x0A || c == 0x0D)
    m_speechBuffer.Add(c);
}

// Remove characters from the scrollback buffer
void FrotzApp::ScrollbackRemove(int remove)
{
	int size = m_scrollback.GetSize() - remove;
	if (size < 0)
		size = 0;
	m_scrollback.SetSize(size,8192);
}

// Initialize and run the interpreter
BOOL FrotzApp::InitInstance()
{
	::CoInitialize(NULL);
	AfxInitRichEdit();
    

	// Set the registry key names
	SetRegistryKey("W. Scott Dillman");
	free((void*)m_pszProfileName);
	m_pszProfileName = strdup("FrotzTTS");

	// Read in settings
	ReadSettings();
    InitializeSAPI();
	// Register file types
	RegisterFileTypes();

	// Load international resources, if needed
	LoadInternationalResources();

	// Get a game to run
	if (PromptForGame(true))
	{
		// Run the interpreter until it exits or aborts
		ExitStatus status = Restarting;
		while (status == Restarting)
		{
			Initialize();
			option_save_quetzal = m_quetzal ? 1 : 0;
			status = RunInterpreter();
			reset_memory();
			FrotzGfx::ClearCache();
			FrotzSound::Stop(0);
			CloseBlorbFile();
		}

		// Write out settings
		if (status != Aborted)
			WriteSettings();
	}

	// TTS
    myTTS->release();
	
	FrotzSound::ShutDown();

	::CoUninitialize();
	return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// Message handlers
/////////////////////////////////////////////////////////////////////////////

// Load a new game
void FrotzApp::OnFileNew() 
{
	if (PromptForGame(false))
	{
		StoreWindowSize();
		throw RestartFrotz();
	}
}

// Load a saved game
void FrotzApp::OnFileOpen() 
{
	if (m_lineInput)
		theWnd->InputString("restore\r");
}

void FrotzApp::OnUpdateFileOpen(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_lineInput ? TRUE : FALSE);
}

// Save the current game
void FrotzApp::OnFileSave() 
{
	if (m_lineInput)
		theWnd->InputString("save\r");
}

void FrotzApp::OnUpdateFileSave(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_lineInput ? TRUE : FALSE);
}

void FrotzApp::OnFileStop()
{
	if (istream_replay)
		replay_close();
}

void FrotzApp::OnUpdateFileStop(CCmdUI *pCmdUI)
{
	pCmdUI->Enable(istream_replay ? TRUE : FALSE);
}

void FrotzApp::OnViewOptions()
{
	OptionsDialog dialog(AfxGetMainWnd());

	// Show the current display settings
	dialog.m_propFontName = m_propFontName;
	dialog.m_fixedFontName = m_fixedFontName;
	dialog.m_fontSize.Format("%d",m_fontSize);
	dialog.m_v6Scale.Format("%d",m_v6scale);
	dialog.m_textColour.SetCurrentColour(m_defaultFore);
	dialog.m_backColour.SetCurrentColour(m_defaultBack);
	dialog.m_fastScroll = m_fastScroll;
	dialog.m_morePrompts = m_morePrompts;
	dialog.m_leftMargin = m_leftMargin;
	dialog.m_rightMargin = m_rightMargin;

	// Show the current interpreter settings
	dialog.m_interpreter = h_interpreter_number-1;
	dialog.m_reportErrors = err_report_mode;
	dialog.m_ignore = option_ignore_errors ? TRUE : FALSE;
	dialog.m_expand = option_expand_abbreviations ? TRUE : FALSE;
	dialog.m_tandy = m_tandy;

	// Show the current startup settings
	dialog.m_register = m_register;

    // TTS
    dialog.m_Volume = m_Volume;
    dialog.m_voiceToken = m_voiceToken;
    dialog.m_voiceName = m_voiceName;
    dialog.m_Rate = m_Rate;
    dialog.m_bDialogSpeak = m_bDialogSpeak;

	// Show the options dialog
	if (dialog.DoModal() == IDOK)
	{
		// Get the new font size
		int fontSize = 0;
		sscanf(dialog.m_fontSize,"%d",&fontSize);
		if (fontSize < 8)
			fontSize = 8;
		if (fontSize > 28)
			fontSize = 28;

		// Check if the font settings have changed
		bool fontUpdate = false;
		if (m_propFontName != dialog.m_propFontName)
			fontUpdate = true;
		if (m_fixedFontName != dialog.m_fixedFontName)
			fontUpdate = true;
		if (m_fontSize != fontSize)
			fontUpdate = true;

		// Check if the margin settings have changed
		bool marginUpdate = false;
		if (m_leftMargin != dialog.m_leftMargin)
			marginUpdate = true;
		if (m_rightMargin != dialog.m_rightMargin)
			marginUpdate = true;

		// Update the display settings
		sscanf(dialog.m_v6Scale,"%d",&m_v6scale);
		m_defaultFore = Make5BitColour(dialog.m_textColour.GetCurrentColour());
		m_defaultBack = Make5BitColour(dialog.m_backColour.GetCurrentColour());
		hx_fore_colour = TrueToRGB5(GetDefaultColour(true));
		hx_back_colour = TrueToRGB5(GetDefaultColour(false));
		m_fastScroll = dialog.m_fastScroll ? true : false;
		m_morePrompts = dialog.m_morePrompts ? true : false;
		m_leftMargin = dialog.m_leftMargin;
		m_rightMargin = dialog.m_rightMargin;

		// Update the interpreter settings
		h_interpreter_number = dialog.m_interpreter+1;
		err_report_mode = dialog.m_reportErrors;
		option_ignore_errors = dialog.m_ignore ? 1 : 0;
		option_expand_abbreviations = dialog.m_expand ? 1 : 0;
		m_tandy = dialog.m_tandy ? true : false;

		// Update the startup settings
		m_register = dialog.m_register ? true : false;

		// Update the font settings, if changed
		if (fontUpdate)
		{
			m_propFontName = dialog.m_propFontName;
			m_fixedFontName = dialog.m_fixedFontName;
			m_fontSize = fontSize;

			if (theWnd->CreateFonts() == false)
				throw AbortFrotz();

			FrotzWnd::TextSettings savedText = theWnd->GetTextSettings();
			theWnd->ApplyTextSettings(FrotzWnd::TextSettings(0,FIXED_WIDTH_FONT));
			h_font_width = (zbyte)theWnd->GetCharWidth('0');
			h_font_height = (zbyte)theWnd->GetFontHeight();
			theWnd->ApplyTextSettings(savedText);
		}
		if (fontUpdate || marginUpdate)
			theWnd->ResizeDisplay();

		// Update the current foreground and background colours
		if (theWnd->GetTextSettings().foreDefault)
			theWnd->GetTextSettings().fore = theApp.GetDefaultColour(true);
		if (theWnd->GetTextSettings().backDefault)
			theWnd->GetTextSettings().back = theApp.GetDefaultColour(false);
		theWnd->ApplyTextSettings();

        if (h_version == V3)
        {
            if (IsTandyBitSet())
                h_config |= CONFIG_TANDY;
            else
                h_config &= ~CONFIG_TANDY;

        }

        // Write changes back into the game header
        if (zmp != NULL)
        {
            if (h_version == V3)
                SET_BYTE(H_CONFIG,h_config)
                if (h_version >= V4)
                    SET_BYTE(H_INTERPRETER_NUMBER,h_interpreter_number)
                    if (h_version >= V5)
                    {
                        set_header_extension(HX_FORE_COLOUR,hx_fore_colour);
                        set_header_extension(HX_BACK_COLOUR,hx_back_colour);
                    }
        }
        // TTS
        m_Volume = dialog.m_Volume;
        m_Rate = dialog.m_Rate;
        m_voiceName = dialog.m_voiceName;
        m_voiceToken = dialog.m_voiceToken;
        m_bDialogSpeak = dialog.m_bDialogSpeak;

        myTTS->ChangeVoice(m_voiceToken.GetBuffer(),m_Rate,m_Volume);

        DWORD dTest = 0;

        if(m_bDialogSpeak)
        {
          //  OutputDebugString("TRUE");
          //   dTest = myTTS->addFlag(IF_DIALOG_SPEECH);
        }
        else
        {
         //   dTest = myTTS->removeFlag(IF_DIALOG_SPEECH);
        }

        myTTS->removeFlag(IF_DIALOG_SPEECH);

		/*
        char cBuf[100];
        sprintf(cBuf,"%d",myTTS->getFlags());
        OutputDebugString(cBuf);
		*/

        if(myTTS->checkFlags(IF_DIALOG_SPEECH))
            OutputDebugString("Flag Checked");
        
	}

	// Update any pending input line
	theWnd->InputType(FrotzWnd::Input::Reset);
}

void FrotzApp::OnViewScrollback()
{
	// Convert the scrollback buffer to ASCII
	int len = m_scrollback.GetSize();
	char* text = new char[len+1];
	::ZeroMemory(text,len+1);
	if (::WideCharToMultiByte(CP_ACP,0,m_scrollback.GetData(),len,text,len,NULL,NULL) > 0)
	{
		ScrollbackDialog dialog(text);
		dialog.DoModal();
	}
	delete[] text;
}

void FrotzApp::OnAppAbout()
{
	AboutDialog dialog(AfxGetMainWnd());
	dialog.DoModal();
}

void FrotzApp::OnLinksArchive()
{
	::ShellExecute(0,NULL,"http://www.ifarchive.org/",NULL,NULL,SW_SHOWNORMAL);
}

void FrotzApp::OnLinksZcode()
{
	::ShellExecute(0,NULL,
		"http://www.ifarchive.org/indexes/if-archiveXgamesXzcode.html",
		NULL,NULL,SW_SHOWNORMAL);
}

void FrotzApp::OnLinksBaf()
{
	::ShellExecute(0,NULL,"http://wurb.com/if/",NULL,NULL,SW_SHOWNORMAL);
}

void FrotzApp::OnLinksIFComp()
{
	::ShellExecute(0,NULL,"http://www.ifcomp.org/",NULL,NULL,SW_SHOWNORMAL);
}

void FrotzApp::OnLinksArtShow()
{
	::ShellExecute(0,NULL,"http://members.aol.com/iffyart/",NULL,NULL,SW_SHOWNORMAL);
}

void FrotzApp::OnLinksInform()
{
	::ShellExecute(0,NULL,"http://www.inform-fiction.org/",NULL,NULL,SW_SHOWNORMAL);
}

void FrotzApp::OnLinksFaq()
{
	::ShellExecute(0,NULL,"http://www.firthworks.com/roger/",NULL,NULL,SW_SHOWNORMAL);
}

/////////////////////////////////////////////////////////////////////////////
// About dialog
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNAMIC(AboutDialog, CDialog)
AboutDialog::AboutDialog(CWnd* pParent) : CDialog(AboutDialog::IDD, pParent)
{
}

AboutDialog::~AboutDialog()
{
}

void AboutDialog::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	DDX_Control(pDX, IDC_LOGO, m_logo);
}

BEGIN_MESSAGE_MAP(AboutDialog, CDialog)
END_MESSAGE_MAP()

BOOL AboutDialog::OnInitDialog()
{
	CDialog::OnInitDialog();

	m_logo.SetBitmap(::LoadBitmap(AfxGetInstanceHandle(),MAKEINTRESOURCE(IDB_FROTZ)));
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Options dialog
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNAMIC(OptionsDialog, CDialog)

OptionsDialog::OptionsDialog(CWnd* pParent)	: CDialog(OptionsDialog::IDD, pParent)
, m_textColour(FALSE), m_backColour(FALSE), m_fastScroll(FALSE), m_morePrompts(FALSE)
, m_leftMargin(0), m_rightMargin(0)
, m_interpreter(0), m_reportErrors(0)
, m_expand(FALSE), m_tandy(FALSE), m_ignore(FALSE), m_register(FALSE)
,m_Volume(100),m_Rate(0),m_bDialogSpeak(TRUE)
{
}

OptionsDialog::~OptionsDialog()
{
}

void OptionsDialog::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	DDX_Control(pDX, IDC_PROP_FONT, m_propFont);
	DDX_Control(pDX, IDC_FIXED_FONT, m_fixedFont);
	DDX_CBString(pDX, IDC_FONT_SIZE, m_fontSize);
	DDX_CBString(pDX, IDC_V6_SCALE, m_v6Scale);
	DDX_Check(pDX, IDC_FAST_SCROLL, m_fastScroll);
	DDX_Check(pDX, IDC_MORE_PROMPT, m_morePrompts);
	DDX_Text(pDX, IDC_LEFT_MARGIN, m_leftMargin);
	DDX_Text(pDX, IDC_RIGHT_MARGIN, m_rightMargin);
	DDX_CBIndex(pDX, IDC_TERP_NUMBER, m_interpreter);
	DDX_CBIndex(pDX, IDC_ERRORS, m_reportErrors);
	DDX_Check(pDX, IDC_EXPAND, m_expand);
	DDX_Check(pDX, IDC_TANDY, m_tandy);
	DDX_Check(pDX, IDC_IGNORE_RUNTIME, m_ignore);
	DDX_Check(pDX, IDC_REGISTER_FILETYPES, m_register);
    DDX_Check(pDX, IDC_DIALOGSPEAK, m_bDialogSpeak);

    // Get voice name
    DDX_Text(pDX,IDC_COMBO_VOICES,m_voiceName);

}

BEGIN_MESSAGE_MAP(OptionsDialog, CDialog)
	ON_WM_HELPINFO()
    ON_BN_CLICKED(IDC_DIALOGSPEAK, OnBnClickedDialogspeek)
END_MESSAGE_MAP()

BOOL OptionsDialog::OnInitDialog()
{
	CDialog::OnInitDialog();

	// Get all the possible fonts
	CDC* dc = GetDC();
	LOGFONT font;
	::ZeroMemory(&font,sizeof(LOGFONT));
	font.lfCharSet = ANSI_CHARSET;
	::EnumFontFamiliesEx(dc->GetSafeHdc(),&font,(FONTENUMPROC)ListFonts,(LPARAM)this,0);
	ReleaseDC(dc);

	// Initialize the font controls
	if (m_propFont.SelectString(-1,m_propFontName) == CB_ERR)
		m_propFont.SetCurSel(0);
	if (m_fixedFont.SelectString(-1,m_fixedFontName) == CB_ERR)
		m_fixedFont.SetCurSel(0);

	// Initialize the colour controls
	m_textColour.SubclassDlgItem(IDC_TEXT_COLOUR,this);
	m_backColour.SubclassDlgItem(IDC_BACK_COLOUR,this);


    // Initialize the speech controls
    m_voices.SubclassDlgItem(IDC_COMBO_VOICES,this);
    m_rateSlide.SubclassDlgItem(IDC_RATE,this);
    m_volumeSlide.SubclassDlgItem(IDC_VOLUME,this);

    HRESULT hr = SpInitTokenComboBox( ::GetDlgItem( m_hWnd,IDC_COMBO_VOICES ), SPCAT_VOICES );
    m_voices.SelectString(0,m_voiceName);

    m_rateSlide.SetRange(SPMIN_RATE,SPMAX_RATE,TRUE);
    m_volumeSlide.SetRange(SPMIN_VOLUME,SPMAX_VOLUME,TRUE);

    m_rateSlide.SetPageSize(5);
    m_rateSlide.SetPos(m_Rate);

    m_volumeSlide.SetPageSize(10);
    m_volumeSlide.SetPos(m_Volume);

	return TRUE;
}

// Called when the dialog has been closed with the OK button
void OptionsDialog::OnOK()
{
	// Read the font controls
	m_propFont.GetWindowText(m_propFontName);
	m_fixedFont.GetWindowText(m_fixedFontName);

    // Get voice token
    ISpObjectToken* pToken = SpGetCurSelComboBoxToken( ::GetDlgItem(m_hWnd, IDC_COMBO_VOICES ) );
    WCHAR * ppszCoMemTokenId[1];

    if(pToken)
    {
        pToken->GetId(&ppszCoMemTokenId[0]);
        m_voiceToken = ppszCoMemTokenId[0];
        CoTaskMemFree(ppszCoMemTokenId[0]);
    }

    // Get slider data
    m_Rate = m_rateSlide.GetPos();
    m_Volume = m_volumeSlide.GetPos();

	CDialog::OnOK();
}

HWND WINAPI AfxHtmlHelp(HWND hWnd, LPCTSTR szHelpFilePath, UINT nCmd, DWORD_PTR dwData);

BOOL OptionsDialog::OnHelpInfo(HELPINFO* pHelpInfo)
{
	static DWORD helpIds[] =
	{
		IDC_PROP_FONT,1,
		IDC_FIXED_FONT,2,
		IDC_FONT_SIZE,3,
		IDC_V6_SCALE,4,
		IDC_TEXT_COLOUR,5,
		IDC_BACK_COLOUR,6,
		IDC_TERP_NUMBER,7,
		IDC_ERRORS,8,
		IDC_IGNORE_RUNTIME,9,
		IDC_EXPAND,10,
		IDC_TANDY,11,
		IDC_FAST_SCROLL,12,
		IDC_LEFT_MARGIN,13,
		IDC_RIGHT_MARGIN,14,
		IDC_MORE_PROMPT,15,
		0,0
	};

	if (pHelpInfo->iContextType == HELPINFO_WINDOW)
	{
		// Is there a help topic for this control?
		DWORD* id = helpIds;
		while (*id != 0)
		{
			if (pHelpInfo->iCtrlId == *id)
			{
				CString helpFile(AfxGetApp()->m_pszHelpFilePath);
				helpFile.Append("::/options.txt");

				// Show the help popup
				AfxHtmlHelp((HWND)pHelpInfo->hItemHandle,helpFile,
					HH_TP_HELP_WM_HELP,(DWORD_PTR)helpIds);
				return TRUE;
			}
			id += 2;
		}
	}
	return TRUE;
}

// Called when enumerating fonts, populates the font drop down lists in the dialog
int CALLBACK OptionsDialog::ListFonts(ENUMLOGFONTEX *font, NEWTEXTMETRICEX *metric, DWORD fontType, LPARAM param)
{
	OptionsDialog* dialog = (OptionsDialog*)param;

	// Only allow scaleable fonts (TrueType, etc.)
	bool allow = false;
	if (fontType & TRUETYPE_FONTTYPE)
		allow = true;
	else if (!(fontType & RASTER_FONTTYPE))
		allow = ((metric->ntmTm.ntmFlags & NTM_PS_OPENTYPE|NTM_TT_OPENTYPE|NTM_TYPE1) != 0);

	if (allow)
	{
		dialog->m_propFont.AddString(font->elfLogFont.lfFaceName);
		if (font->elfLogFont.lfPitchAndFamily & FIXED_PITCH)
			dialog->m_fixedFont.AddString(font->elfLogFont.lfFaceName);
	}
	return 1;
}

/////////////////////////////////////////////////////////////////////////////
// Scrollback dialog
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNAMIC(ScrollbackDialog, CDialog)
ScrollbackDialog::ScrollbackDialog(const char* text, CWnd* pParent)
: m_textTop(0), CDialog(ScrollbackDialog::IDD, pParent), m_text(text)
{
}

ScrollbackDialog::~ScrollbackDialog()
{
}

void ScrollbackDialog::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
}

BEGIN_MESSAGE_MAP(ScrollbackDialog, CDialog)
    ON_WM_SIZE()
    ON_BN_CLICKED(IDC_COPY, OnCopy)
END_MESSAGE_MAP()

BOOL ScrollbackDialog::OnInitDialog()
{
    CDialog::OnInitDialog();

    // Subclass the rich edit text control
    if (m_edit.SubclassDlgItem(IDC_TEXT,this) == FALSE)
        return FALSE;

    // Get the relative position of the top of the text control
    CRect size;
    m_edit.GetWindowRect(size);
    ScreenToClient(size);
    m_textTop = size.top;

    // Change the window icon
    SetIcon(::LoadIcon(AfxGetInstanceHandle(),MAKEINTRESOURCE(IDI_INFOCOM)),TRUE);

    // Get the size of the display
    int w = ::GetSystemMetrics(SM_CXSCREEN);
    int h = ::GetSystemMetrics(SM_CYSCREEN);

    // Get the size of the main window

    AfxGetMainWnd()->GetWindowRect(size);

    // Resize the dialog, but no bigger than the display
    if ((size.Width() > w) && (size.Height() > h))
        MoveWindow(0,0,w,h);
    else
        MoveWindow(size);

    // Set the control to format the text so that it fits
    // into the window
    m_edit.SetTargetDevice(NULL,0);

    // Set the background colour
    m_edit.SetBackgroundColor(FALSE,GetSysColor(COLOR_3DFACE));

    // Put the text into the control
    m_edit.SetWindowText(m_text);

    // Scroll the control to the end of the text
    m_edit.SetSel(-1,-1);
    m_edit.SendMessage(EM_SCROLLCARET);

    return TRUE;
}

void ScrollbackDialog::OnSize(UINT nType, int cx, int cy)
{
    CDialog::OnSize(nType,cx,cy);

    // Resize the edit control
    if (m_edit.GetSafeHwnd() != NULL)
        m_edit.SetWindowPos(NULL,0,m_textTop,cx,cy-m_textTop,SWP_NOZORDER);
}

void ScrollbackDialog::OnCopy()
{
    m_edit.Copy();
}

/////////////////////////////////////////////////////////////////////////////
// File dialog for loading a new Z-code game
/////////////////////////////////////////////////////////////////////////////


IMPLEMENT_DYNAMIC(GameFileDialog, CFileDialog)

GameFileDialog::GameFileDialog(LPCTSTR lpszFileName, CWnd* pParentWnd)
: CFileDialog(TRUE,NULL,lpszFileName,OFN_HIDEREADONLY|OFN_ENABLETEMPLATE,CResString(IDS_ZCODE_FILTER),pParentWnd),
	m_title(IDS_ZCODE_TITLE)
{
    
	m_ofn.lpstrTitle = m_title;
	m_ofn.lpTemplateName = MAKEINTRESOURCE(IDD_LOADGAME);
	m_quetzal = ((FrotzApp*)AfxGetApp())->GetUseQuetzal();
    
}

GameFileDialog::~GameFileDialog()
{
}

BEGIN_MESSAGE_MAP(GameFileDialog, CFileDialog)
	ON_BN_CLICKED(IDC_QUETZAL, OnUseQuetzal)
END_MESSAGE_MAP()

void GameFileDialog::OnUseQuetzal()
{
	m_quetzal = ((CButton*)GetDlgItem(IDC_QUETZAL))->GetCheck() == BST_CHECKED;
}

void GameFileDialog::OnInitDone()
{
	((CButton*)GetDlgItem(IDC_QUETZAL))->SetCheck(m_quetzal ? TRUE : FALSE);
	CFileDialog::OnInitDone();
}

INT_PTR GameFileDialog::DoModal()
{
    
	INT_PTR result = CFileDialog::DoModal();
	if (result == IDOK)
    {
        ((FrotzApp*)AfxGetApp())->SetUseQuetzal(m_quetzal);
    }
		
	return result;
}

/////////////////////////////////////////////////////////////////////////////
// String class, loads string from resources in constructor
/////////////////////////////////////////////////////////////////////////////

CResString::CResString(UINT id)
{
	LoadString(id);
}

bool FrotzApp::getSpeechState(void)
{
    return m_speech;
}
void FrotzApp::OnScratch()
{
    WString sText;
    myTTS->getList( sText,CifTTS::if_words_transient, NULL, NULL );

    for(unsigned int t = 0 ; t < sText.length(); t++)
        theWnd->InputUnicode(sText[t]);
}
void FrotzApp::OnVoices()
{
    WString sText;
    myTTS->getList(sText, CifTTS::if_voices, NULL, NULL );
   if(!sText.empty())
   {
        std::string sTmp;
        myTTS->GetVoice(sTmp,m_Rate,m_Volume);

        m_voiceToken = sTmp.c_str();
        m_voiceName = myTTS->UC8_To_ASCII(sText).c_str();
   }
}
void FrotzApp::OnWebHelp()
{
    myTTS->OpenUsingShellExecute("http://www.binaryrevelations.com/iftts/modules.php?name=Forums&file=viewforum&f=2",0,NULL);
}
void FrotzApp::OnUpdate()
{
    String sResult;
    String sMsg;
    int uDif = 0;
    UINT itype = MB_OK;


    if(!myTTS->getVersion(uDif,sResult))
    {
     
        Speak(L"There was a problem getting version info from the web site.",SPF_ASYNC|SPF_PURGEBEFORESPEAK);
        ::MessageBox(GetActiveWindow(),"There was a problem getting version info from the web site.","Error",MB_OK);
    }
    else
    {

        if( uDif  == 0 )
        {
            // Up to date
            sMsg = ". \nLooks like you are up to date. ";
            sMsg += "No update needed. Hit escape to close dialog.";
            
        }
        else if( uDif < 0 )
        {
            // Beta
            sMsg = ". \nLooks like you are running a advanced beta release. ";
            sMsg += "No update needed. Hit escape to close dialog.";

        }
        else
        {
            // Old version
            sMsg = ". \nThere is a new version available. ";
            sMsg += "Hit enter to go to download page.";
            itype = MB_YESNO;
        }

        String s = "The latest version of the program is: ";
        s += sResult; 
        s += " " + sMsg;
        Speak(static_cast<WString>(s),SPF_ASYNC|SPF_PURGEBEFORESPEAK);
        UINT hres = ::MessageBox(GetActiveWindow(),s.c_str(),"Update",itype);
        myTTS->Speak(L"",SPF_PURGEBEFORESPEAK);

        if(hres == IDYES)
        {
            myTTS->OpenUsingShellExecute("http://www.binaryrevelations.com/iftts/modules.php?name=Downloads&d_op=viewdownload&cid=2",0,NULL);
        }

    }
}
void FrotzApp::OnAdd()
{
    GameFileDialog dialog(m_filename,AfxGetMainWnd());
    if (dialog.DoModal() == IDOK)
    {
        // ok add this game to the list
        CString sfilename = dialog.GetPathName();

        WString sText2;
        String sName = sfilename.GetBuffer(0);
        myTTS->getInput(sText2,"What is the title of this new game.");
        String sTitle= sText2;

        if(myTTS->addNewGame(sName,sTitle) == false)
        {
            Speak(L"Game could not be added to supported list. Hit enter to Continue.",SPF_ASYNC|SPF_PURGEBEFORESPEAK);
            ::MessageBox(GetActiveWindow(),"Game could not be added to supported list. Hit enter to Continue.","Error",MB_OK);

        }
    }

}
void FrotzApp::OnRepeat()
{
    myTTS->Repeat();
}
void FrotzApp::OnWordlist()
{
    WString sText;
    myTTS->getList( sText,CifTTS::if_words_static, NULL, NULL );

    // ADD TEXT TO SCREEN
    for(unsigned int t = 0 ; t < sText.length(); t++)
          theWnd->InputUnicode(sText[t]);
}
void FrotzApp::OnAppSpeech()
{
    m_speech = !m_speech;
    

    if(!m_speech)
    {
        myTTS->Speak(L"",SPF_PURGEBEFORESPEAK);
    }
}

void OptionsDialog::OnBnClickedDialogspeek()
{
    
}
