/////////////////////////////////////////////////////////////////////////////
// Windows Frotz
// Frotz sound classes
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "FrotzSound.h"
#include "FrotzWnd.h"
#include "mpsndsys.h"
#include <math.h>

extern "C"
{
#include "blorblow.h"
void end_of_sound(zword routine);
}

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

FrotzSound::FrotzSound(int sound, unsigned short eos, BYTE* data, int length, bool del)
{
	m_sound = sound;
	m_eos = eos;
	m_data = data;
	m_delete = del;
	m_length = length;
}

FrotzSound::~FrotzSound()
{
	if (m_delete)
		delete[] m_data;
}

// Get the number of the sound
int FrotzSound::GetSoundNumber(void)
{
	return m_sound;
}

// Get the Z-code to routine at the end
unsigned short FrotzSound::GetEndRoutine(void)
{
	return m_eos;
}

// Get the sound volume in decibels
int FrotzSound::GetDecibelVolume(int volume)
{
	if (volume < 1)
		volume = 1;
	else if (volume > 8)
		volume = 8;

	// Convert volume from the range 1 to 8 into decibels
	double decibels = -10.0 * (log(8.0/(double)volume) / log(2.0));
	if (decibels > 0.0)
		decibels = 0.0;
	else if (decibels < -100.0)
		decibels = -100.0;
	return (int)decibels;
}

// Initialize the sound engine
bool FrotzSound::Init(FrotzWnd* wnd)
{
	if (CDSoundEngine::GetSoundEngine().Initialize())
	{
		wnd->SetTimer(FrotzWnd::SoundTimer,100,NULL);
		return true;
	}

	::MessageBox(AfxGetMainWnd()->GetSafeHwnd(),
		CResString(IDS_FAIL_DIRECTSOUND),CResString(IDS_FROTZ),MB_ICONWARNING|MB_OK);
	return false;
}

// Stop the sound engine
void FrotzSound::ShutDown(void)
{
	Stop(0);
	FrotzSoundMOD::CloseMODPlug();
	CDSoundEngine::GetSoundEngine().Destroy();
}

// Start playing a sound
void FrotzSound::Play(int sound, bb_map_t* map, int repeat, int volume, unsigned short eos)
{
	if (CDSoundEngine::GetSoundEngine().GetStatus() == CDSoundEngine::STATUS_READY)
	{
		bb_result_t result;
		if (bb_load_resource_snd(map,bb_method_Memory,&result,sound,NULL) == bb_err_None)
		{
			BYTE* data = (BYTE*)result.data.ptr;
			int length = result.length;
			unsigned int id = map->chunks[result.chunknum].type;

			// Look for a recognized format
			if (id == bb_make_id('F','O','R','M'))
			{
				delete m_soundEffect;
				m_soundEffect = new FrotzSoundAIFF(sound,eos,data,length);
				m_soundEffect->Play(repeat,volume);
			}
			else if (id == bb_make_id('M','O','D',' '))
			{
				delete m_soundMusic;
				m_soundMusic = new FrotzSoundMOD(sound,eos,data,length,false);
				m_soundMusic->Play(repeat,volume);
			}
			else if (id == bb_make_id('S','O','N','G'))
			{
				delete m_soundMusic;

				MODBuilder build;
				m_soundMusic = build.CreateMODFromSONG(sound,eos,data,length,map);
				if (m_soundMusic)
					m_soundMusic->Play(repeat,volume);
			}
		}
	}
}

// Stop playing a sound
void FrotzSound::Stop(int sound)
{
	if (m_soundEffect != NULL)
	{
		if ((sound == 0) || (m_soundEffect->GetSoundNumber() == sound))
		{
			delete m_soundEffect;
			m_soundEffect = NULL;
		}
	}
	if (m_soundMusic != NULL)
	{
		if ((sound == 0) || (m_soundMusic->GetSoundNumber() == sound))
		{
			delete m_soundMusic;
			m_soundMusic = NULL;
		}
	}
}

// Called to check if a sound has finished
void FrotzSound::Timer(void)
{
	if (m_soundEffect != NULL)
	{
		if (m_soundEffect->IsPlaying() == false)
		{
			zword routine = m_soundEffect->GetEndRoutine();
			delete m_soundEffect;
			m_soundEffect = NULL;
			end_of_sound(routine);
		}
	}
	if (m_soundMusic != NULL)
	{
		if (m_soundMusic->IsPlaying() == false)
		{
			zword routine = m_soundMusic->GetEndRoutine();
			delete m_soundMusic;
			m_soundMusic = NULL;
			end_of_sound(routine);
		}
	}
}

FrotzSound* FrotzSound::m_soundEffect = NULL;
FrotzSound* FrotzSound::m_soundMusic = NULL;

/////////////////////////////////////////////////////////////////////////////
// Class for AIFF sounds
/////////////////////////////////////////////////////////////////////////////

FrotzSoundAIFF::FrotzSoundAIFF(int sound, unsigned short eos, BYTE* data, int length) :
	FrotzSound(sound,eos,data,length,false)
{
	m_renderPtr = NULL;
	m_renderMin = NULL;
	m_renderMax = NULL;
	m_duration = 0;
}

FrotzSoundAIFF::~FrotzSoundAIFF()
{
	RemoveFromList();
}

bool FrotzSoundAIFF::Play(int repeat, int volume)
{
	SampleData data;
	if (GetSampleData(data) == false)
		return false;

	// Create a buffer
	if (CreateBuffer(data.channels,(int)data.rate,data.bits) == false)
		return false;

	// Set the duration of the sample
	if (repeat > 0)
		m_duration = (DWORD)ceil((data.samples * repeat * 1000.0) / data.rate);
	else
		m_duration = -1;

	// Set up the current position for rendering wave data
	m_renderPtr = data.data;
	m_renderMin = m_renderPtr;
	m_renderMax = data.data + ((data.bits>>3)*data.samples*data.channels);

	// Fill the buffer with sample data
	m_repeat = (repeat < 0) ? -1 : repeat - 1;
	if (FillBuffer(GetBufferSize()) == false)
		return false;

	// Set the volume for the buffer
	SetBufferVolume(GetDecibelVolume(volume) * 100L);

	// Start the buffer playing
	return PlayBuffer();
}

bool FrotzSoundAIFF::IsPlaying(void)
{
	return IsBufferPlaying();
}

// Write sample data into the supplied PCM sample buffers
void FrotzSoundAIFF::WriteSampleData(unsigned char* sample, int len)
{
	switch (m_Format.wBitsPerSample)
	{
	case 8:
		{
			for (int i = 0; i < len; i++)
			{
				if (CheckRenderPtr())
					*(sample++) = (unsigned char)((*(m_renderPtr++)) ^ 0x80);
				else
					*(sample++) = 0x80;
			}
		}
		break;
	case 16:
		{
			unsigned short* writePtr = (unsigned short*)sample;
			for (int i = 0; i < len; i += 2)
			{
				if (CheckRenderPtr())
				{
					*(writePtr++) = (unsigned short)
						(((*m_renderPtr)<<8) | *(m_renderPtr+1));
					m_renderPtr += 2;
				}
				else
					*(writePtr++) = 0;
			}
		}
		break;
	}
}

// Test that the current point into the AIFF buffer is valid
bool FrotzSoundAIFF::CheckRenderPtr(void)
{
	if (m_renderPtr >= m_renderMax)
	{
		// Fail if this is the end of the last repeat
		if (m_repeat == 0)
			return false;

		// If not looping forever, decrement the repeat counter
		if (m_repeat > 0)
			m_repeat--;

		// Reset the pointer
		m_renderPtr = m_renderMin;
	}
	return true;
}

// Check if the sound has finished playing
bool FrotzSoundAIFF::IsSoundOver(DWORD tick)
{
	if (m_Playing == false)
		return true;

	// Check if sound is playing forever
	if (m_duration < 0)
		return false;
	return (tick > m_StartTime + m_duration);
}

// Get a type identifier for the sound
int FrotzSoundAIFF::GetType(void)
{
	return (int)'A';
}

// Get details of the sample
bool FrotzSoundAIFF::GetSampleData(SampleData& data)
{
	if (m_data == NULL)
		return false;

	// Check for AIFF header
	if (strncmp((char*)m_data,"FORM",4) != 0)
		return false;
	if (strncmp((char*)(m_data+8),"AIFF",4) != 0)
		return false;

	// Find the COMM chunk
	BYTE* chunk = FindChunk("COMM");
	if (chunk == NULL)
		return false;

	// Read in details of the sample
	data.channels = ReadShort(chunk);
	data.samples = ReadLong(chunk+2);
	data.bits = ReadShort(chunk+6);
	data.rate = ReadExtended(chunk+8);

	// Find the SSND chunk
	data.data = FindChunk("SSND");
	if (data.data == NULL)
		return false;

	// Set up default repeat information
	data.repeat1 = 0;
	data.repeat2 = 0;

	// Find the INST chunk
	chunk = FindChunk("INST");
	if (chunk != NULL)
	{
		// Look for a sustain loop
		if (ReadShort(chunk+8) != 0)
		{
			unsigned short mark1 = ReadShort(chunk+10);
			unsigned short mark2 = ReadShort(chunk+12);

			// Find the MARK chunk
			chunk = FindChunk("MARK");
			if (chunk != NULL)
			{
				unsigned short markers = ReadShort(chunk);

				BYTE* mark = chunk+2;
				for (int i = 0; i < markers; i++)
				{
					unsigned short id = ReadShort(mark);
					unsigned long pos = ReadLong(mark+2);

					if (id == mark1)
						data.repeat1 = pos;
					else if (id == mark2)
						data.repeat2 = pos;

					unsigned char nameLen = mark[6]+1;
					if ((nameLen % 2) == 1)
						nameLen++;
					mark += nameLen+6;
				}
			}
		}
	}

	return true;
}

// Find an AIFF chunk
BYTE* FrotzSoundAIFF::FindChunk(LPCTSTR chunk)
{
	BYTE* data = m_data+12;
	while (true)
	{
		if (strncmp((char*)data,chunk,4) == 0)
			return data+8;

		// Move to the next chunk
		data += ReadLong(data+4)+8;

		if ((data - m_data) > m_length-8)
			break;
	}
	return NULL;
}

unsigned short FrotzSoundAIFF::ReadShort(const unsigned char *bytes)
{
	return (unsigned short)(
		((unsigned short)(bytes[0] & 0xFF) << 8) |
		((unsigned short)(bytes[1] & 0xFF)));
}

unsigned long FrotzSoundAIFF::ReadLong(const unsigned char *bytes)
{
	return (unsigned long)(
		((unsigned long)(bytes[0] & 0xFF) << 24) |
		((unsigned long)(bytes[1] & 0xFF) << 16) |
		((unsigned long)(bytes[2] & 0xFF) << 8) |
		((unsigned long)(bytes[3] & 0xFF)));
}

/* 
 * Copyright (C) 1988-1991 Apple Computer, Inc.
 * All rights reserved.
 *
 * Machine-independent I/O routines for IEEE floating-point numbers.
 *
 * NaN's and infinities are converted to HUGE_VAL or HUGE, which
 * happens to be infinity on IEEE machines.  Unfortunately, it is
 * impossible to preserve NaN's in a machine-independent way.
 * Infinities are, however, preserved on IEEE machines.
 *
 * These routines have been tested on the following machines:
 *    Apple Macintosh, MPW 3.1 C compiler
 *    Apple Macintosh, THINK C compiler
 *    Silicon Graphics IRIS, MIPS compiler
 *    Cray X/MP and Y/MP
 *    Digital Equipment VAX
 *
 * Implemented by Malcolm Slaney and Ken Turkowski.
 *
 * Malcolm Slaney contributions during 1988-1990 include big- and little-
 * endian file I/O, conversion to and from Motorola's extended 80-bit
 * floating-point format, and conversions to and from IEEE single-
 * precision floating-point format.
 *
 * In 1991, Ken Turkowski implemented the conversions to and from
 * IEEE double-precision format, added more precision to the extended
 * conversions, and accommodated conversions involving +/- infinity,
 * NaN's, and denormalized numbers.
 */
#define UnsignedToFloat(u) (((double)((long)(u - 2147483647L - 1))) + 2147483648.0)
double FrotzSoundAIFF::ReadExtended(const unsigned char *bytes)
{
	double f;
	int expon;
	unsigned long hiMant, loMant;

	expon = ((bytes[0] & 0x7F) << 8) | (bytes[1] & 0xFF);
	hiMant = ReadLong(bytes+2);
	loMant = ReadLong(bytes+6);

	if (expon == 0 && hiMant == 0 && loMant == 0)
		f = 0;
	else
	{
		if (expon == 0x7FFF) /* Infinity or NaN */
			f = -1;
		else
		{
			expon -= 16383;
			f = ldexp(UnsignedToFloat(hiMant),expon -= 31);
			f += ldexp(UnsignedToFloat(loMant),expon -= 32);
		}
	}

	if (bytes[0] & 0x80)
		return -f;
	return f;
}

/////////////////////////////////////////////////////////////////////////////
// Class for MOD music
/////////////////////////////////////////////////////////////////////////////

FrotzSoundMOD::FrotzSoundMOD(int sound, unsigned short eos, BYTE* data, int length, bool del) :
	FrotzSound(sound,eos,data,length,del)
{
	m_duration = 0;
}

FrotzSoundMOD::~FrotzSoundMOD()
{
	RemoveFromList();
}

bool FrotzSoundMOD::Play(int repeat, int volume)
{
	if (m_data == NULL)
		return false;

	InitMODPlug();
	if (IsMODPlugLoaded() == false)
		return false;

	// Only one MOD can be playing at any given time, so stop
	// any currently playing MODs
	CDSoundEngine& engine = CDSoundEngine::GetSoundEngine();
	engine.StopSounds(GetType());

	// Load the song into MODPlug
	if (m_api->LoadSong(m_data,m_length) != MPPERR_NOERROR)
		return false;

	// Set the number of repeats and the duration
	if (repeat > 0)
	{
		m_api->SetRepeatCount(repeat-1);
		m_duration = m_api->GetSongLength() * repeat * 1000;
	}
	else
	{
		m_api->SetRepeatCount(-1);
		m_duration = -1;
	}

	// Create a buffer
	WAVEFORMATEX& format = engine.GetPrimaryFormat();
	if (CreateBuffer(format.nChannels,format.nSamplesPerSec,format.wBitsPerSample) == false)
		return false;

	// Fill the buffer with sample data
	if (FillBuffer(GetBufferSize()) == false)
		return false;

	// Set the volume for the buffer
	SetBufferVolume(GetDecibelVolume(volume) * 100L);

	// Start the buffer playing
	return PlayBuffer();
}

bool FrotzSoundMOD::IsPlaying(void)
{
	return IsBufferPlaying();
}

void FrotzSoundMOD::WriteSampleData(unsigned char* sample, int len)
{
	WAVEFORMATEX& format = CDSoundEngine::GetSoundEngine().GetPrimaryFormat();

	// Render audio
	int bytes = m_api->Render(sample,len) * format.nBlockAlign;

	// Has the MOD been completely rendered?
	if (bytes < len)
	{
		for (int i = bytes; i < len; i++)
		{
			sample[i] = (unsigned char)
				((format.wBitsPerSample == 8) ? 0x80 : 0x00);
		}
	}
}

// Check if the sound has finished playing
bool FrotzSoundMOD::IsSoundOver(DWORD tick)
{
	if (m_Playing == false)
		return true;

	// Check if sound is playing forever
	if (m_duration < 0)
		return false;
	return (tick > m_StartTime + m_duration);
}

// Get a type identifier for the sound
int FrotzSoundMOD::GetType(void)
{
	return (int)'M';
}

// MODPlug functions and data

HINSTANCE FrotzSoundMOD::m_dll = 0;
IModMixer* FrotzSoundMOD::m_api = NULL;
bool FrotzSoundMOD::m_showError = true;

// Initialize the MODPlug DLL
void FrotzSoundMOD::InitMODPlug(void)
{
	if (IsMODPlugLoaded())
		return;

	// Load the DLL
	m_dll = ::LoadLibrary("mppsdk.dll");
	if (m_dll != 0)
	{
		// Get the exported function to access MODPlug
		MPP_GETMODAPIFUNC getAPI = (MPP_GETMODAPIFUNC)
			::GetProcAddress(m_dll,MPP_GETMODAPIFUNCNAME);
		if (getAPI != NULL)
		{
			// Get the IModMixer interface
			getAPI(&m_api);
			if (m_api != NULL)
			{
				// Check the version number
				if (m_api->GetVersion() >= MPPAPI_VERSION)
				{
					// Set up the player
					WAVEFORMATEX& format = CDSoundEngine::GetSoundEngine().
						GetPrimaryFormat();
					MPPERR result = m_api->SetWaveFormat(format.nSamplesPerSec,
						format.nChannels,format.wBitsPerSample);
					if (result == MPPERR_NOERROR)
					{
						// Set the options for MODPlug
						result = m_api->SetMixerOptions(
							MPPMIX_SURROUND|MPPMIX_HIGHQUALITY|MPPMIX_BASSEXPANSION);
						if (result == MPPERR_NOERROR)
						{
							// Set the amplification level to maximum
							m_api->SetPreampLevel(400);
							return;
						}
					}
				}
			}
		}
	}

	// Show an error if MODPlug failed
	if (m_showError)
	{
		::MessageBox(AfxGetMainWnd()->GetSafeHwnd(),
			CResString(IDS_FAIL_MODPLUG),CResString(IDS_FROTZ),MB_ICONWARNING|MB_OK);

		m_showError = false;
	}

	// If loading MODPlug has failed, clean up
	CloseMODPlug();
}

// Close the MODPlug DLL
void FrotzSoundMOD::CloseMODPlug(void)
{
	if (m_api != NULL)
	{
		m_api->Release();
		m_api = NULL;
	}

	if (m_dll != 0)
	{
		::FreeLibrary(m_dll);
		m_dll = 0;
	}
}

// Check if MODPlug is loaded
bool FrotzSoundMOD::IsMODPlugLoaded(void)
{
	return (m_api != NULL);
}

/////////////////////////////////////////////////////////////////////////////
// Class to build a MOD from a SONG and AIFF samples
/////////////////////////////////////////////////////////////////////////////

MODBuilder::MODBuilder() : m_mod(NULL)
{
}

MODBuilder::~MODBuilder()
{
	delete[] m_mod;

	int index = 0;
	FrotzSoundAIFF* aiff = NULL;

	// Discard all the loaded samples
	POSITION pos = m_samples.GetStartPosition();
	while (pos != NULL)
	{
		m_samples.GetNextAssoc(pos,index,aiff);
		delete aiff;
	}
}

// Convert a SONG and AIFF samples into a MOD
FrotzSoundMOD* MODBuilder::CreateMODFromSONG(int sound, unsigned short eos, BYTE* data, int length, bb_map_t* map)
{
	const int SAMPLE_NAME = 0;
	const int SAMPLE_LENGTH = 22;
	const int SAMPLE_REPEAT_POS = 26;
	const int SAMPLE_REPEAT_LEN = 28;
	const int SAMPLE_SIZE = 30;

	int modLen = length;

	// Loop over the samples
	for (int i = 0; i < 31; i++)
	{
		BYTE* sample = data + 20 + i*SAMPLE_SIZE;
		if (*(sample + SAMPLE_NAME) != 0)
		{
			// Get the sample resource index
			int index = 0;
			if (sscanf((const char*)(sample + SAMPLE_NAME),"SND%d",&index) != 1)
				return NULL;

			// Attempt to load the sample
			bb_result_t result;
			if (bb_load_resource_snd(map,bb_method_Memory,&result,index,NULL) != bb_err_None)
				return NULL;
			if (map->chunks[result.chunknum].type != bb_make_id('F','O','R','M'))
				return NULL;
			FrotzSoundAIFF* aiff = new FrotzSoundAIFF(index,0,(BYTE*)result.data.ptr,result.length);
			m_samples[index] = aiff;

			// Get details of the sample
			FrotzSoundAIFF::SampleData data;
			if (aiff->GetSampleData(data) == false)
				return NULL;

			// Add to the total size of the MOD
			modLen += data.samples + 2;
		}
	}

	// Create a buffer for the MOD and put the SONG data at the start
	m_mod = new BYTE[modLen];
	::CopyMemory(m_mod,data,length);

	// Copy the samples into the buffer
	int writePos = length;
	for (int i = 0; i < 31; i++)
	{
		BYTE* sample = data + 20 + i*SAMPLE_SIZE;
		if (*(sample + SAMPLE_NAME) != 0)
		{
			// Get the sample resource index
			int index = 0;
			sscanf((const char*)(sample + SAMPLE_NAME),"SND%d",&index);

			// Get the sample
			FrotzSoundAIFF* aiff = m_samples[index];
			FrotzSoundAIFF::SampleData data;
			aiff->GetSampleData(data);

			// Update the sample length
			WriteHeaderValue(i,SAMPLE_LENGTH,data.samples+2);

			// Update repeat information
			if ((data.repeat1 != 0) && (data.repeat2 != 0))
			{
				WriteHeaderValue(i,SAMPLE_REPEAT_POS,data.repeat1+2);
				WriteHeaderValue(i,SAMPLE_REPEAT_LEN,data.repeat2-data.repeat1);
			}

			// Copy into the buffer, converting to 8-bit mono
			*(m_mod+(writePos++)) = 0;
			*(m_mod+(writePos++)) = 0;
			for (unsigned long j = 0; j < data.samples; j++)
				*(m_mod+writePos+j) = *(data.data+(j*data.channels*(data.bits>>3)));
			writePos += data.samples;
		}
	}

	FrotzSoundMOD* mod = new FrotzSoundMOD(sound,eos,m_mod,modLen,true);
	m_mod = NULL;
	return mod;
}

// Write a value into the MOD header
void MODBuilder::WriteHeaderValue(int sample, int offset, unsigned long value)
{
	const int SAMPLE_SIZE = 30;
	unsigned short val = (unsigned short)(value>>1);

	BYTE* header = m_mod + 20 + (sample*SAMPLE_SIZE);
	header[offset] = (BYTE)(val >> 8);
	header[offset+1] = (BYTE)(val - (header[offset] << 8));
}
