/////////////////////////////////////////////////////////////////////////////
// Windows Frotz
// TrueType font analysis class
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "TrueTypeFont.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

DWORD TrueTypeFont::m_cmapName = MAKETABLENAME('c','m','a','p');

TrueTypeFont::TrueTypeFont() : m_table(NULL)
{
}

TrueTypeFont::~TrueTypeFont()
{
	delete[] m_table;
}

// Load the font tables
bool TrueTypeFont::LoadTables(CDC& dc)
{
	delete[] m_table;
	m_table = NULL;

	DWORD size;
	GetUnicodeCoverage(dc,NULL,0,&size);
	m_table = new BYTE[size];
	if (!GetUnicodeCoverage(dc,(LPCMAP4)m_table,size,&size))
	{
		delete[] m_table;
		m_table = NULL;
		return false;
	}
	return true;
}

// Determine if the font has a valid CMAP structure
bool TrueTypeFont::HasValidCmap(void)
{
	return (m_table != NULL);
}

// Get the index of a glyph
USHORT TrueTypeFont::GetUnicodeGlyphIndex(USHORT ch)
{
	if (m_table == NULL)
		return 0;

	// Find the cmap segment that has the character code
	USHORT segment = 0;
	if (!FindFormat4Segment((LPCMAP4)m_table,ch,&segment))
		return 0;

	// Get pointers to the cmap data
	USHORT* idRangeOffset = GetIdRangeOffsetArray(m_table);
	USHORT* idDelta = GetIdDeltaArray(m_table);
	USHORT* startCount = GetStartCountArray(m_table);

	if (idRangeOffset[segment] == 0)
		return (idDelta[segment] + ch) % 65536;

	USHORT id = *(idRangeOffset[segment]/2 + 
		(ch - startCount[segment]) + &idRangeOffset[segment]);
	if (id)
		return (idDelta[segment] + id) % 65536;
	return 0;
}

// Get the Unicode tables for the font
bool TrueTypeFont::GetUnicodeCoverage(CDC& dc, LPCMAP4 buffer, DWORD size, DWORD *needed)
{
	// Get the number of subtables in the CMAP table
	USHORT encode = 0;
	if (dc.GetFontData(m_cmapName,sizeof(USHORT),&encode,sizeof(USHORT)) != sizeof(USHORT))
	{
		*needed = 0;
		return false;
	}
	encode = SWAPWORD(encode);

	// Get the Unicode encoding
	CMAPENCODING encoding;
	DWORD unicode = encode;
	for (int i = 0; i < encode; i++)
	{
		if (!GetFontEncoding(dc,&encoding,i))
		{
			*needed = 0;
			return false;
		}

		if (encoding.platformId == 3)
		{
			if ((encoding.encodingId == 1) || (encoding.encodingId == 0))
				unicode = i;
		}
	}
	if (unicode >= encode)
	{
		*needed = 0;
		return false;
	}

	// Get the header entries for the Unicode encoding
	CMAP4 format4;
	if (!GetFontFormat4Header(dc,&format4,encoding.offset))
	{
		*needed = 0;
		return false;
	}
	if (format4.format != 4)
	{
		*needed = 0;
		return false;
	}

	// Work out the buffer size
	*needed = format4.length;    
	if ((*needed > size) || (buffer == NULL))
		return false;

	// Get the entire subtable
	BYTE* format4Subtable = new BYTE[format4.length];
	if (!GetFontFormat4Subtable(dc,(LPCMAP4)format4Subtable,encoding.offset))
	{
		*needed = 0;
		delete[] format4Subtable;
		return false;
	}

	// Copy the retrieved table into the buffer
	::CopyMemory(buffer,format4Subtable,((LPCMAP4)format4Subtable)->length);
	delete[] format4Subtable;
	return true;
}

// Get the encoding for a font
bool TrueTypeFont::GetFontEncoding(CDC& dc, CMAPENCODING *encoding, int encode)
{
	// Get the structure data from the TrueType font
	DWORD result = dc.GetFontData(m_cmapName,
		CMAPHEADERSIZE + ENCODINGSIZE*encode,encoding,sizeof(CMAPENCODING));
	bool success = (result == sizeof(CMAPENCODING));

	// Swap the fields
	encoding->platformId = SWAPWORD(encoding->platformId);
	encoding->encodingId = SWAPWORD(encoding->encodingId);
	encoding->offset = SWAPLONG(encoding->offset);
	return success;
}

// Get the header for a font
bool TrueTypeFont::GetFontFormat4Header(CDC& dc, LPCMAP4 format4, DWORD offset)
{
	bool success = true;

	USHORT* field = (USHORT*)format4;
	for (int i = 0; i < 7; i++)
	{
		// Get the field from the subtable
		DWORD result = dc.GetFontData(m_cmapName,
			offset + sizeof(USHORT)*i,field,sizeof(USHORT));

		*field = SWAPWORD(*field);
		field++;
		success = (result == sizeof(USHORT)) && success;
	}
	return success;
}

// Get the format subtable for a font
bool TrueTypeFont::GetFontFormat4Subtable(CDC& dc, LPCMAP4 format4Subtable, DWORD offset)
{
	// Retrieve the header values in swapped order
	if (!GetFontFormat4Header(dc,format4Subtable,offset))
		return false;

	// Get the rest of the table
	USHORT length = format4Subtable->length - (7*sizeof(USHORT));
	DWORD result = dc.GetFontData(m_cmapName,
		offset + 7*sizeof(USHORT),(LPBYTE)format4Subtable->arrays,length);       
	if (result != length)
		return false;

	SwapArrays(format4Subtable);
	return true;
}

// Find the segment for a character
bool TrueTypeFont::FindFormat4Segment(LPCMAP4 table, USHORT ch, USHORT *seg)
{
	USHORT segCount = table->segCountX2/2;
	USHORT* startCount = GetStartCountArray((LPBYTE)table);
	USHORT* endCount = GetEndCountArray((LPBYTE)table);

	for (int i = 0; (i < segCount) && (endCount[i] < ch); i++);

	if (i >= segCount)
		return false;
	if (startCount[i] > ch)
		return false;

	*seg = i;
	return true;
}

// Byte swap the font arrays
void TrueTypeFont::SwapArrays(LPCMAP4 format4)
{
	DWORD segCount = format4->segCountX2/2;
	USHORT *startCount = GetStartCountArray((LPBYTE)format4);
	USHORT *endCount = GetEndCountArray((LPBYTE)format4);
	USHORT *idDelta = GetIdDeltaArray((LPBYTE)format4);
	USHORT *idRangeOffset = GetIdRangeOffsetArray((LPBYTE)format4);

	for (DWORD i = 0; i < segCount; i++)
	{
		startCount[i] = SWAPWORD(startCount[i]);
		endCount[i] = SWAPWORD(endCount[i]);
		idDelta[i] = SWAPWORD(idDelta[i]);
		idRangeOffset[i] = SWAPWORD(idRangeOffset[i]);
	}

	USHORT* glyphId = idRangeOffset + segCount;
	USHORT* endOfBuffer = (USHORT*)((LPBYTE)format4 + format4->length);
	for (; glyphId < endOfBuffer; glyphId++)
		*glyphId = SWAPWORD(*glyphId);
}

USHORT* TrueTypeFont::GetStartCountArray(LPBYTE buff)
{
	DWORD segCount = ((LPCMAP4)buff)->segCountX2/2;
	return (USHORT*)(buff + 8*sizeof(USHORT) + segCount*sizeof(USHORT));
}

USHORT* TrueTypeFont::GetEndCountArray(LPBYTE buff)
{
	return (USHORT*)(buff + 7*sizeof(USHORT));
}

USHORT* TrueTypeFont::GetIdDeltaArray(LPBYTE buff)
{
	DWORD segCount = ((LPCMAP4)buff)->segCountX2/2;
	return (USHORT*)(buff + 8*sizeof(USHORT) + segCount*2*sizeof(USHORT));
}

USHORT* TrueTypeFont::GetIdRangeOffsetArray(LPBYTE buff)
{
	DWORD segCount = ((LPCMAP4)buff)->segCountX2/2;
	return (USHORT*)(buff + 8*sizeof(USHORT) + segCount*3*sizeof(USHORT));
}
