/*
 * mf2rol
 *
 * Convert a MIDI file to AdLib .ROL
 */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include "midifile.h"

#define MAXTRACK 128
#define CHUNKSIZE 0x8000
#define NR_VOICES 11

char filler44[44] = {
0,0,4,0,
92,114,111,108,108,92,100,101,102,97,117,108,116,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

char filler5[5] = {
48,0,56,0,0};

char filler143[143] = {
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,22,0,0,0,
0,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,2,0,
1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,
1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,
1,0,1,0,1,0,1,0,1,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
84,101,109,112,111,0,43,42,-16,67,-16,67,43,42,70};

char fillernote[15] = {
86,111,105,120,32,32,48,0,-16,67,0,0,-68,51,-53};
char fillerinstrum[15] = {
84,105,109,98,114,101,32,32,48,0,86,111,105,120,32};
char fillervolume[15] = {
86,111,108,117,109,101,32,32,48,0,84,105,109,98,114};
char fillerpitch[15] = {
80,105,116,99,104,32,32,48,0,0,84,105,109,98,114};

/* structure used to store midifile events internally */
struct i_event {
	long i_time;
	int trk;
	char chan;
	char type;
	char v1;
	char v2;
	struct i_event *next;
};
#define getmemEvent() ((struct i_event *) getmem(sizeof(struct i_event)))

/* structure used to store timbre assignment rules internally */
int ntrules;
struct timrule {
	char name[9];
	char track;
	char channel;
	char patch;
	char note;
	char percus;
	int  offset;
} *trulebase;
/* name of timbre bank file */
char bankfile[64];

/* variables defining tempo */
int division;
float basictempo;
int beats_measure;
int notebeat;
int ticks_beat;
long maxquarter;
int clocks_click;
int sper24;

/* flag for percussion mode */
int ispercussion;

/* event types for struct i_event */
#define I_NOTEOFF  0
#define I_PATCH    1
#define I_TEMPO    2
#define I_NOTEON   3
#define I_PRESSURE 4
#define I_PITCH    5
/*
char *etypestr[]={"off", "patch", "tempo", "on", "pressure", "pitch"};
*/

struct i_event *i_curr;
struct i_event *i_trkbase[MAXTRACK];
struct i_event *i_trkptr[MAXTRACK];

int i_ntrks;
char patchmap[16][16];
char chanuse[16];
char curpatch[16];

char *alloc_base;
char *alloc_limit;

typedef int Time;
struct sTempoEvent {
		Time    time;
		float   tempo;
		struct sTempoEvent *next;
	} ;
typedef struct sTempoEvent TempoEvent;
#define getmemTempo() ((TempoEvent *) getmem(sizeof(TempoEvent)))
struct sNoteEvent {
		Time			ontime;
		Time			offtime;
		char             note;
		char             offset;
		struct sNoteEvent *next;
	} ;
typedef struct sNoteEvent NoteEvent;
#define getmemNote() ((NoteEvent *) getmem(sizeof(NoteEvent)))
struct sInstrumEvent {
		Time time;
		char instrumName[ 9];
		char filler;
		int     timbreIndex;
		struct sInstrumEvent *next;
	} ;
typedef struct sInstrumEvent InstrumEvent;
#define getmemInstrum() ((InstrumEvent *) getmem(sizeof(InstrumEvent)))
struct sVolumeEvent {
		Time time;
		float volume;
		struct sVolumeEvent *next;
	} ;
typedef struct sVolumeEvent VolumeEvent;
#define getmemVolume() ((VolumeEvent *) getmem(sizeof(VolumeEvent)))
struct sPitchEvent {
		Time time;
		float pitch;
		struct sPitchEvent *next;
	} ;
typedef struct sPitchEvent PitchEvent;
#define getmemPitch() ((PitchEvent *) getmem(sizeof(PitchEvent)))

TempoEvent *hTempoEvent;
TempoEvent *cTempoEvent;
struct hlisth {
	NoteEvent *hNoteEvent;
	InstrumEvent *hInstrumEvent;
	VolumeEvent *hVolumeEvent;
	PitchEvent *hPitchEvent;
	NoteEvent *cNoteEvent;
	InstrumEvent *cInstrumEvent;
	VolumeEvent *cVolumeEvent;
	PitchEvent *cPitchEvent;
} vocs[NR_VOICES];

struct vst {
	char timbre[9];
	char on;
	int timeon;
	int note;
	int chan;
} vocstat[NR_VOICES];

static FILE *F;

filegetc()
{
	return(getc(F));
}

outofmem()
{
	fprintf(stderr, "\nError: Out of Memory. Sorry!\n");
	exit(1);
}

char *getmem(size)
{
	char *p;
	/*
	static int chunknum = 0;
	*/

	p = alloc_base;
	alloc_base += size;
	if (alloc_base >= alloc_limit) {
		/*
		printf("Allocating 32K chunk %d\n", ++chunknum);
		*/
		alloc_base = malloc(CHUNKSIZE);
		if (alloc_base == NULL)
			outofmem();
		alloc_limit = alloc_base + CHUNKSIZE;
		p = alloc_base;
		alloc_base += size;
	}
	return(p);
}
char *nextarg(s)
char *s;
{
	while ((*s != 0) && (*s != ' ') && (*s != '\t') && (*s != '\n'))
		s++;
	if ((*s == 0) || (*s == '\n'))
		return(NULL);
	while ((*s == ' ') || (*s == '\t'))
		s++;
	if ((*s == 0) || (*s == '\n'))
		return(NULL);
	return(s);
}
copyarg(s1, s2)
char *s1, *s2;
{
	while ((*s2 != ' ') && (*s2 != '\t') && (*s2 != '\n') && (*s2 != 0)) {
		if (islower(*s2))
			*s1 = toupper(*s2);
		else
			*s1 = *s2;
		s1++;
		s2++;
	}
	*s1 = 0;
}
scanerr(s)
char *s;
{
	fprintf(stderr, "Syntax error: %s", s);
	exit(1);
}
readrules(s)
char *s;
{
	FILE *fin;
	char buf[128];
	struct timrule *t;
	char *s1, *s2;
	int v;

	*bankfile = 0;
	ispercussion = 0;
	if ((fin = fopen(s, "r")) == NULL) {
		fprintf(stderr, "Error: Can't open rule file %s.\n", s);
		exit(1);
	}
	printf("Scanning rule file...\n");
	ntrules = 1;
	while (fgets(buf, 128, fin) != NULL)
		if (strncmp(buf, "assign", 6) == 0)
			ntrules++;
	trulebase = (struct timrule *) malloc(ntrules * sizeof(struct timrule));
	ntrules = 0;
	fseek(fin, 0L, 0);
	if (trulebase == NULL)
		outofmem();
	t = trulebase;
	while (fgets(buf, 128, fin) != NULL) {
		if ((*buf == '#') || (*buf == '\n') || (*buf == 0))
			continue;
		if (strncmp(buf, "assign", 6) == 0) {
			if (*bankfile == 0) {
				fprintf(stderr, "Error: Bankfile must come before assign\n");
				exit(1);
			}
			s1 = nextarg(buf);
			s2 = t->name;
			copyarg(s2, s1);
			if (strlen(s2) > 8) {
				fprintf(stderr, "Error: instrument name longer than 8 characters\n");
				exit(1);
			}
			if (strcmp(s2, "MUTE")==0)
				v=0;
			else if ((v=findtimbre(s2)) < 0) {
				fprintf(stderr, "Error: cannot find instrument %s\n", s2);
				exit(1);
			}
			while (*s2 != 0) {
				if (isupper(*s2))
					*s2 = tolower(*s2);
				s2++;
			}
			t->percus = v;
			s1 = nextarg(s1);
			if (s1 == NULL)
				scanerr(buf);
			if (*s1 == '*')
				t->track = -1;
			else
				t->track = atoi(s1);
			s1 = nextarg(s1);
			if (s1 == NULL)
				scanerr(buf);
			if (*s1 == '*')
				t->channel = -1;
			else
				t->channel = atoi(s1);
			s1 = nextarg(s1);
			if (s1 == NULL)
				scanerr(buf);
			if (*s1 == '*')
				t->patch = -1;
			else
				t->patch = atoi(s1);
			s1 = nextarg(s1);
			if (s1 == NULL)
				scanerr(buf);
			if (*s1 == '*')
				t->note = -1;
			else
				t->note = atoi(s1);
			s1 = nextarg(s1);
			if (s1 == NULL)
				t->offset = 0;
			if (*s1 == '*')
				t->offset = 0;
			else
				t->offset = atoi(s1);
			/*
			printf("%s %d %d %d %d %d\n",t->name,t->track,t->channel,t->patch,t->note,t->percus);
			*/
			ntrules++;
			t++;
		} else if (strncmp(buf, "percussion", 10) == 0) {
			s1 = nextarg(buf);
			if (s1 == NULL)
				scanerr(buf);
			if ((*s1 == 'y') || (*s1=='Y'))
				ispercussion = 1;
		} else if (strncmp(buf, "bankfile", 8) == 0) {
			if (*bankfile != 0) {
				fprintf(stderr, "Error: Multiple bankfiles\n");
				exit(1);
			}
			s1 = nextarg(buf);
			if (s1 == NULL)
				scanerr(buf);
			copyarg(bankfile, s1);
			readbank(bankfile);
		} else {
			scanerr(buf);
		}
	}
	printf("Scanning done.\n    %d assignment rules.\n", ntrules);
	fclose(fin);
}
initevent()
{
	int i;
	TempoEvent *te;
	NoteEvent *ne;
	InstrumEvent *ie;
	VolumeEvent *ve;
	PitchEvent *pe;

	hTempoEvent = cTempoEvent = getmemTempo();
	hTempoEvent->next = NULL;
	for (i=0; i<NR_VOICES; i++) {
		vocstat[i].on = 0;
		ne = vocs[i].hNoteEvent = vocs[i].cNoteEvent = getmemNote();
		ne->next = NULL;
		ie = vocs[i].hInstrumEvent = vocs[i].cInstrumEvent = getmemInstrum();
		ie->next = NULL;
		ve = vocs[i].hVolumeEvent = vocs[i].cVolumeEvent = getmemVolume();
		ve->next = NULL;
		pe = vocs[i].hPitchEvent = vocs[i].cPitchEvent = getmemPitch();
		pe->next = NULL;
	}
	for (i=0; i<16; i++)
		curpatch[i] = 0;

}
struct timrule *whattimbre(track, channel, patch, note)
{
	int i;
	struct timrule *p;
	static struct timrule dflt = { "piano1", 0,0,0,0,0,0 };

	p = trulebase;
	i = ntrules;
	while (i > 0) {
		if ( ((p->track<0) || (p->track==track))
			&& ((p->channel<0) || (p->channel==channel))
			&& ((p->patch<0) || (p->patch==patch))
			&& ((p->note<0) || (p->note==note)) )
				return(p);
		p++;
		i--;
	}
	return(&dflt);
}

putevent(time, track, type, channel, arg1, arg2)
long time;
int track, type, channel, arg1, arg2;
{
	int i, j;
	int rtime;
	int patch;
	int v;
	int lim;
	int nfree;
	int least;
	struct timrule *timb;
	TempoEvent *te;
	NoteEvent *ne;
	InstrumEvent *ie;
	VolumeEvent *ve;
	PitchEvent *pe;
	long micro;
	float tempofactor;

	/*
	printf("T=%ld type=%s chan=%d arg1=%d arg2=%d\n",time,etypestr[type],channel,arg1,arg2);
	*/
	rtime = (time * (long)ticks_beat) / (long)division;
	if (rtime<0) {
		fprintf(stderr, "Error: Tune too long for ROL\n");
		exit(1);
	}
	switch (type) {
	case I_NOTEON:
		if (arg1 < 1)
			break;
		/* check if this note is already on */
		for (i=0; i<NR_VOICES; i++)
			if ((vocstat[i].on != 0) && (vocstat[i].note==arg1) &&
				(vocstat[i].chan==channel))
				break;
		/* if already on then ignore this */
		if (i<NR_VOICES)
			break;
		patch = curpatch[channel];
		timb = whattimbre(track, channel+1, patch, arg1);
		/* ignore if mute note */
		if (strcmp(timb->name, "mute")==0)
			break;
		if (timb->percus) {
			/* percussion timbre */
			if (ispercussion)
				/* if percussion timbre then we HAVE to use designated voice */
				v = timb->percus;
			else {
				fprintf(stderr, "Error: Percussion timbre %s used but percussion mode not set\n", timb->name);
				exit(1);
			}
			/*
			printf("voice %d trk %d chan %d patch %d note %d timbre %s\n",
			v, track, channel+1, patch, arg1, timb->name);
			*/
		} else {
			/* non-percussion timbre */
			if (ispercussion)
				lim = 6;
			else
				lim = 9;
			nfree = 0;
			/* count number of free voices */
			for (i=0; i<lim; i++)
				if (vocstat[i].on == 0) {
					v = i;
					nfree++;
				}

			if (nfree > 1) {
				/* more than one free voice */
				/* search for one with one of the required timbre */
				for (i=0; i<lim; i++)
						if ((vocstat[i].on==0) && 
						(strcmp(vocstat[i].timbre, timb->name)==0))
							break;
				if (i<lim) {
					/* found one */
					v = i;
				} else {
					/* no match so just get the first free voice */
					for (i=0; i<lim; i++)
						if (vocstat[i].on == 0)
							break;
					v = i;
				}
			} else if (nfree == 0) {
				/* no free voice so bump somebody off */
				/*
				printf("Bump");
				*/
				least = vocstat[0].timeon;
				v = 0;
				for (i=0; i<lim; i++) {
					/*
					printf(" %d", vocstat[i].timeon);
					*/
					if (vocstat[i].timeon < least) {
						least = vocstat[i].timeon;
						v = i;
					}
				}
				/*
				printf("\nLeast was %d on voice %d\n", least, v);
				*/
			} else {
				/* one free voice, no choice */
				;
			}
		}
		/* v is now set to the selected voice */
		/* bump off old sound if necessary */
		if (vocstat[v].on) {
			(vocs[v].cNoteEvent)->offtime = rtime;
			vocstat[v].on = 0;
		}
		/* change timbre if necessary */
		if (strcmp(vocstat[v].timbre, timb->name) != 0) {
			ie = vocs[v].cInstrumEvent;
			ie->next = getmemInstrum();
			ie = ie->next;
			ie->time = rtime;
			strcpy(ie->instrumName, timb->name);
			ie->filler = 0;
			ie->timbreIndex = 0;
			ie->next = NULL;
			vocs[v].cInstrumEvent = ie;
			strcpy(vocstat[v].timbre, timb->name);
			/*
			printf("New timbre voice %d to %s %lx\n",v,timb->name, ie);
			*/
		}
		ne = vocs[v].cNoteEvent;
		ne->next = getmemNote();
		ne = ne->next;
		ne->ontime = rtime;
		ne->offtime = rtime+1;
		ne->note = arg1;
		ne->offset = timb->offset;
		ne->next = NULL;
		vocs[v].cNoteEvent = ne;
		vocstat[v].on = 1;
		vocstat[v].timeon = rtime;
		vocstat[v].note = arg1;
		vocstat[v].chan = channel;
		break;

	case I_PATCH:
		curpatch[channel] = arg1;
		break;

	case I_NOTEOFF:
		/* look for this note to turn off */
		for (i=0; i<NR_VOICES; i++)
			if ((vocstat[i].on != 0) && (vocstat[i].note==arg1) &&
				(vocstat[i].chan==channel))
				break;
		/* turn off this note if found */
		if (i < NR_VOICES) {
			v = i;
			(vocs[v].cNoteEvent)->offtime = rtime;
			vocstat[v].on = 0;
		}
		break;

	case I_TEMPO:
		micro = ((long) arg1) << 16;
		micro &= 0xffff00;
		micro |= (((long) arg2) << 8) & 0xff00;
		micro &= 0xffff00;
		tempofactor = 60000000.0/((float) micro);
		tempofactor = tempofactor/basictempo;
		te = cTempoEvent;
		te->next = getmemTempo();
		te = te->next;
		te->next = NULL;
		te->time = rtime;
		te->tempo = tempofactor;
		cTempoEvent = te;
		break;

	default:
		break;
	}

}

buildrol(fp)
FILE *fp;
{
	int i;
	int n, l;
	int comp;
	int totime;
	int totime2;
	float f;
	TempoEvent *te;
	NoteEvent *ne;
	InstrumEvent *ie;
	VolumeEvent *ve;
	PitchEvent *pe;
	char pm;

	/* gererate header */
	fwrite(filler44, 1, 44, fp);
	fwrite(&ticks_beat, 2, 1, fp);
	fwrite(&beats_measure, 2, 1, fp);
	fwrite(filler5, 1, 5, fp);
	if (ispercussion)
		pm = 0;
	else
		pm = 1;
	fwrite(&pm, 1, 1, fp);
	fwrite(filler143, 1, 143, fp);
	fwrite(&basictempo, 4, 1, fp);

	/* generate tempo track */
	n = 1;
	te = hTempoEvent;
	while (te->next != NULL) {
		n++;
		te = te->next;
	}
	hTempoEvent->time = 0;
	hTempoEvent->tempo = 1.0;
	te = hTempoEvent;
	if ((n>1) && ((hTempoEvent->next)->time == 0)) {
		te = te->next;
		n--;
	}
	fwrite(&n, 2, 1, fp);
	while (te != NULL) {
		fwrite(te, 6, 1, fp);
		te = te->next;
	}
	/* generate all voice tracks */
	for (i=0; i<NR_VOICES; i++) {
		/* note events */
		fwrite(fillernote, 1, 15, fp);
		ne = vocs[i].hNoteEvent;
		ne->offtime = 0;
		n++;
		while (ne->next != NULL) {
			ne = ne->next;
			n++;
		}
		totime = ne->offtime;
		fwrite(&totime, 2, 1, fp);
		/*
		printf("Buildrol voc %d total note time is %d\n", i, totime);
		*/
		if (totime > 0) {
			/* if first note doesn't start at 0 then generate space */
			totime2 = 0;
			ne = vocs[i].hNoteEvent;
			ne = ne->next;
			if (ne->ontime > 0) {
				n = 0;
				fwrite(&n, 2, 1, fp);
				n = ne->ontime;
				totime2 += n;
				fwrite(&n, 2, 1, fp);
			}
			ne = vocs[i].hNoteEvent;
			while (ne->next != NULL) {
				ne = ne->next;
				n = ne->note + ne->offset;
				l = ne->offtime - ne->ontime;
				comp = 0;
				if (l > 0) {
					fwrite(&n, 2, 1, fp);
					totime2 += l;
					fwrite(&l, 2, 1, fp);
				} else {
					if (ne->next != NULL) {
						l = ((ne->next)->ontime) - (ne->offtime);
						if (l > 1) {
							comp = 1;
							l = 1;
							n = ne->note + ne->offset;
							fwrite(&n, 2, 1, fp);
							totime2 += l;
							fwrite(&l, 2, 1, fp);
						}
					}
				}
				if (ne->next != NULL) {
					n = 0;
					l = ((ne->next)->ontime) - (ne->offtime);
					l -= comp;
					if (l > 0) {
						fwrite(&n, 2, 1, fp);
						totime2 += l;
						fwrite(&l, 2, 1, fp);
					}
				}
			}
			/*
			printf("actual summed time %d\n", totime2);
			*/
			if (totime != totime2) {
				fprintf(stderr, "Inconsistent track length voice:%d last:%d sum:%d\n", i, totime, totime2);
			}
		}
		/* Instrument Events */
		fwrite(fillerinstrum, 1, 15, fp);
		ie = vocs[i].hInstrumEvent;
		n = 0;
		while (ie->next != NULL) {
			n++;
			ie = ie->next;
		}
		if (n == 0) {
			n++;
			ie = vocs[i].hInstrumEvent;
			strcpy(ie->instrumName, "piano1");
			ie->filler = 0;
			ie->timbreIndex = 0;
			ie->time = 0;
		} else {
			ie = vocs[i].hInstrumEvent;
			ie = ie->next;
			ie->time = 0;
		}
		/*
		printf("Buildrol instrum voc %d ninstrum %d\n", i, n);
		*/
		fwrite(&n, 2, 1, fp);
		while (ie != NULL) {
			/*
			printf("  instrum %s\n", ie->instrumName);
			*/
			fwrite(ie, 14, 1, fp);
			ie = ie->next;
		}
		fwrite(fillervolume, 1, 15, fp);
		n = 1;
		fwrite(&n, 2, 1, fp);
		n = 0;
		fwrite(&n, 2, 1, fp);
		f = 0.75;
		fwrite(&f, 4, 1, fp);
		fwrite(fillerpitch, 1, 15, fp);
		n = 1;
		fwrite(&n, 2, 1, fp);
		n = 0;
		fwrite(&n, 2, 1, fp);
		f = 1.0;
		fwrite(&f, 4, 1, fp);

	} /* for */
}
main(argc,argv)
char **argv;
{
	FILE *efopen();
	FILE *fp;
	int i,j;
	long mintime;
	struct i_event *p;
	int minidx;
	char *q;
	char ofilen[128];
	char mintype;

	printcopy();
	if ( argc > 1 )
		F = efopen(argv[1],"rb");
	else {
		fprintf(stderr, "MF2ROL - Covert Standard MIDI File to AdLib .ROL\n");
		fprintf(stderr, "Usage:\n    mf2rol <midifile> <config-file>\n");
		exit(1);
	}
	ntrules = -1;
	trulebase = NULL;
	if (argc > 2)
		readrules(argv[2]);

	i_ntrks = 0;
	for (i=0; i<16; i++)
		for (j=0; j<16; j++)
			patchmap[i][j] = -1;
	for (i=0; i<16; i++)
		chanuse[i] = 0;

	beats_measure = 0;
	ticks_beat = 12;
	maxquarter = 0;
	clocks_click = 0;
	sper24 = 0;
	notebeat = 0;

	alloc_base = malloc(CHUNKSIZE);
	if (alloc_base == NULL)
		outofmem();
	alloc_limit = alloc_base + CHUNKSIZE;

	for (i=0; i<MAXTRACK; i++) {
		i_trkbase[i] = NULL;
		i_trkptr[i] = NULL;
	}

	initfuncs();
	Mf_getc = filegetc;
	printf("Scanning Midi File: %s\n", argv[1]);
	midifile();
	fclose(F);
	printf("Scanning done.\nChannels used:");
	for (i=0; i<16; i++)
		if (chanuse[i])
			printf(" %d", i+1);
	printf("\nPatches (programs) used:\n");
	minidx = 0;
	for (i=0; i<16; i++)
		if (patchmap[i][0] != -1) {
			minidx = 1;
			printf(" Channel %d:", i+1);
			for (j=0; j<16; j++)
				if (patchmap[i][j] != -1)
					printf(" %d", patchmap[i][j]);
			printf("\n");
		}
	if (minidx == 0)
		printf(" None.\n");

	/* if no tempo was specified use 120 BPM or 500K microseconds */
	if (maxquarter == 0)
		maxquarter = 500000L;
	maxquarter &= 0xffff00;
	basictempo = 60000000.0/((float) maxquarter);

	initevent();

	/* play out the events in time-order */
	printf("Assigning voices.\n");
	for (i=0; i<i_ntrks; i++)
		i_trkptr[i] = i_trkbase[i]->next;
	do {
		mintime = 0x7fffffffL;
		mintype = 100;
		p = NULL;
		for (i=0; i<i_ntrks; i++) {
			if (i_trkptr[i] == NULL)
				continue;
			if ((i_trkptr[i]->i_time < mintime) ||
				((i_trkptr[i]->i_time == mintime) && (i_trkptr[i]->type < mintype))) {
				p = i_trkptr[i];
				mintime = p->i_time;
				mintype = p->type;
				minidx = i;
			}
		}
		if (p != NULL) {
			putevent(p->i_time, p->trk, p->type, p->chan, p->v1, p->v2);
			/*
			printf("Time=%ld Track=%d Event=%d Chan=%d V1=%d V2=%d\n", p->i_time, minidx+1,
			p->type, p->chan, p->v1, p->v2);
			*/
			i_trkptr[minidx] = p->next;
		}
	} while (p != NULL);
	strcpy(ofilen, argv[1]);
	q = ofilen + strlen(ofilen);
	while (q > ofilen) {
		q--;
		if (*q == '.')
			break;
	}
	if (*q == '.')
		*q = 0;
	strcat(ofilen, ".rol");
	fp = fopen(ofilen, "wb");
	if (fp == NULL) {
		fprintf(stderr, "Error: Cannot create output file %s\n", ofilen);
		exit(1);
	}
	printf("Building .ROL file: %s\n", ofilen);
	buildrol(fp);
	fclose(fp);
	printf("Conversion complete.\n");
	exit(0);
}

FILE *
efopen(name,mode)
char *name;
char *mode;
{
	FILE *f;

	if ( (f=fopen(name,mode)) == NULL ) {
		(void) fprintf(stderr,"Error: Can't open MIDI file %s.\n",name);
		exit(1);
	}
	return(f);
}

error(s)
char *s;
{
	fprintf(stderr,"Error: %s\n",s);
}

txt_header(format,ntrks,div)
{
	/*
	printf("Header format=%d ntrks=%d division=%d\n",format,ntrks,div);
	*/
	division = div;
}

txt_trackstart()
{
	struct i_event *p;
	/*
	printf("Track start\n");
	*/
	if (i_ntrks>=MAXTRACK) {
		fprintf(stderr,"Error: Too many tracks\n");
		exit(1);
	}
	p = getmemEvent();
	i_trkbase[i_ntrks] = p;
	i_trkptr[i_ntrks] = p;
	p->next = NULL;
}

txt_trackend()
{
	/*
	printf("Track end\n");
	*/
	i_ntrks++;
}

txt_noteon(chan,pitch,vol)
{
	struct i_event *p;
	/*
	prtime();
	printf("Note on, chan=%d pitch=%d vol=%d\n",chan+1,pitch,vol);
	*/
	i_curr = getmemEvent();
	p = i_trkptr[i_ntrks];
	p->next = i_curr;
	i_curr->next = NULL;
	i_trkptr[i_ntrks] = i_curr;

	i_curr->i_time = Mf_currtime;
	i_curr->chan = chan;
	i_curr->v1 = pitch;
	i_curr->v2 = vol;
	i_curr->trk = i_ntrks+1;
	if (vol == 0)
		i_curr->type = I_NOTEOFF;
	else
		i_curr->type = I_NOTEON;
	chanuse[chan] = 1;
}

txt_noteoff(chan,pitch,vol)
{
	struct i_event *p;
	/*
	prtime();
	printf("Note off, chan=%d pitch=%d vol=%d\n",chan+1,pitch,vol);
	*/
	i_curr = getmemEvent();
	p = i_trkptr[i_ntrks];
	p->next = i_curr;
	i_curr->next = NULL;
	i_trkptr[i_ntrks] = i_curr;

	i_curr->i_time = Mf_currtime;
	i_curr->chan = chan;
	i_curr->v1 = pitch;
	i_curr->v2 = vol;
	i_curr->trk = i_ntrks+1;
	i_curr->type = I_NOTEOFF;
}

txt_pressure(chan,pitch,press)
{
	/*
	prtime();
	printf("Pressure, chan=%d pitch=%d press=%d\n",chan+1,pitch,press);
	*/
}

txt_parameter(chan,control,value)
{
	/*
	prtime();
	printf("Parameter, chan=%d c1=%d c2=%d\n",chan+1,control,value);
	*/
}

txt_pitchbend(chan,msb,lsb)
{
	struct i_event *p;
	/*
	prtime();
	printf("Pitchbend, chan=%d msb=%d lsb=%d\n",chan+1,msb,lsb);
	*/
	/*
	i_curr = getmemEvent();
	p = i_trkptr[i_ntrks];
	p->next = i_curr;
	i_curr->next = NULL;
	i_trkptr[i_ntrks] = i_curr;

	i_curr->i_time = Mf_currtime;
	i_curr->chan = chan;
	i_curr->v1 = msb;
	i_curr->v2 = lsb;
	i_curr->type = I_PITCH;
	i_curr->trk = i_ntrks+1;
	*/
}

txt_program(chan,program)
{
	struct i_event *pp;
	char *p;
	int i;

	/*
	prtime();
	printf("Program, chan=%d program=%d\n",chan+1,program);
	*/
	i_curr = getmemEvent();
	pp = i_trkptr[i_ntrks];
	pp->next = i_curr;
	i_curr->next = NULL;
	i_trkptr[i_ntrks] = i_curr;

	i_curr->i_time = Mf_currtime;
	i_curr->chan = chan;
	i_curr->v1 = program;
	i_curr->type = I_PATCH;
	i_curr->trk = i_ntrks+1;
	/* install program number for this channel into patch table */
	p = &(patchmap[chan][0]);
	for (i=0; i<16; i++,p++)
		if (*p == program)
			return(0);
	p = &(patchmap[chan][0]);
	for (i=0; i<16; i++,p++)
		if (*p == -1) {
			*p = program;
			return(0);
		}
}

txt_chanpressure(chan,press)
{
	struct i_event *p;
	/*
	prtime();
	printf("Channel pressure, chan=%d pressure=%d\n",chan+1,press);
	*/
	/*
	i_curr = getmemEvent();
	p = i_trkptr[i_ntrks];
	p->next = i_curr;
	i_curr->next = NULL;
	i_trkptr[i_ntrks] = i_curr;

	i_curr->i_time = Mf_currtime;
	i_curr->chan = chan;
	i_curr->v1 = press;
	i_curr->type = I_PRESSURE;
	i_curr->trk = i_ntrks+1;
	*/
}

txt_sysex(leng,mess)
char *mess;
{
	/*
	prtime();
	printf("Sysex, leng=%d\n",leng);
	*/
}

txt_metamisc(type,leng,mess)
char *mess;
{
	/*
	prtime();
	printf("Meta event, unrecognized, type=0x%02x leng=%d\n",type,leng);
	*/
}

txt_metaspecial(type,leng,mess)
char *mess;
{
	/*
	prtime();
	printf("Meta event, sequencer-specific, type=0x%02x leng=%d\n",type,leng);
	*/
}

txt_metatext(type,leng,mess)
char *mess;
{
	static char *ttype[] = {
		NULL,
		"Text Event",           /* type=0x01 */
		"Copyright",            /* type=0x02 */
		"Track Name",
		"Instrument",           /* ...       */
		"Lyric",
		"Marker",
		"Cue Point",            /* type=0x07 */
		"Unrecognized"
	};
	int unrecognized = (sizeof(ttype)/sizeof(char *)) - 1;
	register int n, c;
	register char *p = mess;

	if ( type < 1 || type > unrecognized )
		type = unrecognized;
	printf(" Trk:%2d: %s: ",i_ntrks+1, ttype[type]);
	for ( n=0; n<leng; n++ ) {
		c = *p++;
		printf( (isprint(c)||isspace(c)) ? "%c" : "\\0x%02x" , c);
	}
	printf("\n");
}

txt_metaseq(num)
{
	/*
	prtime();
	printf("Meta event, sequence number = %d\n",num);
	*/
}

txt_metaeot()
{
	/*
	prtime();
	printf("Meta event, end of track\n");
	*/
}

txt_keysig(sf,mi)
{
	/*
	prtime();
	printf("Key signature, sharp/flats=%d  minor=%d\n",sf,mi);
	*/
}

txt_tempo(long tempo)
{
	struct i_event *p;
	/*
	prtime();
	printf("Tempo, microseconds-per-MIDI-quarter-note=%ld\n",tempo);
	*/
	i_curr = getmemEvent();
	p = i_trkptr[i_ntrks];
	p->next = i_curr;
	i_curr->next = NULL;
	i_trkptr[i_ntrks] = i_curr;

	i_curr->i_time = Mf_currtime;
	i_curr->trk = i_ntrks+1;
	i_curr->chan = 0;
	i_curr->v1 = (tempo>>16) & 0xff;
	i_curr->v2 = (tempo>>8) & 0xff;
	i_curr->type = I_TEMPO;
	if (tempo > maxquarter)
		maxquarter = tempo;
}

txt_timesig(nn,dd,cc,bb)
{
	int denom = 1;
	while ( dd-- > 0 )
		denom *= 2;
	/*
	prtime();
	printf("Time signature=%d/%d  MIDI-clocks/click=%d  32nd-notes/24-MIDI-clocks=%d\n",
		nn,denom,cc,bb);
	*/
	if (nn > beats_measure) {
		beats_measure = nn;
		notebeat = denom;
		clocks_click = cc;
		sper24 = bb;
	}
}

txt_smpte(hr,mn,se,fr,ff)
{
	/*
	prtime();
	printf("SMPTE, hour=%d minute=%d second=%d frame=%d fract-frame=%d\n",
		hr,mn,se,fr,ff);
	*/
}

txt_arbitrary(leng,mess)
char *mess;
{
	/*
	prtime();
	printf("Arbitrary bytes, leng=%d\n",leng);
	*/
}

prtime()
{
	/*
	printf("Time=%ld  ",Mf_currtime);
	*/
}

printcopy()
{
	static char notice[] = "MIDI File to .ROL converter.  V1.1   A. A. Kapauan 1991";
	fprintf(stderr, "\n%s\n\n", notice);
}

initfuncs()
{
	Mf_error = error;
	Mf_header =  txt_header;
	Mf_trackstart =  txt_trackstart;
	Mf_trackend =  txt_trackend;
	Mf_noteon =  txt_noteon;
	Mf_noteoff =  txt_noteoff;
	Mf_pressure =  txt_pressure;
	Mf_parameter =  txt_parameter;
	Mf_pitchbend =  txt_pitchbend;
	Mf_program =  txt_program;
	Mf_chanpressure =  txt_chanpressure;
	Mf_sysex =  txt_sysex;
	Mf_metamisc =  txt_metamisc;
	Mf_seqnum =  txt_metaseq;
	Mf_eot =  txt_metaeot;
	Mf_timesig =  txt_timesig;
	Mf_smpte =  txt_smpte;
	Mf_tempo =  txt_tempo;
	Mf_keysig =  txt_keysig;
	Mf_seqspecific =  txt_metaspecial;
	Mf_text =  txt_metatext;
	Mf_arbitrary =  txt_arbitrary;
}
