{$A-,B-,D-,F-,G+,I-,K+,L-,N-,P-,Q-,R-,S-,T-,V+,W-,X+,Y-}
{$M 8192,8192}
program Midi_Mon;

{$D Hubi's MIDI Monitor}

{$DEFINE LOWLEVEL}

uses Winapi
   , WinTypes
   , WinProcs
   , OWindows
   , ODialogs
   , MMSystem
   , HWMD
   ;

{$R MIDI_MON.RES}

const

{ Resource IDs }

  id_Dialog = 10;
  id_Menu = 20;

{ Menu item IDs }

  idm_input_none = 101;
  idm_output_none = 201;
  idm_start = 300;
  idm_clear = 400;

  idm_f_note = 501;
  idm_f_key = 502;
  idm_f_ctrl = 503;
  idm_f_pgm = 504;
  idm_f_chn = 505;
  idm_f_bend = 506;
  idm_f_syst = 507;
  idm_f_rtim = 508;
  idm_f_syx = 509;
  idm_f_mim = 510;
  idm_f_mom = 511;

  { submenu positions }
  mpos_input = 0;
  mpos_output = 1;
  mpos_filter = 4;

  { Dialog IDs }
  idd_list = 101;
  { max No of Lines in List }
  max_list = 100;

  { size of SYSEX buffers }
  syx_size = 2000; { more can't be displayed with 100 lines }
  syx_nums = 3;
  msg_queue_len = 100; { 8.. 120 }

{ TMidiMonDlg is the main window of the application }

TYPE
  PMidiMonDlg = ^TMidiMonDlg;
  TMidiMonDlg = object(TDlgWindow)
    wInputNums : integer;
    wCurInput  : integer;
    hmid : HMIDI;

    wOutputNums : integer;
    wCurOutput  : integer;
    hmod : HMIDI;

    midihdr : array [0..syx_nums-1] of TMIDIHDR;

    mp_list : PListBox ;
    fCycleBuffers : boolean;
    wFilter : Word; { Bitmask corresponding to Filter Menu; 1=checked }
    fStarted : boolean;

    lTimeStamp : Longint; { Timestamp }

    constructor Init;
    destructor done; virtual;
    procedure SetupWindow; virtual;
    function GetClassName: PChar; virtual;

    procedure DoStart(var Msg: TMessage);
      virtual cm_First + idm_Start;
    procedure DoClear(var Msg: TMessage);
      virtual cm_First + idm_Clear;

    procedure ListBeginWrite;
    procedure ListEndWrite;
    procedure ListWrite(a_text:PChar);
    procedure ListVPrintf(fmt:PChar; var arglist);
    procedure ListPrintMMsg(desc:PChar;MidiMsg:Longint);
    procedure ListPrintLMsg(desc:Pchar;data:PChar;Len:WORD);

    procedure SetupInputs;
    procedure SetInput(newCurInput:integer);
    procedure SetupOutputs;
    procedure SetOutput(newCurOutput:integer);

    procedure AddBuffer(lpmh:PMidiHdr);
    procedure SendBuffer(lpmh:PMidiHdr);

    procedure OpenInput;
    procedure OpenOutput;
    procedure CloseOutput;
    procedure CloseInput;

    function Filter(bByte:BYTE):boolean; { true if Msg is to display }
    procedure CheckFilter(bpos:integer);

    procedure wmCommand(var Msg:TMessage); virtual wm_first+wm_command;

    procedure mmMimOpen(var Msg:TMessage); virtual wm_first+MM_MIM_OPEN;
    procedure mmMimClose(var Msg:TMessage); virtual wm_first+MM_MIM_CLOSE;
    procedure mmMimData(var Msg:TMessage); virtual wm_first+MM_MIM_DATA;
    procedure mmMimError(var Msg:TMessage); virtual wm_first+MM_MIM_ERROR;
    procedure mmMimLongData(var Msg:TMessage); virtual wm_first+MM_MIM_LONGDATA;
    procedure mmMimLongError(var Msg:TMessage); virtual wm_first+MM_MIM_LONGERROR;
    procedure mmMimTimeStamp(var Msg:TMessage); virtual wm_first+MM_MIM_TIMESTAMP;
              {non-MMSYSTEM msg. = WM_USER+1}

    procedure mmMomOpen(var Msg:TMessage); virtual wm_first+MM_MOM_OPEN;
    procedure mmMomClose(var Msg:TMessage); virtual wm_first+MM_MOM_CLOSE;
    procedure mmMomDone(var Msg:TMessage); virtual wm_first+MM_MOM_DONE;

    function ShowMIM:boolean;
    function ShowMOM:boolean;
  end;

{ TMidiMonApp is the application object. It creates a main window of
  type TMidiMonDlg. }

  TMidiMonApp = object(TApplication)
    procedure InitMainWindow; virtual;
  end;


function MidiMsgLen(status:byte):byte;
const chan_len : array [8..$E] of Byte = (3,3,3,3,2,2,3);
const sys_len : array [0..3] of Byte = (4,2,3,2);
begin
   if      status >=$F4 then MidiMsgLen:= 1
   else if status >=$F0 then MidiMsgLen:= sys_len [ status and 3 ]
   else if status >=$80 then MidiMsgLen:= chan_len[ status shr 4 ]
   else                      MidiMsgLen:= 4
end;


{ TMidiMonDlg }

{ dialog constructor. }

constructor TMidiMonDlg.Init;
begin
  TDlgWindow.Init(nil, PChar(id_Dialog));
  mp_list := New(PListBox,InitResource(@self,idd_list));
  mp_list^.Attr.Style := mp_list^.Attr.Style and (not LBS_SORT);

  wInputNums := 0;
  wCurInput := 0;
  hmid := 0;

  wOutputNums := 0;
  wCurOutput := 0;
  hmod := 0;

  wFilter := 0;
  fStarted := false;

  lTimeStamp := 0;
end;

destructor TMidiMonDlg.Done;
var i:integer;
begin
  CloseInput;
  CloseOutput;
  TDlgWindow.Done;
  for i:=0 to syx_nums-1 do begin
      GlobalFreePtr(midihdr[i].lpData);
      midihdr[i].lpData := nil;
  end;
end;
{ SetupWindow is called right after the Convert dialog is created. }

procedure TMidiMonDlg.SetupWindow;
var i:integer;
    mh : THandle;
begin
   inherited SetupWindow;
   SetupInputs;
   SetInput(0);
   SetupOutputs;
   SetOutput(0);
   for i:=0 to syx_nums-1 do begin
       midihdr[i].lpData := GlobalAllocPtr(GMEM_MOVEABLE or GMEM_SHARE,syx_size);
       midihdr[i].dwBufferLength := syx_size;
       midihdr[i].dwUser := i;
       midihdr[i].dwFlags := 0;
       midihdr[i].lpNext := nil;
       midihdr[i].reserved := 0;
   end;
end;

{ add available Midi Inputs to Input Menu }
procedure TMidiMonDlg.SetupInputs;
VAR mc : TMIDIINCAPS;
    i: integer;
    h_pmenu : HMENU;
begin
   h_pmenu := GetSubMenu(GetMenu(hWindow),mpos_input);
   wInputNums := midiInGetNumDevs;
   for i:=0 to wInputNums-1 do
   begin
      midiInGetDevCaps(i,@mc,sizeof(mc));
      if (h_pmenu<>0) then
         AppendMenu(h_pmenu,MF_STRING,idm_input_none+1+i,mc.szPName);
   end;
end;

{ Check Midi Input Menu }
procedure TMidiMonDlg.SetInput(newCurInput:integer);
VAR
    i: integer;
    h_pmenu : HMENU;
    chk : Word;
begin
   h_pmenu := GetSubMenu(GetMenu(hWindow),mpos_input);
   if newCurInput>wInputNums then newCurInput:=0;
   wCurInput := newCurInput;
   for i:=0 to wInputNums do begin
       if h_pmenu<>0 then begin
          if i=wCurInput then chk := MF_CHECKED else chk := MF_UNCHECKED;
          CheckMenuItem(h_pmenu,idm_input_none+i,MF_BYCOMMAND or chk);
       end;
   end;
end;

{ add available Midi Outputs to Output Menu }
procedure TMidiMonDlg.SetupOutputs;
VAR mc : TMIDIOUTCAPS;
    i: integer;
    h_pmenu : HMENU;
begin
   h_pmenu := GetSubMenu(GetMenu(hWindow),mpos_output);
   wOutputNums := midiOutGetNumDevs + 1;
   for i:=-1 to wOutputNums-2 do { start with MIDI_MAPPER }
   begin
      midiOutGetDevCaps(i,@mc,sizeof(mc));
      if (h_pmenu<>0) then
         AppendMenu(h_pmenu,MF_STRING,idm_output_none+2+i,mc.szPName);
   end;
end;

{ Check Midi Output Menu }
procedure TMidiMonDlg.SetOutput(newCurOutput:integer);
VAR
    i: integer;
    h_pmenu : HMENU;
    chk : Word;
begin
   h_pmenu := GetSubMenu(GetMenu(hWindow),mpos_output);
   if newCurOutput>wOutputNums then newCurOutput:=0;
   wCurOutput := newCurOutput;
   for i:=0 to wOutputNums  do begin
       if h_pmenu<>0 then begin
          if i=wCurOutput then chk := MF_CHECKED else chk := MF_UNCHECKED;
          CheckMenuItem(h_pmenu,idm_output_none+i,MF_BYCOMMAND or chk);
       end;
   end;
end;

procedure modErr(HWindow:HWnd; err:WORD; caption:PChar);
var errbuf : array [0..255] of char;
begin
   if midiOutGetErrorText(err,errbuf,sizeof(errbuf)-1)<>0 then begin
      wvsprintf(errbuf,'Unknown Error %04Xh',err);

   end;
   if caption=nil then caption:='Midi Output Error';
   MessageBox(HWindow,errbuf,caption,MB_ICONSTOP or MB_OK);
end;

{ Open Midi Output }
procedure TMidiMonDlg.OpenOutput;
VAR err : WORD;
begin
   if hmod<>0 then CloseOutput;
   if wCurOutput >= 1 then begin
{$ifdef LOWLEVEL}
      err := midiOutOpen(@hmod,wCurOutput-2,LongInt(@MidiCallback),LongInt(hWindow),CALLBACK_FUNCTION);
{$else}
      err := midiOutOpen(@hmod,wCurOutput-2,LongInt(hWindow),0,CALLBACK_WINDOW);
{$ENDIF}
      if err<>0 then modErr(hWindow,err,'midiOutOpen');
   end;
end;

{ Close Midi Output }
procedure TMidiMonDlg.CloseOutput;
VAR
    i: integer;
    err : WORD;
begin
   if hmod <> 0 then begin
      err := midiOutReset(hmod);
      if err<>0 then modErr(hWindow,err,'midiOutReset');
      { at this point, we must wait for DONE_MESSAGES }
      err := midiOutClose(hmod);
      if err<>0 then modErr(hWindow,err,'midiOutClose');
      { at this point, we must wait for CLOSE_MESSAGES }
      hmod := 0;
  end;
end;

procedure midErr(HWindow:HWnd; err:WORD; caption:PChar);
var errbuf : array [0..MaxErrorLength] of char;
begin
   if midiInGetErrorText(err,errbuf,sizeof(errbuf)-1)<>0 then begin
      wvsprintf(errbuf,'Unknown Error %04Xh',err);
   end;
   if caption=nil then caption:='Midi Input Error';
   MessageBox(HWindow,errbuf,caption,MB_ICONSTOP or MB_OK);
end;

{ Open Midi Input }
procedure TMidiMonDlg.OpenInput;
VAR err : WORD;
begin
   if hmid<>0 then CloseInput;

   { error - MM_MIM Messages must be processed first ! (how?) }

   if wCurInput >= 1 then begin
{$ifdef LOWLEVEL}
      err := midiInOpen(@hmid,wCurInput-1,LongInt(@MidiCBTimeStamp),LongInt(hWindow),CALLBACK_FUNCTION);
{$else}
      err := midiInOpen(@hmid,wCurInput-1,LongInt(hWindow),0,CALLBACK_WINDOW);
{$ENDIF}
      if err<>0 then midErr(hWindow,err,'midiInOpen');
      if hmid=0 then exit;
      fCycleBuffers := true;

   end;
end;

procedure TMidiMonDlg.AddBuffer(lpmh:PMIDIHDR);
var err:WORD;
begin
   if (hmid<>0) and (lpmh<>nil) then begin
          lpmh^.dwBufferLength := syx_size;
          lpmh^.dwBytesrecorded := 0;
          {lpmh^.dwFlags := 0;}
          err := midiInPrepareHeader(hmid,lpmh,sizeof(TMIDIHDR));
          if err<>0 then midErr(hWindow,err,'midiInPrepareHeader');
          err := midiInAddBuffer(hmid,lpmh,sizeof(TMIDIHDR));
          {if err<>0 then midErr(hWindow,err,'midiInAddBuffer');}
   end;
end;

procedure TMidiMonDlg.SendBuffer(lpmh:PMIDIHDR);
var err:WORD;
begin
   if (hmod<>0) and (lpmh<>nil) then begin
          lpmh^.dwFlags := 0;
          err := midiOutPrepareHeader(hmod,lpmh,sizeof(TMIDIHDR));
          if err<>0 then modErr(hWindow,err,'midiOutPrepareHeader');
          err := midiOutLongMsg(hmod,lpmh,sizeof(TMIDIHDR));
          if err<>0 then modErr(hWindow,err,'midiOutLongMsg');
   end else
       if fCycleBuffers then addBuffer(lpmh);
end;

{ Close Midi Input }
procedure TMidiMonDlg.CloseInput;
VAR
    err : WORD;
begin
   if hmid <> 0 then begin
      fCycleBuffers := false;
      err := midiInReset(hmid);
      { at this point, we must wait for LONGDATA_MESSAGES }
      if err<>0 then midErr(hWindow,err,'midiInReset');
      err := midiInClose(hmid);
      { at this point, we must wait for DONE_MESSAGES }
      if err<>0 then midErr(hWindow,err,'midiInClose');
      hmid := 0;
  end;
end;

{ Return window class name. This name corresponds to the class name
  specified for the Convert dialog in the resource file. }

function TMidiMonDlg.GetClassName: PChar;
begin
  GetClassName := 'MidiMonDlg';
end;

procedure TMidiMonDlg.ListBeginWrite;
begin
   if mp_list=nil then exit;
   SendMessage(mp_list^.HWindow,WM_SETREDRAW,0,0);
end;

procedure TMidiMonDlg.ListEndWrite;
begin
   if mp_list=nil then exit;
   SendMessage(mp_list^.HWindow,WM_SETREDRAW,1,0);
(*   InvalidateRect(mp_list^.HWindow,nil,true);
   UpdateWindow(mp_list^.HWindow);
*)
end;

procedure TMidiMonDlg.ListWrite(a_text:PChar);
begin
   if mp_list=nil then exit;
   mp_list^.SetSelIndex(-1);
   if mp_list^.GetCount >=max_list then mp_list^.DeleteString(0);
   mp_list^.SetSelIndex(mp_list^.AddString(a_text));
end;

procedure TMidiMonDlg.ListVPrintf(fmt:PChar;var arglist);
var buf: array [0..127] of char;
begin
   wvsprintf(buf,fmt,arglist);
   ListWrite(buf);
end;

procedure TMidiMonDlg.ListPrintMMsg(desc:Pchar;midimsg:Longint);
const msg_format : PChar = '%02X %02X %02X %02X';
var arglist: record case integer of
        0:(w:array [0..3] of word);
        1:(t:Longint; s:array [0..1] of PChar)
        end;
    dbuf : array [0..13] of char;
    err : WORD;
    p_dbuf:PChar;
    bstatus:Byte;
begin
   bstatus := LoByte(LoWord(midimsg));
   if fStarted and Filter(bstatus) then begin
    arglist.w[0] := bstatus;
    arglist.w[1] := HiByte(LoWord(midimsg));
    arglist.w[2] := LoByte(HiWord(midimsg));
    arglist.w[3] := HiByte(HiWord(midimsg));
    wvsprintf(dbuf,PChar(LongInt(msg_format)+5*(4-MidiMsgLen(bstatus))),arglist);
    arglist.t := lTimeStamp;
    arglist.s[0] := desc;
    arglist.s[1] := dbuf;
    ListBeginWrite;
    if ShowMIM then ListVPrintf('%8ld %s : %s',arglist) else
    ListVPrintf('%s',arglist.s[1]);
    ListEndWrite;
   end;
end;

procedure TMidiMonDlg.ListPrintLMsg(desc:Pchar;data:PChar;Len:WORD);
const bytes_perline = 20;
var arglist: record t:Longint; s: PChar; w: WORD; end;
    dbuf : array [0..3*bytes_perline] of char;
    wLine,wData,argData:WORD;
begin
   if (not fStarted) or (not Filter($F0)) then exit;
   ListBeginWrite;
   arglist.t := lTimeStamp;
   arglist.s := desc;
   arglist.w := Len;
   if ShowMIM then ListVPrintf('%8ld - %s : %u bytes',arglist);
   wData := 0;
   While (wData<Len) do begin
         wLine := 0;
         while (wData<Len) and (wLine<bytes_perline) do begin
            argData := Byte(data[wData]);
            wvsprintf(@dbuf[3*wLine],' %02X',argData);
            Inc(wLine);
            Inc(wData);
         end;
         ListWrite(dbuf);
   end;
   ListEndWrite;
end;

procedure TMidiMonDlg.DoStart(var Msg: TMessage);
var p_item : PChar;
BEGIN
   fStarted := not fStarted;
   if fStarted then p_item:='&Stop' else p_item:='&Start';
   ModifyMenu(GetMenu(hWindow),idm_start,MF_BYCOMMAND,idm_start,p_item);
   DrawMenuBar(hWindow);
END;

procedure TMidiMonDlg.DoClear(var Msg: TMessage);
BEGIN
   mp_list^.ClearList;
END;

{ dispatch MidiInput-Output Menu Commands }
procedure TMidiMonDlg.wmCommand(var Msg:TMessage);
var i:integer;
BEGIN
   if (Msg.LParam = 0) then begin { a menu command }
      i := Msg.WParam - idm_input_none;
      if ( i >= 0) and ( i <= wInputNums ) then begin
         SetInput(i);
         OpenInput;
         if hmid=0 then SetInput(0);
         Msg.Result := 0;
         exit;
      end;
      i := Msg.WParam - idm_output_none;
      if ( i >= 0) and ( i <= wOutputNums ) then begin
         SetOutput(i);
         OpenOutput;
         if hmod=0 then SetOutput(0);
         Msg.Result := 0;
         exit;
      end;
      if (Msg.WParam >= idm_f_note) and (Msg.WParam <= idm_f_mom) then begin
         CheckFilter(Msg.WParam - idm_f_note);
         Msg.Result := 0;
         exit;
      end;
   end;
   inherited wmCommand(Msg);
END;

function TMidiMonDlg.ShowMIM:boolean;
begin
   ShowMIM:=(wFilter and $0200)=0;
end;
function TMidiMonDlg.ShowMOM:boolean;
begin
   ShowMOM:=(wFilter and $0400)=0;
end;

procedure TMidiMonDlg.mmMimOpen(var Msg:TMessage);
var err: Word;
    i : integer;
begin
   if ShowMIM then
      ListVPrintf('MM_MIM_OPEN (%04X)',Msg.WParam);
   for i:=0 to syx_nums-1 do AddBuffer(@midihdr[i]);
   err := midiInStart(hmid);
   if err<>0 then midErr(hWindow,err,'midiInStart');
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMimClose(var Msg:TMessage);
begin
   if ShowMIM then
      ListVPrintf('MM_MIM_CLOSE (%04X)',Msg.WParam);
   Msg.Result := 0;
end;

{ if output open send Msg }
procedure TMidiMonDlg.mmMimData(var Msg:TMessage);
var err : WORD;
begin
   if hmod<>0 then begin
      err:=midiOutShortMsg(hmod,Msg.Lparam);
      if err<>0 then modErr(hWindow,err,'midiOutShortMsg');
      end;
   ListPrintMMsg('MM_MIM_DATA',Msg.LParam);
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMimError(var Msg:TMessage);
begin
   if ShowMIM then
      ListPrintMMsg('MM_MIM_ERROR',Msg.LParam);
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMimLongData(var Msg:TMessage);
var
    dwRec : LongInt;
    bnum : Longint;
    lpmh : PMIDIHDR;
    err : word;
begin
   lpmh := PMIDIHDR(Msg.LParam);
   dwRec := lpmh^.dwBytesRecorded;
   bnum := lpmh^.dwUser;
   if hmid<>0 then begin
      err := midiInUnprepareHeader(hmid,lpmh,sizeof(TMIDIHDR));
      if err<>0 then midErr(hWindow,err,'midiInUnprepareHeader');
      end;
   if (dwRec<>0) and (hmod<>0) then begin
      lpmh^.dwBufferLength := dwRec;
      SendBuffer(lpmh);
      end
   else if fCycleBuffers then AddBuffer(lpmh);
   if ShowMIM then
      ListPrintLMsg('MM_MIM_LONGDATA',lpmh^.lpdata,WORD(dwRec))
   else
      ListPrintLMsg('',lpmh^.lpdata,WORD(dwRec));
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMimLongError(var Msg:TMessage);
var dwRec : LongInt;
    bnum : LongInt;
    lpmh : PMIDIHDR;
    err : WORD;
begin
   lpmh := PMIDIHDR(Msg.LParam);
   dwRec := lpmh^.dwBytesRecorded;
   bnum := lpmh^.dwUser;
   {ListVPrintf('MM_MIM_LONGERROR %ld - %5ld bytes',bnum);}
   if ShowMIM then
      ListPrintLMsg('MM_MIM_LONGERROR',lpmh^.lpdata,WORD(dwRec));
   err := midiInUnprepareHeader(Msg.WParam,lpmh,sizeof(TMIDIHDR));
   if err<>0 then midErr(hWindow,err,'midiInUnprepareHeader');
   if fCycleBuffers then AddBuffer(lpmh);
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMimTimeStamp(var Msg:TMessage);
begin
   lTimeStamp := Msg.LParam;
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMomOpen(var Msg:TMessage);
begin
   if ShowMOM then ListVPrintf('MM_MOM_OPEN (%04X)',Msg.WParam);
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMomClose(var Msg:TMessage);
begin
   if ShowMOM then ListVPrintf('MM_MOM_CLOSE (%04X)',Msg.WParam);
   Msg.Result := 0;
end;

procedure TMidiMonDlg.mmMomDone(var Msg:TMessage);
var dwSent : LongInt;
    bnum : LongInt;
    lpmh : PMIDIHDR;
    err : Word;
begin
   lpmh := PMIDIHDR(Msg.LParam);
   dwSent := lpmh^.dwBufferLength;
   bnum := lpmh^.dwUser;
   if ShowMOM then ListVPrintf('MM_MOM_DONE %ld - %ld bytes sent',bnum);
   err := midiOutUnprepareHeader(Msg.WParam,lpmh,sizeof(TMIDIHDR));
   if err<>0 then modErr(hWindow,err,'midiOutUnprepareHeader');
   if fCycleBuffers then AddBuffer(lpmh);
   Msg.Result := 0;
end;

function TMidiMonDlg.Filter(bByte:BYTE):boolean;
var i : WORD;
begin
   i := 0;
   case bByte of
      $80..$9F : i:=1;
      $A0..$AF : i:=2;
      $B0..$BF : i:=4;
      $C0..$CF : i:=8;
      $D0..$DF : i:=16;
      $E0..$EF : i:=32;
      $F1..$F6 : i:=64;
      $F8..$FF : i:=128;
      $F0,$F7 :  i:=256;
   end;
   Filter := (i and WFilter)=0;
end;

procedure TMidiMonDlg.CheckFilter(bpos:integer);
VAR
    i: WORD;
    h_pmenu : HMENU;
begin
   h_pmenu := GetSubMenu(GetMenu(hWindow),mpos_filter);
   if h_pmenu<>0 then begin
      i := GetMenuState(h_pmenu,bpos,MF_BYPOSITION);
      i := i xor MF_CHECKED;
      CheckMenuItem(h_pmenu,bpos,MF_BYPOSITION or i);
      if (i and MF_CHECKED)<>0 then
         wFilter := wFilter or (1 shl bpos)
      else
         wFilter := wFilter and (not (1 shl bpos));
   end;
end;

{ TMidiMonApp }

{ Create a Midimon dialog as the application's main window. }

procedure TMidiMonApp.InitMainWindow;
begin
  MainWindow := New(PMidiMonDlg, Init);
end;

var
  MidiMonApp: TMidiMonApp;

begin
  SetMessageQueue(msg_queue_len);
  MidiMonApp.Init('MidiMonApp');
  MidiMonApp.Run;
  MidiMonApp.Done;
end.
