// **********************************************
// Definitions of class types

#ifndef _DATABASE_H_
#define _DATABASE_H_

#define WIN31
#include <owl.h>
#include <array.h>
#include <iostream.h>
#include "dbllist.hpp"

typedef int BOOL;          //  override Borland type.

const MELODY           = 0;
const PART             = 1;
const STAFF            = 2;
const POINTOBJECT      = 3;
const CONTINUOUSOBJECT = 4;

const INSTAFF        = 0;
const COMMONMULTIPLE = 1;
const ABOVESTAFF     = 2;
const ABOVEMULTIPLE  = 3;
const BELOWSTAFF     = 4;
const BELOWMULTIPLE  = 5;

const ONEPERSTAFF = 0x81;

const FULLDURATION      = 0x100;
const HALFDURATION      = 0x80;
const QUARTERDURATION   = 0x40;
const EIGHTHDURATION    = 0x20;
const SIXTEENTHDURATION = 0x10;
const THIRTY2DURATION   = 0x08;

const KEYSOL = 0;
const KEYFA  = 1;

const BEATC    = 0;
const BEATCBAR = 1;
const BEAT28   = 2;
const BEAT24   = 3;
const BEAT38   = 4;
const BEAT34   = 5;
const BEAT48   = 6;
const BEAT44   = 7;
const BEAT68   = 8;

const BAR       = 0;
const DOUBLEBAR = 1;
const STARTBAR  = 2;
const ENDBAR    = 3;

const FORTE      = 1;       // the Minus sign is to ger the opposit effect.
const PIANO      = -1;
const FORTISSIMO = 2;
const PIANISSIMO = -2;
const CRESCENDO  = 1;
const DIMINUENDO = -1;

// The order of the next line parameters is VERY important.
// don't change it unless u sure u know what u're doing.
typedef enum {PENCIL, ERASER, HAND, TEXT, MARK_BLOCK, CONNECT, NOTE,
              PASTE_BLOCK, KEYSIG} EDITMODE;
typedef enum {SINGLESTAFF, FIRSTSTAFF, MIDSTAFF, LASTSTAFF} STAFFLOC;

const MAXPARTNAME     = 10;
const MAXMULTIPLICITY = 4;
const MAXTEXT         = 15;

// **********************************************
// Definitions of global variables

//-------------------------------------------------------------------
//          class    I n d e x e d L i s t
//-------------------------------------------------------------------
class IndexedList : public TDoubleList {
  public:
   IndexedList();
   IndexedList(int anUpper, int aLower, int aDelta);
   friend TDoubleListIterator;
   int number() const;
   void insertAt(Object &, int);
   void detachAt(int);
   void destroyAt(int);
   void printOn(ostream &) const;
};


//-------------------------------------------------------------------
//          class    I n d e x e d L i s t
//-------------------------------------------------------------------
class TDoubleListIterator {
    const TDoubleList* _doubleList;
    Item* _currItem;
  public:
    TDoubleListIterator(const TDoubleList *);
    TDoubleListIterator(const IndexedList *);
    ~TDoubleListIterator();
    virtual Object* first();
    virtual Object* next();
    virtual BOOL last();
};


//-------------------------------------------------------------------
//                       M e l o d y  classes.
//-------------------------------------------------------------------
class MelodyParameters {
   unsigned staffWidth, _tempo;

  protected:
   BOOL LoadFrom(istream &, void (*)());
   void printOn(ostream &);
  public:
   MelodyParameters() { _tempo = 120; };
   unsigned GetStaffWidth() {return staffWidth;}
   void SetStaffWidth(unsigned newWidth) {staffWidth = newWidth;}
   int tempo() { return _tempo; };
   void SetTempo(int tempo) { _tempo = tempo; };
};

class Melody : public MelodyParameters {
  public:
    // Those are parametrs that are not saved.
    BOOL melodyModified;
    BOOL melodyExists;
    BOOL scoreDisplay;
    int scoreMultiplicity;
    int scoreStaves;
    int displayedPart;
    unsigned pixelsPerStaff;   // Height of a staff in pixels
    unsigned pixelsPerObject;  // Width of an object in pixels
    IndexedList part;
    Melody();
    ~Melody();
    BOOL LoadFrom(istream &, void (*)());
    void printOn(ostream &);
};

extern Melody melody;        

//-------------------------------------------------------------------
//          P a r t      classes
//-------------------------------------------------------------------
class PartParameters {
  protected:
   char _name[MAXPARTNAME+1];
   int _multiplicity;
   unsigned editYMin;
   BOOL LoadFrom(istream &, void (*)());
   void printOn(ostream &);
   int _instNum;
  public:
   char *name() {return _name;}
   int &multiplicity() {return _multiplicity;}
   void SetPartY(unsigned Y) {editYMin = Y;}
   unsigned GetPartY() {return editYMin;}
   virtual void SetInstNum(int inst) { _instNum = inst; };
   virtual int GetInstNum() { return _instNum; };
};

class Part : public Object, public PartParameters {
  public:
    IndexedList staff;
    Part();
    Part(const char *, int);
    ~Part();
    virtual void ChangeAttrib(const char *, int);
    virtual classType isA() const {return PART;}
    virtual char *nameOf() const {return "Part";}
    virtual hashValueType hashValue() const {return 0;}
    virtual int isEqual(const Object &obj) const
     {return (obj.isA() == PART);}
    BOOL LoadFrom(istream &, void (*)());
    virtual void printOn(ostream &) const;
};
//-------------------------------------------------------------------
//          S t a f f    classes
//-------------------------------------------------------------------
class StaffParameters {
  protected:
   unsigned _X, _Y, _width;
   BOOL LoadFrom(istream &, void (*)());
   void printOn(ostream &);
  public:
   unsigned &X() {return _X;}
   unsigned &Y() {return _Y;}
   unsigned &width() {return _width;}
};

class Staff : public Object, public StaffParameters {
  public:
   IndexedList pointObject;
   IndexedList continuousObject;
   Staff();
   Staff(int, unsigned);
   ~Staff();
   BOOL Draw(HDC, int, BOOL, unsigned);
   virtual classType isA() const {return STAFF;}
   virtual char *nameOf() const {return "Staff";}
   virtual hashValueType hashValue() const {return 0;}
   virtual int isEqual(const Object &obj) const
    {return (obj.isA() == STAFF);}
   BOOL LoadFrom(istream &, void (*)());
   virtual void printOn(ostream &) const;
};

//-------------------------------------------------------------------
//          M u s i c a l    O b j e c t s    classes
//-------------------------------------------------------------------
// **********************************************
// The MusicalObject class is the base class from which all musical
// object types are derived. It defines a few virtual functions that
// should be given own versions in the derived classes.
// The tree starting at MusicalObject divides into two main subtrees:
// point objects (derived from PointObject), who have only one (if at all)
// X coordinate, and continuous objects (derived from ContinuousObject),
// which have a starting and ending X coordinate.

class MusicalObject : public Object {
  protected:
   int _location;
  public:
   int &location() {return _location;}
   virtual void Draw(HDC, int, int, int, int) = 0;
   virtual classType isA() const = 0;
   virtual char *nameOf() const = 0;
   virtual hashValueType hashValue() const = 0;
   virtual int isEqual(const Object &) const = 0;
   virtual void printOn(ostream &) const = 0;
   virtual void clipOn(void far *&) const = 0;
};

class PointObject : public MusicalObject {
  protected:
   int _X;
  public:
   int &X() {return _X;}
   virtual void Draw(HDC, int ,int, int, int) = 0;
   virtual void Format(int &X) {_X = X;}
   virtual int Width() {return melody.pixelsPerObject;}
   virtual void MIDIPlay(int &type, int &, int &, int &) { type = 0; };
   virtual int Duration() {return 0;}
   virtual classType isA() const {return POINTOBJECT;}
   virtual char *nameOf() const {return "PointObject";}
   virtual hashValueType hashValue() const {return 0;}
   virtual int isEqual(const Object &obj) const
    {return (obj.isA() == POINTOBJECT);}
   virtual void printOn(ostream &) const = 0;
   virtual void clipOn(void far *&) const = 0;
   // a virtual function used only by key. (to change its signature)
   virtual void ChangeSignature(int, int) {};

};

class ContinuousObject : public MusicalObject {
  protected:
   int _Xleft, _Xright;
  public:
   int &Xleft() {return _Xleft;}
   int &Xright() {return _Xright;}
   virtual void Draw(HDC, int, int, int, int) = 0;
   virtual void FormatLeft(int &Xleft) {_Xleft = Xleft;}
   virtual void FormatRight(int &Xright) {_Xright = Xright;}
   virtual classType isA() const {return CONTINUOUSOBJECT;}
   virtual char *nameOf() const {return "ContinuousObject";}
   virtual hashValueType hashValue() const {return 0;}
   virtual int isEqual(const Object &obj) const
      {return (obj.isA() == CONTINUOUSOBJECT);}
   virtual void printOn(ostream &) const = 0;
   virtual void clipOn(void far *&) const = 0;
};


// **********************************************
// The SymbolClass class is a generic type
// from which symbol class types are derived.

class SymbolClass {
  protected:
   int symbolID;
   char *_bitmapName;
   char symbolType;
  public:
   int GetID() {return symbolID;}
   char GetType() {return symbolType;}
   int BitmapName(LPSTR, int);
   void DrawSymbol(HDC, HDC, BOOL);
   virtual MusicalObject *CreateObject(int, int, int, unsigned) = 0;
};

extern class NullSymbol : virtual public SymbolClass {
  public:
    NullSymbol();
    void DrawSymbol(HDC, int);
#pragma argsused
// The "argsused" pragma is only allowed between function definitions,
// and it affects only the next function.
// It disables the warning message  "Parameter 'parameter' is never used"
    virtual MusicalObject *CreateObject(int staff, int X, int Y, unsigned width) {return NULL;}
} nullSymbol;

extern SymbolClass *symbolArray[]; // The symbol array (SYMCLASS.CPP)
extern SymbolClass *NoteSymbols[];
extern SymbolClass *NoteSharpSymbols[];
extern SymbolClass *KeySymbols[];
extern SymbolClass *BeatSymbols[];
extern SymbolClass *BarSymbols[];
extern SymbolClass *LoudnessSymbols[];
extern SymbolClass *TextSymbols[];

#endif