/*-----------------------------------------------------*\
 | Muzika 2.0 - A musical editor. using Windows & OWL  |
 |                                                     |
 |   Main :                                            |
 |                                                     |
\*-----------------------------------------------------*/
#define WIN31
#include <owl.h>
#include <string.h>
#include <stdlib.h>
#include <values.h>
#include <filedial.h>

#include "muzika.h"
#include "database.h"
#include "menu.h"
#include "file.hpp"
#include "toolsdlg.hpp"
#include "toolbar.hpp"
#include "pntwnd.hpp"
#include "objects.h"
#include "print.h"
#include "layout.h"
#include "statwin.h"
#include "midi.hpp"
#include "block.hpp"
#include "mididlg.hpp"

#define COMMANDMSG(arg) (arg.WParam)

extern SymbolClass* BlockSymbols[];
extern TBlock MarkedBlock;

HWND hMainWnd;
HANDLE hInst;

Melody melody;

extern void PrintSinglePart(HDC hPrinterDC, int part);
extern void PrintScore(HDC hPrinterDC);

void LoadError();

char programTitle[MAXPATH+7] = "MUZIKA "; // Main window caption

/*-----------------------------------------------------*\
 |  Main Muzika Object - TMuzikaApplication            |
\*-----------------------------------------------------*/

class TMuzikaApplication : public TApplication
{
  public:
    TMuzikaApplication (LPSTR lpszName, HANDLE hInstance,
      HANDLE hPrevInstance, LPSTR lpszCmdLine,
      int nCmdShow);
    virtual void InitMainWindow ();
    virtual BOOL CanClose();
};

class TMuzikaWindow : public TWindow
{
  private:
    TPrinter *Printer;
    TPaintWindow *PaintWindow;
    TStatusWindow *StatusWindow;
    TToolBar *ToolBar;
    TNotesToolDialog *ptNotesDlg;
    TToolDialog *ptClefsDlg,*ptBeatsDlg,*ptBarsDlg,
            *ptDynamicsDlg, *ptActiveTool;
    int ActiveSymbol;
    TToolDialog *Tool;
    char *filepath;
    char _muzika_path[MAXPATH];
  public:
   TMuzikaWindow (PTWindowsObject pwParent, LPSTR lpszTitle,
       PTModule pmModule, LPSTR muzika_path);
   ~TMuzikaWindow();
   virtual LPSTR GetClassName();
   virtual void GetWindowClass(WNDCLASS&);
   void EnableMenus();
   virtual void WMSize( TMessage& Message ) = [WM_FIRST+WM_SIZE];
   virtual void CMFileNew(TMessage&) = [CM_FIRST + IDM_FILE_NEW];
   virtual void CMFileOpen(TMessage&) = [CM_FIRST + IDM_FILE_OPEN];
   virtual void CMFileSave(TMessage&) = [CM_FIRST + IDM_FILE_SAVE];
   virtual void CMFileSaveAs(TMessage&) = [CM_FIRST + IDM_FILE_SAVEAS];
   virtual void CMFilePrint(TMessage&) = [CM_FIRST + IDM_FILE_PRINT];
   virtual void CMFilePrinterSetup(TMessage&) = [CM_FIRST + IDM_FILE_PRINTERSETUP];
   virtual void CMFileCreateMidi(TMessage&) = [CM_FIRST + IDM_FILE_CREATEMIDI];
   virtual void CMFileMidiSetup(TMessage&) = [CM_FIRST + IDM_FILE_MIDISETUP];
   virtual void SaveAs();

   virtual void CMEditCopy(TMessage&) = [CM_FIRST + IDM_EDIT_COPY];
   virtual void CMEditCut(TMessage&) = [CM_FIRST + IDM_EDIT_CUT];
   virtual void CMEditPaste(TMessage&) = [CM_FIRST + IDM_EDIT_PASTE];
   virtual void CMEditDelete(TMessage&) = [CM_FIRST + IDM_EDIT_DELETE];

   virtual void CMLayoutParts(TMessage&) =[CM_FIRST + IDM_LAYOUT_PARTS];
   virtual void CMLayoutPage(TMessage&) =[CM_FIRST + IDM_LAYOUT_PAGE];
   virtual void CMLayoutReformat(TMessage&) =[CM_FIRST + IDM_LAYOUT_REFORMAT];

   virtual void CMNotes(TMessage&) = [CM_FIRST + IDM_SYMBOLS_NOTES];
   virtual void CMClefs(TMessage&) = [CM_FIRST + IDM_SYMBOLS_CLEFS];
   virtual void CMBeats(TMessage&) = [CM_FIRST + IDM_SYMBOLS_BEATS];
   virtual void CMBars(TMessage&)  = [CM_FIRST + IDM_SYMBOLS_BARS];
   virtual void CMDynamics(TMessage&) = [CM_FIRST + IDM_SYMBOLS_DYNAMICS];
   virtual void CMKeySignature(TMessage&) = [CM_FIRST + IDM_SYMBOLS_KEYSIG];

   virtual void CMFileAbout(TMessage&) = [CM_FIRST + IDM_HELP_ABOUT];

   virtual void CMFileExit(TMessage&) = [CM_FIRST + IDM_FILE_EXIT];
// Utility Functions
   BOOL AskSave();
};

TMuzikaWindow *MuzWindow;

class TActiveSymDlg : public TDialog
{
   int ActiveSymbol;
  public:
    TActiveSymDlg(PTWindowsObject ptParent, LPSTR lpszName)
                   : TDialog(ptParent, lpszName) { ActiveSymbol = -1; };
    int GetActiveSymbol() { return ActiveSymbol; };
    virtual void CMActSymPencil(TMessage&) = [ID_FIRST+500];
    virtual void CMActSymTools(TMessage&) = [ID_FIRST+501];
    virtual void CMActSymEraser(TMessage&) = [ID_FIRST+502];
    virtual void WMSetCursor(TMessage& Msg) = [WM_SETCURSOR];
};


/*-----------------------------------------------------*\
 |  Main Muzika Function - WinMain                     |
\*-----------------------------------------------------*/

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
       LPSTR lpszCmdLine, int nCmdShow)
{
  HINSTANCE hBorLibrary;

  //Load Borland custom controls
  hBorLibrary = LoadLibrary("bwcc.dll");
  if((int) hBorLibrary <= 32)
    MessageBox(NULL, "Unable to load Borland Controls", "System Error", MB_OK | MB_ICONHAND);
  TMuzikaApplication Muzika("MUZIKA", hInstance, hPrevInstance,
          lpszCmdLine, nCmdShow);
  hInst = hInstance;
  Muzika.Run();
  if((int) hBorLibrary > 32)
    FreeLibrary(hBorLibrary);
  return Muzika.Status;
}

TMuzikaApplication::TMuzikaApplication(LPSTR lpszName, HANDLE hInstance,
                          HANDLE hPrevInstance, LPSTR lpszCmdLine,
             int nCmdShow)
  :TApplication(lpszName, hInstance, hPrevInstance, lpszCmdLine, nCmdShow)
{
  HAccTable = LoadAccelerators( hInstance, "#100");
}

void TMuzikaApplication::InitMainWindow()
{
  char path[MAXPATH];

  getcwd(path, MAXPATH);
  MuzWindow = new TMuzikaWindow(NULL, "MUZIKA", NULL, path);
  MainWindow = MuzWindow;
}

BOOL TMuzikaApplication::CanClose()
{
  return MuzWindow->AskSave();
}

TMuzikaWindow::TMuzikaWindow(PTWindowsObject pwParent, LPSTR lpszTitle,
            PTModule pmModule, LPSTR muzika_path)
  :TWindow(pwParent, lpszTitle, pmModule)
{
  RECT rc;
  GetWindowRect(GetDesktopWindow(), &rc);
  Attr.X = rc.left;
  Attr.Y = rc.top;
  Attr.W = rc.right;
  Attr.H = rc.bottom;
  strcpy(_muzika_path, muzika_path);
  Attr.Style |= WS_MAXIMIZE;
  ActiveSymbol = -1;
  Tool = NULL;
  Printer = new TPrinter;
  StatusWindow = new TStatusWindow(this, &melody);
  PaintWindow = new TPaintWindow(this, &ActiveSymbol, &Tool, StatusWindow);
  ToolBar = new TToolBar(this, &ActiveSymbol);
  filepath = programTitle+7;
  ptNotesDlg = NULL;
  ptClefsDlg = ptBeatsDlg =
           ptBarsDlg = ptDynamicsDlg = ptActiveTool = NULL;
  hMainWnd = HWindow;

}

TMuzikaWindow::~TMuzikaWindow()
{
  while (melody.part.number())
    melody.part.destroyAt(melody.part.number()-1);
  delete Printer;
  delete StatusWindow;
  delete PaintWindow;
  delete ToolBar;
}

LPSTR TMuzikaWindow::GetClassName()
{
  return "MUZIKA:MAIN";
}

void TMuzikaWindow::GetWindowClass(WNDCLASS& wc)
{
  TWindow::GetWindowClass(wc);
  wc.hIcon=LoadIcon(wc.hInstance, "I_MUZIKA");
  wc.lpszMenuName = "M_MUZIKA";
  wc.hbrBackground = COLOR_BACKGROUND + 1;
  wc.style |= WS_MAXIMIZE;
}

void TMuzikaWindow::EnableMenus()
{
  HMENU Menu = GetMenu(HWindow);
  if (!melody.melodyExists) {
   EnableMenuItem(Menu, IDM_FILE_NEW, MF_BYCOMMAND | MF_ENABLED);
   EnableMenuItem(Menu, IDM_FILE_OPEN, MF_BYCOMMAND | MF_ENABLED);
   EnableMenuItem(Menu, IDM_FILE_SAVE, MF_GRAYED);
   EnableMenuItem(Menu, IDM_FILE_SAVEAS, MF_BYCOMMAND | MF_GRAYED);
   EnableMenuItem(Menu, IDM_FILE_PRINT, MF_BYCOMMAND | MF_GRAYED);
   EnableMenuItem(Menu, IDM_FILE_PRINTERSETUP, MF_BYCOMMAND | MF_GRAYED);
   EnableMenuItem(Menu, IDM_FILE_CREATEMIDI, MF_BYCOMMAND | MF_GRAYED);
   EnableMenuItem(Menu, IDM_FILE_MIDISETUP, MF_BYCOMMAND | MF_GRAYED);
   EnableMenuItem(Menu, IDM_FILE_EXIT, MF_BYCOMMAND | MF_ENABLED);
   EnableMenuItem(Menu, IDM_EDIT_COPY, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_EDIT_CUT, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_EDIT_PASTE, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_EDIT_DELETE, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_LAYOUT_PARTS, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_LAYOUT_PAGE, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_LAYOUT_REFORMAT, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_SYMBOLS_NOTES, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_SYMBOLS_CLEFS, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_SYMBOLS_BEATS, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_SYMBOLS_BARS, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_SYMBOLS_DYNAMICS, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_SYMBOLS_KEYSIG, MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
   EnableMenuItem(Menu, IDM_HELP_ABOUT, MF_BYCOMMAND | MF_ENABLED);
  } else {
   EnableMenuItem(Menu, IDM_FILE_NEW, MF_BYCOMMAND | MF_ENABLED);
   EnableMenuItem(Menu, IDM_FILE_OPEN, MF_BYCOMMAND | MF_ENABLED);
   EnableMenuItem(Menu, IDM_FILE_SAVE, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_FILE_SAVEAS, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_FILE_PRINT, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_FILE_PRINTERSETUP, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_FILE_CREATEMIDI, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_FILE_MIDISETUP, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_EDIT_COPY, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_EDIT_CUT, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_EDIT_PASTE, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_EDIT_DELETE, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_LAYOUT_PARTS, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_LAYOUT_PAGE, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_LAYOUT_REFORMAT, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_SYMBOLS_NOTES, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_SYMBOLS_CLEFS, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_SYMBOLS_BEATS, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_SYMBOLS_BARS, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_SYMBOLS_DYNAMICS, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_SYMBOLS_KEYSIG, MF_BYCOMMAND);
   EnableMenuItem(Menu, IDM_HELP_ABOUT, MF_BYCOMMAND | MF_ENABLED);
  }
}

void TMuzikaWindow::WMSize( TMessage& )
{
  RECT R;

  GetClientRect(HWindow, &R);
  MoveWindow(ToolBar->HWindow, 4, 4, 31*NUM_OF_TOOLS+1, 32, TRUE);
  MoveWindow(PaintWindow->HWindow, 4, 40, R.right-8, R.bottom-44, TRUE);
  MoveWindow(StatusWindow->HWindow, R.right-254, 4, 250, 32, TRUE);
  EnableMenus();
}

void TMuzikaWindow::CMFileNew(TMessage&)
{
  int iRetVal;
  TDialogNew *ptDialogNew;
  char buffer[80];

  if (AskSave()) {
    ptDialogNew = new TDialogNew(this, "D_FILENEW");
    iRetVal = GetApplication()->ExecDialog((PTWindowsObject)ptDialogNew);
    if (iRetVal != 0) {
      // Set the melody staff width according to what the user selected
      melody.SetStaffWidth(iRetVal);
      // Initialize the database:
      // create a melody with only one UNNAMED part in it,
      // and reset the various Boolean flags.
      if (iRetVal > Attr.W) {
        PaintWindow->Scroller->SetRange(iRetVal/PaintWindow->Scroller->XUnit,
                           PaintWindow->Scroller->YRange);
      };
      while (melody.part.number())
        melody.part.destroyAt(melody.part.number()-1);
      melody.melodyExists = TRUE;
      melody.part.insertAt(*new Part("UNNAMED", 1), 0);
      melody.scoreDisplay = FALSE;
      melody.displayedPart = 0;
      filepath[0] = 0;
      melody.melodyModified = FALSE;
      MarkedBlock.Unmark();
      // Refresh the screen
      InvalidateRect(HWindow, NULL, TRUE);
    }
  }
  EnableMenus();
}

void TMuzikaWindow::CMFileOpen(TMessage&)
{
  char TempName[MAXPATH];

  _fstrcpy(TempName, "*.MUZ");
  if (GetApplication()->ExecDialog(new TFileDialog(this, SD_FILEOPEN,
           TempName)) == IDOK) {
      ifstream in;
      in.open(TempName, ios::in | ios::nocreate | ios::binary);
      if (in.fail())
        MessageBox(HWindow, "File does not exist.", NULL,
          MB_ICONEXCLAMATION | MB_OK);
      else {
        // The file exists: close it, and ask if the user
        // wants to save the current melody (in case it
        // has been modified).
        in.close();
        if (!AskSave())
          return;
        melody.melodyModified = FALSE;
        // Clear the workspace
        while (melody.part.number())
          melody.part.destroyAt(melody.part.number()-1);
        // Use the LoadFrom() virtual function to load the melody
        HCURSOR hPrevCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
        char drive[MAXDRIVE], dir[MAXDIR], ext[MAXEXT];
        fnsplit(TempName, drive, dir, filepath, ext);
        in.open(TempName, ios::in | ios::binary);
        if (melody.LoadFrom(in, LoadError)) {
          // The melody has been successfully loaded:
          // reset a few Boolean flags and make
          // part 0 as the displayed part
          melody.melodyExists = TRUE;
          melody.scoreDisplay = FALSE;
          melody.displayedPart = 0;
          InvalidateRect(PaintWindow->HWindow , NULL, TRUE);
          SetCursor(hPrevCursor);
       }
      }
      MarkedBlock.Unmark();
      in.close();
      InvalidateRect(PaintWindow->HWindow, NULL, TRUE);
      PaintWindow->UpdateScrollerRange();
    };
  EnableMenus();
  SetWindowText(HWindow, programTitle);
}

void TMuzikaWindow::CMFileSave(TMessage&)
{
  ofstream out;

  if (filepath[0]) {
   // The melody already has a file name:
   // open the stream and use Melody::printOn to save the melody
   char path[MAXPATH];

   wsprintf(path, "%s.muz", filepath);
   HCURSOR hPrevCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
   out.open(path, ios::out | ios::binary);
   melody.printOn(out);
   out.close();
   melody.melodyModified = FALSE;
   SetCursor(hPrevCursor);
  } else {
   // No current file name:
   // open a "Save As..." dialog box
   SaveAs();
  }
}

void TMuzikaWindow::CMFileSaveAs(TMessage&)
{
  SaveAs();
}

void TMuzikaWindow::SaveAs()
{
  char path[MAXPATH];
  char drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];

  if (GetApplication()->ExecDialog(new TFileDialog(this, SD_FILESAVE,
                                    path)) == IDOK) {
    // Append a ".MUZ" extension to the file name if necessary
    if (!(fnsplit(path, drive, dir, file, ext) & EXTENSION))
      strcpy(ext, ".MUZ");
    fnmerge(path, drive, dir, file, ext);
    strcpy(filepath, file);
    // Check if the file already exists
    ofstream out;
    out.open(path, ios::out | ios::noreplace | ios::binary);
    if (out.fail())
      // The file already exists:
      // confirm that it should be overwritten
      if (MessageBox(HWindow, "File already exists. Overwrite?",
        "Warning", MB_ICONINFORMATION | MB_YESNOCANCEL) != IDYES) {
        out.close();
        return;
      }
    out.close();
    HDC hPrevCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));;
    out.open(path, ios::out | ios::trunc | ios::binary);
    // Use the printOn() virtual function to save the melody
    melody.printOn(out);
    out.close();
    melody.melodyModified = FALSE;
    SetWindowText(HWindow, programTitle);
    SetCursor(hPrevCursor);
  }
}

void TMuzikaWindow::CMFilePrint(TMessage&)
{
  PTMuzikaPrintOut Printout = 0;

  if ( Printer )
  {
    int mult = (*(Part *)&melody.part[melody.displayedPart]).multiplicity();
    int pageNum = 0;
    for (int i=0; i<melody.part.number(); i++) {
      int staffs = (*(Part *)&melody.part[i]).staff.number();
      pageNum += (int)((double)staffs/(8/mult*mult) + 0.99999);
    };
    Printout = new TMuzikaPrintOut("Muzika Print");
    if ( Printout )
    {
      Printout->SetBanding( TRUE );
      Printout->SetPageNum( pageNum );
      Printer->Print(this, Printout);
      delete Printout;
    }
  }
}

void TMuzikaWindow::CMFilePrinterSetup(RTMessage)
{
  if ( Printer ) Printer->Setup(this);
}


void TMuzikaWindow::CMFileCreateMidi(TMessage&)
// DialogCreateMIDI is the "File/Create MIDI..." dialog box function,
// processing message intended for this dialog box.
{
  char path[MAXPATH];
  char drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];

  if (filepath[0])
    strcpy(path, filepath);
  else
    strcpy(path, "*.mid");
  if (GetApplication()->ExecDialog(new TFileDialog(this, SD_FILESAVE,
                        path)) == IDOK) {
   // Append a ".MID" extension to the file name if necessary
   if (!(fnsplit(path, drive, dir, file, ext) & EXTENSION))
     strcpy(ext, ".MID");
   fnmerge(path, drive, dir, file, ext);
//   Check if the file already exists
   ofstream out;
   out.open(path, ios::out | ios::noreplace | ios::binary);
   if (out.fail())
    // The file already exists:
    // confirm that it should be overwritten
    if (MessageBox(HWindow, "File already exists. Overwrite?",
       "Warning", MB_ICONINFORMATION | MB_YESNOCANCEL) != IDYES) {
       out.close();
       return;
    }
    out.close();
    HDC hPrevCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));;
    // Use the printOn() virtual function to save the melody
    Midi.CreateFile(path);
    SetCursor(hPrevCursor);
  }
}

void TMuzikaWindow::CMFileMidiSetup(TMessage&)
{
  GetApplication()->ExecDialog((PTWindowsObject)
                               (new TMidiSetupDlg(this, "D_MIDISETUP", _muzika_path)));
}

void TMuzikaWindow::CMEditCopy(TMessage&)
{
  // Obtain the staff.
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  Staff *s = (Staff *) &p.staff[MarkedBlock.markBeginStaff];

  PaintWindow->NewPointObject(BlockSymbols[3], MarkedBlock.markBeginX,
                    p.GetPartY()+s->Y() );
  ActiveSymbol = PASTE_BLOCK;
}

void TMuzikaWindow::CMEditCut(TMessage&)
{
  // Obtain the staff.
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  Staff *s = (Staff *) &p.staff[MarkedBlock.markBeginStaff];

  PaintWindow->NewPointObject(BlockSymbols[2], MarkedBlock.markBeginX,
                    p.GetPartY()+s->Y() );
  InvalidateRect(PaintWindow->HWindow, NULL, TRUE);
  ActiveSymbol = PASTE_BLOCK;
}

void TMuzikaWindow::CMEditPaste(TMessage&)
{
  ActiveSymbol = PASTE_BLOCK;
}

void TMuzikaWindow::CMEditDelete(TMessage&)
{
  // Obtain the staff.
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  Staff *s = (Staff *) &p.staff[MarkedBlock.markBeginStaff];

  PaintWindow->NewPointObject(BlockSymbols[4], MarkedBlock.markBeginX,
                    p.GetPartY()+s->Y() );
  InvalidateRect(PaintWindow->HWindow, NULL, TRUE);
}


void TMuzikaWindow::CMLayoutParts(TMessage&)
{
  GetApplication()->ExecDialog((PTWindowsObject)
                     (new TPartsDialog(this, "D_LAYOUTPARTS")));
}

void TMuzikaWindow::CMLayoutPage(TMessage&)
{
  GetApplication()->ExecDialog((PTWindowsObject)
                     (new TPageDialog(this, "D_LAYOUTPAGE")));
  InvalidateRect(PaintWindow->HWindow , NULL, TRUE);
  InvalidateRect(StatusWindow->HWindow , NULL, TRUE);
  PaintWindow->UpdateScrollerRange();
}

void TMuzikaWindow::CMLayoutReformat(TMessage&)
{

}

void TActiveSymDlg::WMSetCursor(TMessage& )
{
  SetCursor(LoadCursor(NULL, IDC_ARROW));
}

void TActiveSymDlg::CMActSymPencil(TMessage&)
{
  ActiveSymbol = PENCIL;
}

void TActiveSymDlg::CMActSymTools(TMessage&)
{
  ActiveSymbol = -1;
}

void TActiveSymDlg::CMActSymEraser(TMessage&)
{
  ActiveSymbol = ERASER;
}


void TMuzikaWindow::CMNotes(TMessage&)
{
  char buffer[80];

  ActiveSymbol = 6;
  if (ptNotesDlg && IsWindow(ptNotesDlg->HWindow)) {
   SetActiveWindow(ptNotesDlg->HWindow);
  } else {
   ptNotesDlg = new TNotesToolDialog(this, "D_NOTES", 100, NoteSymbols, &Tool, &ActiveSymbol, ToolBar->HWindow);
   GetApplication()->MakeWindow(ptNotesDlg);
  }
  ptActiveTool = ptNotesDlg;
}

void TMuzikaWindow::CMClefs(TMessage&)
{
  ActiveSymbol = 6;
  if (ptClefsDlg && IsWindow(ptClefsDlg->HWindow)) {
   SetActiveWindow(ptClefsDlg->HWindow);
  } else {
   ptClefsDlg = new TToolDialog(this, "D_CLEFS", 200, KeySymbols, &Tool, &ActiveSymbol, ToolBar->HWindow);
   GetApplication()->MakeWindow(ptClefsDlg);
  }
  ptActiveTool = ptClefsDlg;
}

void TMuzikaWindow::CMBeats(TMessage&)
{
  ActiveSymbol = 6;
  if (ptBeatsDlg && IsWindow(ptBeatsDlg->HWindow)) {
    SetActiveWindow(ptBeatsDlg->HWindow);
  } else {
    ptBeatsDlg = new TToolDialog(this, "D_BEATS", 300, BeatSymbols, &Tool, &ActiveSymbol, ToolBar->HWindow);
    GetApplication()->MakeWindow(ptBeatsDlg);
  }
  ptActiveTool = ptBeatsDlg;
}

void TMuzikaWindow::CMBars(TMessage&)
{
  ActiveSymbol = 6;
  if (ptBarsDlg && IsWindow(ptBarsDlg->HWindow)) {
    SetActiveWindow(ptBarsDlg->HWindow);
  } else {
    ptBarsDlg = new TToolDialog(this, "D_BARS", 400, BarSymbols, &Tool, &ActiveSymbol, ToolBar->HWindow);
    GetApplication()->MakeWindow(ptBarsDlg);
  }
  ptActiveTool = ptBarsDlg;
}

void TMuzikaWindow::CMDynamics(TMessage&)
{
  ActiveSymbol = 6;
  if (ptDynamicsDlg && IsWindow(ptDynamicsDlg->HWindow)) {
    SetActiveWindow(ptDynamicsDlg->HWindow);
  } else {
    ptDynamicsDlg = new TToolDialog(this, "D_DYNAMICS", 500, LoudnessSymbols, &Tool, &ActiveSymbol, ToolBar->HWindow);
    GetApplication()->MakeWindow(ptDynamicsDlg);
  }
  ptActiveTool = ptDynamicsDlg;
}

void TMuzikaWindow::CMKeySignature(TMessage&)
{
  int selected=0;
  char helpStr[40];

  TSignatureDialog *SignatureDialog;
  SignatureDialog = new TSignatureDialog(this, "D_KEY_SIG", &selected);
  // The return from signatureDialog changes selected to -
  //   1..7  - num of sharps.
  //  11..17 - num of flats + 10;
  if (GetApplication()->ExecDialog((PTWindowsObject)SignatureDialog)) {
    ActiveSymbol = KEYSIG;
    PaintWindow->transfer(selected);
  }
}

void TMuzikaWindow :: CMFileAbout(TMessage&)
{
  GetApplication()->ExecDialog((PTWindowsObject)(new TDialog(this, "D_ABOUT")));
}

void TMuzikaWindow :: CMFileExit(TMessage&)
{
  SendMessage(HWindow, WM_SYSCOMMAND, SC_CLOSE, 0L);
}

// **********************************************
// AskSave requests the user to confirm qutting
// the current melody after modification. Return TRUE
// if whatever the user requests succeeds (allowing therefore
// to proceed with the previous operation), or FALSE if either
// the user chooses CANCEL or the save fails.
BOOL TMuzikaWindow::AskSave()
{
  int reply;

  // If melody has been modified, display an asking message box
  // and save the melody if the user selects YES
  if (melody.melodyModified) {
    if ((reply = MessageBox(HWindow, "Current melody has been modified. "
                           "Do you wish to save it?", "Warning",
               MB_ICONINFORMATION | MB_YESNOCANCEL)) == IDYES) {
      SaveAs();
      return TRUE;
    } else {
      if (reply == IDNO) {
        return TRUE;
      } else {
        return FALSE;
      }
    }
  }
  return TRUE;
}


// **********************************************
// LoadError is a generic load error recovery function,
// called whenever there is an error loading a file (probably
// because the file does not conform to MUZIKA format).

void LoadError()
{
  // Display an error message box
  SetCursor(LoadCursor(NULL, IDC_ARROW));
  MessageBox(hMainWnd, "The file is not a valid MUZIKA file.", NULL,
   MB_SYSTEMMODAL | MB_OK);
  // Delete the currently stored melody
  while (melody.part.number())
   melody.part.destroyAt(melody.part.number()-1);
  melody.melodyExists = FALSE;
  InvalidateRect(hMainWnd, NULL, TRUE);
}