// **********************************************
// File: OBJCLASS.CPP
// Musical objects library

#include "objects.h"
#include "database.h"

#include <string.h>
#include <math.h>

// **********************************************
// LoadObject loads an object from a file.
// It reads the object type and constructs the appropriate object,
// letting it to go on and load its data (using the version of the constructor
// with an input stream as a parameter). The type of the object
// (point or continuous) is returned in *type.
MusicalObject *LoadObject(istream &in, void (*LoadError)(), int *type)
{
  // Identify the object type
  switch (in.get()) {
    case O_NOTE:
      *type = POINTOBJECT;
      return new Note(in);
    case O_NOTE_SHARP:
      *type = POINTOBJECT;
      return new NoteSharp(in);
    case O_NOTE_NATURAL:
      *type = POINTOBJECT;
      return new NoteNatural(in);
    case O_NOTE_FLAT:
      *type = POINTOBJECT;
      return new NoteFlat(in);
    case O_PAUSE:
      *type = POINTOBJECT;
      return new Pause(in);
    case O_KEY:
      *type = POINTOBJECT;
      return new Key(in);
    case O_BEAT:
      *type = POINTOBJECT;
      return new Beat(in);
    case O_BAR:
      *type = POINTOBJECT;
      return new Bar(in);
    case O_LOUDNESS:
      *type = POINTOBJECT;
      return new Loudness(in);
    case O_CRESCENDO:
      *type = CONTINUOUSOBJECT;
      return new Crescendo(in);
    case O_TEXT:
      *type = POINTOBJECT;
      return new Text(in);
  }
  // Unknown object: call error function
  (*LoadError)();
  return NULL;
}

// **********************************************
// PasteObject creates an object from a clipboard entry.
// It reads the object type and constructs the appropriate object,
// letting it to go on and load its data (using the version of the constructor
// with a far char pointer as a parameter). The type of the object
// (point or continuous) is returned in *type.
MusicalObject *PasteObject(void far *&clipboard, int *type)
{
  // Identify the object type
  switch (*((char far *) clipboard)++) {
    case O_NOTE:
      *type = POINTOBJECT;
      return new Note(clipboard);
    case O_NOTE_SHARP:
      *type = POINTOBJECT;
      return new NoteSharp(clipboard);
    case O_NOTE_NATURAL:
      *type = POINTOBJECT;
      return new NoteNatural(clipboard);
    case O_NOTE_FLAT:
      *type = POINTOBJECT;
      return new NoteFlat(clipboard);
    case O_PAUSE:
      *type = POINTOBJECT;
      return new Pause(clipboard);
    case O_KEY:
      *type = POINTOBJECT;
      return new Key(clipboard);
    case O_BEAT:
      *type = POINTOBJECT;
      return new Beat(clipboard);
    case O_BAR:
      *type = POINTOBJECT;
      return new Bar(clipboard);
    case O_LOUDNESS:
      *type = POINTOBJECT;
      return new Loudness(clipboard);
    case O_CRESCENDO:
      *type = CONTINUOUSOBJECT;
      return new Crescendo(clipboard);
    case O_TEXT:
      *type = POINTOBJECT;
      return new Text(clipboard);
  }
  // Unknown object: return a null pointer
  return NULL;
}

// **********************************************
// Following are the member functions of the various musical objects,
// all derived from the MusicalObject base class, either via
// PointObject or ContinuousObject. The following functions
// are defined for each musical object type:
// - A constructor from direct object data
// - A constructor from an input stream (reads the object data from the stream)
// - A constructor from the clipboard (reads the object data from memory)
// - Draw draws the object in the given display context, and the place
// - Format sets the object in the given X coordinate during formatting.
// - MIDIPlay writes the MIDI event described by the object to the given file.
// - printOn writes the object data to the given file (when the melody is saved).
// - clipOn writes the object data in the clipboard (when the object is cut or copied).
//
// Any new objects in future extensions should be added here,
// and define their versions of all the above functions.

// **********************************************
// Definitions of the Note class member functions

GenNote :: GenNote(int X, int Y, int duration, int dot_value)
{
  // Copy the given parameters to the object variables
  _location = INSTAFF;
  _X = X;
  _Y = (Y >= 0) ? (Y+1)/3*3 : -((1-Y)/3*3);
  _duration = duration;
  _num_of_dots = dot_value;
}

GenNote :: GenNote(istream &in)
{
  _location = INSTAFF;
  // Read the object data from the stream
  in.read((char *) &_X, sizeof(_X));
  in.read((char *) &_Y, sizeof(_Y));
  in.read((char *) &_duration, sizeof(_duration));
  in.read((char *) &_num_of_dots, sizeof(_num_of_dots) );
}

GenNote :: GenNote(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = INSTAFF;
  // Read the object data from the clipboard entry
  _X = *iclip++;
  _Y = *iclip++;
  _duration = *iclip++;
  _num_of_dots = *iclip++;
}


void GenNote :: DrawDot(HDC hDC, HANDLE hInst, int X, int Y)
{
   HBITMAP hBitmap;

   if (_duration == FULLDURATION) {
     Y++;
     hBitmap = LoadBitmap(hInst, "B_DOT4PAUSES");
   } else if (_Y > 12)
       hBitmap = LoadBitmap(hInst, "B_DOTUP");
     else
       hBitmap = LoadBitmap(hInst, "B_DOTDOWN");


   HDC hBitmapDC = CreateCompatibleDC(hDC);


   HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);

   BitBlt(hDC, X, Y, 32, 20, hBitmapDC, 0, 0, SRCAND);

   SelectObject(hBitmapDC, hOldBitmap);
   DeleteDC(hBitmapDC);
   DeleteObject(hBitmap);
};


void GenNote :: Draw(HDC hDC, int staffX, int staffY, int, int)
{
  // Draw the staff extension lines if necessary
  if (_Y < 0)
    for (int y = -6; y >= _Y; y -= 6) {
      MoveTo(hDC, _X+staffX-5, staffY+y);
      LineTo(hDC, _X+staffX+6, staffY+y);
    }
  if (_Y > 24)
    for (int y = 30; y <= _Y; y += 6) {
      MoveTo(hDC, _X+staffX-5, staffY+y);
      LineTo(hDC, _X+staffX+6, staffY+y);
    }
  // Load the appropriate bitmap
  HDC hBitmapDC = CreateCompatibleDC(hDC);
  HBITMAP hBitmap;
  HANDLE hInst;
  hInst = GetApplicationObject()->hInstance;
  int yCenter;
  switch (_duration) {
    case FULLDURATION:
      hBitmap = LoadBitmap(hInst , "B_FULLNOTE");
      yCenter = 10;
      break;
    case HALFDURATION:
      if (_Y > 12) {
        hBitmap = LoadBitmap(hInst, "B_HALFNOTEUP");
        yCenter = 15;
      } else {
        hBitmap = LoadBitmap(hInst, "B_HALFNOTE");
        yCenter = 4;
      }
      break;
    case QUARTERDURATION:
      if (_Y > 12) {
        hBitmap = LoadBitmap(hInst, "B_QUARTERNOTEUP");
        yCenter = 15;
      }  else {
        hBitmap = LoadBitmap(hInst, "B_QUARTERNOTE");
        yCenter = 4;
      }
      break;
    case EIGHTHDURATION:
      if (_Y > 12) {
        hBitmap = LoadBitmap(hInst, "B_EIGHTHNOTEUP");
        yCenter = 15;
      }  else {
        hBitmap = LoadBitmap(hInst, "B_EIGHTHNOTE");
        yCenter = 4;
      }
      break;
    case SIXTEENTHDURATION:
      if (_Y > 12) {
        hBitmap = LoadBitmap(hInst, "B_SIXTEENTHNOTEUP");
        yCenter = 15;
      }  else {
        hBitmap = LoadBitmap(hInst, "B_SIXTEENTHNOTE");
        yCenter = 4;
      }
      break;
    case THIRTY2DURATION:
      if (_Y > 12) {
        hBitmap = LoadBitmap(hInst, "B_THIRTY2NOTEUP");
        yCenter = 15;
      }  else {
        hBitmap = LoadBitmap(hInst, "B_THIRTY2NOTE");
        yCenter = 4;
      }
      break;
  }

  // Draw the bitmap
  HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);
  BitBlt(hDC, _X+staffX-16, _Y+staffY-yCenter, 32, 20, hBitmapDC, 0, 0, SRCAND);
  DrawExtended(hDC, hInst, _X+staffX-16, _Y+staffY-yCenter+1);
  if (_num_of_dots == 1) {
    DrawDot(hDC, hInst, _X+staffX-16, _Y+staffY-yCenter-1);
  }
  SelectObject(hBitmapDC, hOldBitmap);
  DeleteDC(hBitmapDC);
  DeleteObject(hBitmap);
}

void GenNote :: Format(int &X)
{
  // No special actions required: just copy the X coordinate
  _X = X;
}

void GenNote :: printOn(ostream &out) const
{
  // Write the object type ID (O_NOTE) and data to the stream
  PutOutType(out);
  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_Y, sizeof(_Y));
  out.write((char *) &_duration, sizeof(_duration));
  out.write((char *) &_num_of_dots, sizeof(_num_of_dots));
}

void GenNote :: clipOn(void far *&clipboard) const
{
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_xxxNOTE) and data to the clipboard entry
  PutClipType(clipboard);
  *iclip++ = _X;
  *iclip++ = _Y;
  *iclip++ = _duration;
  *iclip++ = _num_of_dots;
}

//  MIDIPlay Retruns The Obejet Type to the caller ( midi.cpp)
//  plus the correct defualts values for Frequency (Pitch) & Duration.
void GenNote :: MIDIPlay(int &type, int &frequency, int &duration,
                         int &index)
{
  static const noteNumber[7] = {0, 2, 4, 5, 7, 9, 11};
                                   // C ,D, E, F, G, A, B  offsets
                                   // see table #2 in external doc.
  static const Bass_correct[7] = {20, 21, 21, 20, 20, 21, 21};
                                   // C ,D, E, F, G, A, B  correction values
                                   // between Treble & Bass clefs.
  static const Bass_map[7] = { 2, 3, 4, 5, 6, 0, 1 };
                                   // C -> E, D -> F ,... (on bass clef)

  int octave = (72-_Y)/21+2; // 72 = 30 + 2*21
                                   // 30 is the Y hight of middle C (C4)
                                   // 21 = 7 notes per Octave * 3 pixels.
  index  = ((72+2*21) -21*octave-_Y)/3;
  frequency = 12*octave +noteNumber[index]; // 12= midi_val per oct.
  frequency = frequency - Bass_correct[index]*(type == KEYFA);
  duration = _duration ;               // right value for un-dotted notes.
  for (int i=0; i < GetDotsNum(); i++ )
    duration = duration * (double)(1+pow(2, -(i+1) ))  ;
  if (type == KEYFA) index = Bass_map[index];
  type = O_NOTE;
}

struct textMode {
  int octave;     // [0..6]
  int length;     // 1 whole, 2 half, 4 quarter, ...
}


void GenNote :: TextWrite(ostream &out, textMode &mode) const
{
  int octave = (72-_Y)/21+2; // 72 = 30 + 2*21
  // Write the object type ID (O_NOTE) and data to the stream  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_Y, sizeof(_Y));
  out.write((char *) &_duration, sizeof(_duration));
  out.write((char *) &_num_of_dots, sizeof(_num_of_dots));
}


// **********************************************
//
// Definitions of the   N o t e    class member functions
//  Note::MIDIPlay  is Inherited from GenNote.

void Note :: PutClipType(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  *bclip++ = O_NOTE;
}
// ***************************************************
// Definitions of the NoteSharp class member functions
void NoteSharp :: DrawExtended(HDC hDC, HANDLE hInst, int X, int Y)
{
   HBITMAP hBitmap;

   if (_duration == FULLDURATION) {
     hBitmap = LoadBitmap(hInst, "B_SHARP");
     Y += 3;
   } else if (_Y > 12) {
     hBitmap = LoadBitmap(hInst, "B_SHARPUP");
   } else {
     Y -= 3;
     hBitmap = LoadBitmap(hInst, "B_SHARP");
   }

   HDC hBitmapDC = CreateCompatibleDC(hDC);


   HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);

   BitBlt(hDC, X, Y, 32, 20, hBitmapDC, 0, 0, SRCAND);

   SelectObject(hBitmapDC, hOldBitmap);
   DeleteDC(hBitmapDC);
   DeleteObject(hBitmap);
};

void NoteSharp :: MIDIPlay(int &type, int &frequency, int &duration,
                           int &index )
{
  //  Get values from your father.
  GenNote::MIDIPlay(type,frequency,duration,index);
  type = O_NOTE_SHARP ;    // change your type.
  frequency++ ;            // add +1 to your Frequency because
                           // This is the mening of SHARP sign.
}

void NoteSharp :: PutClipType(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  *bclip++ = O_NOTE_SHARP;
}

// *****************************************************
// Definitions of the NoteNatural class member functions

void NoteNatural :: DrawExtended(HDC hDC, HANDLE hInst, int X, int Y)
{
   HBITMAP hBitmap;

   if (_duration == FULLDURATION) {
     hBitmap = LoadBitmap(hInst, "B_NATURAL");
     Y += 3;
   } else if (_Y > 12)
     hBitmap = LoadBitmap(hInst, "B_NATURALUP");
   else {
     hBitmap = LoadBitmap(hInst, "B_NATURAL");
     Y -= 3;
   }


   HDC hBitmapDC = CreateCompatibleDC(hDC);

   HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);

   BitBlt(hDC, X, Y, 32, 20, hBitmapDC, 0, 0, SRCAND);

   SelectObject(hBitmapDC, hOldBitmap);
   DeleteDC(hBitmapDC);
   DeleteObject(hBitmap);
};

void NoteNatural :: MIDIPlay(int &type, int &frequency, int &duration,
                             int &index )
{
  //  Get values from your father.
  GenNote::MIDIPlay(type,frequency,duration,index);
  type = O_NOTE_NATURAL ;    // change your type.
}

void NoteNatural :: PutClipType(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  *bclip++ = O_NOTE_NATURAL;
}

// *****************************************************
// Definitions of the NoteFlat class member functions
void NoteFlat :: DrawExtended(HDC hDC, HANDLE hInst, int X, int Y)
{
   HBITMAP hBitmap;

   if (_duration == FULLDURATION) {
     hBitmap = LoadBitmap(hInst, "B_FLAT");
     Y++;
   } else if (_Y > 12)
     hBitmap = LoadBitmap(hInst, "B_FLATUP");
   else {
     Y -= 3;
     hBitmap = LoadBitmap(hInst, "B_FLAT");
   }

   HDC hBitmapDC = CreateCompatibleDC(hDC);

   HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);

   BitBlt(hDC, X, Y, 32, 20, hBitmapDC, 0, 0, SRCAND);

   SelectObject(hBitmapDC, hOldBitmap);
   DeleteDC(hBitmapDC);
   DeleteObject(hBitmap);
};


void NoteFlat :: MIDIPlay(int &type, int &frequency, int &duration,
                          int &index)
{
  //  Get values from your father.
  GenNote::MIDIPlay(type,frequency,duration,index);
  type = O_NOTE_FLAT ;     // change your type.
  frequency-- ;            // decrement 1 from your Frequency
                           // This is the mening of FLAT sign.
}

void NoteFlat :: PutClipType(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  *bclip++ = O_NOTE_FLAT;
}

// **********************************************
// Definitions of the Pause class member functions
Pause :: Pause(int X, int duration, int dot_value)
{
  // Copy the given parameters to the object variables
  _location = INSTAFF;
  _X = X;
  _duration = duration;
  _num_of_dots = dot_value;
}

Pause :: Pause(istream &in)
{
  _location = INSTAFF;
  // Read the object data from the stream
  in.read((char *) &_X, sizeof(_X));
  in.read((char *) &_duration, sizeof(_duration));
  in.read((char *) &_num_of_dots, sizeof(_num_of_dots));
}

Pause :: Pause(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = INSTAFF;
  // Read the object data from the clipboard entry
  _X = *iclip++;
  _duration = *iclip++;
  _num_of_dots = *iclip++;
}

void Pause :: DrawDot(HDC hDC, HANDLE hInst, int X, int Y)
{
   HBITMAP hBitmap;
   hBitmap = LoadBitmap(hInst, "B_DOT4PAUSES");
   HDC hBitmapDC = CreateCompatibleDC(hDC);
   HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);
   BitBlt(hDC, X, Y, 32, 20, hBitmapDC, 0, 0, SRCAND);
   SelectObject(hBitmapDC, hOldBitmap);
   DeleteDC(hBitmapDC);
   DeleteObject(hBitmap);
};

void Pause :: Draw(HDC hDC, int staffX, int staffY, int, int)
{
  // Draw the pause bitmap according to duration
  HANDLE hInst = GetApplicationObject()->hInstance;
  switch (_duration) {
    case FULLDURATION:
      Rectangle(hDC, _X+staffX-5, staffY+7, _X+staffX+5, staffY+9);
      if (_num_of_dots == 1)
        DrawDot(hDC, hInst, _X+staffX-16, staffY-2);
      break;
    case HALFDURATION:
      Rectangle(hDC, _X+staffX-5, staffY+10, _X+staffX+5, staffY+12);
      if (_num_of_dots == 1)
        DrawDot(hDC, hInst, _X+staffX-16, staffY-2);
      break;
    default:
      HDC hBitmapDC = CreateCompatibleDC(hDC);
      HBITMAP hBitmap;
      switch (_duration) {
        case QUARTERDURATION:
          hBitmap = LoadBitmap(hInst, "B_QUARTERPAUSE");
          break;
        case EIGHTHDURATION:
          hBitmap = LoadBitmap(hInst, "B_EIGHTHPAUSE");
          break;
        case SIXTEENTHDURATION:
          hBitmap = LoadBitmap(hInst, "B_SIXTEENTHPAUSE");
          break;
        case THIRTY2DURATION:
          hBitmap = LoadBitmap(hInst, "B_THIRTY2PAUSE");
          break;
      }
      HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);
      BitBlt(hDC, _X+staffX-16, staffY+2, 32, 20, hBitmapDC, 0, 0, SRCAND);
      if (_num_of_dots == 1) {
        DrawDot(hDC, hInst, _X+staffX-16, staffY-2);
      }
      SelectObject(hBitmapDC, hOldBitmap);
      DeleteDC(hBitmapDC);
      DeleteObject(hBitmap);
      break;
  }
}

void Pause :: Format(int &X)
{
  // No special actions required: just copy the X coordinate
  _X = X;
}

void Pause :: MIDIPlay(int &type, int &frequency, int &duration,
                       int &)
{
  // Put a Note Off event with the correct duration in the temporary file
  type = O_PAUSE;
  frequency = 0;
  duration = _duration;
  for (int i=0; i < GetDotsNum(); i++ )
    duration += duration * (double)pow(2, -(i+1))  ;
}

void Pause :: printOn(ostream &out) const
{
  // Write the object type ID (O_PAUSE) and data to the stream
  out.put(O_PAUSE);
  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_duration, sizeof(_duration));
  out.write((char *) &_num_of_dots, sizeof(_num_of_dots));
}

void Pause :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_PAUSE) and data to the clipboard entry
  *bclip++ = O_PAUSE;
  *iclip++ = _X;
  *iclip++ = _duration;
  *iclip++ = _num_of_dots;
}

// **********************************************
// Definitions of the Key class member functions

Key :: Key(int, int type)
{
  // Copy the given parameters to the object variables
  _location = INSTAFF | ONEPERSTAFF;
  _X = LOCATION;
  _type = type;
  _sigType = NONE;
  _sigNum = 0;
}

Key :: Key(istream &in)
{
  _location = INSTAFF | ONEPERSTAFF;
  _X = LOCATION;
  // Read the object data from the stream
  in.read((char *) &_type, sizeof(_type));
  in.read((char *) &_sigType, sizeof(_sigType));
  in.read((char *) &_sigNum, sizeof(_sigNum));
}

Key :: Key(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = INSTAFF | ONEPERSTAFF;
  _X = LOCATION;
  // Read the object data from the clipboard entry
  _type = *iclip++;
  _sigType = *iclip++;
  _sigNum = *iclip++;
}

void Key :: Draw(HDC hDC, int staffX, int staffY, int, int)
{
  HDC hBitmapDC = CreateCompatibleDC(hDC);
  HBITMAP hBitmap;
  HANDLE hInst;
  hInst = GetApplicationObject()->hInstance;
  // Load the appropriate bitmap
  HBITMAP hOldBitmap;
  switch (_type) {
    case KEYSOL:
      hBitmap = LoadBitmap(hInst, "B_KEYSOL_BIG");
      // Draw the bitmap
      hOldBitmap = SelectObject(hBitmapDC, hBitmap);
      BitBlt(hDC, _X+staffX-16, staffY-8, 32, 45, hBitmapDC,0, 0, SRCAND);
      break;
    case KEYFA:
      hBitmap = LoadBitmap(hInst, "B_KEYFA");
      // Draw the bitmap
      hOldBitmap = SelectObject(hBitmapDC, hBitmap);
      BitBlt(hDC, _X+staffX-16, staffY+2, 32, 20, hBitmapDC,0, 0, SRCAND);
      break;
  }
  SelectObject(hBitmapDC, hOldBitmap);
  DeleteDC(hBitmapDC);
  DeleteObject(hBitmap);
  DrawSignature(hDC, _X+staffX+4, staffY-7);
}

void Key :: DrawSignature(HDC hdc, int x, int y)
{
  char sigName[8], bitmapName[11];;
  x += 3;
  switch (_sigType) {
    case SHARP_SIG:
      strcpy(sigName, "B_SHARP");
      y--;
      break;
    case FLAT_SIG:
      strcpy(sigName, "B_FLAT");
      y += 3;
      break;
    default:
      return;
  };
  wsprintf(bitmapName, "%s_%d", sigName, _sigNum);
  HDC hBitmapDC = CreateCompatibleDC(hdc);
  HANDLE hInst = GetApplicationObject()->hInstance;
  HBITMAP hBitmap = LoadBitmap(hInst, bitmapName);
  HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);
  int ySupp = 0;
  if (_type == KEYFA) ySupp = 6;  // move down the key-sig bitmap for bass.
  BitBlt(hdc, x, y+ySupp, 60, 32, hBitmapDC,0, 0, SRCAND);
  SelectObject(hBitmapDC, hOldBitmap);
  DeleteDC(hBitmapDC);
  DeleteObject(hBitmap);
}

// To change a Key attribute (adding flat and sharp)
// use:
//   type = 0 -> flat
//   type = 1 -> sharp
//   num  = 1..7 the number of signatures per staff.
//          ( sharp or flat )
void Key :: ChangeSignature(int type, int num)
{
  if (type == 1)
    _sigType = FLAT_SIG;
  else if (type == 0)
    _sigType = SHARP_SIG;
  else
    return;
  if ((num <= 7) && (num >= 1)) _sigNum = num;
}

void Key :: Format(int &)
{
  // Keys cannot be formatted
}

void Key :: printOn(ostream &out) const
{
  // Write the object type ID (O_KEY) and data to the stream
  out.put(O_KEY);
  out.write((char *) &_type, sizeof(_type));
  out.write((char *) &_sigType, sizeof(_sigType));
  out.write((char *) &_sigNum, sizeof(_sigNum));
}

void Key :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_KEY) to the clipboard entry
  *bclip++ = O_KEY;
  *iclip++ = _type;
  *iclip++ = _sigType;
  *iclip++ = _sigNum;
}

void Key :: MIDIPlay(int &obj_type, int &clefType, int &keySigType, int &numOfSigs)
// it is called by, obj_type, obj_freq, obj_duration, obj_index;
{
  obj_type = O_KEY ;
  clefType = _type ;         // return KEYFA,KEYSOL for MIDI correction.
  keySigType = 0;            // return  0 = No keySig (just Clef)
  numOfSigs  = 0;            //  by default 0 here mean it's a Clef not a Key.
  if (_sigType != NONE )
    keySigType = _sigType;   //  1 = Sharp   2= Flat  keySig  kind.
    numOfSigs  = _sigNum  ;  // how many signatures in the key
}

// **********************************************
// Definitions of the Beat class member functions

Beat :: Beat(int X, int type)
{
  // Copy the given parameters to the object variables
  _location = COMMONMULTIPLE;
  _X = X;
  _type = type;
}

Beat :: Beat(istream &in)
{
  _location = COMMONMULTIPLE;
  // Read the object data from the stream
  in.read((char *) &_X, sizeof(_X));
  in.read((char *) &_type, sizeof(_type));
}

Beat :: Beat(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = COMMONMULTIPLE;
  // Read the object data from the clipboard entry
  _X = *iclip++;
  _type = *iclip++;
}

void Beat :: Draw(HDC hDC, int staffX, int staffY, int, int)
{
  HDC hBitmapDC = CreateCompatibleDC(hDC);
  HBITMAP hBitmap;
  HANDLE hInst;
  hInst = GetApplicationObject()->hInstance;
  // Load the appropriate bitmap
  switch (_type) {
    case BEATC:
      hBitmap = LoadBitmap(hInst, "B_BEATC");
      break;
    case BEATCBAR:
      hBitmap = LoadBitmap(hInst, "B_BEATCBAR");
      break;
    case BEAT28:
      hBitmap = LoadBitmap(hInst, "B_BEAT28");
      break;
    case BEAT24:
      hBitmap = LoadBitmap(hInst, "B_BEAT24");
      break;
    case BEAT38:
      hBitmap = LoadBitmap(hInst, "B_BEAT38");
      break;
    case BEAT34:
      hBitmap = LoadBitmap(hInst, "B_BEAT34");
      break;
    case BEAT48:
      hBitmap = LoadBitmap(hInst, "B_BEAT48");
      break;
    case BEAT44:
      hBitmap = LoadBitmap(hInst, "B_BEAT44");
      break;
    case BEAT68:
      hBitmap = LoadBitmap(hInst, "B_BEAT68");
      break;
  }
  // Draw the bitmap
  HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);
  BitBlt(hDC, _X+staffX-16, staffY+4, 32, 20, hBitmapDC, 0, 0, SRCAND);
  SelectObject(hBitmapDC, hOldBitmap);
  DeleteDC(hBitmapDC);
  DeleteObject(hBitmap);
}

void Beat :: Format(int &X)
{
  // No special actions required: just copy the X coordinate
  _X = X;
}

void Beat :: printOn(ostream &out) const
{
  // Write the object type ID (O_BEAT) and data to the stream
  out.put(O_BEAT);
  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_type, sizeof(_type));
}

void Beat :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_BEAT) and data to the clipboard entry
  *bclip++ = O_BEAT;
  *iclip++ = _X;
  *iclip++ = _type;
}

// **********************************************
// Definitions of the Bar class member functions

Bar :: Bar(int X, unsigned width, int type)
{
  // Choose the _location attribute value according to bar type
  switch (_type = type) {
    case STARTBAR:
      _location = COMMONMULTIPLE | ONEPERSTAFF;
      _X = 0;
      break;
    case ENDBAR:
      _location = COMMONMULTIPLE | ONEPERSTAFF;
      _X = width;
      break;
    default:
      _location = COMMONMULTIPLE;
      _X = X;
      break;
  }
}

Bar :: Bar(istream &in)
{
  _location = COMMONMULTIPLE;
  // Read the object data from the stream
  in.read((char *) &_X, sizeof(_X));
  in.read((char *) &_type, sizeof(_type));
  // Decide about the object _location attribute value according to type
  if (_type == STARTBAR || _type == ENDBAR)
    _location |= ONEPERSTAFF;
}

Bar :: Bar(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = COMMONMULTIPLE;
  // Read the object data from the clipboard entry
  _X = *iclip++;
  _type = *iclip++;
  // Decide about the _location attribute value according to type
  if (_type == STARTBAR || _type == ENDBAR)
    _location |= ONEPERSTAFF;
}

void Bar :: Draw(HDC hDC, int staffX, int staffY, int currStaffHeight, int staffLoc)
{
  HBRUSH hOldBrush = SelectObject(hDC, GetStockObject(BLACK_BRUSH));
  int Yfrom, Yto;
  // Calculate the bar height limits
  Yfrom = staffY;
  switch (staffLoc) {
    case SINGLESTAFF:
    case LASTSTAFF:
      Yto = staffY+24;
      break;
    case FIRSTSTAFF:
    case MIDSTAFF:
      Yto = staffY+currStaffHeight;
      break;
  }
  // Draw the appropriate bar pattern
  switch (_type) {
    case BAR:
      MoveTo(hDC, _X+staffX, Yfrom);
      LineTo(hDC, _X+staffX, Yto);
      break;
    case DOUBLEBAR:
      MoveTo(hDC, _X+staffX-2, Yfrom);
      LineTo(hDC, _X+staffX-2, Yto);
      MoveTo(hDC, _X+staffX+2, Yfrom);
      LineTo(hDC, _X+staffX+2, Yto);
      break;
    case STARTBAR:
      Rectangle(hDC, _X+staffX, Yfrom, _X+staffX+4, Yto);
      MoveTo(hDC, _X+staffX+6, Yfrom);
      LineTo(hDC, _X+staffX+6, Yto);
      break;
    case ENDBAR:
      Rectangle(hDC, _X+staffX-3, Yfrom, _X+staffX+1, Yto);
      MoveTo(hDC, _X+staffX-6, Yfrom);
      LineTo(hDC, _X+staffX-6, Yto);
      break;
  }
  SelectObject(hDC, hOldBrush);
}

void Bar :: Format(int &X)
{
  // Bar can be moved unless it is fixed at the start or end of the staff
  if (_type != STARTBAR && _type != ENDBAR)  _X = X;
}

void Bar :: printOn(ostream &out) const
{
  // Write the object type ID (O_BAR) and data to the stream
  out.put(O_BAR);
  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_type, sizeof(_type));
}

void Bar :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_BAR) and data to the clipboard entry
  *bclip++ = O_BAR;
  *iclip++ = _X;
  *iclip++ = _type;
}

void Bar :: MIDIPlay(int &type, int &, int &, int &)
{
  type = O_BAR;    // change your type.
}

// **********************************************
// Definitions of the Loudness class member functions
Loudness :: Loudness(int X, int loudness)
{
  // Copy the given parameters to the object variables
  _location = BELOWMULTIPLE;
  _X = X;
  _loudness = loudness;
}

Loudness :: Loudness(istream &in)
{
  _location = BELOWMULTIPLE;
  // Read the object data from the stream
  in.read((char *) &_X, sizeof(_X));
  in.read((char *) &_loudness, sizeof(_loudness));
}

Loudness :: Loudness(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = BELOWMULTIPLE;
  // Read the object data from the clipboard entry
  _X = *iclip++;
  _loudness = *iclip++;
}

void Loudness :: Draw(HDC hDC, int staffX, int staffY, int currStaffHeight, int)
{
  HDC hBitmapDC = CreateCompatibleDC(hDC);
  HBITMAP hBitmap;
  HANDLE hInst;
  hInst = GetApplicationObject()->hInstance;
  // Load the appropriate bitmap
  switch (_loudness) {
    case FORTE:
      hBitmap = LoadBitmap(hInst, "B_FORTE");
      break;
    case FORTISSIMO:
      hBitmap = LoadBitmap(hInst, "B_FORTISSIMO");
      break;
    case PIANO:
      hBitmap = LoadBitmap(hInst, "B_PIANO");
      break;
    case PIANISSIMO:
      hBitmap = LoadBitmap(hInst, "B_PIANISSIMO");
      break;
  }
  // Draw the bitmap
  HBITMAP hOldBitmap = SelectObject(hBitmapDC, hBitmap);
  BitBlt(hDC, _X+staffX-16, staffY+currStaffHeight/2-8, 32, 20, hBitmapDC,
    0, 0, SRCAND);
  SelectObject(hBitmapDC, hOldBitmap);
  DeleteDC(hBitmapDC);
  DeleteObject(hBitmap);
}

void Loudness :: Format(int &X)
{
  // No special actions required: just copy the X coordinate
  _X = X;
}

void Loudness :: printOn(ostream &out) const
{
  // Write the object type ID (O_LOUDNESS) and data to the stream
  out.put(O_LOUDNESS);
  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_loudness, sizeof(_loudness));
}

void Loudness :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_LOUDNESS) and data to the clipboard entry
  *bclip++ = O_LOUDNESS;
  *iclip++ = _X;
  *iclip++ = _loudness;
}

// **********************************************
// Definitions of the Crescendo class member functions

Crescendo :: Crescendo(int Xleft, int Xright, int direction)
{
  // Copy the given parameters to the object variables
  _location = BELOWMULTIPLE;
  _Xleft = Xleft;
  _Xright = Xright;
  _direction = direction;
}

Crescendo :: Crescendo(istream &in)
{
  _location = BELOWMULTIPLE;
  // Read the object data from the stream
  in.read((char *) &_Xleft, sizeof(_Xleft));
  in.read((char *) &_Xright, sizeof(_Xright));
  in.read((char *) &_direction, sizeof(_direction));
}

Crescendo :: Crescendo(void far *&clipboard)
{
  int far *&iclip = (int far *) clipboard;
  _location = BELOWMULTIPLE;
  // Read the object data from the clipboard entry
  _Xleft = *iclip++;
  _Xright = *iclip++;
  _direction = *iclip++;
}

void Crescendo :: Draw(HDC hDC, int staffX, int staffY, int currStaffHeight, int)
{
  // Draw the pair of lines according to direction
  MoveTo(hDC, staffX+((_direction == CRESCENDO) ? _Xright : _Xleft),
    staffY+currStaffHeight/2+4);
  LineTo(hDC, staffX+((_direction == CRESCENDO) ? _Xleft : _Xright),
    staffY+currStaffHeight/2+8);
  LineTo(hDC, staffX+((_direction == CRESCENDO) ? _Xright : _Xleft),
    staffY+currStaffHeight/2+12);
}

void Crescendo :: printOn(ostream &out) const
{
  // Write the object type ID (O_CRESCENDO) and data to the stream
  out.put(O_CRESCENDO);
  out.write((char *) &_Xleft, sizeof(_Xleft));
  out.write((char *) &_Xright, sizeof(_Xright));
  out.write((char *) &_direction, sizeof(_direction));
}

void Crescendo :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_CRESCENDO) and data to the clipboard entry
  *bclip++ = O_CRESCENDO;
  *iclip++ = _Xleft;
  *iclip++ = _Xright;
  *iclip++ = _direction;
}

// **********************************************
// Definitions of the Text class member functions

Text :: Text(int X, int Y, char *string)
{
  // Copy the given parameters to the object variables
  _location = INSTAFF;
  _X = X;
  _Y = (Y > 12) ? 30 : -21;
  strcpy(_text, string);
}

Text :: Text(istream &in)
{
  _location = INSTAFF;
  // Read the object data from the stream
  in.read((char *) &_X, sizeof(_X));
  in.read((char *) &_Y, sizeof(_Y));
  in.get(_text, MAXTEXT+1, 0);
  in.get();
}

Text :: Text(void far *&clipboard)
{
  _location = INSTAFF;
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Read the object data from the clipboard entry
  _X = *iclip++;
  _Y = *iclip++;
  _fstrcpy(_text, bclip);
  bclip += _fstrlen(bclip)+1;
}

void Text :: Draw(HDC hDC, int staffX, int staffY, int, int)
{
  // Put the text in a transparent fashion (i.e. not erasing the background)
  SetBkMode(hDC, TRANSPARENT);
  TextOut(hDC, _X+staffX, _Y+staffY, _text, strlen(_text));
}

void Text :: Format(int &X)
{
  // No special actions required: just copy the X coordinate
  _X = X;
}

void Text :: printOn(ostream &out) const
{
  // Write the object type ID (O_TEXT) and data to the stream
  out.put(O_TEXT);
  out.write((char *) &_X, sizeof(_X));
  out.write((char *) &_Y, sizeof(_Y));
  out.write(_text, strlen(_text)+1);
}

void Text :: clipOn(void far *&clipboard) const
{
  char far *&bclip = (char far *) clipboard;
  int far *&iclip = (int far *) clipboard;
  // Write the object type ID (O_TEXT) and data to the clipboard entry
  *bclip++ = O_TEXT;
  *iclip++ = _X;
  *iclip++ = _Y;
  _fstrcpy(bclip, _text);
  bclip += strlen(_text)+1;
}
