// **********************************************
// File: OBJECTS.H
// Musical objects header file
#ifndef _OBJECTS_H_
#define _OBJECTS_H_

#include "muzika.h"
#include "database.h"

MusicalObject *LoadObject(istream &in, void (*LoadError)(), int *type);
MusicalObject *PasteObject(void far *&clipboard, int *type);

#define WidthRound(x) ((x)/melody.pixelsPerObject*melody.pixelsPerObject+melody.pixelsPerObject/2)

// --- Object class codes
const O_NOTE =          5;
const O_PAUSE =         6;
const O_KEY =           7;
const O_BEAT =          8;
const O_BAR =           9;
const O_LOUDNESS =     10;
const O_CRESCENDO =    11;
const O_TEXT =         12;
const O_NOTE_SHARP =   13;
const O_NOTE_NATURAL = 14;
const O_NOTE_FLAT =    15;

// **********************************************
// Definitions of the various musical classes

//----------------------------------------------------------------------
//           G e n e r i c    N o t e.
//----------------------------------------------------------------------
// GenNote is the father of any note (normal,with sharp etc.)
// it is a pure Generic abstract class ( ..... = 0 )
class GenNote : public PointObject {
  protected:
    int _Y;
    int _duration;
    int _num_of_dots;        // for n-dotted notes. default is 0
  public:
    GenNote(int X, int Y, int duration, int dot_value = 0);
    GenNote(istream &);
    GenNote(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void DrawDot(HDC, HANDLE, int, int);
    virtual void DrawExtended(HDC, HANDLE, int, int) = 0;
    virtual void Format(int &);
    virtual void MIDIPlay(int &,int &, int &, int &);
    virtual int  Duration() {return _duration;}
    virtual void printOn(ostream &) const;
    virtual void PutOutType(ostream &) const = 0;
    virtual void clipOn(void far *&) const;
    virtual void PutClipType(void far *&) const = 0;
    virtual void SetDotsNum(int dots) {_num_of_dots=dots;}
    virtual int  GetDotsNum(void) { return _num_of_dots; }
};

class Note : virtual public GenNote {
  public:
    Note(int X, int Y, int duration, int dots) : GenNote(X, Y, duration, dots) { };
    Note(istream &in) :GenNote(in) { };
    Note(void far *&clip) :GenNote(clip) { };
    virtual void DrawExtended(HDC, HANDLE, int, int) { };
    virtual void PutOutType(ostream &out) const { out.put(O_NOTE); };
    virtual void PutClipType(void far *&) const;
};

class NoteSharp : virtual public GenNote {
  public:
    NoteSharp(int X, int Y, int duration, int dots) : GenNote(X, Y, duration, dots) { };
    NoteSharp(istream &in) :GenNote(in) { };
    NoteSharp(void far *&clip) :GenNote(clip) { };
    virtual void DrawExtended(HDC, HANDLE, int, int);
    virtual void MIDIPlay(int &, int &, int &, int &);
    virtual void PutOutType(ostream &out) const { out.put(O_NOTE_SHARP); };
    virtual void PutClipType(void far *&) const;
};

class NoteNatural : virtual public GenNote {
  public:
    NoteNatural(int X, int Y, int duration, int dots) : GenNote(X, Y, duration, dots) { };
    NoteNatural(istream &in) :GenNote(in) { };
    NoteNatural(void far *&clip) :GenNote(clip) { };
    virtual void DrawExtended(HDC, HANDLE, int, int);
    virtual void PutOutType(ostream &out) const { out.put(O_NOTE_NATURAL); };
    virtual void PutClipType(void far *&) const;
    virtual void MIDIPlay(int &,int &, int &, int &);
};

class NoteFlat : virtual public GenNote {
  public:
    NoteFlat(int X, int Y, int duration, int dots) : GenNote(X, Y, duration, dots) { };
    NoteFlat(istream &in) :GenNote(in) { };
    NoteFlat(void far *&clip) :GenNote(clip) { };
    virtual void DrawExtended(HDC, HANDLE, int, int);
    virtual void MIDIPlay(int &, int &, int &, int &);
    virtual void PutOutType(ostream &out) const { out.put(O_NOTE_FLAT); };
    virtual void PutClipType(void far *&) const;
};


//----------------------------------------------------------------------
//           P a u s e    ( the musical term is Rest.)
//----------------------------------------------------------------------
class Pause : virtual public PointObject {
  protected:
    int _duration;
    int _num_of_dots;          // for n-dotted notes. default is 0
  public:
    Pause(int X, int duration, int dot_value = 0);
    Pause(istream &);
    Pause(void far *&);
    virtual void DrawDot(HDC, HANDLE, int, int);
    virtual void Draw(HDC, int, int, int, int);
    virtual void Format(int &);
    virtual void MIDIPlay(int &, int &, int &, int &);
    virtual int  Duration() {return _duration;}
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
    virtual void SetDotsNum(int dots) {_num_of_dots=dots;} ;
    virtual int  GetDotsNum(void) { return _num_of_dots; } ;
};
//----------------------------------------------------------------------
//           K e y  ( the musical term is Clef.)
//----------------------------------------------------------------------
class Key : virtual public PointObject {
  protected:
    enum {LOCATION = 15};
    int _type;
    enum { NONE=0, SHARP_SIG, FLAT_SIG } _sigType;
    int _sigNum;  // number of Sharp / Flats (by _sigType) ranging from 1..7
  public:
    Key(int X, int type);
    Key(istream &);
    Key(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void DrawSignature(HDC, int, int);
    virtual void Format(int &);
    virtual void ChangeSignature(int, int);
    virtual int Width() {return WidthRound(2*melody.pixelsPerObject+LOCATION);}
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
    virtual void MIDIPlay(int &, int &, int &, int &);
};

//----------------------------------------------------------------------
//           B e a t .
//----------------------------------------------------------------------
class Beat : virtual public PointObject {
  protected:
    int _type;
  public:
    Beat(int X, int type);
    Beat(istream &);
    Beat(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void Format(int &);
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
};
//----------------------------------------------------------------------
//           B a r .
//----------------------------------------------------------------------
class Bar : virtual public PointObject {
  protected:
    int _type;
  public:
    Bar(int X, unsigned width, int type);
    Bar(istream &);
    Bar(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void Format(int &);
    virtual int Width()
      {return _type != STARTBAR && _type != ENDBAR ? melody.pixelsPerObject : 0;}
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
    virtual void MIDIPlay(int &, int &, int &, int &);
};
//----------------------------------------------------------------------
//           L o u d n e s s  ( the musical term is Dynamics.)
//----------------------------------------------------------------------
class Loudness : virtual public PointObject {
  protected:
    int _loudness;
  public:
    Loudness(int X, int loudness);
    Loudness(istream &);
    Loudness(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void Format(int &);
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
};

class Crescendo : virtual public ContinuousObject {
  protected:
    int _direction;
  public:
    Crescendo(int Xleft, int Xright, int direction);
    Crescendo(istream &);
    Crescendo(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
};
//----------------------------------------------------------------------
//           T e x t  ( used for Lyrics.)
//----------------------------------------------------------------------
class Text : virtual public PointObject {
  protected:
    int _Y;
    char _text[MAXTEXT];
  public:
    Text(int X, int Y, char *);
    Text(istream &);
    Text(void far *&);
    virtual void Draw(HDC, int, int, int, int);
    virtual void Format(int &);
    virtual void printOn(ostream &) const;
    virtual void clipOn(void far *&) const;
};

#endif