#include "pntwnd.hpp"
#include "block.hpp"

extern Melody melody;
extern HWND hMainWnd;
HWND hPaintWnd;
TBlock MarkedBlock;

extern SymbolClass* BlockSymbols[];

inline int  max(int a, int b) { return (a>b) ? a : b; };
inline int  min(int a, int b) { return (a<b) ? a : b; };

TPaintWindow::TPaintWindow(PTWindowsObject pwParent, int *ActSym,
                           TToolDialog **newTool, TStatusWindow *StatusWindow)
  :TWindow(pwParent, NULL)
{
  Attr.Style |= WS_VSCROLL | WS_HSCROLL | WS_CHILD | WS_VISIBLE | WS_BORDER;
  Scroller = new TScroller(this, 80, 20, 1, 1);
  ActiveSymbol = ActSym;
  _currContSymbol = NULL;
  Tool = newTool;
  theStatusWindow = StatusWindow;
  _isCapture = FALSE;
  trans = 0;
  Xto = -1;
  Yto = -1;
}

int TPaintWindow::GetActiveSymbol()
{
  return *ActiveSymbol;
}

void TPaintWindow::SetActiveSymbol(int newSym)
{
  *ActiveSymbol = newSym;
}

// The transfer methos is used to transfer paitWindow
// with an additional information.
// used when paint window must recieve an additional
// information that can not be transmitted any other way.
void TPaintWindow::transfer(int what)
{
  trans = what;
}

LPSTR TPaintWindow::GetClassName()
{
  return "MUZIKA:PAINT";
}

void TPaintWindow::GetWindowClass(WNDCLASS& wc)
{
  TWindow::GetWindowClass(wc);
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.style |= CS_DBLCLKS;
}

void TPaintWindow::WMSetCursor(TMessage& Msg)
{
  WNDCLASS wc;
  HCURSOR hCursor;
  int iChoice;

  iChoice = GetActiveSymbol();
  if (Msg.LP.Lo == HTCLIENT) {
    switch (iChoice) {
      case PENCIL:
        hCursor = LoadCursor(GetApplication()->hInstance, "C_PENCIL");
        SetCursor(hCursor);
        break;
      case HAND:
        hCursor = LoadCursor(GetApplication()->hInstance, "C_HAND");
        SetCursor(hCursor);
        break;
      case ERASER:
        hCursor = LoadCursor(GetApplication()->hInstance, "C_ERASER");
        SetCursor(hCursor);
        break;
      case TEXT:
        hCursor = LoadCursor(NULL, IDC_IBEAM);
        SetCursor(hCursor);
        break;
      case MARK_BLOCK:
        hCursor = LoadCursor(NULL, IDC_CROSS);
        SetCursor(hCursor);
        break;
      case PASTE_BLOCK:
        hCursor = LoadCursor(GetApplication()->hInstance, "C_PASTE");
        SetCursor(hCursor);
        break;
      case CONNECT:
        hCursor = LoadCursor(GetApplication()->hInstance, "C_CONNECT");
        SetCursor(hCursor);
        break;
      case KEYSIG:
        hCursor = LoadCursor(GetApplication()->hInstance, "C_KEYSIG");
        SetCursor(hCursor);
        break;
      default:
        if (!(*Tool) ||(!(*Tool)->getCursorName()[0])) {
          hCursor = LoadCursor(NULL, IDC_ARROW);
          SetCursor(hCursor);
        } else {
          GetWindowClass(wc);
          wc.hCursor = (*Tool)->LoadCursor(wc.hInstance);
          SetCursor(wc.hCursor);
        }
    }
  } else {
   DefWndProc(Msg);
  }
}

void TPaintWindow::Paint(HDC hdc, PAINTSTRUCT&)
{
  int staffX, staffY, currStaffHeight, staffLoc;
  // Draw the page according to the current settings
  hPaintWnd = HWindow;
  if (melody.melodyExists) {
    if (!melody.scoreDisplay) {
      // Display a single part
      Part &p = *((Part *) &melody.part[melody.displayedPart]);
      int firstStaff = 0, lastStaff = 0;
      // Display the page
      for (int index = 0; index < p.staff.number(); ++index) {
        int firstY, lastY;
        if (index % p.multiplicity() == 0) {
          firstY = -1;
          lastY = MAXINT-24;
        }
        Staff &s = *((Staff *) &p.staff[index]);
        // Draw the staff itself
        if (s.Draw(hdc, p.GetPartY()+(Scroller->YPos)*(Scroller->YUnit), TRUE , s.width())) {
          if (!firstStaff) firstStaff = index+p.multiplicity();
          lastStaff = index+p.multiplicity();
          staffX = s.X();
          staffY = s.Y()-p.GetPartY();
          // Check whether first or last staff in a group
          staffLoc = MIDSTAFF;
          if (index % p.multiplicity() == 0) {
            firstY = staffY;
            staffLoc = FIRSTSTAFF;
          }
          if ((index+1) % p.multiplicity() == 0) {
            lastY = staffY;
            staffLoc = LASTSTAFF;
          }
          if (p.multiplicity() == 1)
            staffLoc = SINGLESTAFF;
          currStaffHeight = (index+1 < p.staff.number()) ?
            ((Staff *) &p.staff[index+1])->Y()-s.Y()
            : melody.pixelsPerStaff;
          // Draw the point objects inside the staff
          IndexedList &pointList = s.pointObject;
          TDoubleListIterator ListIterator(&pointList);
          Object *pointObject;
          for (pointObject = ListIterator.first(); !ListIterator.last();
               pointObject = ListIterator.next() )
            ((PointObject *) pointObject)->Draw(hdc, staffX, staffY, currStaffHeight, staffLoc);
          // Draw the continuous objects inside the staff
          IndexedList &contList = s.continuousObject;
          for (int i = 0; i < contList.number(); ++i)
            ((ContinuousObject *) &contList[i])->Draw(hdc, staffX, staffY, currStaffHeight, staffLoc);
        }
        // Check if the multiple staff is complete
        if ((index+1)%p.multiplicity() == 0) {
          if (firstY != -1 || lastY != MAXINT-24) {
            // Draw the connecting lines of a multiple staff
            MoveTo(hdc, s.X(), firstY);
            LineTo(hdc, s.X(), lastY+24);
            MoveTo(hdc, s.X()+s.width()-1, firstY);
            LineTo(hdc, s.X()+s.width()-1, lastY+24);
          }
          // Show the marked block
          int i = index/p.multiplicity()*p.multiplicity();
          if (i >= MarkedBlock.markBeginStaff && i <= MarkedBlock.markEndStaff) {
            int markFrom, markTo;
            if (i == MarkedBlock.markBeginStaff)
              markFrom = MarkedBlock.markBeginX-melody.pixelsPerObject/2;
            if (i > MarkedBlock.markBeginStaff)
              markFrom = 0;
            if (i == MarkedBlock.markEndStaff)
              markTo = MarkedBlock.markEndX+melody.pixelsPerObject/2;
            if (i < MarkedBlock.markEndStaff)
              markTo = s.width();
            PatBlt(hdc, markFrom+s.X(),
              ((Staff *) &p.staff[i])->Y()-p.GetPartY(),
              markTo-markFrom, s.Y()+25-((Staff *) &p.staff[i])->Y(),
              DSTINVERT);
          }
        }
      }
      theStatusWindow->SetStaffsLoc(firstStaff/p.multiplicity(),
                                    lastStaff/p.multiplicity());
      InvalidateRect(theStatusWindow->HWindow, NULL, TRUE);
    } else {
      // Display score
      int line = 0;
      int staffLeftX = 32;
      int staffRightX = 32+melody.GetStaffWidth();
      int scoreFirstStaff = 0;
      int scoreFirstPart = 0;
      int scoreMultiplicity=0;
      int scoreStaves = 0;
      BOOL screenFull = FALSE;

      for (int index = 0; index < melody.part.number(); ++index) {
        Part &p = *((Part *) &melody.part[index]);
        scoreMultiplicity += p.multiplicity();
        if (p.staff.number()/p.multiplicity() > scoreStaves)
          scoreStaves = p.staff.number()/p.multiplicity();
      }
      for (int scoreIndex = scoreFirstStaff;
        scoreIndex < scoreStaves && !screenFull; ++scoreIndex) {
        // Draw a multiple staff
        for (int partIndex = scoreFirstPart; partIndex < melody.part.number();
             ++partIndex) {
          Part &p = *((Part *) &melody.part[partIndex]);
          for (int staffIndex = scoreIndex*p.multiplicity();
            staffIndex < (scoreIndex+1)*p.multiplicity(); ++staffIndex) {
            // Draw a single staff
            Staff &s = *((Staff *) &p.staff[staffIndex]);
            int tempY = s.Y();
            s.Y() = line += melody.pixelsPerStaff;
            screenFull = !s.Draw(hdc, 0, TRUE, s.width());
            staffX = staffLeftX;
            staffY = line;
            // Draw the point objects inside the staff
            IndexedList &pointList = s.pointObject;
            for (int i = 0; i < pointList.number(); ++i)
              ((PointObject *) &pointList[i])->Draw(hdc, staffX, staffY, currStaffHeight, staffLoc);
            // Draw the continuous objects inside the staff
            IndexedList &contList = s.continuousObject;
            for (i = 0; i < contList.number(); ++i)
              ((ContinuousObject *) &contList[i])->Draw(hdc, staffX, staffY, currStaffHeight, staffLoc);
            s.Y() = tempY;
          }
        }
        // Draw the connecting lines of the score multiple staff
        MoveTo(hdc, staffLeftX, line-(scoreMultiplicity-1)*melody.pixelsPerStaff);
        LineTo(hdc, staffLeftX, line+24);
        MoveTo(hdc, staffRightX, line-(scoreMultiplicity-1)*melody.pixelsPerStaff);
        LineTo(hdc, staffRightX, line+24);
      }
    }
  }
}

void TPaintWindow::WMRButtonDown(TMessage &Msg)
{
  if ( melody.melodyExists ) {
    DeleteMusicalObject(
      MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit),
      MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit));
  }
}

void TPaintWindow::WMLButtonDown(TMessage &Msg)
{
  // Process a WM_LBUTTONDOWN message, indicating that the user
  // has clicked the left mouse button. According to what the
  // current edit symbol is, an editing action is performed.
  SymbolClass *currentSymbol;
  BOOL skipDefault = FALSE;
  if (melody.melodyExists)
    if (!melody.scoreDisplay)
      switch ( GetActiveSymbol() ) {
        case PENCIL:
          // The current symbol is the pencil on a staff:
          // insert a new staff at the current position
          NewMultipleStaff(MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit));
          break;
        case ERASER:
          // The current symbol is the eraser:
          // erase the symbols at the current cursor position
          DeleteMusicalObject(
          MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit),
            MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit));
          break;
        case MARK_BLOCK:
          _isCapture = TRUE;
          capture = NEWCONTINUOUS;
          _currContSymbol = BlockSymbols[0];
          // The symbol corresponds to a point object:
          // just insert the object in the staff
          Xto = Xfrom = MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit);
          Yto = Yfrom = MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit);
          SetCapture(HWindow);
          break;
        case CONNECT:
          Xfrom = MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit);
          Yfrom = MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit);
          SetCapture(HWindow);
          break;
        case HAND:
          // The current symbol is the hand:
          // capture the mouse until the left button is released,
          // then move the objects to their new place.
          Xfrom = MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit);
          Yfrom = MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit);
          SetCapture(HWindow);
          capture = MOVEOBJECT;
          break;
        case KEYSIG:
          Part *p = (Part *)&melody.part[melody.displayedPart];
          int staffIndex = IdentifyStaff(p->staff, MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit));
          Staff *s = (staffIndex >= 0) ? (Staff *) &p->staff[staffIndex] : NULL;
          if (s) {
            int type, stam;
            ((PointObject *)&(s->pointObject[0]))->MIDIPlay(type, stam, stam, stam);
            if (type == O_KEY) {
              ((PointObject *)&s->pointObject[0])->ChangeSignature(trans/10, trans % 10);
            }
          }
          InvalidateRect(HWindow, NULL, TRUE);
          break;
        case TEXT:
          currentSymbol = (*Tool)->GetTextSymbolClass();
          skipDefault = TRUE;
          /* No Break should be here */
        case PASTE_BLOCK:
          if (!skipDefault) {
            currentSymbol = BlockSymbols[1];
            skipDefault = TRUE;
          }
        default:
          // The current symbol is one of the musical object symbols:
          // insert the appropriate object at the current position
          if (!skipDefault) {
            if (!*Tool) return;
            currentSymbol = (*Tool)->getSymbolClass();
          };
          switch (currentSymbol->GetType()) {
            case POINTOBJECT:
              // The symbol corresponds to a point object:
              // just insert the object in the staff
              NewPointObject(currentSymbol,
                MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit),
                MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit));
              break;
            case CONTINUOUSOBJECT:
              // The symbol corresponds to a continuous object:
              // capture the mouse until the left button is released,
              // then create the object between its two extents.
              Xfrom = MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit);
              Yfrom = MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit);
              SetCapture(HWindow);
              capture = NEWCONTINUOUS;
              _currContSymbol = currentSymbol;
              break;
          }
        }
    else
      // No editing is allowed on a score display
      MessageBox(HWindow, "Cannot edit a score display", NULL, MB_ICONSTOP | MB_OK);
}

void TPaintWindow::WMLButtonDblclk(RTMessage Msg)
{
  if ( GetActiveSymbol() == ERASER ) {
    DeleteMultipleStaff(MAKEPOINT(Msg.LP).y);
  }
}

void TPaintWindow::DrawMarkedRect()
{
  RECT r;

  r.left = min(Xfrom, Xto);
  r.right = max(Xfrom, Xto);
  r.top = min(Yfrom, Yto);
  r.bottom = max(Yfrom, Yto);
  HDC hdc = GetDC(HWindow);
  SetROP2(hdc, R2_XORPEN);
  Rectangle(hdc, r.left-(Scroller->XPos)*(Scroller->XUnit),
            r.top-(Scroller->YPos)*(Scroller->YUnit),
            r.right-(Scroller->XPos)*(Scroller->XUnit),
            r.bottom-(Scroller->YPos)*(Scroller->YUnit));
  ReleaseDC(HWindow, hdc);
}

void TPaintWindow::WMMouseMove(TMessage &Msg)
{
  if (_isCapture) {
    RECT r;
    Part *p = (Part *)&melody.part[melody.displayedPart];
    if (Yto < Yfrom + melody.GetStaffWidth() * p->multiplicity() ) {
      if (Xto != -1)   /* Erasing the xor block that was printed */
        DrawMarkedRect();
      Xto = MAKEPOINT(Msg.LP).x + (Scroller->XPos)*(Scroller->XUnit);
      Yto = MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit);
      DrawMarkedRect();
    }
  }
}

void TPaintWindow::WMLButtonUp(TMessage &Msg)
{
  if (GetCapture() == HWindow) {
    switch (capture) {
      case NEWCONTINUOUS:
        // The capture was due to insertion of a continuous object:
        // insert a new continuous object between the two points
        if (_isCapture) DrawMarkedRect();
        Xto = MAKEPOINT(Msg.LP).x+(Scroller->XPos)*(Scroller->XUnit);
        Yto = MAKEPOINT(Msg.LP).y + (Scroller->YPos)*(Scroller->YUnit);
        if (Xto < Xfrom) {
          int temp = Xto;
          Xto = Xfrom;
          Xfrom = temp;
        }
        if (Xto != Xfrom)
          NewContinuousObject(_currContSymbol, Xfrom+melody.pixelsPerObject/2,
                                               Xto-melody.pixelsPerObject/2,
                                               Yfrom);
        break;
      case MOVEOBJECT:
        // The capture was due to an object moving operation:
        // move a musical object to its new place
        Xto = MAKEPOINT(Msg.LP).x;
        Yto = MAKEPOINT(Msg.LP).y;
        MoveMusicalObject(Xfrom, Yfrom, Xto, Yto);
        break;
      case MOVESTAFF:
        // The capture was due to a staff moving operation:
        // move the staff to its new place
        Yto = MAKEPOINT(Msg.LP).y;
        MoveStaff(Yfrom, Yto);
        break;
    }
    _isCapture = FALSE;
    ReleaseCapture();
    Xto = -1;
    Yto = -1;
  }
}

void TPaintWindow::NewPointObject(SymbolClass *symbol, int X, int Y)
{
  // Obtain the staff to insert the object into
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  int staffIndex = IdentifyStaff(p.staff, (Y += p.GetPartY()));
  Staff *s = (staffIndex >= 0) ? (Staff *) &p.staff[staffIndex] : NULL;

  if (!s || X < s->X() || X >= s->X()+s->width()) return;

  // Create the object and decide where to insert it
  int first, last;
  MusicalObject *obj = symbol->CreateObject(staffIndex, WidthRound(X-s->X()), Y -= s->Y(), s->width());
  if (obj) {
    X = ((PointObject *) obj)->X();
    // Check the object _location attribute
    switch (obj->location() & ~ONEPERSTAFF) {
      case INSTAFF:
      case ABOVESTAFF:
      case BELOWSTAFF:
        first = last = staffIndex;
        break;
      case ABOVEMULTIPLE:
        first = last = staffIndex/p.multiplicity()*p.multiplicity();
        break;
      case BELOWMULTIPLE:
        first = last = (staffIndex/p.multiplicity()+1)*p.multiplicity()-1;
        break;
      case COMMONMULTIPLE:
        first = staffIndex/p.multiplicity()*p.multiplicity();
        last = (staffIndex/p.multiplicity()+1)*p.multiplicity()-1;
        break;
    }
    // Insert the objects in the required staff or staves
    for (staffIndex = first; staffIndex <= last; ++staffIndex) {
      s = (Staff *) &p.staff[staffIndex];
      IndexedList &list = s->pointObject;
      PointObject *obj1;
      for (int index = 0;
        index < list.number() && (obj1 = (PointObject *) &list[index])->X() <= X;
        ++index)
        if (obj1->location() & ONEPERSTAFF && obj1->X() == X) {
          MessageBox(HWindow, "Only one such symbol is allowed per staff", NULL,
            MB_ICONEXCLAMATION | MB_OK | MB_TASKMODAL);
          return;
        }
      list.insertAt(
        (staffIndex == first) ? *obj : *symbol->CreateObject(staffIndex, X, Y, s->width()), index);
    }
  }
  melody.melodyModified = TRUE;
  // Refresh screen
  InvalidateRect(HWindow, NULL, !obj);
}

// **********************************************
// NewContinuousObject creates a continuous object corresponding to the
// active symbol at the given coordinate (obtained from the current
// cursor position), and inserts it in the staff according
// to the object _location attribute.
void TPaintWindow::NewContinuousObject(SymbolClass *symbol, int Xleft, int Xright, int Y)
{
  // Obtain the staff to insert the object into
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  int staffIndex = IdentifyStaff(p.staff, (Y += p.GetPartY()));
  Staff *s = (staffIndex >= 0) ? (Staff *) &p.staff[staffIndex] : NULL;

  if (!s || Xleft < s->X() || Xright >= s->X()+s->width()) return;
  // Create the object and decide where to insert it
  int first, last;
  MusicalObject *obj =
    symbol->CreateObject(staffIndex, WidthRound(Xleft-s->X()), WidthRound(Xright-s->X()), 0);
  if (obj) {
    Xleft = ((ContinuousObject *) obj)->Xleft();
    Xright = ((ContinuousObject *) obj)->Xright();
    // Check the object _location attribute
    switch (obj->location() & ~ONEPERSTAFF) {
      case INSTAFF:
      case ABOVESTAFF:
      case BELOWSTAFF:
        first = last = staffIndex;
        break;
      case ABOVEMULTIPLE:
      first = last = staffIndex/p.multiplicity()*p.multiplicity();
        break;
      case BELOWMULTIPLE:
        first = last = (staffIndex/p.multiplicity()+1)*p.multiplicity()-1;
        break;
      case COMMONMULTIPLE:
        first = staffIndex/p.multiplicity()*p.multiplicity();
        last = (staffIndex/p.multiplicity()+1)*p.multiplicity()-1;
        break;
    }
    // Insert the object in the appropriate staff or staves
    for (staffIndex = first; staffIndex <= last; ++staffIndex) {
      s = (Staff *) &p.staff[staffIndex];
      IndexedList &list = s->continuousObject;
      ContinuousObject *obj1;
      for (int index = 0;
        index < list.number() &&
          (obj1 = (ContinuousObject *) &list[index])->Xleft() <= Xleft;
        ++index)
      if (obj1->location() & ONEPERSTAFF && obj1->Xleft() == Xleft) {
          MessageBox(hMainWnd, "Only one such symbol is allowed per staff", NULL,
            MB_ICONEXCLAMATION | MB_OK);
          return;
        }
      list.insertAt((staffIndex == first) ? *obj :
        *symbol->CreateObject(staffIndex, Xleft, Xright, 0), index);
    }
  } else {

  }
  melody.melodyModified = TRUE;
  // Refresh screen
  InvalidateRect(HWindow, NULL, !obj);
}


// **********************************************
// MoveStaff moves a staff from Yfrom to Yto, both coordinates
// obtained from the mouse cursor. Before actually moving,
// the destination is checked to be free from other staves.
void TPaintWindow::MoveStaff(int Yfrom, int Yto)
{
  // Obtain a pointer to the staff to move
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  int staffIndex = IdentifyStaff(p.staff, (Yfrom += p.GetPartY()));

  if (staffIndex < 0) return;
  staffIndex = staffIndex/p.multiplicity()*p.multiplicity();
  Staff *s = (Staff *) &p.staff[staffIndex];
  int lastMoved = staffIndex+p.multiplicity()-1;
  // Verify that the destination is not occupied
  Yto += p.GetPartY();
  for (int indexTo = 0; indexTo < p.staff.number(); indexTo += p.multiplicity())
    if (indexTo != staffIndex) {
      Staff &firstTo = *((Staff *) &p.staff[indexTo]);
      Staff &lastTo = *((Staff *) &p.staff[indexTo+p.multiplicity()-1]);
      if (Yto <= lastTo.Y()+24 &&
        ((Staff *) &p.staff[lastMoved])->Y()-s->Y()+Yto+24 >= firstTo.Y()) {
        // The destination is occupied:
        // display an error message
        MessageBox(hMainWnd, "Cannot move onto another staff", NULL,
          MB_ICONEXCLAMATION | MB_OK);
        return;
    }
  }
  // Set the staff new coordinates and move it
  int distance = Yto-s->Y();
  for (; staffIndex <= lastMoved; ++staffIndex)
    ((Staff *) &p.staff[staffIndex])->Y() += distance;
  staffIndex -= p.multiplicity();
  for (indexTo = 0;
    indexTo < p.staff.number() && ((Staff *) &p.staff[indexTo])->Y() < Yto;
    indexTo += p.multiplicity());
  while (staffIndex <= lastMoved) {
    s = (Staff *) &p.staff[staffIndex];
    // Detach and re-insert staves whose index is between
    // the old and new indexes of the moved staff
    p.staff.detachAt(staffIndex);
    if (indexTo > staffIndex) {
      --lastMoved;
      --indexTo;
    }
    p.staff.insertAt(*s, indexTo);
    if (staffIndex >= indexTo) {
      ++staffIndex;
      ++indexTo;
    }
  }
  // Refresh the display
  MarkedBlock.Unmark();
  melody.melodyModified = TRUE;
  InvalidateRect(HWindow, NULL, TRUE);
}

// **********************************************
// MoveMusicalObject moves a musical object by using
// the CutBlock and PasteBlock functions in BLOCK.CPP.
void TPaintWindow::MoveMusicalObject(int Xfrom, int Yfrom, int Xto, int Yto)
{
  // Verify both points are on the same staff
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  int staffFrom = IdentifyStaff(p.staff, Yfrom+p.GetPartY());
  int staffTo = IdentifyStaff(p.staff, Yto+p.GetPartY());

  if (staffFrom < 0 || staffTo < 0)
    return;
  staffFrom = staffFrom/p.multiplicity()*p.multiplicity();
  staffTo = staffTo/p.multiplicity()*p.multiplicity();
  int staffX = ((Staff *) &p.staff[staffFrom])->X();
  // Use the cut and paste functions to move the objects
  MarkedBlock.Mark(staffFrom, WidthRound(Xfrom-staffX),
                     staffFrom, WidthRound(Xfrom-staffX));
  MarkedBlock.Cut();
  MarkedBlock.Paste(staffTo, WidthRound(Xto-staffX));
  InvalidateRect(HWindow, NULL, TRUE);
}

// **********************************************
// IdentifyStaff finds the index of the staff to which the Y coordinate
// (obtained from the current cursor position) is closest.
int TPaintWindow::IdentifyStaff(IndexedList &staffList, unsigned Y)
{
  unsigned minDistance = MAXINT;
  int minIndex;
  int staffY;
  // Find a staff that is closest to the clicked point,
  // by minimizing the distance from different staves
  for (int index = 0; index < staffList.number(); ++index) {
    // If point actually on the staff, return it
    if ((staffY = ((Staff *) &staffList[index])->Y()) <= Y &&
      staffY+24 >= Y)
      return index;
    int distance = (staffY > Y) ? staffY-Y : Y-staffY-25;
    if (distance < minDistance) {
      minDistance = distance;
      minIndex = index;
    }
  }
  // Return a staff only if within reasonable distance
  return (minDistance < (melody.pixelsPerStaff-25)/2) ? minIndex : -1;
}

// **********************************************
// DeleteMusicalObject deletes the objects that are within
// (pixelsPerObject/2) pixels away from the given coordinate
// (obtained from the current cursor position).

void TPaintWindow::DeleteMusicalObject(int X, int Y)
{
  // Obtain the staff to delete the objects from
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  int staffIndex = IdentifyStaff(p.staff, (Y += p.GetPartY()));
  Staff *s = (staffIndex >= 0) ? (Staff *) &p.staff[staffIndex] : NULL;
  if (!s || X < s->X() || X >= s->X()+s->width()) return;
  // Scan the point objects list and delete any objects in range
  X -= s->X();
  Y -= s->Y();
  for (int index = 0; index < s->pointObject.number(); ++index) {
    PointObject *obj = (PointObject *) &s->pointObject[index];
    if (abs(obj->X()-X) < melody.pixelsPerObject/2)
      // Check the object _location attribute to see if
      // any special treatment is required
      switch (obj->location() & ~ONEPERSTAFF) {
        case ABOVESTAFF:
        case ABOVEMULTIPLE:
          if (Y < 0) {
            s->pointObject.destroyAt(index);
            --index;
          }
          break;
        case BELOWSTAFF:
        case BELOWMULTIPLE:
          if (Y > 24) {
            s->pointObject.destroyAt(index);
            --index;
          }
          break;
        case INSTAFF:
          s->pointObject.destroyAt(index);
          --index;
          break;
      }
  }
  // Scan the continuous objects list and delete any objects in range
  for (index = 0; index < s->continuousObject.number(); ++index) {
    ContinuousObject *obj = (ContinuousObject *) &s->continuousObject[index];
    if (abs(obj->Xleft()-X) < melody.pixelsPerObject/2)
      switch (obj->location() & ~ONEPERSTAFF) {
        case ABOVESTAFF:
        case ABOVEMULTIPLE:
          if (Y < 0) {
            s->continuousObject.destroyAt(index);
            --index;
            melody.melodyModified = TRUE;
          }
          break;
        case BELOWSTAFF:
        case BELOWMULTIPLE:
          if (Y > 24) {
            s->continuousObject.destroyAt(index);
            --index;
            melody.melodyModified = TRUE;
          }
          break;
        case INSTAFF:
          s->continuousObject.destroyAt(index);
          --index;
          melody.melodyModified = TRUE;
          break;
      }
  }
  // Scan all staves in group and delete COMMONMULTIPLE objects
  int first = staffIndex/p.multiplicity()*p.multiplicity();
  int last = (staffIndex/p.multiplicity()+1)*p.multiplicity()-1;
  for (staffIndex = first; staffIndex <= last; ++staffIndex) {
    // Delete COMMONMULTIPLE point objects
    s = (Staff *) &p.staff[staffIndex];
    for (index = 0; index < s->pointObject.number(); ++index) {
      PointObject *obj = (PointObject *) &s->pointObject[index];
      if (abs(obj->X()-X) < melody.pixelsPerObject/2 &&
        (obj->location() & ~ONEPERSTAFF) == COMMONMULTIPLE) {
        s->pointObject.destroyAt(index);
        --index;
        melody.melodyModified = TRUE;
      }
    }
    for (index = 0; index < s->continuousObject.number(); ++index) {
      // Delete COMMONMULTIPLE continuous objects
      ContinuousObject *obj = (ContinuousObject *) &s->continuousObject[index];
      if (abs(obj->Xleft()-X) < melody.pixelsPerObject/2 &&
        (obj->location() & ~ONEPERSTAFF) == COMMONMULTIPLE) {
        s->continuousObject.destroyAt(index);
        --index;
        melody.melodyModified = TRUE;
      }
    }
  }
  // Refresh screen
  InvalidateRect(HWindow, NULL, TRUE);
}

// **********************************************
// NewMultipleStaff creates a new multiple staff, after the user
// has clicked the pencil-on-staff symbol.
// The staff is inserted in the list, and the scroll bar range
// is readjusted.

void TPaintWindow::NewMultipleStaff(int Y)
{
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  // Insert a new staff group in the database and refresh screen
  for (int i = 0; i < p.multiplicity(); ++i)
    InsertEmptyStaff(p.staff, Y+p.GetPartY(), p.multiplicity());
  melody.melodyModified = TRUE;
  UpdateScrollerRange();
  InvalidateRect(HWindow, NULL, TRUE);
}

// **********************************************
// InsertEmptyStaff inserts a new empty staff in the specified list
// given the staff Y location and the part multiplicity.
// The list is kept sorted by the Y locations in ascending order.

void TPaintWindow::InsertEmptyStaff(IndexedList &staffList, unsigned Y, int multiplicity)
{
  // Find the index to insert at
  for (int index = 0;
    index < staffList.number() && ((Staff *) &staffList[index])->Y() < Y;
    index += multiplicity);
  if (index > staffList.number())
    index = staffList.number();
  // Update coordinates of other staves and of the marked block
  for (int other = index; other < staffList.number(); ++other)
    ((Staff *) &staffList[other])->Y() += melody.pixelsPerStaff;
  if (MarkedBlock.markBeginStaff >= index)
    ++MarkedBlock.markBeginStaff;
  if (MarkedBlock.markEndStaff >= index)
    ++MarkedBlock.markEndStaff;
  // Adjust Y   to a multiple of melody.pixelsPerStaff
  if (index) {
    Y = ((Staff *) &staffList[index-1])->Y()+melody.pixelsPerStaff;
  } else
    Y = melody.pixelsPerStaff;
  // Insert the staff
  staffList.insertAt(*new Staff(Y, melody.GetStaffWidth()), index);
//  Scroller->SetRange(Scroller->XRange, ((Staff *)&staffList[staffList.number()-1])->Y() / Scroller->YUnit );
  UpdateScrollerRange();
}

// **********************************************
// DeleteMultipleStaff deletes the multiple staff identified
// by a Y coordinate (obtained from the current cursor position).
// If the staff contains any objects, the user is requested
// to confirm the operation.
void TPaintWindow::DeleteMultipleStaff(int Y)
{
  // Obtain a pointer to the staff to delete
  Part &p = *((Part *) &melody.part[melody.displayedPart]);
  int staffIndex = IdentifyStaff(p.staff, (Y += p.GetPartY()));
  if (staffIndex < 0) return;
  staffIndex = staffIndex/p.multiplicity()*p.multiplicity();
  Staff *s = (Staff *) &p.staff[staffIndex];
  // Check if the multiple staff contains any objects

  for (int i = staffIndex;
       i/p.multiplicity() == staffIndex/p.multiplicity(); ++i) {
    Staff *s = (Staff *) &p.staff[i];
    if (s->pointObject.number() || s->continuousObject.number())
      if (MessageBox(HWindow, "Staff is not empty. Erase anyway?", "WARNING",
        MB_ICONEXCLAMATION | MB_YESNOCANCEL) == IDYES)
        break;
      else return;
  }
  // Erase the staff
  int staffY = s->Y();
  for (i = 0; i < p.multiplicity(); ++i)
    p.staff.destroyAt(staffIndex);
  if (MarkedBlock.markBeginStaff > staffIndex)
    MarkedBlock.markBeginStaff -= p.multiplicity();
  if (MarkedBlock.markEndStaff >= staffIndex)
    MarkedBlock.markEndStaff -= p.multiplicity();
  if (MarkedBlock.markEndStaff < MarkedBlock.markBeginStaff)
    MarkedBlock.Unmark();
  if (p.staff.number() > staffIndex) {
    int staffDiff = ((Staff *) &p.staff[staffIndex])->Y()-staffY;
    for (; staffIndex < p.staff.number(); ++staffIndex)
      ((Staff *) &p.staff[staffIndex])->Y() -= staffDiff;
  }
  // Mark the melody as modified and readjust the scroll bar range
  melody.melodyModified = TRUE;
  UpdateScrollerRange();
  InvalidateRect(HWindow, NULL, TRUE);
}



void TPaintWindow :: UpdateScrollerRange()
{
  Part &p = *((Part *) &melody.part[melody.displayedPart]);

  if ( p.staff.number() > 0 ) {
    Scroller->SetRange(melody.GetStaffWidth()/Scroller->XUnit,
          ((Staff *)&p.staff[p.staff.number()-1])->Y() / Scroller->YUnit );
  }
};

BOOL Staff :: Draw(HDC hDC, int editYMin, BOOL clip, unsigned width)
{
  int line = Y();
  RECT staffRect = {0, line-editYMin, width, line-editYMin+24};
  RECT windowRect, dummyRect;

  // Check whether staff is within screen range
  if (clip) {
    GetClientRect(hPaintWnd, &windowRect);
    editYMin = 0;
  }
  if (!clip || IntersectRect(&dummyRect, &staffRect, &windowRect)) {
    // Either no clip checking or the staff is within clipping boundaries:
    // draw the staff lines
    for (; line < Y()+5*6; line += 6 ) {
      MoveTo(hDC, X(), line-editYMin);
      LineTo(hDC, X()+width, line-editYMin);
    }
    return TRUE;
  }
  return FALSE;
}
