// **********************************************
// File: SYMCLASS.CPP
// Symbols classes library

#include "muzika.h"
#include "dialog.h"
#include "objects.h"
#include <values.h>
#include <stdio.h>

extern HWND hMainWnd; // defined in newmain.cpp (to allow ..)
                      // opening the text dialog.
extern int Dots;      // is the dot button pressed ?
// **********************************************
// Following are the member function definitions for
// the generic SymbolClass class, from which all the symbol
// classes are derived.

// **********************************************
// SymbolClass::BitmapName returns the bitmap name
// (without the "B_" prefix) of the symbol in the resource file.

int SymbolClass :: BitmapName(LPSTR lpBuffer, int)
{
  // Here for historical reasons.
  wsprintf(lpBuffer, "B_%s", _bitmapName);
  return 1;
}

// **********************************************
// SymbolClass::DrawSymbol draws the symbol at its place
// on the screen, derived from its ID code. The symbol is drawn
// in normal or reverse video, according to whether it is active or not.

void SymbolClass :: DrawSymbol(HDC hDC, HDC hBitmapDC, BOOL active)
{
  HBITMAP hBitmap, hOldBitmap;
  char name[19] = "B_";
  // Test if the symbol exists at all
  if (BitmapName(name+2, sizeof(name)-2)) {
    // Load the symbol bitmap from the resource file
    hBitmap = LoadBitmap(GetApplicationObject()->hInstance, name);
    hOldBitmap = SelectObject(hBitmapDC, hBitmap);
    // Display the bitmap in normal or reverse video
    if (active)
      BitBlt(hDC, 2+36*(symbolID%2), 38+24*(symbolID%0x10/2), 32, 20,
        hBitmapDC, 0, 0, NOTSRCCOPY);
    else
      BitBlt(hDC, 2+36*(symbolID%2), 38+24*(symbolID%0x10/2), 32, 20,
        hBitmapDC, 0, 0, SRCCOPY);
    SelectObject(hBitmapDC, hOldBitmap);
    DeleteObject(hBitmap);
  }
}

// **********************************************
// The NullSymbol class, derived from SymbolClass,
// is the default for the "empty slot" symbols.
// It does nothing except drawing an empty rectangle
// as its DrawSymbol function.
NullSymbol :: NullSymbol()
{
  symbolID = MAXINT;
}

void NullSymbol :: DrawSymbol(HDC hDC, int ID)
{
  Rectangle(hDC, 36*(ID%2), 36+24*(ID%0x10/2),
    36*(ID%2+1)+1, 36+24*(ID%0x10/2+1)+1);
}

NullSymbol nullSymbol;

// **********************************************
// Following are definitions of various symbol classes,
// all derived from the generic SymbolClass.
// For each symbol class there is a constructor,
// putting values in the symbolID and symbolType variables,
// and a CreateObject function, which returns a pointer
// to a newly created MusicalObject corresponding to the symbol.
// Any future extensions of symbols should be added here.
class FullNoteSymbol : virtual public SymbolClass {
  public:
    FullNoteSymbol() { symbolID = 16; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} FullNoteSymbol;

MusicalObject *FullNoteSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new Note(X, Y, FULLDURATION, Dots);
}

class FullPauseSymbol : virtual public SymbolClass {
  public:
    FullPauseSymbol() {symbolID = 17; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int, unsigned);
} FullPauseSymbol;

MusicalObject *FullPauseSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Pause(X, FULLDURATION, Dots);
}

class HalfNoteSymbol : virtual public SymbolClass {
  public:
    HalfNoteSymbol() {symbolID = 18; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} HalfNoteSymbol;

MusicalObject *HalfNoteSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new Note(X, Y, HALFDURATION, Dots);
}

class HalfPauseSymbol : virtual public SymbolClass {
  public:
    HalfPauseSymbol() {symbolID = 19; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int, unsigned);
} HalfPauseSymbol;

MusicalObject *HalfPauseSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Pause(X, HALFDURATION, Dots);
}

class QuarterNoteSymbol : virtual public SymbolClass {
  public:
    QuarterNoteSymbol() {symbolID = 20; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} QuarterNoteSymbol;

MusicalObject *QuarterNoteSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new Note(X, Y, QUARTERDURATION, Dots);
}

class QuarterPauseSymbol : virtual public SymbolClass {
  public:
    QuarterPauseSymbol() {symbolID = 21; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int, unsigned);
} QuarterPauseSymbol;

MusicalObject *QuarterPauseSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Pause(X, QUARTERDURATION, Dots);
}

class EighthNoteSymbol : virtual public SymbolClass {
  public:
    EighthNoteSymbol() {symbolID = 22; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} EighthNoteSymbol;

MusicalObject *EighthNoteSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new Note(X, Y, EIGHTHDURATION, Dots);
}

class EighthPauseSymbol : virtual public SymbolClass {
  public:
    EighthPauseSymbol() {symbolID = 23; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int, unsigned);
} EighthPauseSymbol;

MusicalObject *EighthPauseSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Pause(X, EIGHTHDURATION, Dots);
}

class SixteenthNoteSymbol : virtual public SymbolClass {
  public:
    SixteenthNoteSymbol() {symbolID = 24; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} SixteenthNoteSymbol;

MusicalObject *SixteenthNoteSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new Note(X, Y, SIXTEENTHDURATION, Dots);
}

class SixteenthPauseSymbol : virtual public SymbolClass {
  public:
    SixteenthPauseSymbol() {symbolID = 25; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int, unsigned);
} SixteenthPauseSymbol;

MusicalObject *SixteenthPauseSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Pause(X, SIXTEENTHDURATION, Dots);
}

class Thirty2NoteSymbol : virtual public SymbolClass {
  public:
    Thirty2NoteSymbol() {symbolID = 26; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} Thirty2NoteSymbol;

MusicalObject *Thirty2NoteSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new Note(X, Y, THIRTY2DURATION, Dots);
}

class Thirty2PauseSymbol : virtual public SymbolClass {
  public:
    Thirty2PauseSymbol() {symbolID = 27; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int, unsigned);
} Thirty2PauseSymbol;

MusicalObject *Thirty2PauseSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Pause(X, THIRTY2DURATION, Dots);
}

class FullNoteSharpSymbol : virtual public SymbolClass {
  public:
    FullNoteSharpSymbol() {symbolID = 16; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} FullNoteSharpSymbol;

MusicalObject *FullNoteSharpSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteSharp(X, Y, FULLDURATION, Dots);
}

class HalfNoteSharpSymbol : virtual public SymbolClass {
  public:
    HalfNoteSharpSymbol() {symbolID = 18; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} HalfNoteSharpSymbol;

MusicalObject *HalfNoteSharpSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteSharp(X, Y, HALFDURATION, Dots);
}

class QuarterNoteSharpSymbol : virtual public SymbolClass {
  public:
    QuarterNoteSharpSymbol() {symbolID = 20; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} QuarterNoteSharpSymbol;

MusicalObject *QuarterNoteSharpSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteSharp(X, Y, QUARTERDURATION, Dots);
}

class EighthNoteSharpSymbol : virtual public SymbolClass {
  public:
    EighthNoteSharpSymbol() {symbolID = 22; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} EighthNoteSharpSymbol;

MusicalObject *EighthNoteSharpSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteSharp(X, Y, EIGHTHDURATION, Dots);
}

class SixteenthNoteSharpSymbol : virtual public SymbolClass {
  public:
    SixteenthNoteSharpSymbol() {symbolID = 24; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} SixteenthNoteSharpSymbol;

MusicalObject *SixteenthNoteSharpSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteSharp(X, Y, SIXTEENTHDURATION, Dots);
}

class Thirty2NoteSharpSymbol : virtual public SymbolClass {
  public:
    Thirty2NoteSharpSymbol() {symbolID = 26; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} Thirty2NoteSharpSymbol;

MusicalObject *Thirty2NoteSharpSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteSharp(X, Y, THIRTY2DURATION, Dots);
}

class FullNoteNaturalSymbol : virtual public SymbolClass {
  public:
    FullNoteNaturalSymbol() {symbolID = 16; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} FullNoteNaturalSymbol;

MusicalObject *FullNoteNaturalSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteNatural(X, Y, FULLDURATION, Dots);
}

class HalfNoteNaturalSymbol : virtual public SymbolClass {
  public:
    HalfNoteNaturalSymbol() {symbolID = 18; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} HalfNoteNaturalSymbol;

MusicalObject *HalfNoteNaturalSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteNatural(X, Y, HALFDURATION, Dots);
}

class QuarterNoteNaturalSymbol : virtual public SymbolClass {
  public:
    QuarterNoteNaturalSymbol() {symbolID = 20; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} QuarterNoteNaturalSymbol;

MusicalObject *QuarterNoteNaturalSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteNatural(X, Y, QUARTERDURATION, Dots);
}

class EighthNoteNaturalSymbol : virtual public SymbolClass {
  public:
    EighthNoteNaturalSymbol() {symbolID = 22; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} EighthNoteNaturalSymbol;

MusicalObject *EighthNoteNaturalSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteNatural(X, Y, EIGHTHDURATION, Dots);
}

class SixteenthNoteNaturalSymbol : virtual public SymbolClass {
  public:
    SixteenthNoteNaturalSymbol() {symbolID = 24; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} SixteenthNoteNaturalSymbol;

MusicalObject *SixteenthNoteNaturalSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteNatural(X, Y, SIXTEENTHDURATION, Dots);
}

class Thirty2NoteNaturalSymbol : virtual public SymbolClass {
  public:
    Thirty2NoteNaturalSymbol() {symbolID = 26; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} Thirty2NoteNaturalSymbol;

MusicalObject *Thirty2NoteNaturalSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteNatural(X, Y, THIRTY2DURATION, Dots);
}

class FullNoteFlatSymbol : virtual public SymbolClass {
  public:
    FullNoteFlatSymbol() {symbolID = 16; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} FullNoteFlatSymbol;

MusicalObject *FullNoteFlatSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteFlat(X, Y, FULLDURATION, Dots);
}

class HalfNoteFlatSymbol : virtual public SymbolClass {
  public:
    HalfNoteFlatSymbol() {symbolID = 18; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} HalfNoteFlatSymbol;

MusicalObject *HalfNoteFlatSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteFlat(X, Y, HALFDURATION, Dots);
}

class QuarterNoteFlatSymbol : virtual public SymbolClass {
  public:
    QuarterNoteFlatSymbol() {symbolID = 20; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} QuarterNoteFlatSymbol;

MusicalObject *QuarterNoteFlatSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteFlat(X, Y, QUARTERDURATION, Dots);
}

class EighthNoteFlatSymbol : virtual public SymbolClass {
  public:
    EighthNoteFlatSymbol() {symbolID = 22; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} EighthNoteFlatSymbol;

MusicalObject *EighthNoteFlatSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteFlat(X, Y, EIGHTHDURATION, Dots);
}

class SixteenthNoteFlatSymbol : virtual public SymbolClass {
  public:
    SixteenthNoteFlatSymbol() {symbolID = 24; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} SixteenthNoteFlatSymbol;

MusicalObject *SixteenthNoteFlatSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteFlat(X, Y, SIXTEENTHDURATION, Dots);
}

class Thirty2NoteFlatSymbol : virtual public SymbolClass {
  public:
    Thirty2NoteFlatSymbol() {symbolID = 26; symbolType = POINTOBJECT;}
    virtual MusicalObject *CreateObject(int, int X, int Y, unsigned);
} Thirty2NoteFlatSymbol;

MusicalObject *Thirty2NoteFlatSymbol :: CreateObject(int, int X, int Y, unsigned)
{
  return new NoteFlat(X, Y, THIRTY2DURATION, Dots);
}

// -------------------------- Clefs ( keys ) ----------------------
class KeySolSymbol : virtual public SymbolClass {
public:
    KeySolSymbol() {
      _bitmapName = "KEYSOL"; symbolID = 32; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} KeySolSymbol;

MusicalObject *KeySolSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Key(X, KEYSOL);
}

class KeyFaSymbol : virtual public SymbolClass {
  public:
    KeyFaSymbol() {
      _bitmapName = "KEYFA"; symbolID = 33; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} KeyFaSymbol;

MusicalObject *KeyFaSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Key(X, KEYFA);
}

//------------------------ Beats --------------------------------
class BeatCSymbol : virtual public SymbolClass {
  public:
    BeatCSymbol() {
      _bitmapName = "BEATC"; symbolID = 48; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} BeatCSymbol;

MusicalObject *BeatCSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEATC);
}

class BeatCBarSymbol : virtual public SymbolClass {
  public:
    BeatCBarSymbol() {
      _bitmapName = "BEATCBAR"; symbolID = 49; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} BeatCBarSymbol;

MusicalObject *BeatCBarSymbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEATCBAR);
}

class Beat28Symbol : virtual public SymbolClass {
  public:
    Beat28Symbol() {
      _bitmapName = "BEAT28"; symbolID = 50; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat28Symbol;

MusicalObject *Beat28Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT28);
}

class Beat24Symbol : virtual public SymbolClass {
  public:
    Beat24Symbol() {
      _bitmapName = "BEAT24"; symbolID = 51; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat24Symbol;

MusicalObject *Beat24Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT24);
}

class Beat38Symbol : virtual public SymbolClass {
  public:
    Beat38Symbol() {
      _bitmapName = "BEAT38"; symbolID = 52; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat38Symbol;

MusicalObject *Beat38Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT38);
}

class Beat34Symbol : virtual public SymbolClass {
  public:
    Beat34Symbol() {
      _bitmapName = "BEAT34";symbolID = 53; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat34Symbol;

MusicalObject *Beat34Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT34);
}

class Beat48Symbol : virtual public SymbolClass {
  public:
    Beat48Symbol() {
      _bitmapName = "BEAT48"; symbolID = 54; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat48Symbol;

MusicalObject *Beat48Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT48);
}

class Beat44Symbol : virtual public SymbolClass {
  public:
    Beat44Symbol() {
       _bitmapName = "BEAT44"; symbolID = 55; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat44Symbol;

MusicalObject *Beat44Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT44);
}

class Beat68Symbol : virtual public SymbolClass {
  public:
    Beat68Symbol() {
      _bitmapName = "BEAT68"; symbolID = 56; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned);
} Beat68Symbol;

MusicalObject *Beat68Symbol :: CreateObject(int, int X, int, unsigned)
{
  return new Beat(X, BEAT68);
}

//---------------------- Bars ----------------------------------
class BarSymbol : virtual public SymbolClass {
  public:
    BarSymbol() {
       _bitmapName = "BAR"; symbolID = 64; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned width);
} BarSymbol;

MusicalObject *BarSymbol :: CreateObject(int, int X, int, unsigned width)
{
  return new Bar(X, width, BAR);
}

class DoubleBarSymbol : virtual public SymbolClass {
  public:
    DoubleBarSymbol() {
       _bitmapName = "DOUBLEBAR";symbolID = 65; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned width);
} DoubleBarSymbol;

MusicalObject *DoubleBarSymbol :: CreateObject(int, int X, int, unsigned width)
{
  return new Bar(X, width, DOUBLEBAR);
}

class StartBarSymbol : virtual public SymbolClass {
  public:
    StartBarSymbol() {
       _bitmapName = "STARTBAR"; symbolID = 66; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned width);
} StartBarSymbol;

MusicalObject *StartBarSymbol :: CreateObject(int, int X, int, unsigned width)
{
  return new Bar(X, width, STARTBAR);
}

class EndBarSymbol : virtual public SymbolClass {
  public:
    EndBarSymbol() {
       _bitmapName = "ENDBAR"; symbolID = 67; symbolType = POINTOBJECT;}
    MusicalObject *CreateObject(int, int X, int, unsigned width);
} EndBarSymbol;

MusicalObject *EndBarSymbol :: CreateObject(int, int X, int, unsigned width)
{
  return new Bar(X, width, ENDBAR);
}

// **********************************************
// The symbolArray is a global array of symbol classes
// in which a symbol with a given ID is searched.
// The list is sorted in the order of increasing ID,
// and ends with a NullSymbol (with the ID of MAXINT).
SymbolClass *NoteSymbols[] = {
  &FullNoteSymbol, &FullPauseSymbol,
  &HalfNoteSymbol, &HalfPauseSymbol,
  &QuarterNoteSymbol, &QuarterPauseSymbol,
  &EighthNoteSymbol, &EighthPauseSymbol,
  &SixteenthNoteSymbol, &SixteenthPauseSymbol,
  &Thirty2NoteSymbol  , &Thirty2PauseSymbol
};

SymbolClass *NoteSharpSymbols[] = {
  &FullNoteSharpSymbol, NULL,
  &HalfNoteSharpSymbol, NULL,
  &QuarterNoteSharpSymbol, NULL,
  &EighthNoteSharpSymbol, NULL,
  &SixteenthNoteSharpSymbol, NULL,
  &Thirty2NoteSharpSymbol  , NULL
};

SymbolClass *NoteNaturalSymbols[] = {
  &FullNoteNaturalSymbol, NULL,
  &HalfNoteNaturalSymbol, NULL,
  &QuarterNoteNaturalSymbol, NULL,
  &EighthNoteNaturalSymbol, NULL,
  &SixteenthNoteNaturalSymbol, NULL,
  &Thirty2NoteNaturalSymbol  , NULL
};

SymbolClass *NoteFlatSymbols[] = {
  &FullNoteFlatSymbol, NULL,
  &HalfNoteFlatSymbol, NULL,
  &QuarterNoteFlatSymbol, NULL,
  &EighthNoteFlatSymbol, NULL,
  &SixteenthNoteFlatSymbol, NULL,
  &Thirty2NoteFlatSymbol  , NULL
};

SymbolClass *KeySymbols[] = {
  &KeySolSymbol, &KeyFaSymbol
};

SymbolClass *BeatSymbols[] = {
  &BeatCSymbol, &BeatCBarSymbol,
  &Beat28Symbol, &Beat24Symbol,
  &Beat38Symbol, &Beat34Symbol,
  &Beat48Symbol, &Beat44Symbol,
  &Beat68Symbol
};

SymbolClass *BarSymbols[] = {
  &BarSymbol, &DoubleBarSymbol,
  &StartBarSymbol, &EndBarSymbol
};