/*
 * gusthru.c    - Turn your GUS into a sound module.

 * This code is adapted from the midithru code of Hannu Savolainen
 * as noted below.  The voice allocation scheme has been completely
 * rewritten, and support for things like sustain, etc. are added.
 *
 * Greg Wolodkin   1995
 *
 * Originally:
 * midithru.c	- Program for playing internal synth with external
 *		  Midi keybiard.
 * 
 * Copyright by Hannu Savolainen 1993
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <stdio.h>
#include <sys/soundcard.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/errno.h>

#define GSUSTAIN (uint) 0x00100000 
#define GRELEASE (uint) 0x00200000
#define GHISTORY (uint) 0x000FF000
#define GHISINCR (uint) 0x00001000
#define GCHANNEL (uint) 0x00000F00
#define GNOTEVAL (uint) 0x000000FF
#define GESEARCH (uint) 0x00000FFF

typedef enum {START, NEEDDATA1, NEEDDATA2, SYSEX, SYSTEM1, SYSTEM2, MTC} 
  InputState;

void exit_failure (char *);
int first_match(uint, uint);
uint first_available(uint, uint);

SEQ_DEFINEBUF (2048);
SEQ_PM_DEFINES;

int seqfd;
int numv, dev = 0;
uint *voice;

/* Assume 16 MIDI channels */
uint numc = 16;
uint sust[16];
uint patch[16];
uint bender[16];
uint panner[16];

void
seqbuf_dump (void)
{
  if (_seqbufptr)
    if (write (seqfd, _seqbuf, _seqbufptr) == -1)
      exit_failure("write /dev/sequencer");

  _seqbufptr = 0;
}

int
first_match(uint ch, uint note)
{
  uint k, search;

  search = (unsigned) (ch << 8) | note;

  for (k = 0; k < numv; k++)
    if ((voice[k] & GHISTORY) != 0)
      if ((voice[k] & GESEARCH) == search)
        if (!((voice[k] & GSUSTAIN) && (voice[k] & GRELEASE)))
          return k;
  return -1;
}

uint
first_available(uint ch, uint note)
{
  uint k, this, search;
  uint pold = 0, old = 0;

  search = (ch << 8) | note;

  for (k = 0; k < numv; k++) {
    this = voice[k] & GHISTORY;
    if (this == 0)			/* an empty slot */
      return k;
    else if (voice[k] & GESEARCH == search)
      return k;
    else {
      if (this > old) {
        old = this;
        pold = k;
      }
    }
  }
  return pold;
}

void
stop_note(uint ch, uint note, int vel)
{
  uint k, age;
  int target;

  while ((target = first_match(ch, note)) != -1) {
    if ((voice[target] & GSUSTAIN) != 0) {	/* the note is sustained */
      voice[target] |= GRELEASE;		/* just mark it released */
    } else {
      SEQ_STOP_NOTE(dev, target, note, vel);
      SEQ_DUMPBUF();

      age = voice[target] & GHISTORY;
      voice[target] = 0;

      for (k = 0; k < numv; k++)
        if ((voice[k] & GHISTORY) > age)
          voice[k] -= GHISINCR;
    }
  }
}

void
start_note(uint ch, uint note, int vel)
{
  uint k;

  k = first_available(ch, note);
  if ((voice[k] & GHISTORY) != 0) {
    voice[k] &= ~GSUSTAIN;
    stop_note((voice[k] & GCHANNEL) >> 8, voice[k] & GNOTEVAL, 0);
  }

  voice[k] = (ch << 8) | note;

  if (sust[ch])
    voice[k] |= GSUSTAIN;

  SEQ_SET_PATCH(dev, k, patch[ch]);
  SEQ_PITCHBEND(dev, k, bender[ch]);
  SEQ_CONTROL(dev, k, CTL_PAN, panner[ch]);
  SEQ_START_NOTE(dev, k, note, vel);
  SEQ_DUMPBUF();

  for (k = 0; k < numv; k++)
    if (voice[k])
      voice[k] += GHISINCR;
}

void
channel_pressure(uint ch, int pressure)
{
  int k;

  for (k = 0; k < numv; k++) {
    if ((voice[k] & GCHANNEL) == ch << 8) {
      SEQ_CHN_PRESSURE(dev, k, pressure);
      SEQ_DUMPBUF();
    }
  }
}

void
control_change(uint ch, int control, int val)
{
  uint k;

  switch (control) {
  case CTL_SUSTAIN:		/* sustain */
    sust[ch] = (val != 0);
    for (k = 0; k < numv; k++) {
      if ((voice[k] & GHISTORY) != 0) {
        if ((voice[k] & GCHANNEL) == ch << 8) {
          if (val != 0) {			/* sustain is down */
            voice[k] |= GSUSTAIN;
          } else {			/* sustain is up */
            voice[k] &= ~GSUSTAIN;
            if ((voice[k] & GRELEASE) != 0) {
              stop_note((voice[k] & GCHANNEL) >> 8, voice[k] & GNOTEVAL, 0);
            }
          }
        }
      }
    }
    break;

  case CTL_PAN:			/* pan */
    panner[ch] = val;
    /* fallthrough */

  default:
    for (k = 0; k < numv; k++) {
      if ((voice[k] & GCHANNEL) == ch << 8) {
        SEQ_CONTROL(dev, k, control, val);
        SEQ_DUMPBUF();
      }
    }
  }
}

void
pitch_bender(uint ch, int val)
{
  int k;

  bender[ch] = val - 8192;

  for (k = 0; k < numv; k++) {
    if ((voice[k] & GCHANNEL) == (ch << 8)) {
      SEQ_PITCHBEND(dev, k, bender[ch]);
      SEQ_DUMPBUF();
    }
  }
}

void
do_buf (unsigned char *foo)
{
  int value, ch = foo[0] & 0x0f;

  if (ch == 0x09) 
    return;

  switch (foo[0] & 0xf0) {
  case 0x90:			/* Note on */
    if (foo[2])
      start_note(ch, foo[1], foo[2]);
    else
      stop_note(ch, foo[1], foo[2]);
    break;

  case 0xb0:			/* Control change */
    control_change(ch, foo[1], foo[2]);
    break;

  case 0x80:			/* Note off */
    stop_note(ch, foo[1], foo[2]);
    break;

  case 0xe0:			/* Pitch bender */
    value = ((foo[2] & 0x7f) << 7) | (foo[1] & 0x7f);
    pitch_bender(ch, value);
    break;

  case 0xc0:			/* Pgm change */
    if (PM_LOAD_PATCH(dev, ch, foo[1]) < 0)
      if (errno != ESRCH)	/* No such process */
        perror("PM_LOAD_PATCH");

    patch[ch] = foo[1];
    SEQ_SET_PATCH(dev, ch, foo[1]);
    SEQ_DUMPBUF();
    break;

  case 0xd0:			/* Channel pressure */
    channel_pressure(ch, foo[1]);
    break;

  default:

  }

#if 0
  printf("--------------------------\n");
  for (value=0; value < numv; value++)
    if (voice[value] & GHISTORY)
      printf("%2i %8x\n", value, voice[value]);
#endif

}

int
main (int argc, char *argv[])
{
  int k;
  struct synth_info info;
  unsigned char ev[4], buf[256], rs=0;
  InputState state = START;
  int bufp=0;

  info.device = dev;

  if ((seqfd = open ("/dev/sequencer", O_RDWR, 0)) == -1)
    exit_failure("open /dev/sequencer");

  if (ioctl(seqfd, SNDCTL_SYNTH_INFO, &info)==-1)
    exit_failure("info /dev/sequencer");

#if 1
  numv = info.nr_voices;
#else
  numv = 16;			/* stereo patches, all of 'em.. */
#endif

  fprintf(stderr, "Output to %s [%i]\n", info.name, dev);
  fprintf(stderr, "%d voices available\n", numv);

  if ((voice = (uint *) calloc (numv, sizeof(uint))) == NULL)
    exit_failure("Allocating voice memory");

  for (k = 0; k < numc; k++) {
    if (PM_LOAD_PATCH(dev, k, 0) < 0)	/* Load the default instrument */
      if (errno != ESRCH)		/* No such process */
        perror("PM_LOAD_PATCH");

    patch[k] = 0; bender[k] = 0; panner[k] = 64;
    SEQ_SET_PATCH(dev, k, 0);
    SEQ_DUMPBUF();
  }

  while (1) {
    if ((k = read (seqfd, ev, sizeof (ev))) == -1)
      exit_failure("read /dev/sequencer");

#if 0
  for (k = 0; k < 4; k++) 
    fprintf(stderr," %x", ev[k]);
  fprintf(stderr,"\n");
#endif

    if (ev[0] == SEQ_MIDIPUTC) {
      switch (state) {
      case START:
        if (!(ev[1] & 0x80)) {
          switch (rs & 0xf0) {
          case 0x80:
          case 0x90:
          case 0xa0:
          case 0xb0:
          case 0xe0:
            /* explicitly add running state */
            buf[bufp++] = rs; 
            buf[bufp++] = ev[1];
            /*
             * code is the first data byte, but
             * we still need to get the second
             */
            state = NEEDDATA2;
            break;
          case 0xc0:
          case 0xd0:
            /* explicitly add running state */
            buf[bufp++] = rs;
            buf[bufp++] = ev[1];
            state = START;
            do_buf(buf);
            bufp = 0;
            break;
          default:
            /*
             * I don't think running state applies to
             * these events
             */
            buf[bufp++] = ev[1];
            break;
          }
        } else {
          switch (ev[1] & 0xf0) {
          case 0x80:
          case 0x90:
          case 0xa0:
          case 0xb0:
          case 0xe0:
            rs = ev[1];
            buf[bufp++] = rs;
            state = NEEDDATA1;
            break;
          case 0xc0:
          case 0xd0:
            rs = ev[1];
            buf[bufp++] = rs;
            state = NEEDDATA2;
            break;
          default:
            break;
          }
        }
        break;
      case NEEDDATA1:
        buf[bufp++] = ev[1];
        state = NEEDDATA2;
        break;
      case NEEDDATA2:
        buf[bufp++] = ev[1];
        state = START;
        do_buf(buf);
        bufp = 0;
        break;
      case MTC:
      case SYSEX:
      case SYSTEM1:
      case SYSTEM2:
      default:
        break;         
      }
    }
  }
  exit(0);
}

void
exit_failure (char *s)
{
  perror(s);
  exit(-1);
}

