/****************************************************************************** 
 *
 * File:        midiprint.c
 *
 *              Standard MIDI File Pretty Printing
 *
 * Version:     0.1
 * Date:        1994-11-27
 *
 * Project:     MIDI/Sequencer library.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Someday I'll describe the superior features of this library ;-)
 *
 * The short of the long is that it can parse and load a Standard MIDI File
 * and put the resulting events onto some queues. They are defined in
 * midiqueue.h and should be straight forward to understand.
 *
 * The loading can be controlled in some wayes. It can for example be told not
 * to store anything but just run through the file. Thus, it can be used to
 * check a MIDI file for errors or to compute the total playing time. This is
 * exactly what `smfplay' does before it actually plays anything.
 *
 * Save and miscellaneous convertion routines will be added with time...
 *
 * Oh, I forgot - it's bound to be rewritten in C++; MIDI simply cries for
 * an object orientated implementation ;-).
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <stdio.h>

#include "miditypes.h"
#include "midiconst.h"
#include "midiqueue.h"
#include "midiprint.h"

/*** CONSTANTS ****************************************************************/

/* This array is indexed after the most significant 4 BITS of a CHANNEL MESSAGE
 *                                         0 1 2 3 4 5 6 7 8 9 A B C D E F */
/*
static const char MIDI_CHMSG_ARGS[0x10] = {0,0,0,0,0,0,0,0,2,2,2,2,1,1,2,0};
*/

const char *MIDI_META_SMPTE_TXT[5] = 
{
  "Hour",
  "Minutte",
  "Second",
  "Frame",
  "Frame Fraction"
};

const char MIDI_META_KEY_SIG_OFFSET = 7; /* to get C/Am */
const char *MIDI_META_KEY_SIG_MAJOR[15] =
{
  "Cb"
  "Gb",
  "Db",
  "Ab",
  "Eb",
  "Bb",
  "F",
  "C",
  "G",
  "D",
  "A",
  "E",
  "H",
  "F#",
  "C#"
};
const char *MIDI_META_KEY_SIG_MINOR[15] =
{
  "Gbm",
  "Bbm",
  "Fm",
  "Cm",
  "Gm",
  "Dm",
  "Am",
  "Em",
  "Hm",
  "F#m",
  "C#m",
  "G#m",
  "D#m",
  "A#m"
};

/*** MIDI FILE PRETTY PRINTING ************************************************/

void
midi_printdata(int size, char* data)
{
  int i = 0;
  printf("[ ");
  for (i = 0; i < size; i++) printf("%02hX ", data[i]&0xff);
  printf("]");
}

void
midi_printtext(int size, char* data)
{
  int i = 0;
  printf("\"");
  for (i = 0; i < size; i++) printf("%c", data[i]&0xff);
  printf("\"");
}

void
midi_printunknown(midi_event *event, enum MIDI_PRINT level)
{
  if (!level) return;
  printf("UNKNOWN ");
  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };

  if (level > MIDI_PRINT_META)

  printf(" FORMAT: ");
  midi_printdata(event->size, event->data);
  printf("\n");
}

void
midi_printmeta(midi_event *event, enum MIDI_PRINT level)
{
  static long i = 0;
  if (!level) return;
  if (level > MIDI_PRINT_SPARSE) printf("META  ");

  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };

  if (!event->data || !event->size)
    {
      printf("NULL DATA!\n");
      return;
    };

  switch (event->data[0]&0xff)
    {
    case MID_META_SEQ_NUMBER: 
      if (level < MIDI_PRINT_META) break;
      printf("SEQUENCE #       ");
      if (event->size == 3)
	{
	  i = ((event->data[1]&0xff) << 8) | (event->data[2]&0xff);
	  printf("%ld", i);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);

      break;
      
    case MID_META_TEXT:
      printf("TEXT:            ");
      midi_printtext(event->size - 1, &event->data[1]);
      level++;
      break;
    case MID_META_COPYRIGHT:
      printf("COPYRIGHT:       ");
      midi_printtext(event->size - 1, &event->data[1]);
      level++;
      break;
    case MID_META_SEQ_NAME:
      printf("SEQUENCE NAME:   ");
      midi_printtext(event->size - 1, &event->data[1]);
      level++;
      break;
    case MID_META_INST_NAME:
      if (level < MIDI_PRINT_META) break;
      printf("INSTRUMENT NAME: ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
    case MID_META_LYRIC:
      if (level < MIDI_PRINT_META) break;
      printf("LYRIC:           ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
    case MID_META_MARKER:
      if (level < MIDI_PRINT_META) break;
      printf("MARKER:          ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
    case MID_META_CUE_POINT:
      if (level < MIDI_PRINT_META) break;
      printf("CUE POINT:       ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
      
    case MID_META_CH_PREFIX:
      if (level < MIDI_PRINT_META) break;
      printf("CHANNEL PREFIX # ");
      if (event->size == 2)
	printf("%hd", event->data[1]&0xff);
      else
	midi_printdata(event->size - 1, &event->data[1]);

      break;
    case MID_META_EOT:
      if (level < MIDI_PRINT_META) break;
      printf("END OF TRACK");
      break;
    case MID_META_SET_TEMPO:
      if (level < MIDI_PRINT_META) break;
      printf("TEMPO            ");
      if (event->size == 4)
	{
	  i  = (event->data[1]&0xff) << 16;
	  i |= (event->data[2]&0xff) << 8;
	  i |= (event->data[3]&0xff);

	  i = ( 60.0 * 1000000.0 / (float)i );
	  printf("%03ld BPM", i);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);
      break;
    case MID_META_SMPTE_OFFSET:
      if (level < MIDI_PRINT_META) break;
      printf("SMPTE OFFSET     ");
      if (event->size == 6)
	{
	  for (i = 1; i < event->size; i++) 
	    printf("%s: %02hu ", MIDI_META_SMPTE_TXT[i-1], event->data[i]&0xff);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);

      break;
    case MID_META_TIME_SIG:
      if (level < MIDI_PRINT_META) break;
      printf("TIME SIGNATURE   ");
      if (event->size == 5)
	printf("%d/%d", event->data[1]&0xff, 1<<(event->data[2]&0xff));
      else
	midi_printdata(event->size - 1, &event->data[1]);
      break;
    case MID_META_KEY_SIG:
      if (level < MIDI_PRINT_META) break;
      printf("KEY SIGNATURE    ");
      if (event->size == 3)
	{
	  if (event->data[2]&0xff)
	    {
	      i = (signed char)(event->data[1]&0xff) + MIDI_META_KEY_SIG_OFFSET;
	      printf("%s", MIDI_META_KEY_SIG_MINOR[i]);
	    }
	  else
	    {
	      i = (signed char)(event->data[1]&0xff) + MIDI_META_KEY_SIG_OFFSET;
	      printf("%s", MIDI_META_KEY_SIG_MAJOR[i]);
	    };

	  if ((signed char)event->data[1]&0xff < 0)
	    printf(" (%hd b)", -event->data[1]);
	  else if ((signed char)event->data[1]&0xff > 0)
	    printf(" (%hd #)", event->data[1]);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);
      break;
    case MID_META_SEQ_SPECIFIC:
      if (level < MIDI_PRINT_META) break;
      printf("SEQ. SPECIFIC    ");
      midi_printdata(event->size - 1, &event->data[1]);
      break;

    default:
      if (level < MIDI_PRINT_META) break;
      printf("UNKNOWN          ");
      midi_printdata(event->size, &event->data[0]);
      break;
    };
  if (level > MIDI_PRINT_SPARSE) printf("\n");
}

void
midi_printsysex(midi_event *event, enum MIDI_PRINT level)
{
  if (!level) return;
  if (level > MIDI_PRINT_SPARSE) printf("SYSEX ");
  
  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };
  
  if (!event->data || !event->size)
    {
      printf("NULL DATA!\n");
      return;
    };
  
  if (level > MIDI_PRINT_META)
    {
      switch (event->data[0]&0xff)
	{
	case MID_SYS_SYSEX:
	  midi_printdata(event->size, &event->data[0]);
	  break;
	case MID_EVENT_OTHER:
	  if (level < MIDI_PRINT_META) break;
	  midi_printdata(event->size, &event->data[0]);
	  break;
	default:
	  if (level < MIDI_PRINT_META) break;
	  midi_printdata(event->size - 1, &event->data[1]);
	  break;
	};
    };

  if (level > MIDI_PRINT_SPARSE) printf("\n");
}

void
midi_printvoice(midi_event *event, enum MIDI_PRINT level)
{
  static int bank_sel_run = 0;
  static int bank_sel_mm = 0;
  static int bank_sel_ll = 0;
  static int data_mm = 0;
  static int data_ll = 0;
  static int nrpn_run = 0;
  static int nrpn_mm = 0;
  static int nrpn_ll = 0;
  static int rpn_run = 0;
  static int rpn_mm = 0;
  static int rpn_ll = 0;
  static int cmd = 0;
  int j = 0;
  int i = 0;

  if (!level) return;
  if (level > MIDI_PRINT_META) printf("VOICE ");

  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };

  if (!event->data || !event->size)
    {
      printf("NULL DATA!\n");
      return;
    };

  if (level > MIDI_PRINT_META)
    {
      
      if ((event->data[0]&0xff) < MID_EVENT_CHMSG_FIRST)
	{
	  printf("RUNNING ");
	  i = 0;
	}
      else
	{
	  cmd = event->data[0]&0xff;
	  i = 1;
	};

      printf("CHANNEL %02d ", (cmd & 0x0f) + 1);
      
      switch (cmd & 0xf0)
	{
	case MID_EVENT_NOTE_OFF:
	  printf("NOTE #%03hd    OFF. VELOCITY   %03hd", event->data[i++]&0xff,
		 event->data[i++]&0xff);
	  break;
	case MID_EVENT_NOTE_ON:
	  printf("NOTE #%03hd    ON.  VELOCITY   %03hd", event->data[i++]&0xff,
		 event->data[i++]&0xff);
	  break;
	case MID_EVENT_POLY_PRESSURE:
	  printf("NOTE #%03hd    POLY AFTERTOUCH %hd", event->data[i++]&0xff,
		 event->data[i++]&0xff);
	  break;
	case MID_EVENT_CONTROL:
	  switch (event->data[i++]&0xff)
	    {
	    case 0x00:
	      bank_sel_mm = event->data[i]&0xff;
	      if (bank_sel_run)
		{
		  printf("BANK SELECT  %d", (bank_sel_mm << 7) | bank_sel_ll);
		  bank_sel_run = 0;
		}
	      else
		{
		  printf("(bank select running MSB)");
		  bank_sel_run = 1;
		};
	      break;
	    case 0x20:
	      bank_sel_ll = event->data[i]&0xff;
	      if (bank_sel_run)
		{
		  printf("BANK SELECT  %d", (bank_sel_mm << 7) | bank_sel_ll);
		  bank_sel_run = 0;
		}
	      else
		{
		  printf("(bank select running LSB)");
		  bank_sel_run = 1;
		};
	      break;
	    case 0x01:
	      printf("MODULATION   %hd", event->data[i]&0xff);
	      break;
	    case 0x05:
	      printf("PORTA TIME   %hd", event->data[i]&0xff);
	      break;
	      
	    case 0x06:
	      data_mm = event->data[i]&0xff;
	      if (nrpn_run)
		{
		  nrpn_run = 0;
		  switch (nrpn_mm << 8 | nrpn_ll)
		    {
		    case 0x0108:
		      printf("VIBRA RANGE  %hd", data_mm - 0x40);
		      break;
		    case 0x0109:
		      printf("VIBRA DEPTH  %hd", data_mm - 0x40);
		      break;
		    case 0x010A:
		      printf("VIBRA DELAY  %hd", data_mm - 0x40);
		      break;
		    case 0x0120:
		      printf("TVF CUTOFF   %hd", data_mm - 0x40);
		      break;
		    case 0x0121:
		      printf("TVF RESONAN  %hd", data_mm - 0x40);
		      break;
		    case 0x0163:
		      printf("TVF ATTACK   %hd", data_mm - 0x40);
		      break;
		    case 0x0164:
		      printf("TVF DECAY    %hd", data_mm - 0x40);
		      break;
		    case 0x0166:
		      printf("TVF RELEASE  %hd", data_mm - 0x40);
		      break;
		    default:
		      switch (nrpn_mm)
			{
			case 0x18:
			  printf("PITCH COARSE %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			case 0x1A:
			  printf("TVA LEVEL    %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			case 0x1C:
			  printf("PANPOT       %hd (drum instrument key %hd)", 
				 data_mm - 0x40, nrpn_ll);
			  break;
			case 0x1D:
			  printf("REVERB LEVEL %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			case 0x1E:
			  printf("CHORUS LEVEL %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			default:
			  printf("unknown NRPN message. Probably not Roland GS");
			};
		    };
		}
	      else
		printf("(data running MSB)");

	      break;
	    case 0x26:
	      data_ll = event->data[i]&0xff;
	      if (rpn_run)
		{
		  rpn_run = 0;
		  switch ( (rpn_mm << 8) | rpn_ll )
		    {
		    case 0x0000:
		      printf("PITCH BEND SENSE     %hd", data_mm);
		      break;
		    case 0x0001:
		      j = ((data_mm&0xff) << 7) | (data_ll&0xff);
		      printf("MASTER FINE TUNING   %d", j - 0x2000);
		      break;
		    case 0x0002:
		      printf("MASTER COARSE TUNING %d semitones", data_mm - 0x40);
		      break;
		    case 0x7F7F:
		      printf("RPN RESET");
		      break;
		    default:
		      printf("unknown RPN message");
		    };
		}
	      else
		printf("(data running LSB)");

	      break;
	    case 0x07:
	      printf("VOLUME       %hd", event->data[i]&0xff);
	      break;
	    case 0x0A:
	      printf("PANPOT       %hd", (event->data[i]&0xff) - 0x40);
	      break;
	    case 0x0B:
	      printf("EXPRESSION   %hd", event->data[i]&0xff);
	      break;
	    case 0x40:
	      printf("HOLD1        %s", (event->data[i]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case 0x41:
	      printf("PORTAMENTO   %s", (event->data[i]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case 0x42:
	      printf("SOSTENUTO    %s", (event->data[i]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case 0x43:
	      printf("SOFT         %s", (event->data[i]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case 0x54:
	      printf("PORTA CONTRL %hd", event->data[i]&0xff);
	      break;
	    case 0x5B:
	      printf("REVERB LEVEL %hd", event->data[i]&0xff);
	      break;
	    case 0x5D:
	      printf("CHORUS LEVEL %hd", event->data[i]&0xff);
	      break;
	    case 0x63:
	      nrpn_mm = event->data[i]&0xff;
	      printf("(nrpn running MSB)");
	      nrpn_run = 1;
	      break;
	    case 0x62:
	      nrpn_ll = event->data[i]&0xff;
	      printf("(nrpn running LSB)");
	      nrpn_run = 1;
	      break;
	    case 0x78:
	      printf("ALL SOUNDS OFF");
	      break;
	    case 0x79:
	      printf("RESET ALL CONTROLLERS");
	      break;
	    case 0x7B:
	      printf("ALL NOTES OFF");
	      break;
	    case 0x7C:
	      printf("OMNI OFF");
	      break;
	    case 0x7D:
	      printf("OMNI ON");
	      break;
	    case 0x7F:
	      printf("POLY");
	      break;
	    default:
	      printf("CTRL CHANGE  0x%02x 0x%02hx", cmd&0xff, event->data[i]&0xff);
	    };
	  break;
	case MID_EVENT_PROG_CHG:
	  printf("PROG CHANGE  %hd", event->data[i]&0xff);
	  break;
	case MID_EVENT_CH_PRESSURE:
	  printf("AFTERTOUCH   %hd", event->data[i]&0xff);
	  break;
	case MID_EVENT_PITCH_BEND:
	  j = ( (event->data[i++]&0xff) << 7)|(event->data[i]&0xff);
	  printf("PITCH BEND   %d", j - 0x2000);
	  break;
	};
      printf("\n");
    };
}

void
midi_printevent(midi_event *event, enum MIDI_PRINT level)
{
  if (!event || !level) return;
  if (level > MIDI_PRINT_META) printf("TIME %08lx, SIZE %08x, ", event->time, event->size);

  switch (event->type&0xff)
    {
    case MIDI_TYP_VOICE:
      midi_printvoice(event, level);
      break;
    case MIDI_TYP_SYSEX:
      midi_printsysex(event, level);
      break;
    case MIDI_TYP_META:
      midi_printmeta(event, level);
      break;
    default:
      midi_printunknown(event, level);
      break;
    };
}

void
midi_printrawevent(byte type, unsigned long time, word size, char *data, enum MIDI_PRINT level)
{
  midi_event event;
  event.prev = event.next = NULL;
  event.type = type;
  event.time = time;
  event.size = size;
  event.data = data;
  midi_printevent(&event, level);
}

/*** END OF FILE **************************************************************/
