/****************************************************************************** 
 *
 * File:        midiqueue.c
 *
 * Version:     0.1
 * Date:        1994-11-27
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MIDI-401 Device driver for Linux 1.1.64 and higher.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * DESCRIPTION OF THE DATA FORMAT USED INTERNALLY BY THE MPU-401 DEVICE DRIVER:
 *
 *    Everything in this driver is very closely related to the way the MPU-401
 *    operates. I have tried to minimize the effects of the rather obfuscated 
 *    Standard MIDI File format whenever possible.
 *
 *    Here's how the driver (and MPU-401) sees the world:
 *
 *    The driver has three internal, dynamic queues. They are `out', `ctl'
 *    and `rec'.
 *
 *    rec: MIDI IN queue. When recording either in step mode or real time, 
 *         incomming events are placed here. The can be retrieved with
 *         ioctl command MPUIOC_GET_EVENT - see mpuioctl.h for details.
 *
 *         VOICE EVENTS and SYSEX events are stored.
 *
 *    out: MIDI VOICE OUT queue. VOICE EVENTS to be played are buffered here.
 *
 *    ctl: CONTROL queue. CONTROL EVENTS are tempo change commands, time 
 *         signature and metronome handling. SYSEX data to be transmitted.
 *
 *    + VOICE EVENTS:
 *
 *      Only ``real'' voice events are transmitted trough the voice queue.
 *      (first byte of non-running event must be in between 0x80 to 0xef).
 *      Refer to the Standard MIDI File spec. for a detailed discussion about
 *      voice events.
 *
 *    + SYSEX EVENTS:
 *
 *      Sysex events are specialized, synthesize/device specific commands.
 *
 *      Refer to the Standard MIDI File spec. for a detailed discussion about
 *      META events.
 *
 *    + META EVENTS:
 *  
 *      Tempo information etc. is stored as META events in a MIDI file.
 *
 *      Refer to the Standard MIDI File spec. for a detailed discussion about
 *      META events.
 *  
 *    The following structure is used to ``transport'' data:
 *
 *      struct midi_event
 *        {
 *          struct midi_event *prev;
 *          struct midi_event *next;
 *          enum MIDI_TYPE type;
 *          long time;
 *          word size;
 *          char *data;
 *        };
 *
 *    `type': Identifies whether it is a VOICE, SYSEX or META event
 *            It can be either MIDI_TYP_VOICE, MIDI_TYP_SYSEX or MIDI_TYP_META.
 *    `time': Represents total number of MIDI clocks from the beginning.
 *    `size': The size of the character array pointed to by the data field.
 *    `data': Character array of data.
 *
 *
 *    VOICE EVENT: Data are simply stored as if read from disk.
 *
 *    SYSEX EVENT:
 *
 *       1st byte: 0xf0 or 0xf7
 *       2nd byte: 2nd SysEx data byte
 *        :
 *      last byte: 0xf7
 *
 *    META EVENTS:
 *
 *       1st byte: META Type, fx. 0x03 (== sequence name)
 *       2nd byte: 1st META data byte
 *        :
 *      last byte: The size-1'th META data byte.
 *
 *    When operating with events to be passed to the driver, *ALWAYS* use the
 *    queue manipulation routines in my midi library (I *believe* they are correct!).
 *
 * NOTE: In order to use the following functions you *must* provide
 *       two functions that are assumed to be linked in. They are:
 *
 *         extern midi_event *midi_event_alloc();
 *         extern void midi_event_free(midi_event *event);
 *
 *       If you are using them from the library libmidifile.a however, they
 *       are already linked in (default behavior is to call exit(-1) in case
 *       of no memory available).
 *
 *       midi_event_alloc() / * is expected to do: * /
 *         {
 *             midi_event *tmp = (midi_event *)malloc(sizeof(midi_event));
 *
 *	       / * If you want special error handling you can place it here! * /
 *
 *             return tmp;
 *         }
 *
 *       midi_event_free(midi_event *event) / * is expected to do: * /
 *         {
 *            if (!event) return;
 *            if (event->data) free(event->data);
 *            free(event);
 *         }
 *
 *       There reason for doing that is than most of the queue code is shared
 *       with my MPU-401 device driver which cannot use normal malloc/free 
 *       since its running within the kernel.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <stddef.h>
#include <errno.h>

#include "miditypes.h"
#include "midiqueue.h"

/*** MIDI-401 DEVICE MIDI EVENT QUEUE ******************************************/

int
midi_queue_put_least(midi_queue *queue, midi_event *event)
{
  static midi_event *tempe = NULL;
  
  if (!queue || !event) return -EINVAL;
  
  if (!queue->count++) /* FIRST EVER */
    {
      queue->head = queue->tail = event;
      event->prev = event->next = NULL;
    }
  else if (event->time < queue->head->time) /* SMALLEST */
    {
      queue->head->prev = event;
      event->prev = NULL;
      event->next = queue->head;
      queue->head = event;
    }
  else
    {
      /* Here goes some assumptions:
       *
       * + Most often the operations on the queue are monotoneous, i.e. either a series of
       *   insertions or a series of removals.
       *
       * + Insertions most often happens to be sorted with respect to time. There is actually
       *   only problems when events are merged from for example two MIDI tracks or to
       *   recordings.
       *
       * + Each time one starts to put a new series on the queue, the current field must be
       *   cleared.
       *
       * + It is then assumed that there is a very high probability that the next event should
       *   be inserted right after the previous - or at least some few events further down.
       *   So instead of scanning the whole queue everytime an insertions is made, we first
       *   check if we can use the previous inserted event as an offset.
       */

      tempe = queue->current;
      while (tempe && event->time < tempe->time) tempe = tempe->prev;

      if (!tempe) tempe = queue->head;

      /* Event gets lowest position in queue if there is equal time elements */
      while (tempe && event->time > tempe->time) tempe = tempe->next;

      if (tempe)
	{
	  event->next = tempe;
	  event->prev = tempe->prev;
	  tempe->prev = event;
	  if (event->prev)
	    {
	      event->prev->next = event;
	    }
	  else
	    {
	      queue->head = event; /* This *should* have been caught above */
	    };
	}
      else
	{
	  /* This *should* have been caught above ... play it safe though */
	  queue->tail->next = event;
	  event->next = NULL;
	  event->prev = queue->tail;
	  queue->tail = event;
	};
    }

  queue->current = event;
  return queue->count;
}

int
midi_queue_put_least_event(midi_queue *q, byte tp, unsigned long tm, word sz, char *dt)
{
  static midi_event *event = NULL;

  event = midi_event_alloc();
  if (!event) return -ENOMEM;

  event->type = tp;
  event->time = tm;
  event->size = sz;
  event->data = dt;

  return midi_queue_put_least(q, event);
}

int
midi_queue_put(midi_queue *queue, midi_event *event)
{
  static midi_event *tempe = NULL;
  
  if (!queue || !event) return -EINVAL;
  
  if (!queue->count++) /* FIRST EVER */
    {
      queue->head = queue->tail = event;
      event->prev = event->next = NULL;
    }
  else if (event->time >= queue->tail->time) /* BIGGEST */
    {
      queue->tail->next = event;
      event->next = NULL;
      event->prev = queue->tail;
      queue->tail = event;
    }
  else if (event->time < queue->head->time) /* SMALLEST */
    {
      queue->head->prev = event;
      event->prev = NULL;
      event->next = queue->head;
      queue->head = event;
    }
  else /* IN BETWEEN */
    {
      /* Here goes some assumptions:
       *
       * + Most often the operations on the queue are monotoneous, i.e. either a series of
       *   insertions or a series of removals.
       *
       * + Insertions most often happens to be sorted with respect to time. There is actually
       *   only problems when events are merged from for example two MIDI tracks or to
       *   recordings.
       *
       * + Each time one starts to put a new series on the queue, the current field must be
       *   cleared.
       *
       * + It is then assumed that there is a very high probability that the next event should
       *   be inserted right after the previous - or at least some few events further down.
       *   So instead of scanning the whole queue everytime an insertions is made, we first
       *   check if we can use the previous inserted event as an offset.
       */

      tempe = queue->current;
      while (tempe && event->time < tempe->time) tempe = tempe->prev;

      if (!tempe) tempe = queue->head;

      /* Event gets highest position in queue if there is equal time elements */
      while (tempe && event->time >= tempe->time) tempe = tempe->next;

      if (tempe)
	{
	  event->next = tempe;
	  event->prev = tempe->prev;
	  tempe->prev = event;
	  if (event->prev)
	    event->prev->next = event;
	  else
	    {
	      queue->head = event; /* This *should* have been caught above */
	    };
	}
      else
	{
	  /* This *should* have been caught above ... play it safe though */
	  queue->tail->next = event;
	  event->next = NULL;
	  event->prev = queue->tail;
	  queue->tail = event;
	};
    };

  queue->current = event;
  return queue->count;
}


int
midi_queue_put_event(midi_queue *q, byte tp, unsigned long tm, word sz, char *dt)
{
  static midi_event *event = NULL;

  event = midi_event_alloc();
  if (!event) return -ENOMEM;

  event->type = tp;
  event->time = tm;
  event->size = sz;
  event->data = dt;

  return midi_queue_put(q, event);
}

int midi_queue_at_put(midi_queue *queue, midi_event *atevent, midi_event *event)
{
  if (!queue || !event)
    return -EINVAL;
  else if (!queue->count || !atevent || atevent->time != event->time || atevent == queue->tail)
    return midi_queue_put(queue, event);
  else if (atevent == queue->head)
    return midi_queue_put_least(queue, event);


  event->prev = atevent->prev;
  event->next = atevent;
  atevent->prev = event;
  if (event->prev) event->prev->next = event;
  return ++queue->count;
}

int
midi_queue_at_put_event(midi_queue *q, midi_event *atevent,
			byte tp, unsigned long tm, word sz, char *dt)
{
  static midi_event *event = NULL;

  event = midi_event_alloc();
  if (!event) return -ENOMEM;

  event->type = tp;
  event->time = tm;
  event->size = sz;
  event->data = dt;

  return midi_queue_at_put(q, atevent, event);
}

midi_event *
midi_queue_get(midi_queue *queue)
{
  static midi_event *tmp;

  if (!queue) return NULL;
  else if (queue->count < 1) return NULL;
  else if (queue->count-- == 1)
    {
      tmp = queue->head;
      queue->head = queue->tail = NULL;
      queue->count = 0;
    }
  else
    {
      tmp = queue->head;
      queue->head = queue->head->next;
      queue->head->prev = NULL;
    };

  tmp->prev = tmp->next = NULL;

  queue->current = NULL;

  return tmp;
}

midi_event *
midi_queue_get_highest(midi_queue *queue)
{
  static midi_event *tmp;

  if (!queue) return NULL;
  else if (queue->count < 1) return NULL;
  else if (queue->count-- == 1)
    {
      tmp = queue->tail;
      queue->head = queue->tail = NULL;
      queue->count = 0;
    }
  else
    {
      tmp = queue->tail;
      queue->tail = queue->tail->prev;
      queue->tail->next = NULL;
    };

  tmp->prev = tmp->next = NULL;

  queue->current = NULL;

  return tmp;
}

void
midi_queue_remove(midi_queue *queue)
{
  midi_event_free(midi_queue_get(queue));
}
void
midi_queue_remove_highest(midi_queue *queue)
{
  midi_event_free(midi_queue_get_highest(queue));
}
void
midi_queue_remove_event(midi_queue *queue, midi_event *event)
{
  if (!queue) return;
  
  if (event == queue->head)
    {
      midi_queue_remove(queue);
    }
  else if (event == queue->tail)
    {
      midi_queue_remove_highest(queue);
    }
  else /* We trust it really is a member of this queue - its too expensive to check... */
    {
      if (event->next) event->next->prev = event->prev;
      if (event->prev) event->prev->next = event->next;
      event->prev = event->next = NULL;
      midi_event_free(event);
    };
}

midi_event *
midi_queue_peek(midi_queue *queue)
{
  if (!queue) return NULL;
  queue->current = NULL;
  return queue->head;
}
midi_event *
midi_queue_peek_highest(midi_queue *queue)
{
  if (!queue) return NULL;
  queue->current = NULL;
  return queue->tail;
}

void
midi_queue_swap(midi_queue *queue, midi_event *a, midi_event *b)
{
  static midi_event t;

  if (!queue || !a || !b) return;

  if (a == queue->head) queue->head = b;
  if (a == queue->tail) queue->tail = b;

  if (b == queue->head) queue->head = a;
  if (b == queue->tail) queue->tail = a;

  if (a->next) a->next->prev = b;
  if (a->prev) a->prev->next = b;

  if (b->next) b->next->prev = a;
  if (b->prev) b->prev->next = a;

  t.next = a->next;
  t.prev = a->prev;

  a->prev = b->prev;
  a->next = b->next;

  b->prev = t.prev;
  b->next = t.next;
}

void
midi_queue_reset(midi_queue *queue)
{
  if (!queue) return;
  queue->head = queue->tail = queue->current = NULL;
  queue->count = 0;
}

void
midi_queue_flush(midi_queue *queue)
{
  midi_event *tmp = NULL;

  if (!queue) return;

  queue->current = queue->head;
  while (queue->current)
    {
      tmp = queue->current;
      queue->current = queue->current->next;
      tmp->next = tmp->prev = NULL;
      midi_event_free(tmp);
    };

  midi_queue_reset(queue);
}

/*** End of File **************************************************************/
