/****************************************************************************** 
 *
 * File:        midiqueue.c
 *
 * Version:     0.1
 * Date:        1994-11-27
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MIDI-401 Device driver for Linux 1.1.64 and higher.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * DESCRIPTION OF THE DATA FORMAT USED INTERNALLY BY THE MPU-401 DEVICE DRIVER:
 *
 *    Everything in this driver is very closely related to the way the MPU-401
 *    operates. I have tried to minimize the effects of the rather obfuscated 
 *    Standard MIDI File format whenever possible.
 *
 *    Here's how the driver (and MPU-401) sees the world:
 *
 *    The driver has three internal, dynamic queues. They are `out', `ctl'
 *    and `rec'.
 *
 *    rec: MIDI IN queue. When recording either in step mode or real time, 
 *         incomming events are placed here. The can be retrieved with
 *         ioctl command MPUIOC_GET_EVENT - see mpuioctl.h for details.
 *
 *         VOICE EVENTS and SYSEX events are stored.
 *
 *    out: MIDI VOICE OUT queue. VOICE EVENTS to be played are buffered here.
 *
 *    ctl: CONTROL queue. CONTROL EVENTS are tempo change commands, time 
 *         signature and metronome handling. SYSEX data to be transmitted.
 *
 *    + VOICE EVENTS:
 *
 *      Only ``real'' voice events are transmitted trough the voice queue.
 *      (first byte of non-running event must be in between 0x80 to 0xef).
 *      Refer to the Standard MIDI File spec. for a detailed discussion about
 *      voice events.
 *
 *    + SYSEX EVENTS:
 *
 *      Sysex events are specialized, synthesize/device specific commands.
 *
 *      Refer to the Standard MIDI File spec. for a detailed discussion about
 *      META events.
 *
 *    + META EVENTS:
 *  
 *      Tempo information etc. is stored as META events in a MIDI file.
 *
 *      Refer to the Standard MIDI File spec. for a detailed discussion about
 *      META events.
 *  
 *    The following structure is used to ``transport'' data:
 *
 *      struct midi_event
 *        {
 *          struct midi_event *prev;
 *          struct midi_event *next;
 *          enum MIDI_TYPE type;
 *          long time;
 *          word size;
 *          char *data;
 *        };
 *
 *    `type': Identifies whether it is a VOICE, SYSEX or META event
 *            It can be either MIDI_TYP_VOICE, MIDI_TYP_SYSEX or MIDI_TYP_META.
 *    `time': Represents total number of MIDI clocks from the beginning.
 *    `size': The size of the character array pointed to by the data field.
 *    `data': Character array of data.
 *
 *
 *    VOICE EVENT: Data are simply stored as if read from disk.
 *
 *    SYSEX EVENT:
 *
 *       1st byte: 0xf0 or 0xf7
 *       2nd byte: 2nd SysEx data byte
 *        :
 *      last byte: 0xf7
 *
 *    META EVENTS:
 *
 *       1st byte: META Type, fx. 0x03 (== sequence name)
 *       2nd byte: 1st META data byte
 *        :
 *      last byte: The size-1'th META data byte.
 *
 *    When operating with events to be passed to the driver, *ALWAYS* use the
 *    queue manipulation routines in my midi library (I *believe* they are correct!).
 *
 * NOTE: In order to use the following functions you *must* provide
 *       two functions that are assumed to be linked in. They are:
 *
 *         extern midi_event *midi_event_alloc();
 *         extern void midi_event_free(midi_event *event);
 *
 *       If you are using them from the library libmidifile.a however, they
 *       are already linked in (default behavior is to call exit(-1) in case
 *       of no memory available).
 *
 *       midi_event_alloc() / * is expected to do: * /
 *         {
 *             midi_event *tmp = (midi_event *)malloc(sizeof(midi_event));
 *
 *	       / * If you want special error handling you can place it here! * /
 *
 *             return tmp;
 *         }
 *
 *       midi_event_free(midi_event *event) / * is expected to do: * /
 *         {
 *            if (!event) return;
 *            if (event->data) free(event->data);
 *            free(event);
 *         }
 *
 *       There reason for doing that is than most of the queue code is shared
 *       with my MPU-401 device driver which cannot use normal malloc/free 
 *       since its running within the kernel.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#ifndef __MIDI_QUEUE_H
#define __MIDI_QUEUE_H

#include "miditypes.h"

/*** TYPEDEFS *****************************************************************/

enum MIDI_TYPE
{
  MIDI_TYP_VOICE,
  MIDI_TYP_SYSEX,
  MIDI_TYP_META
};

struct midi_event
{
  struct midi_event *prev;
  struct midi_event *next;
  enum MIDI_TYPE type;
  long time; /* not delta time! */
  word size;
  char *data;
};
typedef struct midi_event midi_event;

struct midi_queue
{
  struct midi_event *head;
  struct midi_event *tail;
  long count;
  struct midi_event *current;
};
typedef struct midi_queue midi_queue;

/*** PROTOTYPES ***************************************************************/

/* USER DEFINED ROUTINES - MUST BE LINKED IN! */

extern midi_event *midi_event_alloc();
extern void midi_event_free(midi_event *event);

/* END OF USER DEFINED ROUTINES */

/* Event gets lowest position (rank) in queue if there is equal timed elements */
extern int midi_queue_put_least(midi_queue *queue, midi_event *event);
extern int midi_queue_put_least_event(midi_queue *q, byte tp, unsigned long tm, word sz, char *dt);

/* Event gets highest position (rank) in queue if there is equal timed elements */
extern int midi_queue_put(midi_queue *queue, midi_event *event);
extern int midi_queue_put_event(midi_queue *q, byte tp, unsigned long tm, word sz, char *dt);

extern int midi_queue_at_put(midi_queue *queue, midi_event *atevent, midi_event *event);
extern int midi_queue_at_put_event(midi_queue *q, midi_event *atevent, byte tp, unsigned long tm, word sz, char *dt);

/* Get (detach) event with lowest time value */
extern midi_event *midi_queue_get(midi_queue *queue);

extern midi_event *midi_queue_get_highest(midi_queue *queue);

/* Remove event with lowest time value */
extern void midi_queue_remove(midi_queue *queue);

extern void midi_queue_remove_highest(midi_queue *queue);
extern void midi_queue_remove_event(midi_queue *queue, midi_event *event);

/* Peek at event with lowest time value */
extern midi_event *midi_queue_peek(midi_queue *queue);

extern midi_event *midi_queue_peek_highest(midi_queue *queue);

extern void midi_queue_swap(midi_queue *queue, midi_event *a, midi_event *b);

extern void midi_queue_reset(midi_queue *queue);
extern void midi_queue_flush(midi_queue *queue);

#endif /* __MIDI_QUEUE_H */

/*** End of File **************************************************************/
