/****************************************************************************** 
 *
 * File:        mpu401.c
 *
 *              Roland MPU-401 MIDI Interface Device Driver.
 *
 * Version:     0.1 alpha
 * Date:        1994-11-14
 *
 * Project:     Roland MPU-401 Device driver for Linux 1.1.64 and higher.
 *              This driver is a modular part of the kernel.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <fcntl.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/major.h>
#include <linux/ioctl.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/delay.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/string.h>

#include <linux/module.h>

#include "configure.h"
#include "midiconst.h"
#include "miditypes.h"
#include "midiqueue.h"
#include "mpuioctl.h"
#include "mpu401.h"

/*** CONSTANTS ****************************************************************/

/* This array is indexed after the most significant 4 BITS of a CHANNEL MESSAGE
 *                                         0 1 2 3 4 5 6 7 8 9 A B C D E F */
static const char MIDI_CHMSG_ARGS[0x10] = {0,0,0,0,0,0,0,0,2,2,2,2,1,1,2,0};

/* MPU-401 Status bits */

#define MPU_DRR 0x40
#define MPU_DSR 0x80

#define MPU_IOP_RETRY 100000

/*  SysEx buffer - in mpu_irqhandler() - very rarely used if ever.
 * Unfortunately there is no way of telling how many bytes the MPU-401 will send, but I have 
 * never seen messages exceeding 20-30 bytes, so if you get a ``out of sysex buffer'' in your 
 * kernel log you can simply adjust this buffer size. -- 128 bytes is probably an overhead */

#define MPU_SYSEX_BUFFER 128

#define MAX_QUEUE_EVENTS 1000

/*** VARIABLES ****************************************************************/

static const char driver_version_str[] =
"MODULAR MPU-401 DEVICE DRIVER v%d.%d%c <burgaard@daimi.aau.dk>\n";

static word mpu_IRQ = 0;
static word mpu_IRQ_mutex = 0;
static word mpu_IRQ_fake = 0;

static long mpu_total_KMALLOC = 0;
static long mpu_total_KFREE = 0;

#ifdef __MPU_STATISTICS

static long mpu_IRQ_count = 0;
static long mpu_IRQ_fake_count = 0;

static long mpu_DRR_timeout_count = 0;
static long mpu_DSR_timeout_count = 0;

static long mpu_DRR_loop_count = 0;
static long mpu_DSR_loop_count = 0;

static long mpu_running_hit = 0;
static long mpu_voice_count = 0;
static long mpu_sysex_count = 0;
static long mpu_meta_count = 0;

static long mpu_nop_count = 0;

#endif /* __MPU_STATISTICS */

/*** MPU-401 DEVICE STATUS INFORMATION ****************************************/

static struct mpu_status mpu_status =
{
  MPU_STA_NONE,
  MPU_DRV_VERSION, MPU_DRV_SUBVERSION, MPU_DRV_RELEASE,
  0, 0, '\0',
  0, 0
};

static struct mpu_queues
{
  midi_queue out;
  long out_time;

  midi_queue ctl;
  long ctl_time;

  midi_queue rec;
  long rec_time;
}
mpu_queues =
{
  { NULL, NULL, 0 }, 0,
  { NULL, NULL, 0 }, 0,
  { NULL, NULL, 0 }, 0
};

static struct mpu_sleeps
{
  struct wait_queue *record;
  struct wait_queue *play;
  struct wait_queue *demand;
  struct mpu_demand demand_load;
}
mpu_sleeps =
{
  NULL, NULL,
  NULL,
  { 0, 0 }
};

static struct mpu_metronome
{
  enum MPU_METRONOME mode;
  long mysectempo;
  byte tempo;
  byte division;
  byte MIDIperMETRO;
  byte METROperMEAS;
  byte INT4perCLOCKtoHOST;
}
mpu_metronome = 
{
  MPU_METRO_NONE,
  500000,
  100,
  120,
  12,
  8,
  60
};

static word mpu_IOP = 0;
static word mpu_DTP = 0;
static word mpu_CMP = 0;
static word mpu_STP = 0;

static int mpu_old_voice_cmd = 0;

static midi_event *control_old_event = NULL;
static char control_run = 0;

/* This buffer is used for recieving sysex messages. */
static char mpu_sysexbuf[MPU_SYSEX_BUFFER]; 

static word mpu_DEBUG = 0;

/*** ERROR HANDING ************************************************************/

static void
mpu_debug_dump()
{
  printk("\n");
  printk("********* MPU-401 DEVICE DRIVER INFORMATION *********\n");
  printk("\n");
  printk("   Run-time Debug level %d\n", mpu_DEBUG);
  printk("   state                0x%08x\n", mpu_status.state);
  printk("\n");
  printk("   Accumulated kmalloc  %ld\n", mpu_total_KMALLOC);
  printk("   Accumulated kfree_s  %ld\n", mpu_total_KFREE);
  printk("   Current balance      %ld\n", mpu_total_KMALLOC - mpu_total_KFREE);
  printk("\n");
  printk("   record sleep         %s\n", (mpu_sleeps.record) ? "Yes" : "No");
  printk("   sleep2play_end       %s\n", (mpu_sleeps.play) ? "Yes" : "No");
  printk("   sleep2demand         %s, [%ld, %ld]\n", (mpu_sleeps.demand) ? "Yes" : "No",
	 mpu_sleeps.demand_load.voice, mpu_sleeps.demand_load.control);
  printk("\n");
  printk("   out.count, out time  %ld, %ld\n", mpu_queues.out.count, mpu_queues.out_time);
  printk("   ctl.count, ctl time  %ld, %ld\n", mpu_queues.ctl.count, mpu_queues.ctl_time);
  printk("   rec.count, rec time  %ld, %ld\n", mpu_queues.rec.count, mpu_queues.rec_time);
  printk("\n");
  printk("   metronome mode       0x%08x\n", mpu_metronome.mode);
  printk("   myseconds / beat     %ld\n", mpu_metronome.mysectempo);
  printk("   tempo, division      %hd, %hd\n", mpu_metronome.tempo, mpu_metronome.division);
  printk("   MIDI clocks / METRO  %hd\n", mpu_metronome.MIDIperMETRO);
  printk("   METRO / MEASURE      %hd\n", mpu_metronome.METROperMEAS);
  printk("   INT CLK*4 / CLK HOST %hd\n", mpu_metronome.INT4perCLOCKtoHOST);

#ifdef __MPU_STATISTICS

  printk("\n");
  printk("   Real, fake IRQ count %ld, %ld\n", mpu_IRQ_count, mpu_IRQ_fake_count);
  printk("   DRR, DSR timeouts    %ld, %ld\n", mpu_DRR_timeout_count, mpu_DSR_timeout_count);
  printk("   DRR, DSR loop count  %ld, %ld\n", mpu_DRR_loop_count, mpu_DSR_loop_count);
  printk("\n");
  printk("   Voice events played  %ld\n", mpu_voice_count);
  printk("   Voice running hits   %ld\n", mpu_running_hit);
  printk("   SysEx events played  %ld\n", mpu_sysex_count);
  printk("   Meta events played   %ld\n", mpu_meta_count);
  printk("   No operation events  %ld\n", mpu_nop_count);

#endif
  printk("\n");
  printk("*****************************************************\n");
}

static void
mpu_printkevent(midi_event *event)
{
  int i = 0;
  if (!event)
    {
      printk("   Event DUMP - NULL EVENT!\n");
      return;
    };

  printk("   Event DUMP - Type %hx, time %ld, size %d\n",
	 event->type&0xff, event->time, event->size);
  printk("   Data [ ");
  for (i = 0; i < event->size; i++) printk("%02hx ", event->data[i]&0xff);
  printk("]\n");
}

inline static long
mpu_total_event_cnt()
{
  return mpu_queues.out.count + mpu_queues.ctl.count + mpu_queues.rec.count;
}

inline static void
mpu_clear_run()
{
  mpu_old_voice_cmd = 0;
}

static void
mpu_data_error(byte data)
{
  mpu_clear_run();
  printk("   Unknown MPU-401 device message (0x%02x) not processed!\n\n", data);
  printk("   Please report above code, device driver and MPU-401 version\n");
  printk("   information and a brief description of what caused this\n");
  printk("   message to: burgaard@daimi.aau.dk\n\n");
}

inline static int
mpu_can_store()
{
  if ( mpu_total_event_cnt() > MAX_QUEUE_EVENTS )
    {
      printk("EVENT QUEUE OVERFLOW!!!\n");
      mpu_debug_dump();
      return 0;
    };
  return 1;
}

inline static int
mpu_can_rec_store()
{
  if (!mpu_can_store()) return 0;
  return (mpu_status.state & (MPU_STA_REC | MPU_STA_STEPREC));
}

/*** MPU-401 DEVICE LOWLEVEL METHODS ******************************************/

inline static char *
mpu_kmalloc(size_t size)
{
  char *tmp = kmalloc(size, GFP_ATOMIC);
  if (!tmp)
    {
      printk("Out of memory!");
      mpu_debug_dump();
    };
  mpu_total_KMALLOC += size;

  return tmp;
}

midi_event *
midi_event_alloc()
{
  return (midi_event *)mpu_kmalloc(sizeof(midi_event));
}

void 
midi_event_free(midi_event *event)
{
  if (!event) return;

  if (event->data) kfree_s(event->data, event->size);
  mpu_total_KFREE += event->size;

  kfree_s(event, sizeof(midi_event));
  mpu_total_KFREE += sizeof(midi_event);
}

/* The following two wait-loops are pretty *ugly*, but response time seems to
 * swing wildy from 10 to over 500000 loops on my machine (i486DX 33MHz). */

static inline int 
mpu_wait_DRR()
{
  long i = MPU_IOP_RETRY;
  while ( (inb(mpu_STP) & MPU_DRR) && --i );

#ifdef __MPU_STATISTICS
  if (MPU_IOP_RETRY - i > mpu_DRR_loop_count) mpu_DRR_loop_count = MPU_IOP_RETRY - i;
#endif

  if (i) return 0;

  printk("mpu_wait_DRR() time out\n");

#ifdef __MPU_STATISTICS
      mpu_DRR_timeout_count++;
#endif

  return -EBUSY;
}

static inline int 
mpu_wait_DSR()
{
  long i = MPU_IOP_RETRY;
  while ( (inb(mpu_STP) & MPU_DSR) && --i );

#ifdef __MPU_STATISTICS
  if (MPU_IOP_RETRY - i > mpu_DSR_loop_count) mpu_DSR_loop_count = MPU_IOP_RETRY - i;
#endif

  if (i) return 0;

#ifdef __MPU_STATISTICS
  mpu_DSR_timeout_count++;
#endif
  printk("mpu_wait_DSR() time out\n");

  return 0;
}

static inline byte
mpu_rd_dta()         
{
  return inb(mpu_DTP);
}

static inline void
mpu_wr_dta(byte dta)
{
  outb(dta, mpu_DTP);
}

static inline void
mpu_wr_cmd(byte cmd)
{
  outb(cmd, mpu_CMP);
}

/*** MPU-401 DEVICE INTERNAL METHODS ******************************************/

/* Delay (msec) routine. Mostly taken from ftape-io.c by Bas Laarhoven */
static void
mpu_msdelay(word time)
{
  unsigned long flags;
  int ticks = 1 + (time + (1000/HZ) - 1) / (1000/HZ);

  if (mpu_IRQ_mutex)
    {
      printk("!!!PANIC!!! mpu_msdelay called from interrupt!\n");
      mpu_debug_dump();
      return;
    };

  if (DEBUG_EXTREME) printk("MILLISECOND DELAY REQUEST %d\n", time);
  
  /* error in range [0..1] (1000/HZ) */
  if (time < (1000/HZ))
    {
      /*  Time too small for scheduler, do a busy wait ! */
      udelay(1000 * time);
    }
  else
    {
      current->timeout = jiffies + ticks;
      current->state = TASK_INTERRUPTIBLE;
      save_flags( flags);
      sti();
      do
	{
	  while (current->state != TASK_RUNNING)
	    {
	      schedule();
	    };
	  if (current->signal & ~current->blocked)
	    {
	      printk("MPU delay sleep awoken by non-blocked signal\n");
	      break; /* exit on signal */
	    };
	}    
      while (current->timeout > 0);
      restore_flags(flags);
    };
}

static void mpu_irqhandler();

static inline int
mpu_RcDta(void)
{
  return (mpu_wait_DSR()) ? -EBUSY : mpu_rd_dta();
}

static int
mpu_SnCmd(byte cmd)
{
  static int i = 0;

  /* This is important since we might want to call the interrupt handler
   * and we certainly do not want this to be called recursively */

  if (mpu_IRQ_mutex)
    {
      printk("!!!PANIC!!! mpu_SnCmd called from interrupt!\n");
      mpu_debug_dump();
      return -EINVAL;
    };

  cli();

  if (mpu_wait_DRR()) return -EBUSY;
  mpu_wr_cmd(cmd);

  while ( (i = mpu_RcDta()) != MPU_MSG_ACKNOWLEDGE )
    {
      if (i == -EBUSY) break;
      mpu_IRQ_fake = 1;
      mpu_irqhandler();
      mpu_IRQ_fake = 0;
    };
  
  sti();

  return (i == -EBUSY)  ? i : 0;
}

static int
mpu_RqDta(byte cmd)
{
  static int i = 0;

  /* This is important since we might want to call the interrupt handler
   * and we certainly do not want this to be called recursively */

  if (mpu_IRQ_mutex)
    {
      printk("!!!PANIC!!! mpu_RqDta called from interrupt!\n");
      mpu_debug_dump();
      return -EINVAL;
    };

  cli();

  if (mpu_wait_DRR()) return -EBUSY;
  mpu_wr_cmd(cmd);

  while ( (i = mpu_RcDta()) != MPU_MSG_ACKNOWLEDGE )
    {
      if (i == -EBUSY) break;
      mpu_IRQ_fake = 1;
      mpu_irqhandler();
      mpu_IRQ_fake = 0;
    };

  i = mpu_RcDta();

  sti();

  return i;
}

static inline int
mpu_SnDta(byte dta)
{
  if (mpu_wait_DRR()) return -EBUSY;
  mpu_wr_dta(dta);
  return 0;
}

/*** IRQ HANDLING *************************************************************
 *
 * Some precaution should be taken in the following methods, as they are most
 * likely called from the interrupt handler. I.e. *never* call sleep etc.
 *
 ******************************************************************************/

static void
mpu_midi_in(long time)
{
  static byte  run = 0;
  static byte  dta = 0;
  static byte  siz = 0;
  static byte  i   = 0;
  static char *buf = NULL;

  dta = mpu_RcDta()&0xff;

  if (dta > MID_EVENT_CHMSG_LAST)
    {
      switch (dta)
	{
	case MPU_MRK_MEASURE:
	  /* do nothing */
	  break;

	case MPU_MRK_DATA_END:
	  /* Stop recording */
	  break;
	};
    }
  else if (dta >= MID_EVENT_CHMSG_FIRST)
    {
      run = dta;		/* save new status */

      siz = MIDI_CHMSG_ARGS[(dta>>4)&0x0F];

      if (mpu_can_rec_store() && (buf = mpu_kmalloc(siz)) )
	{
	  buf[0] = dta;
	  for (i = 0; i < siz; i++) buf[i+1] = mpu_RcDta();
	  midi_queue_put_event(&mpu_queues.rec, MIDI_TYP_VOICE, time, siz, buf);
	}
      else
	for (i = 0; i < siz; i++) mpu_RcDta(); /* flush rest of incoming data */
    }
  else if (run)
    {
      siz = MIDI_CHMSG_ARGS[(run>>4)&0x0F];
      
      if ( mpu_can_rec_store() && (buf = mpu_kmalloc(siz)) )
	{
	  buf[0] = run;
	  buf[1] = dta;
	  for (i = 0; i < siz-1; i++) buf[i+2] = mpu_RcDta();
	  midi_queue_put_event(&mpu_queues.rec, MIDI_TYP_VOICE, time, siz, buf);
	}
      else
	for (i = 0; i < siz; i++) mpu_RcDta(); /* flush rest of incoming data */
    }
  else
    mpu_data_error(dta);

  if (mpu_sleeps.record)
    {
      wake_up_interruptible(&mpu_sleeps.record);
      mpu_sleeps.record = NULL;
    };
}

static void
mpu_midi_out(byte track)
{
  static midi_event *event = NULL;
  static long dt = 0;
  static word i = 0;

  if (track || !mpu_queues.out.count || !(event = midi_queue_peek(&mpu_queues.out)) )
    {
      if (!mpu_sleeps.play)
	{
	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MRK_NO_OP);

#ifdef __MPU_STATISTICS
	  mpu_nop_count++;
#endif

	  mpu_queues.out_time += MPU_MSG_MAX_TIME;
	}
      else
	{
	  mpu_clear_run();
	  mpu_status.state &= ~(MPU_STA_PLAY);

	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MSG_DATA_END);

	  wake_up_interruptible(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      return;
    };
  
  dt = event->time - mpu_queues.out_time;
  if (dt < 0) dt = 0; /* if delta time < 0 then we're behind schedule :-( */
  if (dt < MPU_MSG_MAX_TIME)
    {
      i = 0;
      mpu_SnDta(dt);
      if ((event->data[0]&0xff) == mpu_old_voice_cmd)
	{
	  i++;
#ifdef __MPU_STATISTICS
	  mpu_running_hit++;
#endif
	}
      else
	mpu_old_voice_cmd = event->data[0]&0xff;
      
      for (; i < event->size; i++) mpu_SnDta(event->data[i]&0xff);
      
      mpu_queues.out_time += dt;
      midi_queue_remove(&mpu_queues.out);

#ifdef __MPU_STATISTICS
      mpu_voice_count++;
#endif

      if ((mpu_queues.out.count < mpu_sleeps.demand_load.voice) && mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};
    }
  else
    {
      mpu_SnDta(MPU_MSG_MAX_TIME);
      mpu_SnDta(MPU_MRK_NO_OP);

#ifdef __MPU_STATISTICS
      mpu_nop_count++;
#endif

      mpu_queues.out_time += MPU_MSG_MAX_TIME;
    };
}

/* The method for sending conductor data is a little bit hairy at a first glance.
 * (but it isn't :-) */
static void
mpu_control_out()
{
  static midi_event *event = NULL;
  static long dt = 0;
  static int i = 0;

  if (control_run && control_old_event)
    {
      /* send current control data */
      for (i = 0; i < control_old_event->size; i++) mpu_SnDta(control_old_event->data[i]&0xff);

      midi_event_free(control_old_event);
      control_old_event = NULL; control_run = 0;
      mpu_clear_run(); 
    };
  
  if (!mpu_queues.ctl.count || !(event = midi_queue_peek(&mpu_queues.ctl)) )
    {
      if (control_old_event) /* should never happen */
	{
	  midi_event_free(control_old_event);
	  control_old_event = NULL;
	};
      
      control_run = 0;
      
      if (!mpu_sleeps.play || mpu_queues.out.count)
	{
	  mpu_SnDta(MPU_MSG_TIME_OVERFLOW);
	  mpu_queues.ctl_time += MPU_MSG_MAX_TIME + 1;

#ifdef __MPU_STATISTICS
	  mpu_nop_count++;
#endif
	}
      else
	{
	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MSG_DATA_END);
	  
	  if (mpu_sleeps.play)
	    {
	      mpu_clear_run(); 
	      mpu_status.state &= ~(MPU_STA_PLAY);
	      
	      wake_up_interruptible(&mpu_sleeps.play);
	      mpu_sleeps.play = NULL;
	    };
	};
      return;
    };
  
  dt = event->time - mpu_queues.ctl_time;
  if (dt < 0) dt = 0; /* if delta time < 0 then we're behind schedule :-( */
  if (dt < MPU_MSG_MAX_TIME)
    {
      mpu_clear_run(); 

      mpu_queues.ctl_time += dt;
      switch (event->type)
	{
	case MIDI_TYP_META:
	  /* There should not be any old events left but we play it safe... */
	  if (control_old_event) midi_event_free(control_old_event);
	  switch (event->data[0]&0xff)
	    {
	    case MID_META_SET_TEMPO:
	      if (event->size != 4)
		{
		  mpu_SnDta(dt);
		  mpu_SnDta(MPU_MRK_NO_OP);
		  printk("MPU Invalid META TEMPO DATA\n");
		  mpu_printkevent(event);
		  break;
		};

	      mpu_metronome.mysectempo = 
		((event->data[1]&0xff)<<16) |
		  ((event->data[2]&0xff)<<8) | 
		    ((event->data[3]&0xff));

	      mpu_metronome.tempo = (byte)(( 60.0 * 1000000.0 )/(float)(mpu_metronome.mysectempo));
	      mpu_SnDta(dt);
	      mpu_SnDta(MPU_SET_TEMPO);
	      mpu_SnDta(mpu_metronome.tempo);

	      if (DEBUG_VERBOSE) printk("TEMPO %hd BPM = %ld microseconds per beat\n",
					mpu_metronome.tempo, mpu_metronome.mysectempo);

#ifdef __MPU_STATISTICS
	      mpu_meta_count++;
#endif

	      break;
	    default:
	      mpu_SnDta(dt);
	      mpu_SnDta(MPU_MRK_NO_OP);

#ifdef __MPU_STATISTICS
	      mpu_nop_count++;
#endif

	      if (DEBUG_EXTREME) printk("NOP\n");
	      break;
	    };

	  midi_queue_remove(&mpu_queues.ctl);
	  control_old_event = NULL; control_run = 0;
	  break;
	case MIDI_TYP_SYSEX:
	  mpu_SnDta(dt);
	  mpu_SnDta(MPU_SND_SYSEX);
	  control_old_event = midi_queue_get(&mpu_queues.ctl);
	  control_run = 1;

#ifdef __MPU_STATISTICS
	  mpu_sysex_count++;
#endif

	  break;
	case MIDI_TYP_VOICE:
	  mpu_SnDta(dt);
	  mpu_SnDta(MPU_SND_DATA + 1); /* send on MPU-401 TRACK 2 */
	  control_old_event = midi_queue_get(&mpu_queues.ctl);
	  control_run = 1;

#ifdef __MPU_STATISTICS
	  mpu_voice_count++;
#endif

	  break;
	default:
	  mpu_SnDta(dt);
	  mpu_SnDta(MPU_MRK_NO_OP);

#ifdef __MPU_STATISTICS
	  mpu_nop_count++;
#endif

	  printk("MPU control run event is invalid\n");
	  mpu_printkevent(event);
	  return;
	  break;
	};

      if ((mpu_queues.ctl.count < mpu_sleeps.demand_load.control) && mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};
    }
  else
    {
      mpu_SnDta(MPU_MSG_TIME_OVERFLOW);
      mpu_queues.ctl_time += MPU_MSG_MAX_TIME + 1;

#ifdef __MPU_STATISTICS
      mpu_nop_count++;
#endif
    };
}

static void
mpu_irqhandler()
{
  static char *buf = NULL;
  static byte dta  = 0;
  static word cnt  = 0;
  static word size = 0;

  /* If DSR = 1, then IRQ was not from MPU. */
  if (inb(mpu_STP) & MPU_DSR) return;
  dta = mpu_rd_dta();

  mpu_IRQ_mutex = 1;

#ifdef __MPU_STATISTICS
  if (!mpu_IRQ_fake) mpu_IRQ_count++;
  else mpu_IRQ_fake_count++;
#endif

  if (dta <= MPU_MSG_MAX_TIME)
    {
      if (DEBUG_EXTREME) printk("MIDI IN\n");
      mpu_queues.rec_time += dta;
      mpu_midi_in(mpu_queues.rec_time);
    }
  else if (dta < MPU_MSG_TIME_OVERFLOW)
    {
      if (DEBUG_EXTREME) printk("MIDI OUT\n");
      mpu_midi_out(dta&0x0F);
    }
  else switch (dta)
    {
    case MPU_MSG_TIME_OVERFLOW:
      mpu_queues.rec_time += 240;
      if (DEBUG_EXTREME) printk("TIMING OVERFLOW\n");
      break;

    case MPU_REQ_CONDUCTOR:
      if (DEBUG_EXTREME) printk("CONDUCTOR\n");
      mpu_control_out();
      break;

    case MPU_MSG_ALL_END:
      mpu_status.state &= ~(MPU_STA_PLAY);
      if (DEBUG_EXTREME) printk("ALL END\n");

      mpu_clear_run();

      if (mpu_sleeps.play)
	{
	  wake_up_interruptible(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      if (mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};
      break;

    case MPU_MSG_CLOCK_TO_HOST:
      if (DEBUG_EXTREME) printk("CLOCK TO HOST\n");
      break;

    case MPU_MSG_ACKNOWLEDGE:
      if (DEBUG_EXTREME) printk("ACKNOWLEDGE\n");
      break;

    case MPU_MSG_MIDI_SYS_MSG:
      /*
       * When receiving System Exclusive messages, the MPU-401 does not send
       * timing information. Therefore the timing information stored is only
       * an approximation. One could have requested the record count to get the
       * exact time, but that is not advisable since a deadlock might occur!
       * (crossed MPU_MSG_ACKNOWLEDGEs and data requests) */
      size = 0;

      if (DEBUG_EXTREME) printk("MIDI IN SYSEX\n");

      mpu_clear_run();

      while ( (dta = mpu_RcDta()) != MID_SYS_EOX )
	{
	  if (size < MPU_SYSEX_BUFFER) mpu_sysexbuf[size++] = dta;
	  cnt++;
	};

      if (cnt < MPU_SYSEX_BUFFER)
	{
	  if ( !(buf = mpu_kmalloc(size)) ) break;

	  memmove(buf, &mpu_sysexbuf[0], size);
	  mpu_queues.out.current = mpu_queues.ctl.current = mpu_queues.rec.current = NULL;
	  midi_queue_put_event(&mpu_queues.rec, MIDI_TYP_SYSEX, mpu_queues.rec_time, size, buf);
	  if (mpu_sleeps.record)
	    {
	      wake_up_interruptible(&mpu_sleeps.record);
	      mpu_sleeps.record = NULL;
	    };
	}
      else
	{
	  printk("   *** OUT OF SYSEX BUFFER ***\n");
	  printk("   Buffer size is %d bytes, needed %d bytes.\n", MPU_SYSEX_BUFFER, cnt);
	  printk("   You can adjust the MPU_SYSEX_BUFFER constant in mpu401.c\n");
	  printk("\n");
	  mpu_debug_dump();
	};
	
      break;

    default:
      mpu_data_error(dta);
      break;
    };

  mpu_IRQ_mutex = 0;
}

static int
mpu_request_irq()
{
  if (DEBUG_STANDARD) printk("mpu_request_irq()\n");
  if ( request_irq(mpu_IRQ, mpu_irqhandler, SA_INTERRUPT, "MPU-401") )
    {
      printk("mpu_request_irq(%02d) failed.\n", mpu_IRQ);
      return -EBUSY;
    };
  return 0;
}

static void
mpu_free_irq()
{
  if (DEBUG_STANDARD) printk("mpu_free_irq(%02d)\n", mpu_IRQ);
  free_irq(mpu_IRQ);
}

/*** GENERAL MIDI/ROLAND GS ``MACROS'' ****************************************
 *
 ******************************************************************************/

static const byte gm_reset[] = 
{
  0xf0, 0x7e, 0x7f, 0x09, 0x01, 0xf7
};

static const byte gs_reset1[] = 
{
  0xf0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f, 0x00, 0x41, 0xf7
};

static const byte gs_reset2[] = 
{
  0xf0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x01, 0x10, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x00,
  0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x02, 0x17, 0xf7
};

static int
mpu_send_sysex(int size, const byte *msg)
{
  int i;
  if (mpu_SnCmd(MPU_SND_SYSEX)) return -EBUSY;
  for (i = 0; i < size; i++) if (mpu_SnDta(msg[i]&0xff)) return -EBUSY;
  mpu_msdelay(50);
  mpu_wait_DRR();
  mpu_clear_run();

  return 0;
}

/*** MPU-401 DEVICE INTERFACE METHODS *****************************************
 *
 ******************************************************************************/

static int
mpu_reset()
{
  mpu_status.state = MPU_STA_NONE;
  disable_irq(mpu_IRQ);

  if (mpu_sleeps.record) wake_up_interruptible(&mpu_sleeps.record);
  if (mpu_sleeps.play) wake_up_interruptible(&mpu_sleeps.play);
  if (mpu_sleeps.demand) wake_up_interruptible(&mpu_sleeps.demand);

  if (mpu_SnCmd(MPU_CLR_RESET))
    {
      printk("mpu_reset()\n   reset command (%02x) failed.\n", MPU_CLR_RESET);
      return -ENODEV;
    };

  mpu_msdelay(20); /* sleep 20 milliseconds */

  mpu_wait_DRR();		/* Make sure the MPU-401 is ready after the reset command */

  /* Enable mpu_irqhandler to process *all* incoming midi events */
  mpu_SnCmd(MPU_INI_TIMING_BYTE_ON);

  mpu_clear_run();

  midi_queue_flush(&mpu_queues.ctl);
  midi_queue_flush(&mpu_queues.out);
  midi_queue_flush(&mpu_queues.rec);

  mpu_queues.out_time = mpu_queues.ctl_time = mpu_queues.rec_time = 0;
  enable_irq(mpu_IRQ);
  return 0;
}

/* detect calls sleep, so *never* call it from an interrupt! */
static int
mpu_detect()
{
  int i = 0;

  if (DEBUG_STANDARD) printk("mpu_detect()\n");

  /* First see if a RESET commands behaves well */
  if ( mpu_reset() ) return -ENODEV;

  mpu_msdelay(20); /* sleep 20 milliseconds */

  if ( (i = mpu_RqDta(MPU_REQ_VERSION)) < 0 )
    {
      printk("mpu request version (%02x) failed.\n", MPU_REQ_VERSION);
      return -ENODEV;
    };

  mpu_status.mpu_version = (i&0xf0) >> 4;
  mpu_status.mpu_subversion = i&0x0f;

  if (i < 0x14)
    {
      printk("   This MPU-401 device is older than the 1.4 version\n");
      printk("   and may therefore not work reliably in all situations\n");
    };

  mpu_msdelay(20); /* sleep 20 milliseconds */

  if ( (i = mpu_RqDta(MPU_REQ_REVISION)) < 0)
    {
      printk("mpu request revision (%02x) failed.\n", MPU_REQ_REVISION);
      return -ENODEV;
    };

  mpu_status.mpu_revision = (i) ? i - 1 + 'A' : 0;

  printk("MPU-401 v%hd.%hd%c detected at IRQ 0x%02x I/O PORT 0x%04x\n",
	 mpu_status.mpu_version, mpu_status.mpu_subversion, mpu_status.mpu_revision,
	 mpu_status.irq, mpu_status.ioport);

  return 0;
}

/*** DEVICE INTERFACE *********************************************************
 *
 * Data format used by read and write:
 *
 *    count type   description
 *    -------------------------------------------------------------------------
 *    1     byte   Event type: MIDI_TYP_VOICE or MIDI_TYP_CONDUCT.
 *    1     long   Event time: Time values are total time.
 *    1     word   Event size: Number of data bytes following.
 *    size  char   Event data: MIDI VOICE data or MPU Control messages.
 *
 * int mpu_read(struct inode * inode, struct file * filp, char * buffer, int count)
 *    Writes recorded MIDI data to buffer and returns number of bytes written.
 *
 * int mpu_read(struct inode * inode, struct file * filp, char * buffer, int count)
 *    Transfers count bytes from buffer to internal play queue. Returns actual number of
 *    bytes transfered.
 *
 * int mpu_open(struct inode * inode, struct file * filp)
 *    Opens and initializes the MPU-401 interface and returns succes conforming to 
 *    <linux/errno.h>.
 *
 * void mpu_release(struct inode * inode, struct file * filp)
 *    Resets and releases the MPU-401 interface.
 *
 * int mpu_ioctl(struct inode * inode, struct file * filp, unsigned int cmd, char * argp)
 *    Tell the driver what's going on ;-)
 *
 * int mpu_register_device()
 *    This routine registers the character device used for communication with the
 *    device driver.
 *
 * void mpu_unregister_device()
 *    Take a guess ;).
 *
 ******************************************************************************/

#ifdef __WE_DONT_READ_OR_WRITE

static int
mpu_read(struct inode * inode, struct file * filp, char * buffer, int count)
{
  midi_event *event = 0;
  char *tmp = 0;
  
  if ( !mpu_queues.rec.count ) interruptible_sleep_on(&mpu_sleeps.record);
  
  tmp = buffer;
  while(count > 0 && mpu_queues.rec.count)
    {
      event = midi_queue_peek(&mpu_queues.rec);
      if (sizeof(char) + sizeof(long) + sizeof(word) + event->size > count) break;
      
      cli();
      put_fs_byte(event->type, tmp); tmp += sizeof(char);
      put_fs_long(event->time, tmp); tmp += sizeof(long);
      put_fs_word(event->size, tmp); tmp += sizeof(word);
      memcpy_tofs(tmp, event->data, event->size); tmp += event->size;
      sti();
      
      count -= sizeof(char) + sizeof(long) + sizeof(word) + event->size;
      midi_queue_remove(&mpu_queues.rec);
    };
  
  return tmp - buffer;
}

static int
mpu_write(struct inode * inode, struct file * filp, char * buffer, int count)
{
  char *tmp = buffer;
  char *dta = NULL;
  
  char type = MIDI_TYP_VOICE;
  long time = 0;
  word size = 0;
  
  while ( count > 0 )
    {
      type = get_fs_byte(tmp); tmp += sizeof(char);
      time = get_fs_long(tmp); tmp += sizeof(long);
      size = get_fs_word(tmp); tmp += sizeof(word);
      
      count -= sizeof(char);
      count -= sizeof(long);
      count -= sizeof(word);
      
      /* This will never happen (shouldn't at least ;-) */
      if (!size)
	{
	  printk("   mpu_write(...) SOURCE ERROR: Failed on midi_event with zero size\n");
	  break;
	};
      
      if (count < size)
	{
	  printk("   data size exceeds transfer count (%d > %d)\n", size, count);
	  break;
	};
      if ( !(dta = mpu_kmalloc(size)) ) break;
      
      memcpy_fromfs(dta, tmp, size); tmp += size;
      count -= size;
      
      if (type == MIDI_TYP_VOICE)
	midi_queue_put_event(&mpu_queues.out, type, time, size, dta);
      else
	midi_queue_put_event(&mpu_queues.ctl, type, time, size, dta);
    };
  
  return tmp - buffer;
}

#endif /* __WE_DONT_READ_OR_WRITE */

static int
mpu_ioctl_guest
(struct inode * inode, struct file * filp, word request, unsigned long argp)
{
  if (DEBUG_EXTREME) printk("STATUS IOCONTROL(..., 0x%08x, 0x%08lx)\n", request, argp);

  if ((char)IOCGROUP(request) != 'M')
    {
      printk("MPU Unknown ioctl request, 0x%08x (bad ``magic'' ID)\n", request);
      return -EINVAL;
    };

  switch (request)
    {
    case MPUIOC_GET_STATUS: /* _IOR('M', 0x01, struct mpu_status) */
      if (!argp) return -EINVAL;
      memcpy_tofs((void *)argp, &mpu_status, sizeof(struct mpu_status));
      break;
      
    case MPUIOC_DEBUG_DUMP: /* _IO('M', 0xf0) */
      mpu_debug_dump();
      break;
    case MPUIOC_SET_DEBUG_LVL: /* _IOW('M', 0xf1, word) */
      if (!argp) return -EINVAL;
      mpu_DEBUG = get_fs_word(argp);
      break;
    case MPUIOC_GET_DEBUG_LVL: /* _IOR('M', 0xf2, word) */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_DEBUG, argp);
      break;
    default:
      return -EINVAL;
    };
  return 0;
}

static int
mpu_ioctl(struct inode * inode, struct file * filp, word request, unsigned long argp)
{
  static struct mpu_demand *demand = NULL;
  static midi_event *pevent = NULL; 
  static midi_event *tevent = NULL;
  static midi_event event;
  static char *data = NULL;
  static word i  = 0;
  static int res = 0;

  if (DEBUG_VERBOSE) printk("IOCONTROL(..., 0x%08x, 0x%08lx)\n", request, argp);

  if ( MINOR(inode->i_rdev) == MPU_DEV_STATUS ) return mpu_ioctl_guest(inode, filp, request, argp);

  if ((char)IOCGROUP(request) != 'M')
    {
      printk("MPU Unknown ioctl request, 0x%08x (bad ``magic'' ID)\n", request);
      return -EINVAL;
    };

  switch (request)
    {
    case MPUIOC_RESET:		/* _IO('M', 0x00) */
      mpu_reset();
      mpu_status.state = MPU_STA_OPEN;
      break;

    case MPUIOC_GET_STATUS:	/* _IOR('M', 0x01, struct mpu_status) */
      if (!argp) return -EINVAL;
      memcpy_tofs((void *)argp, &mpu_status, sizeof(struct mpu_status));
      break;

    case MPUIOC_MIDI_THRU_ON:	/*  _IO('M', 0x02) */
      mpu_SnCmd(MPU_SWI_MIDI_THRU + 1);
      break;
    case MPUIOC_MIDI_THRU_OFF:	/* _IO('M', 0x03) */
      mpu_SnCmd(MPU_SWI_MIDI_THRU);
      break;

    case MPUIOC_BENDER_THRU_ON: /*  _IO('M', 0x04) */
      mpu_SnCmd(MPU_SWI_BENDER + 1);
      break;
    case MPUIOC_BENDER_THRU_OFF: /* _IO('M', 0x05) */
      mpu_SnCmd(MPU_SWI_BENDER);
      break;

    case MPUIOC_INIT_PLAY:	/* _IOW('M', 0x10, long) */
      if (mpu_status.state & MPU_STA_REC)
	{
	  mpu_status.state &= ~(MPU_STA_REC);
	  if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_OFF);
	  if (mpu_sleeps.record)
	    {
	      wake_up_interruptible(&mpu_sleeps.record);
	      mpu_sleeps.record = NULL;
	    };
	};
      mpu_SnCmd(MPU_SEQ_STOP_ALL);
      mpu_queues.out_time = mpu_queues.ctl_time = argp;
      mpu_status.state |= MPU_STA_PLAY;

      mpu_clear_run();
      mpu_SnCmd(MPU_SET_TRACK_ACTIVE); mpu_SnDta(1);
      mpu_SnCmd(MPU_SWI_CONDUCTOR + 1);
      mpu_SnCmd(MPU_CLR_PLAY_CNT);
      mpu_SnCmd(MPU_SEQ_START_PLAY);
      break;

    case MPUIOC_CONT_PLAY:	/* _IOW('M', 0x11, long) */
      if (mpu_status.state & MPU_STA_REC)
	{
	  mpu_status.state &= ~(MPU_STA_REC);
	  mpu_SnCmd(MPU_SEQ_STOP_REC);
	  if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_OFF);
	  if (mpu_sleeps.record)
	    {
	      wake_up_interruptible(&mpu_sleeps.record);
	      mpu_sleeps.record = NULL;
	    };
	};
      mpu_queues.out_time = mpu_queues.ctl_time = argp;
      mpu_clear_run();
      if ( !(mpu_status.state & MPU_STA_PLAY) )
	{
	  mpu_status.state |= MPU_STA_PLAY;
	  mpu_SnCmd(MPU_SEQ_CONT_PLAY);
	};
      break;

    case MPUIOC_STOP_PLAY:	/* _IO('M', 0x12) */
      if (mpu_status.state & MPU_STA_PLAY)
	{
	  mpu_status.state &= ~(MPU_STA_PLAY);
	  mpu_SnCmd(MPU_SEQ_STOP_PLAY);
	};
      
      if (mpu_sleeps.play)
	{
	  wake_up_interruptible(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      if (mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};
      break;

    case MPUIOC_INIT_OVERDUB:	/* _IOW('M', 0x13, long) */
      if (mpu_status.state & MPU_STA_REC)
	{
	  mpu_status.state &= ~(MPU_STA_REC);
	  if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_OFF);
	};
      mpu_SnCmd(MPU_SEQ_STOP_ALL);

      mpu_queues.out_time = mpu_queues.ctl_time = mpu_queues.rec_time = argp;
      mpu_status.state |= MPU_STA_PLAY|MPU_STA_REC;

      mpu_clear_run();
      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_ON);
      mpu_SnCmd(MPU_SET_TRACK_ACTIVE); mpu_SnDta(1);
      mpu_SnCmd(MPU_SWI_CONDUCTOR + 1);
      mpu_SnCmd(MPU_CLR_PLAY_CNT);
      mpu_SnCmd(MPU_SEQ_START_OVERDUB);
      break;

    case MPUIOC_CONT_OVERDUB:	/* _IOW('M', 0x14, long) */
      mpu_status.state |= MPU_STA_PLAY|MPU_STA_REC;
      mpu_queues.out_time = mpu_queues.ctl_time = argp;

      mpu_clear_run();
      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_ON);
      mpu_SnCmd(MPU_SEQ_CONT_PLAY);
      break;

    case MPUIOC_STOP_OVERDUB:	/* _IO('M', 0x15) */
      mpu_status.state &= ~(MPU_STA_PLAY|MPU_STA_REC);

      mpu_SnCmd(MPU_SEQ_STOP_OVERDUB);

      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_OFF);
      if (mpu_sleeps.play)
	{
	  wake_up_interruptible(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      if (mpu_sleeps.record)
	{
	  wake_up_interruptible(&mpu_sleeps.record);
	  mpu_sleeps.record = NULL;
	};
      if (mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};

      break;

    case MPUIOC_INIT_RECORD:	/* _IOW('M', 0x16, long) */
      if (mpu_sleeps.play)
	{
	  wake_up_interruptible(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      if (mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};
      mpu_SnCmd(MPU_SEQ_STOP_ALL);

      mpu_status.state &= ~(MPU_STA_PLAY);
      mpu_status.state |= MPU_STA_REC;

      mpu_queues.rec_time = argp;
      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_ON);
      mpu_SnCmd(MPU_SEQ_START_REC);

      break;

    case MPUIOC_CONT_RECORD:	/* _IOW('M', 0x17, long) */
      if (mpu_status.state & MPU_STA_PLAY)
	{
	  mpu_status.state &= ~(MPU_STA_PLAY);
	  mpu_SnCmd(MPU_SEQ_STOP_PLAY);
	  if (mpu_sleeps.play)
	    {
	      wake_up_interruptible(&mpu_sleeps.play);
	      mpu_sleeps.play = NULL;
	    };
	  if (mpu_sleeps.demand)
	    {
	      wake_up_interruptible(&mpu_sleeps.demand);
	      mpu_sleeps.demand = NULL;
	    };
	};
      mpu_status.state |= MPU_STA_REC;
      mpu_queues.rec_time = argp;

      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_ON);
      mpu_SnCmd(MPU_SEQ_CONT_REC);
      break;

    case MPUIOC_STOP_RECORD:	/* _IO('M', 0x18) */
      mpu_status.state &= ~(MPU_STA_REC);

      mpu_SnCmd(MPU_SEQ_STOP_REC);
      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_OFF);
      if (mpu_sleeps.record)
	{
	  wake_up_interruptible(&mpu_sleeps.record);
	  mpu_sleeps.record = NULL;
	};
      break;

    case MPUIOC_STOP_ALL:	/* _IO('M', 0x19) */
      mpu_status.state &= ~(MPU_STA_REC|MPU_STA_PLAY);
      mpu_SnCmd(MPU_SEQ_STOP_ALL);
      if (mpu_metronome.mode & MPU_METRO_REC) mpu_SnCmd(MPU_SET_METRO_OFF);
      if (mpu_sleeps.play)
	{
	  wake_up_interruptible(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      if (mpu_sleeps.record)
	{
	  wake_up_interruptible(&mpu_sleeps.record);
	  mpu_sleeps.record = NULL;
	};
      if (mpu_sleeps.demand)
	{
	  wake_up_interruptible(&mpu_sleeps.demand);
	  mpu_sleeps.demand = NULL;
	};
      break;

    case MPUIOC_SET_METRONOME:	/* _IOW('M', 0x1a, int) */
      if (!argp) return -EINVAL;
      mpu_metronome.mode = get_fs_word(argp);
      break;

    case MPUIOC_GET_METRONOME:	/* _IOR('M', 0x1b, int) */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_metronome.mode, argp);
      break;

    case MPUIOC_SLEEP2PLAY_END: /* _IO('M', 0x1c) */
      if ( (mpu_status.state & MPU_STA_PLAY) && !mpu_sleeps.play)
	{
	  interruptible_sleep_on(&mpu_sleeps.play);
	  mpu_sleeps.play = NULL;
	};
      break;

    case MPUIOC_SLEEP2DEMAND:	/* _IOWR('M', 0x1d, struct mpu_demand) */
      if (!argp) return -EINVAL;
	
      demand = (struct mpu_demand *)argp;
	
      mpu_sleeps.demand_load.voice = get_fs_long(&demand->voice);
      mpu_sleeps.demand_load.control = get_fs_long(&demand->control);

      if (!mpu_sleeps.demand_load.voice && !mpu_sleeps.demand_load.control)
	break;
	
      if (mpu_status.state & MPU_STA_PLAY) interruptible_sleep_on(&mpu_sleeps.demand);
      else return -EINVAL;

      if (mpu_status.state & MPU_STA_PLAY)
	{
	  put_fs_long(mpu_queues.out.count, &demand->voice);
	  put_fs_long(mpu_queues.ctl.count, &demand->control);
	}
      else
	{
	  put_fs_long(-1, &demand->voice);
	  put_fs_long(-1, &demand->control);
	};
	
      break;
	
    case MPUIOC_GET_OUT_TIME:	/* _IOR('M', 0x20, long) */
      if (!argp) return -EINVAL;
      break;

    case MPUIOC_GET_CTL_TIME:	/* _IOR('M', 0x21, long) */
      if (!argp) return -EINVAL;
      break;
    case MPUIOC_GET_REC_TIME:	/* _IOR('M', 0x22, long) */
      if (!argp) return -EINVAL;
      break;
	
    case MPUIOC_SET_OUT_TIME:	/* _IOW('M', 0x23, long) */
      if (!argp) return -EINVAL;
      break;
    case MPUIOC_SET_CTL_TIME:	/* _IOW('M', 0x24, long) */
      if (!argp) return -EINVAL;
      break;
    case MPUIOC_SET_REC_TIME:	/* _IOW('M', 0x25, long) */
      if (!argp) return -EINVAL;
      break;
	
    case MPUIOC_CLR_OUT_TIME:	/* _IO('M', 0x26) */
      if (!argp) return -EINVAL;
      break;
    case MPUIOC_CLR_CTL_TIME:	/* _IO('M', 0x27) */
      if (!argp) return -EINVAL;
      break;
    case MPUIOC_CLR_REC_TIME:	/* _IO('M', 0x28) */
      if (!argp) return -EINVAL;
      break;
	
      /* tempo (in BPM) between 8 and 240 */
    case MPUIOC_SET_TEMPO:	/* _IOW('M', 0x29, byte) */
      if (!argp) return -EINVAL;
      mpu_SnCmd(MPU_SET_TEMPO);
      return mpu_SnDta(mpu_metronome.tempo = get_fs_byte(argp));

    case MPUIOC_GET_TEMPO:	/* _IOR('M', 0x2a, byte) */
      if (!argp) return -EINVAL;
      put_fs_byte(mpu_metronome.tempo, argp);
      break;

    case MPUIOC_SET_DIVISION:	/* _IOW('M', 0x2b, word) */
      if (!argp) return -EINVAL;
      i = get_fs_word(argp);
      if (i % 24 || i < 48 || i > 192)
	{
	  printk("MPU Invalid Division Request: %d", get_fs_byte(argp));
	  return -EINVAL;
	}
      else
	{
	  mpu_metronome.division = i;
	  return mpu_SnCmd( MPU_SET_TIME_BASE + (i / 24) - 2 );
	};
      break;

    case MPUIOC_GET_DIVISION:	/* _IOR('M', 0x2c, word) */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_metronome.division, argp);
      break;

    case MPUIOC_SET_MIDI_METRO: /* _IOW('M', 0x2d, byte) */
      if (!argp) return -EINVAL;
      mpu_metronome.MIDIperMETRO = get_fs_byte(argp);
      mpu_SnCmd(MPU_SET_MIDI_METRO);
      mpu_SnDta(mpu_metronome.MIDIperMETRO);
      break;
    case MPUIOC_GET_MIDI_METRO: /* _IOR('M', 0x2e, byte) */
      if (!argp) return -EINVAL;
      put_fs_byte(mpu_metronome.MIDIperMETRO, argp);
      break;

    case MPUIOC_SET_METRO_MEAS: /* _IOW('M', 0x2f, byte) */
      if (!argp) return -EINVAL;
      mpu_metronome.METROperMEAS = get_fs_byte(argp);
      mpu_SnCmd(MPU_SET_METRO_MEAS);
      mpu_SnDta(mpu_metronome.METROperMEAS);
      break;
    case MPUIOC_GET_METRO_MEAS: /* _IOR('M', 0x30, byte) */
      if (!argp) return -EINVAL;
      put_fs_byte(mpu_metronome.METROperMEAS, argp);
      break;

    case MPUIOC_MIDI_IN_ON:	/* _IO('M', 0x40) */
      mpu_status.state |= MPU_STA_STEPREC;
      break;
    case MPUIOC_MIDI_IN_OFF:	/* _IO('M', 0x41) */
      mpu_status.state &= ~(MPU_STA_STEPREC);
      break;
    case MPUIOC_MIDI_IN_STATUS: /* _IOR('M', 0x42, word) */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_status.state, argp);
      break;

    case MPUIOC_PUT_IMMEDIATE:	/* _IOW('M', 0x43, struct midi_event) */
      if ( !argp ) return -EINVAL;
	
      memcpy_fromfs(&event, (midi_event *)argp, sizeof(midi_event));
	
      if ( !event.size ) return -EINVAL;
      if ( !event.data ) return -EINVAL;

      mpu_clear_run();
	
      if ( event.type == MIDI_TYP_VOICE )
	res = mpu_SnCmd(MPU_SND_DATA | 0x02); /* want to send data to sequencer track 2 */
      else if ( event.type == MIDI_TYP_SYSEX )
	res = mpu_SnCmd(MPU_SND_SYSEX); /* want to send sysex */
      else
	{
	  printk("MPU Attempt to send CONTROL event immediate (use ioctl instead!)\n");
	  return -EINVAL;
	};
      for (i = 0; i < event.size && !res; i++) res = mpu_SnDta(get_fs_byte(event.data++));
	
      return res;

    case MPUIOC_GET_OUT_CNT:	/* _IOR('M', 0x44, long) */
      if (!argp) return -EINVAL;
      put_fs_long(mpu_queues.out.count, argp);
      break;
    case MPUIOC_GET_CTL_CNT:	/* _IOR('M', 0x45, long) */
      if (!argp) return -EINVAL;
      put_fs_long(mpu_queues.ctl.count, argp);
      break;
    case MPUIOC_GET_REC_CNT:	/* _IOR('M', 0x46, long) */
      if (!argp) return -EINVAL;
      put_fs_long(mpu_queues.rec.count, argp);
      break;

    case MPUIOC_GET_REC_DATASIZE: /* _IOR('M', 0x50, long) */
      if (!argp) return -EINVAL;

      if (mpu_queues.rec.count && (pevent = midi_queue_peek(&mpu_queues.rec)) )
	put_fs_long(pevent->size, argp);
      else
	put_fs_long(0, argp);
      break;
	
    case MPUIOC_PUT_EVENT:	/* _IOW('M', 0x51, struct midi_event) */
      if ( !argp ) return -EINVAL;
      if ( !mpu_can_store() ) return -EBUSY;

      memcpy_fromfs(&event, (midi_event *)argp, sizeof(midi_event));
	
      if ( !event.size ) return -EINVAL;
      if ( !event.data ) return -EINVAL;
	
      if ( !(data = mpu_kmalloc(event.size)) ) return -ENOMEM;
      else memcpy_fromfs(data, event.data, event.size);

      mpu_queues.out.current = mpu_queues.ctl.current = mpu_queues.rec.current = NULL;

      if (DEBUG_EXTREME) printk("PUT EVENT TYPE %d TIME %ld\n", event.type, event.time);
	
      if ( event.type == MIDI_TYP_VOICE )
	midi_queue_put_event(&mpu_queues.out, event.type, event.time, event.size, data);
      else
	midi_queue_put_event(&mpu_queues.ctl, event.type, event.time, event.size, data);
      break;
	
    case MPUIOC_GET_REC_EVENT:	/* _IOWR('M', 0x52, struct midi_event) */
      if ( !argp ) return -EINVAL;
	
      pevent = (midi_event *)argp; /* DESTINATION */
      tevent = midi_queue_peek(&mpu_queues.rec); /* SOURCE */
	
      if ( !mpu_queues.rec.count || !tevent || tevent->size > get_fs_word(&pevent->size) )
	{
	  put_fs_word(0, &pevent->size);
	  return 0;
	};
	
      put_fs_byte(tevent->type, &pevent->type);
      put_fs_long(tevent->time, &pevent->time);
      memcpy_tofs(pevent->data, tevent->data, tevent->size);
      break;

    case MPUIOC_GM_ON:		/* _IO('M', 0xe0) */
      mpu_send_sysex(sizeof(gm_reset), gm_reset);
      break;
    case MPUIOC_GS_RESET:	/* _IO('M', 0xe1) */
      mpu_send_sysex(sizeof(gs_reset1), gs_reset1);
      mpu_send_sysex(sizeof(gs_reset2), gs_reset2);
      break;
	
    case MPUIOC_DEBUG_DUMP:	/* _IO('M', 0xf0) */
      mpu_debug_dump();
      break;
    case MPUIOC_SET_DEBUG_LVL:	/* _IOW('M', 0xf1, word) */
      if (!argp) return -EINVAL;
      mpu_DEBUG = get_fs_word(argp);
      break;
    case MPUIOC_GET_DEBUG_LVL:	/* _IOR('M', 0xf2, word) */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_DEBUG, argp);
      break;

    default:
      printk("Unknown MPU-401 device driver ioctl request 0x%04x.\n", request);
      return -EINVAL;
    };
  
  return 0;
}

static int
mpu_open(struct inode * inode, struct file * filp)
{
  if ( MINOR(inode->i_rdev) == MPU_DEV_RW )
    {
      if ( mpu_status.state & MPU_STA_OPEN ) return -EBUSY;
      else if ( mpu_reset() ) return -EIO;
      else
	{
	  mpu_status.state = MPU_STA_OPEN;
	  MOD_INC_USE_COUNT;
	};
    }
  else if ( MINOR(inode->i_rdev) == MPU_DEV_STATUS )
    {
      /* anonymous login. Restrictions apply ;-) */
      MOD_INC_USE_COUNT;
    }
  else
    return -ENXIO;

  return 0;
}

static void
mpu_release(struct inode * inode, struct file * filp)
{
  if ( MINOR(inode->i_rdev) == MPU_DEV_RW ) mpu_reset();
  MOD_DEC_USE_COUNT;
}

static struct file_operations mpu_fops = 
{
  NULL,        /* mpu_lseek */ 
  NULL,        /* mpu_read */
  NULL,        /* mpu_write */
  NULL,        /* mpu_readdir */
  NULL,        /* mpu_select */
  mpu_ioctl,
  NULL,        /* mpu_mmap */
  mpu_open,
  mpu_release
};

static int
mpu_register_device()
{
  if (DEBUG_STANDARD) printk("mpu_register_device()\n");
  if (register_chrdev(MPU_DEV_MAJOR, "mpu401", &mpu_fops))
    {
      printk("mpu Unable to register sound character device \"mpu401\"\n");
      return -EBUSY;
    };
  return 0;
}

static void
mpu_unregister_device()
{
  if (DEBUG_STANDARD) printk("mpu_unregister_device()\n");
  if (unregister_chrdev(MPU_DEV_MAJOR, "mpu401")) 
    {
      printk("mpu   Unable to unregister sound character device \"mpu401\"\n");
    };
}

/*** RUNTIME CONTROL **********************************************************
 *
 * void mpu_get_rc()
 *    This routine looks after a file named `/etc/mpu.conf' as defined in 
 *    configure.h. If the file does not exist, it will revert to default
 *    settings for IRQ and I/O PORT, which are also defined in configure.h
 *
 * The file is expected to have the following format (binary):
 *
 *    1st byte: 'M'
 *    2nd byte: 'P'
 *    3rd byte: 'U'
 *    4th byte: '!'
 *    5th byte: IRQ number (MSB) (not used - set to zero!)
 *    6th byte: IRQ number (LSB) (must be between 0x02 and 0x07)
 *    7th byte: I/O Port number (MSB) (always 0x03)
 *    8th byte: I/O Port number (LSB) (may be one of 0x30, 0x32, 0x34 or 0x36)
 *
 * If the file is not minimum 8 bytes long, it will be rejected.
 *
 ******************************************************************************/

static char
mpu_get_conf()
{
  static char buf[8];
  static char didread = 0;

  static struct inode * inode = NULL;
  static struct file file;

  static unsigned short fs;


  if (DEBUG_STANDARD)
    {
      printk("mpu_get_conf()\n"); 
      printk("configuration file is %s\n", MPU_RC_PATH);
    };

  fs = get_fs();

  if ( (open_namei(MPU_RC_PATH, 1, S_IRUGO, &inode, NULL)) )
    {
      printk("couldn't open file\n");
      inode = NULL;
      goto end_read;
    };

  if (inode->i_size != 8)
    {
      printk("invalid format (please use mpuconfig).\n");
      goto end_read;
    }

  if (!S_ISREG(inode->i_mode)) /* is it a regular file? */
    {
      printk("this is not a regular file.\n");
      goto end_read;
    }

  if (!inode->i_op || !inode->i_op->default_file_ops) /* does it have file operations? */
    {
      printk("   cannot read file (i_op is NULL).\n");
      goto end_read;
    }

  file.f_mode = 1; /* 1 = read acces */
  file.f_flags = 0;
  file.f_count = 1;
  file.f_inode = inode;
  file.f_pos = 0;
  file.f_reada = 0;
  file.f_op = inode->i_op->default_file_ops;

  if ( file.f_op->open && file.f_op->open(inode, &file) ) /* can we open this file? */
    {
      printk("   file not found.\n");
      goto end_read;
    };

  if (!file.f_op->read) /* does it have file operations? */
    {
      printk("   cannot read file (f_op->read is NULL).\n");
      goto end_read;
    }

  set_fs(KERNEL_DS);

  if ( file.f_op->read(inode, &file, &buf[0], 8) < 8 ) /* did we succeed in reading? */
    {
      printk("   error while reading file.\n");
      goto close_read;
    };

  if (buf[0] == 'M' && buf[1] == 'P' && buf[2] == 'U' && buf[3] == '!')
    {
      didread = 1;
      
      mpu_IRQ = buf[4] << 8;
      mpu_IRQ |= buf[5];
      if (mpu_IRQ < 2 || mpu_IRQ > 7)
	{
	  printk("   Wild IRQ request (0x%02hx).\n", mpu_IRQ);
	  printk("   If your card *really* supports this, please\n");
	  printk("   contact <burgaard@daimi.aau.dk> and I will\n");
	  printk("   be happy to include you IRQ number.\n");
	  didread = 0;
	};

      mpu_IOP = buf[6] << 8;
      mpu_IOP |= buf[7];
      if (mpu_IOP % 2 || mpu_IOP < 0x330 || mpu_IOP > 0x336)
	{
	  printk("   Unknown I/O Port request (0x%04hx).\n", mpu_IOP);
	  printk("   If your card *really* supports this, please\n");
	  printk("   contact <burgaard@daimi.aau.dk> and I will\n");
	  printk("   be happy to include you I/O port number.\n");
	  didread = 0;
	};
    }
  else
    printk("   invalid format (please use mpucondig).\n");

 close_read:
  if (file.f_op->release) file.f_op->release(inode, &file);

 end_read:
  set_fs(fs);
  iput(inode);

  if (!didread)
    {
      printk("   using default values.\n");
      printk("\n");
      printk("   *** Please update configuration with mpuconfig ***\n");
      mpu_IRQ = MPU_IRQ;
    };

  mpu_DTP = mpu_IOP;
  mpu_CMP = mpu_IOP + 1;
  mpu_STP = mpu_IOP + 1;

  mpu_status.ioport = mpu_IOP;
  mpu_status.irq = mpu_IRQ;

  return didread;
}

/*** MODULAR KERNEL INTERFACE *************************************************/

int
init_module()
{
  int i = 0;

  printk(driver_version_str, mpu_status.drv_version, mpu_status.drv_subversion,
	 mpu_status.drv_release);

  if (DEBUG_STANDARD) printk("init_module()\n"); 

  if (!mpu_get_conf()) return -EINVAL;

  midi_queue_reset(&mpu_queues.out);
  midi_queue_reset(&mpu_queues.ctl);
  midi_queue_reset(&mpu_queues.rec);

  if ( (i = mpu_register_device()) ) return i;
  if ( (i = mpu_request_irq()) )
    {
      unregister_chrdev(SOUND_MAJOR, "mpu401");
      return -EBUSY;
    };

  if ( (i = mpu_detect()) )
    {
      mpu_free_irq();
      unregister_chrdev(SOUND_MAJOR, "mpu401");
      return -ENODEV;
    };

  return i;
}

void
cleanup_module()
{
  if (DEBUG_STANDARD) printk("cleanup_module()\n");

  if (mpu_status.state) mpu_reset();

  if ( MOD_IN_USE )
    {
      printk("   Module is in use, remove delayed.\n");
      return;
    };

  mpu_free_irq();
  mpu_unregister_device();

  midi_queue_flush(&mpu_queues.out);
  midi_queue_flush(&mpu_queues.ctl);
  midi_queue_flush(&mpu_queues.rec);
}

/*** End of File **************************************************************/
