/****************************************************************************** 
 *
 * File:        mpu401.h
 *
 *              Roland MPU-401 MIDI Interface Device Driver.
 *
 * Version:     0.1 alpha
 * Date:        1994-11-14
 *
 * Project:     Roland MPU-401 Device driver for Linux 1.1.64 and higher.
 *              This driver is a modular part of the kernel.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#ifndef __MPU401_H
#define __MPU401_H

#include "miditypes.h"

/*** CONSTANTS ****************************************************************/

/* status flags */
enum MPU_STATE
{
  MPU_STA_NONE    = 0x0000,
  MPU_STA_OPEN    = 0x0001,
  MPU_STA_PLAY    = 0x0100,
  MPU_STA_REC     = 0x0200,
  MPU_STA_STEPREC = 0x1000 /* store MIDI IN data also when not in RECORD mode */
};

enum MPU_METRONOME
{
  MPU_METRO_NONE = 0x0000,
  MPU_METRO_REC  = 0x0001,
  MPU_METRO_PLAY = 0x0002 /* NOT YET IMPLEMENTED !!! */
};

/*** TYPE DEFINITIONS *********************************************************/

struct mpu_status
{
  enum MPU_STATE state;

  byte drv_version;
  byte drv_subversion;
  char drv_release;

  byte mpu_version;
  byte mpu_subversion;
  char mpu_revision;

  word irq;
  word ioport;
};

struct mpu_demand
{
  long voice;
  long control;
};

/*** ENUMERATIONS *************************************************************/

/* Bitmaps for different sequencer modes */
enum MPU_MODE
{
   MPU_MOD_STOP_MIDI   = 1,           /* %xxxx01 */
   MPU_MOD_START_MIDI  = 1<<1,        /* %xxxx10 */
   MPU_MOD_CONT_MIDI   = (1<<1) | 1,  /* %xxxx11 */

   MPU_MOD_STOP_PLAY   = 1<<2,        /* %xx01xx */
   MPU_MOD_START_PLAY  = 1<<3,        /* %xx10xx */

   MPU_MOD_STOP_REC    = 1<<4,        /* %01xxxx */
   MPU_MOD_START_REC   = 1<<5,        /* %10xxxx */
   MPU_MOD_STANDBY_REC = MPU_MOD_START_REC
};

/* These are the commands issued for start/stop/continue of sequencing */
enum MPU_SEQUENCE
{
   MPU_SEQ_START_ALL     = MPU_MOD_START_MIDI|MPU_MOD_START_PLAY|MPU_MOD_START_REC,
   MPU_SEQ_STOP_ALL      = MPU_MOD_STOP_MIDI|MPU_MOD_STOP_PLAY|MPU_MOD_STOP_REC,

   MPU_SEQ_START_PLAY    = MPU_MOD_START_MIDI|MPU_MOD_START_PLAY,
   MPU_SEQ_CONT_PLAY     = MPU_MOD_CONT_MIDI|MPU_MOD_START_PLAY,
   MPU_SEQ_STOP_PLAY     = MPU_MOD_STOP_MIDI|MPU_MOD_STOP_PLAY,

   MPU_SEQ_STANDBY_REC   = MPU_MOD_STANDBY_REC,
   MPU_SEQ_START_REC     = MPU_MOD_START_MIDI|MPU_MOD_START_REC,
   MPU_SEQ_CONT_REC      = MPU_MOD_CONT_MIDI|MPU_MOD_START_REC,
   MPU_SEQ_STOP_REC      = MPU_MOD_STOP_MIDI|MPU_MOD_STOP_REC,

   MPU_SEQ_START_OVERDUB = MPU_SEQ_START_ALL,
   MPU_SEQ_CONT_OVERDUB  = MPU_MOD_CONT_MIDI|MPU_MOD_START_PLAY|MPU_MOD_START_REC,
   MPU_SEQ_STOP_OVERDUB  = MPU_SEQ_STOP_ALL
};

/* All switches below can *ONLY* be escaped with an MPU_CMD_RESET!!! */
enum MPU_INIT_SWITCH
{
   MPU_INI_NOTES_OFF_ALL  = 0x30,
   MPU_INI_NO_REAL_TIME   = 0x32,
   MPU_INI_ALL_THRU_OFF   = 0x33,
   MPU_INI_TIMING_BYTE_ON = 0x34, /* Send dummy leading time byte (0x00) when in 
                                   * DATA_IN STOP mode */
   MPU_INI_MODE_MESS_ON   = 0x35, /* Don't mask MIDI IN to HOST */
   MPU_INI_EXL_THRU_ON    = 0x37, /* Send all exclusive msg from IN to OUT */
   MPU_INI_COMMON_TO_HOST = 0x38, /* Send System Common msg to HOST */
   MPU_INI_REAL_TIME      = 0x39, /* Send REAL TIME msg to HOST */

   /* Set MPU in dumb UART MODE, only recognizing MPU_CMD_RESET (0xFF)
    * Note: When issuing an MPU_CMD_RESET to escape UART mode, *NO* ACK will be
    * sent from MPU! */
   MPU_INI_UART_MODE      = 0x3F
};

/* These COMMANDs can be issued to SET MPU states, modes etc. */
enum MPU_SET
{
   MPU_SET_TIME_BASE     = 0xC2,

   /* SET_TIME_BASE (0xc0) *MUST* be added with 0x00 thru 0x06,
    * Selecting one of the following clock resolutions:
    *
    * 0: Time base  48 per beat
    * 1: Time base  72 per beat
    * 2: Time base  96 per beat
    * 3: Time base 120 per beat
    * 4: Time base 144 per beat
    * 5: Time base 168 per beat
    * 6: Time base 192 per beat */

   /* The following values *MUST* be or'd with desired MIDI CHANNEL */
   MPU_SET_CH_REF_TBL_A  = 0x40,
   MPU_SET_CH_REF_TBL_B  = 0x50,
   MPU_SET_CH_REF_TBL_C  = 0x60,
   MPU_SET_CH_REF_TBL_D  = 0x70,

   /* METRONOME actions */
   MPU_SET_METRO_ON      = 0x83, /* Without accents */
   MPU_SET_METRO_OFF     = 0x84,
   MPU_SET_METRO_ACC_ON  = 0x85, /* (!) With accents */

   /* The following *MUST* be followed by additional DATA to MPU ("args") */
   MPU_SET_IMMEDIATE_FIRST = 0xE0,
   MPU_SET_TEMPO	 = 0xE0,
   MPU_SET_REL_TEMPO	 = 0xE1,
   MPU_SET_GRADUATION    = 0xE3, /* Set tempo graduation rate */

   /* The following are used to set misc. CLOCK conversions */
   MPU_SET_MIDI_METRO    = 0xE4, /* Set MIDI CLOCKs per METRONOME BEEP */
   MPU_SET_METRO_MEAS    = 0xE6, /* Set Beat per measure */
   MPU_SET_INT_HOST      = 0xE7, /* Set INTERNAL CLOCKS per CLOCK TO HOST */
   MPU_SET_TRACK_ACTIVE  = 0xEC, /* followed by bitmaped byte */
   MPU_SET_SEND_PLAY_CNT = 0xED,
   MPU_SET_CH_1_8        = 0xEE, /* Bitmap of accepted channels 1 thru 8 */
   MPU_SET_CH_9_16       = 0xEF,  /* Bitmap of accepted channels 9 thru 16 */
   MPU_SET_IMMEDIATE_LAST = 0xEF
};

/* FSK are only implemented on "true" MPU-IPC / MPU-401 interfaces!
 * I suggest not to use this feature unless *absolutely* needed!!! */
enum MPU_CLOCK
{
   MPU_CLK_INT_CLOCK    = 0x80,
   MPU_CLK_FSK_CLOCK    = 0x81, /* (!) Frequency Shifted Key - TAPE SYNC */
   MPU_CLK_MIDI_CLOCK   = 0x82,

   MPU_CLK_FSK_TO_INT   = 0x92, /* Resolution is INTERNAL TIMEBASE */
   MPU_CLK_FSK_TO_MIDI  = 0x93  /* Resolution is MIDI (24 per beat) */
};

/* switches are values for OFF, value + 1 is ON */
enum MPU_SWITCH
{
   MPU_SWI_BENDER        = 0x86,
   MPU_SWI_MIDI_THRU     = 0x88,
   MPU_SWI_DATA_IN_STOP  = 0x8A,
   MPU_SWI_SEND_MEAS_END = 0x8C,
   MPU_SWI_CONDUCTOR     = 0x8E,

   MPU_SWI_REAL_TIME_AFF = 0x90,

   MPU_SWI_CLK_TO_HOST   = 0x94,
   MPU_SWI_EXL_TO_HOST   = 0x96,

   MPU_SWI_CHAN_REF_A    = 0x98, /* Channel reference table A */
   MPU_SWI_CHAN_REF_B    = 0x9A, /* Channel reference table B */
   MPU_SWI_CHAN_REF_C    = 0x9C, /* Channel reference table C */
   MPU_SWI_CHAN_REF_D    = 0x9E  /* Channel reference table D */
};

/* This is the DATA that the HOST can receive from the MPU, */
enum MPU_MESSAGE
{
   MPU_MSG_MAX_TIME      = 0xEF,

   MPU_MSG_TIME_OVERFLOW = 0xF8,
   MPU_MSG_CONDUCTOR_REQ = 0xF9,
   MPU_MSG_DATA_END	 = 0xFC,
   MPU_MSG_ALL_END	 = 0xFC,
   MPU_MSG_CLOCK_TO_HOST = 0xFD,
   MPU_MSG_ACKNOWLEDGE   = 0xFE,
   MPU_MSG_MIDI_SYS_MSG  = 0xFF
};

/* This is the DATA that the HOST can receive during recording */
enum MPU_MARK
{
   MPU_MRK_NO_OP    = 0xF8,
   MPU_MRK_MEASURE  = 0xF9,
   MPU_MRK_DATA_END = 0xFC
};

/* Signal MPU that HOST WANT TO SEND DATA */
enum MPU_SEND
{
   /* SND_DATA (0xD0) *MUST* be or'd with MPU_TRACK[n], 1 <= n <= 8
    * May NOT be issued on active tracks!!!
    * e.g. MPU_SND_DATA|MPU_TRACK[5] signals MPU that DATA will be sent
    * to TRACK number 5 by HOST */
   MPU_SND_DATA  = 0xD0,
   MPU_SND_SYSEX = 0xDF
};

/* These REQUESTs can either be send/received to/from MPU */
enum MPU_REQUEST
{
   /* The following can only be SEND to MPU */
   MPU_REQ_PLAY_CNT = 0xA0,

   /* REQ_PLAY_CNT (0xA0) *MUST* by or'd with MPU_TRACK[n], 1 <= n <= 8
    *
    * i.e. MPU_REQ_PLAY_CNT|MPU_TRACK[3] request play counter for MPU TRACK 3
    * (do NOT confuse tracks with MIDI channels !!!) */

   MPU_REQ_REC_CNT    = 0xA0, /* RECORD counter is cleared after request! */
   MPU_REQ_VERSION    = 0xAC,
   MPU_REQ_REVISION   = 0xAD,
   MPU_REQ_TEMPO      = 0xAF,

   /* The following can only be REVEICED from MPU */
   MPU_REQ_TRK        = 0xF0,

   /* REQ_TRK (0xF0) send by MPU is or'd with TRACK number 0x00 thru 0x07
    * (internally TRACK id's are zerobased!)
    *
    * i.e. MPU_REQ_TRK == 0xF3 request TRACK DATA for MPU TRACK 3
    * (do NOT confuse tracks with MIDI channels !!!) */

   MPU_REQ_CONDUCTOR  = 0xF9
};

/* MPU Clear commands. Normally used by MPU_Reset() */
enum MPU_CLEAR
{
    MPU_CLR_REL_TEMP = 0xB1,
    MPU_CLR_PLAY_CNT = 0xB8,
    MPU_CLR_PLAY_MAP = 0xB9,
    MPU_CLR_REC_CNT  = 0xBA,
    MPU_CLR_RESET    = 0xFF
};

#endif /* __MPU401_H */

/*** End of File **************************************************************/
