/****************************************************************************** 
 *
 * File:        mpuinfo.c
 *
 *              Get MPU-401 information
 *
 * Version:     1.2
 * Date:        1994-12-01
 *
 * Project:     Roland MPU-401 Device driver for Linux 1.1.64 and higher.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "configure.h"
#include "mpuioctl.h"
#include "mpu401.h"

static void
usage(int exitcode)
{
  printf("MPU-401 Device driver information 1.2\n");
  printf("Copyright (c) 1994 Kim Burgaard\n");
  printf("\n");
  printf("SYNOPSIS:\n");
  printf("\n");
  printf("   mpuinfo [options]\n");
  printf("\n");
  printf("OPTIONS:\n");
  printf("\n");
  printf("   -s <n> -dgh --setdebug <n> --dump --getdebug --help\n");
  printf("\n");
  printf("   -s --setdebug <n> Set debug information level to n, where\n");
  printf("                     n is between 0 (disable) and 6 (full).\n");
  printf("   -g --getdebug     Display current debug level\n");
  printf("   -d --dump         Tell the driver to dump detailed information\n");
  printf("                     to the kernel log.\n");
  printf("\n");
  printf("   -h --help         This help text.\n");
  printf("\n");
  printf("DESCRIPTION:\n");
  printf("\n");
  printf("   Tool used to extract driver/device information and to control\n");
  printf("   driver debug facilities\n");
  printf("\n");
  printf("AUTHORS:\n");
  printf("\n");
  printf("   Kim Burgaard, <burgaard@daimi.aau.dk>\n");
  printf("\n");
  printf("   Bugs, comments and/or suggestions to burgaard@daimi.aau.dk\n");
  printf("\n");
  exit(exitcode);
}

static char
isoption(char *s)
{
  if ( !s || !strlen(s) || s[0] != '-' ) return 0;
  return 1;
}

static char
islongoption(char *s)
{
  if ( !s || strlen(s) < 3 || s[0] != '-' || s[1] != '-' ) return 0;
  return 1;
}

static int setdebug = -1;
static int getdebug = 0;
static int dump = 0;

static int
parsearg(int argc, char *argv[])
{
  int i, j, doskip;

  i = 1;
  while (i < argc)
    {
      if ( islongoption(argv[i]) )
	{
	  if ( !strcmp("--setdebug", argv[i]) )
	    {
	      if (setdebug != -1)
		{
		  fprintf(stderr, "Amigeous option %s\n", argv[i]);
		  return 1;
		}
	      else
		{
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Argument expected\n");
		      return 1;
		    }
		  else if (sscanf(argv[i], "%i", &setdebug))
		    {
		      if (setdebug < 0 || setdebug > 6)
			{
			  fprintf(stderr, "Argument out of range\n");
			  return 1;
			};
		    }
		  else
		    {
		      fprintf(stderr, "Invalid argument %s\n", argv[i]);
		      return 1;
		    };
		};
	    }
	  else if ( !strcmp("--dump", argv[i]) )
	    dump = 1;
	  else if ( !strcmp("--getdebug", argv[i]) )
	    getdebug = 1;
	  else if ( !strcmp("--help", argv[i]) )
	    usage(0);
	  else
	    {
	      fprintf(stderr, "Unknown option `%s'\n", argv[i]);
	      fprintf(stderr, "Try `mpuinfo --help' for more information\n");
	      return 1;
	    };
	}
      else if ( isoption(argv[i]) )
	{
	  doskip = 0;
	  for (j = 1; j < strlen(argv[i]); j++)
	    {
	      switch (argv[i][j])
		{
		case 's':
		  if (setdebug != -1)
		    {
		      fprintf(stderr, "Amigeous option %s\n", argv[i]);
		      return 1;
		    }
		  else
		    {
		      i++;
		      if (!(i < argc))
			{
			  fprintf(stderr, "Argument expected\n");
			  return 1;
			}
		      else if (sscanf(argv[i], "%i", &setdebug))
			{
			  if (setdebug < 0 || setdebug > 6)
			    {
			      fprintf(stderr, "Argument out of range\n");
			      return 1;
			    };
			}
		      else
			{
			  fprintf(stderr, "Invalid argument %s\n", argv[i]);
			  return 1;
			};
		      doskip = 1;
		      i--;
		    };
		  break;
		case 'g':
		  getdebug = 1;
		  break;
		case 'd':
		  dump = 1;
		  break;
		case 'h':
		  usage(0);
		  break;
		default:
		  fprintf(stderr, "Unknown option `-%c'\n", argv[i][j]);
		  fprintf(stderr, "Try `mpuinfo --help' for more information\n");
		  return 1;
		};
	    };
	  if (doskip) i++;
	};
      i++;
    };
  return 0;
}

int
main(int argc, char *argv[])
{
  struct mpu_status mpu;
  int mpudev = -1;
  int i = 0;

  if (parsearg(argc, argv)) return 1;
  
  mpudev = open("/dev/mpu401stat", O_RDWR);

  if (mpudev != -1)
    {
      if ( !(i = ioctl(mpudev, MPUIOC_GET_STATUS, &mpu)) )
	{
	  printf("MODULAR MPU-401 DEVICE DRIVER version %hd.%hd%c\n",
		 mpu.drv_version, mpu.drv_subversion, mpu.drv_release);
	  printf("MPU-401 version %hd.%hd%c at IRQ %hd I/O port 0x%03x\n",
		 mpu.mpu_version, mpu.mpu_subversion, mpu.mpu_revision,
		 mpu.irq, mpu.ioport);

	  if (setdebug != -1 || getdebug || dump) printf("\n");

	  if (setdebug != -1)
	    {
	      ioctl(mpudev, MPUIOC_SET_DEBUG_LVL, &setdebug);
	      printf("Driver debug level set to %d\n", setdebug);
	    };
	  if (getdebug)
	    {
	      ioctl(mpudev, MPUIOC_GET_DEBUG_LVL, &getdebug);
	      printf("Current driver debug level is %d\n", getdebug);
	    };
	  if (dump)
	    {
	      ioctl(mpudev, MPUIOC_DEBUG_DUMP, 0);
	      printf("Detailed driver information dumped on kernel log\n");
	    };
	}
      else
	fprintf(stderr, "Could not get driver information.\n");

      close(mpudev);

      return i;
    };

  perror("Could not open `/dev/mpu401stat\'");
  return -1;
}

/*** END OF FILE **************************************************************/
