/****************************************************************************** 
 *
 * File:        smfplay.c
 *
 *              Standard MIDI file player.
 *
 * Version:     0.7
 * Date:        1994-11-17
 *              1994-12-03
 *              1994-12-04
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MPU-401 Device driver for Linux 1.1.64 and higher.
 *
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 *
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Description: This is a ``little'' example application that utilizes my Roland
 *              MPU-401 Device Driver.
 *
 *              It should give a pretty clear picture of what you can do with
 *              /dev/mpu401data if you ignore all the MIDI file parsing.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "mpu401.h"
#include "midiconst.h"
#include "miditypes.h"
#include "midiqueue.h"
#include "midiprint.h"
#include "midifile.h"
#include "mpuioctl.h"

/*
 * # of events to transfer to driver per demand.
 *
 * If you think replay gets sloppy sometimes, especially if a lot
 * of notes is being played at once, try make this amount bigger.
 * Recommended range is 50 to 200 events. There's no point in making
 * it bigger than 200 events (except for momentarily vasting kernel-
 * memory! ;-)
 */
#define PACK_SIZE 100

/*** GLOBAL VARIABLES *********************************************************/

static char skipcheck = 0;
static char noplay    = 0;
static char loop      = 0;
static char dorandom  = 0;
static char quiet     = 0;
static char verbose   = 0;

static int  mpudev    = 0;
static int  filec     = 0;

static char didplay   = 0;
static char dostop    = 0;

static midi_file        midifile;
static midi_file_timing timing;
static midi_file_data   queues;

static float totaltime = 0.0; /* in seconds */

/*** MEMORY HANDLING **********************************************************/

static char *
my_malloc(long size)
{
  static char *tmp = NULL;

  if ( !(tmp = malloc(size)) )
    {
      fprintf(stderr, "Out of memory!\n");
      exit(-3);
    };

  return tmp;
}

/*** ARGUMENT PARSING *********************************************************/

static void
usage(int exitcode)
{
  printf("Standard MIDI File Player version 0.7\n");
  printf("Copyright (c) 1994 Kim Burgaard\n");
  printf("\n");
  printf("SYNOPSIS:\n");
  printf("\n");
  printf("   smfplay [options] filename...\n");
  printf("\n");
  printf("OPTIONS:\n");
  printf("\n");
  printf("   -snlrqvh --skipcheck --noplay --loop --random --quiet\n");
  printf("   --verbose --help\n");
  printf("\n");
  printf("   -s --skipcheck  Disables total checking of files\n");
  printf("   -n --noplay     Do not play files - just check them\n");
  printf("   -l --loop       Loop replaying forever\n");
  printf("   -r --random     Play files in random order\n");
  printf("   -q --quiet      No output to console.\n");
  printf("   -v --verbose    Verbose output to console. Two or more\n");
  printf("                   produces increased output\n");
  printf("   -h --help       This help text.\n");
  printf("\n");
  printf("DESCRIPTION:\n");
  printf("\n");
  printf("   This is a Standard MIDI File player for the Roland MPU-401\n");
  printf("   MIDI Device driver also written by Kim Burgaard.\n");
  printf("   Supports General MIDI/Roland GS format.\n");
  printf("\n");
  printf("   The replayer can safely be stopped with either `kill -TERM <pid>'\n");
  printf("   or with ^c on the console - without getting hanging notes (if you\n");
  printf("   have a GM/GS compliant synth that is!).\n");
  printf("\n");
  printf("AUTHORS:\n");
  printf("\n");
  printf("   Kim Burgaard, <burgaard@daimi.aau.dk>\n");
  printf("\n");
  printf("   Bugs, comments and/or suggestions to burgaard@daimi.aau.dk\n");
  printf("\n");
  exit(exitcode);
}

static char
isfilename(char *s)
{
  if ( !s || !strlen(s) || s[0] == '-' ) return 0;
  return 1;
}

static char
isoption(char *s)
{
  if ( !s || !strlen(s) || s[0] != '-' ) return 0;
  return 1;
}

static char
islongoption(char *s)
{
  if ( !s || strlen(s) < 3 || s[0] != '-' || s[1] != '-' ) return 0;
  return 1;
}

static int
parsearg(int argc, char *argv[])
{
  int i, j;
  if (argc < 2) usage(1);

  for (i = 1; i < argc; i++)
    if ( islongoption(argv[i]) )
      {
	if ( !strcmp("--skipcheck", argv[i]) )
	  skipcheck = 1;
	if ( !strcmp("--noplay", argv[i]) )
	  noplay = 1;
	else if ( !strcmp("--verbose", argv[i]) )
	  {
	    if (!quiet)
	      {
		verbose++;
		if (verbose > MIDI_PRINT_ALL)
		  fprintf(stderr, "Repeating %s more than %d times will have no effect.\n",
			  argv[i], MIDI_PRINT_ALL);
	      }
	    else fprintf(stderr, "Ambigeous option %s with -v or --verbose\n", argv[i]);
	  }
	else if ( !strcmp("--quiet", argv[i]) )
	  {
	    quiet = 1;
	    if (verbose) fprintf(stderr, "Ambigeous option %s with -q or --quiet\n", argv[i]);
	    verbose = 0;
	  }
	else if ( !strcmp("--loop", argv[i]) )
	  {
	    if (!noplay) loop = 1;
	    else fprintf(stderr, "Ambigeous option %s with -n or --noplay\n", argv[i]);
	  }
	else if ( !strcmp("--random", argv[i]) )
	  {
	    if (!noplay) dorandom = 1;
	    else fprintf(stderr, "Ambigeous option %s with -n or --noplay\n", argv[i]);
	  }
	else if ( !strcmp("--help", argv[i]) )
	  usage(0);
	else
	  {
	      fprintf(stderr, "Unknown option `%s' :\n\n", argv[i]);
	      fprintf(stderr, "Try `smfplay --help' for more information\n");
	      exit(-1);
	  };
      }
    else if ( isoption(argv[i]) )
      {
	for (j = 1; j < strlen(argv[i]); j++)
	  {
	    switch (argv[i][j])
	      {
	      case 's':
		skipcheck = 1;
		break;
	      case 'n':
		noplay = 1;
		break;
	      case 'v':
		if (!quiet)
		  {
		    verbose++;
		    if (verbose > MIDI_PRINT_ALL)
		      fprintf(stderr, "Repeating %s more than %d times will have no effect.\n",
			      argv[i], MIDI_PRINT_ALL);
		  }
		else fprintf(stderr, "Ambigeous option %s with -v or --verbose\n", argv[i]);
		break;
	      case 'q':
		quiet = 1;
		if (verbose) fprintf(stderr, "Ambigeous option %s with -q or --quiet\n", argv[i]);
		verbose = 0;
		break;
	      case 'l':
		if (!noplay) loop = 1;
		else fprintf(stderr, "Ambigeous option %s with -n or --noplay\n", argv[i]);
		break;
	      case 'r':
		if (!noplay) dorandom = 1;
		else fprintf(stderr, "Ambigeous option %s with -n or --noplay\n", argv[i]);
		break;
	      case 'h':
		usage(0);
		break;
	      default:
		fprintf(stderr, "Unknown option `-%c' :\n\n", argv[i][j]);
		fprintf(stderr, "Try `smfplay --help' for more information\n");
		exit(-1);
	      };
	  };
      }
    else
      filec++;

  if (!filec)
    { 
      fprintf(stderr, "Try `smfplay --help' for more information\n");
      exit(-1);
    };

  return 0;
}

/*** MIDI FILE HANDLING *******************************************************/

struct name_e
{
  struct name_e *prev;
  struct name_e *next;
  char *name;
  float playtime;
};

typedef struct name_e name_e;

struct name_list
{
  name_e *first;
  name_e *last;
  long count;
};

struct name_list names = { NULL, NULL, 0 };

static long
name_list_add(char *name, float playt)
{
  name_e *tmp = NULL;

  if (!name) return names.count;

  tmp = (name_e *)my_malloc(sizeof(name_e));
  tmp->next = NULL;
  tmp->name = name;
  tmp->playtime = playt;

  if (!names.count)
    {
      tmp->prev = NULL;
      names.first = names.last = tmp;
    }
  else
    {
      names.last->next = tmp;
      tmp->prev = names.last;
      names.last = tmp;
    };

  names.count++;

  return names.count;
}

static long
name_list_del(name_e *nm)
{
  if (!nm) return names.count;

  if (names.count > 1)
    {
      if (nm->prev) nm->prev->next = nm->next;
      if (nm->next) nm->next->prev = nm->prev;

      if (nm == names.first)
	{
	  names.first = names.first->next;
	}
      else if (nm == names.last)
	{
	  names.last = names.last->prev;
	};
    };

  free(nm);
  
  return --names.count;
}

static void name_list_clear()
{
  name_e *cur = names.first;
  name_e *tmp = NULL;

  while ( (tmp = cur) && names.count-- )
    {
      cur = cur->next;
      free(tmp);
    };

  names.first = names.last = NULL;
  names.count = 0;
}

static int
mpuaction()
{
  struct mpu_demand demand;
  midi_event *event = NULL;
  long total = 0;
  int i = 0;

  total = queues.voice.count;

  ioctl(mpudev, MPUIOC_RESET, 0); /* play it safe */

  if (!quiet)
    {
      if (midifile.title && midifile.copyright)
        {
          printf("[ %s, %s ] ", midifile.title, midifile.copyright);
          free(midifile.title);
          free(midifile.copyright);
        }
      else if (midifile.copyright)
        {
          printf("[ %s, %s ] ", midifile.name, midifile.copyright);
          free(midifile.copyright);
        }
      else if (midifile.title)
        {
          printf("[ %s ] ", midifile.title);
          free(midifile.title);
        }
      else
	printf("[ %s ]", midifile.name);
    };

  midifile.title = midifile.copyright = NULL;

  /* usually there is not very many meta and sysex events */
  while ( (event = midi_queue_get(&queues.sysex)) )
    {
      if (event->type == MIDI_TYP_SYSEX ||
	  (event->type == MIDI_TYP_META && (event->data[0]&0xff) == MID_META_SET_TEMPO))
	{
	  ioctl(mpudev, MPUIOC_PUT_EVENT, event);
	};
      midi_event_free(event);
    };
  while ( (event = midi_queue_get(&queues.meta)) )
    {
      ioctl(mpudev, MPUIOC_PUT_EVENT, event);
      midi_event_free(event);
    };
  
  /* preload 2 * PACK_SIZE voice events */
  for (i = 0; i < 2*PACK_SIZE && queues.voice.count > 0; i++)
    {
      if ( !(event = midi_queue_get(&queues.voice)) ) break;
      ioctl(mpudev, MPUIOC_PUT_EVENT, event);
      midi_event_free(event);
    };
  
  fflush(NULL);
  
  ioctl(mpudev, MPUIOC_SET_DIVISION, &timing.division);
  ioctl(mpudev, MPUIOC_INIT_PLAY, 0);

  while ( queues.voice.count )
    {
      demand.voice = (queues.voice.count > PACK_SIZE) ? PACK_SIZE : queues.voice.count;
      demand.control = 0;	/* disable control queue demands */

      i = ((((float)total - (float)queues.voice.count)*100.0)/(float)total);
      if (!quiet) printf("%02d%%", (i < 100 && i >= 0) ? i : 0);
      fflush(NULL);		/* force update */

      if (ioctl(mpudev, MPUIOC_SLEEP2DEMAND, &demand)) return -EINVAL;
      if (!quiet) printf("\b\b\b");

      if (dostop) return 0;
      if (demand.voice == -1) break;

      for (i = 0; i < PACK_SIZE && queues.voice.count > 0; i++)
	{
	  if ( !(event = midi_queue_get(&queues.voice)) ) break;
	  ioctl(mpudev, MPUIOC_PUT_EVENT, event);
	  midi_event_free(event);
	};
    };
  
  /* waiting is all ;-) */
  if (demand.voice != -1) ioctl(mpudev, MPUIOC_SLEEP2PLAY_END, 0);
  
  didplay = 1;
  
  sleep(1); /* Don't cut the last note played abrutly */
  
  ioctl(mpudev, MPUIOC_STOP_ALL, 0);

  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);

  ioctl(mpudev, MPUIOC_GM_ON, 0);    /* ``Clean up'' the synth. */
  ioctl(mpudev, MPUIOC_GS_RESET, 0); /* Even more if it's a Roland */

  if (!quiet) printf("\n");

  return 0;
}

static char buffer[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

static int
load(char *nm, char doplay)
{
  char *tn = nm;
  int i = 0;

  if ( !(midifile.file = fopen(nm, "rb")) )
    {
      fprintf(stderr, "Could not open `%s': ", nm);
      perror("");
      return -EINVAL;
    };

  if ( (tn = strrchr(nm, '/')) ) nm = ++tn; /* strip leading path */
  midifile.name = nm;

  if (skipcheck)
    {
      i = 0;

      if (
	  fread(&buffer[0], 10, 1, midifile.file) != 1 ||
	  memcmp(&buffer[0], "MThd\0\0\0\006", 8) != 0
	  )
	{
	  fprintf(stderr, "`%s' is not a Standard MIDI File\n", midifile.name);
	  i = -EINVAL;
	}
      else if ( (buffer[8]&0xff) || (buffer[9]&0xff) > 2 )
	{
	  fprintf(stderr, "MIDI File format 2 or higher not supported!\n");
	  i = -EINVAL;
	}
      else if ( ferror(midifile.file) )
	{
	  fprintf(stderr, "Error while reading `%s': ", midifile.name);
	  perror("");
	  i = ferror(midifile.file);
	};
      
      fclose(midifile.file);
      return i;
    };

  midifile.title = midifile.copyright = NULL;
  
  midifile.format = 0;
  midifile.tracks = 0;
  timing.division = 96;
  timing.newdivision = 120;
  timing.time = 0.0;
  
  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);
  
  queues.dovoice = queues.dosysex = queues.dometa = doplay;
  
  if ( (i = midi_load(&midifile, &timing, &queues, verbose)) == 0 && doplay && !noplay) 
    {
      mpuaction();
    }
  else if ( ferror(midifile.file) )
    {
      fprintf(stderr, "Error while reading `%s': ", midifile.name);
      perror("");
      i = ferror(midifile.file);
    };
  
  fclose(midifile.file);
  return i;
}

static long
validatefiles(int argc, char *argv[])
{
  char *tm = NULL;
  byte h = 0;
  byte m = 0;
  byte s = 0;
  int i = 0;

  totaltime = 0;

  if (!quiet) printf("Checking files...\n");

  for (i = 1; i < argc; i++)
    {
      if (!isfilename(argv[i])) continue;
      if (load(argv[i], 0) == 0)
	{
	  name_list_add(argv[i], timing.time);

	  if (skipcheck) continue;
	  totaltime += timing.time;
	  
	  if (midifile.title)
	    tm = midifile.title;
	  else
	    {
	      if ( (tm = strrchr(argv[i], '/')) ) tm++;
	      else tm = argv[i];
	    };
	  
	  h = timing.time / (60*60);
	  m = (timing.time / 60) - (h*60);
	  s = ((byte)timing.time % 60);
	  
	  if (!quiet)
	    {
	      printf("%02hd:%02hd:%02hd  ", h, m, s);

	      if (midifile.title && midifile.copyright)
		{
		  printf("[ %s, %s ]\n", midifile.title, midifile.copyright);
		  free(midifile.title);
		  free(midifile.copyright);
		}
	      else if (midifile.copyright)
		{
		  printf("[ %s, %s ]\n", midifile.name, midifile.copyright);
		  free(midifile.copyright);
		}
	      else if (midifile.title)
		{
		  printf("[ %s ]\n", midifile.title);
		  free(midifile.title);
		}
	      else
		printf("[ %s ]\n", midifile.name);
	    };

	  midifile.title = midifile.copyright = NULL;

	  midi_queue_flush(&queues.voice);
	  midi_queue_flush(&queues.sysex);
	  midi_queue_flush(&queues.meta);
	}
    };
  
  if (names.count > 1 && !skipcheck)
    {
      totaltime += 2 * (names.count - 1); /* sleep(1.05 sec) + loading etc... */ 
      
      h = totaltime / (60*60);
      m = (totaltime / 60) - (h*60);
      s = ((byte)totaltime % 60);
      
      if (!quiet)
	{
	  printf("========\n");
	  printf("%02hd:%02hd:%02hd  Total playing time for %ld files.\n\n",
		 h, m, s, names.count);
	};
    };
  
  return names.count;
}

static long
getrand(long range)
{
  return rand() % range;
}

void
closedown()
{
  if (mpudev != -1)
    {
      dostop = 1;
      ioctl(mpudev, MPUIOC_RESET, 0);
      ioctl(mpudev, MPUIOC_GM_ON, 0);    /* if it's GM compliant it'll shut up */
      ioctl(mpudev, MPUIOC_GS_RESET, 0); /* if it's GS compliant it'll shut even more up */
      close(mpudev);
    };
  printf("\n");
  exit(1);
}

/*** MAIN *********************************************************************/

int
main(int argc, char *argv[])
{
  name_e *old = NULL;
  name_e *cur = NULL;
  name_e *tmp = NULL;
  long i = 0;

  if ( parsearg(argc, argv) ) return 1;

  srandom(time(NULL));
  mpudev = -1;

  /* Prevent hanging sounds if terminated by fx. Ctrl-C */
  signal(SIGINT,  (void(*)()) closedown);
  signal(SIGTERM, (void(*)()) closedown);
  signal(SIGABRT, (void(*)()) closedown);
  signal(SIGQUIT, (void(*)()) closedown);

  midi_queue_reset(&queues.voice);
  midi_queue_reset(&queues.sysex);
  midi_queue_reset(&queues.meta);
  
  if (validatefiles(argc, argv))
    {
      verbose = 0;   /* don't need this anymore! */
      skipcheck = 0; /* don't *want* this anymore! */

      if (!noplay)
	{
	  if ( (mpudev = open("/dev/mpu401data", O_RDWR)) == -1 )
	    {
	      perror("Could not open `/dev/mpu401'");
	      name_list_clear();
	      exit(-3);
	    };

	  /* make the list cyclic */
	  names.first->prev = names.last;
	  names.last->next = names.first;
	  
	  cur = names.first;
	  
	  if (dorandom)
	    {
	      i = getrand(names.count);
	      while (i--) cur = cur->next;
	    };
	  
	  while (names.count && cur)
	    {
	      if ( load(cur->name, 1) || !loop )
		{
		  tmp = cur;
		  cur = cur->next;
		  name_list_del(tmp);
		  continue;
		};
	      
	      if (dorandom && names.count > 1)
		{
		  old = cur;
		  i = getrand(names.count);
		  while (i--) cur = cur->next;
		  if (cur->next == old) cur = cur->next;
		}
	      cur = cur->next;
	    };
	  
	  ioctl(mpudev, MPUIOC_GS_RESET, 0);
	  close(mpudev);
	};
      
      name_list_clear();
    }
  else
    fprintf(stderr, "\nNo files to play.\n");

  return 0;
}

/*** END OF FILE **************************************************************/
