/*
 * linux io
 */

#include "glib.h"
#include "sblast.h"
#ifdef GUS
#include "ultra.h"
#endif
#include "midicode.h"

#ifndef NOSOUND
#include <linux/soundcard.h>
#endif
#include <fcntl.h>
#include <sys/time.h>

#ifdef SBLAST
extern int sb_drum_note;
extern int sb_drum_dur;
extern int sb_transpose;
extern int sb_other_voice;
#endif

#ifdef OPL3
extern int o3_drum_note;
extern int o3_drum_dur;
extern int o3_transpose;
extern int o3_other_voice;
#endif

/*
 * some of the following code adapted from:
 * 
 * fmplay and sbiset, by Hannu Savolainen (hsavolai#cs.helsinki.fi)
 * Modifications, bugfixes, and ANSIfied by Rob Hooft (hooft#chem.ruu.nl)
 *
 */
int seq_fd;
static int nrsynths, nrmidis;
int ext_dev = -1, gus_dev = -1, sb_dev = -1;
static unsigned char sbbuf[408];
static int sbptr = 0;
static unsigned sb_time = 0;
static int o3_mode = 0;
static int max_cells[3];
static int cell_pitch[3][32];

static int seq_is_open = 0;
static int tried_it_already = 0;

#ifdef GUS
extern int gus_fixed_key;
void gus_vol(int chan, int prog, int cell, int vol);
void card_expression(int card, int chan, int cell);
void card_main_volume(int card, int chan, int cell);
void gus_max_voices(int num);
void gus_bend(int cell);
#ifndef GUS_NUM_VOICES
#define GUS_NUM_VOICES 32
#endif
#endif

int
opensb()
{
	int i;
	struct synth_info fm_info;
	void sb_resync();

	if (seq_is_open) return(1);
	if (tried_it_already) return(0);
	tried_it_already = 1;
#ifndef NOSOUND
#ifdef MIDIIN
	if ((seq_fd=open("/dev/sequencer", O_RDWR, 0))==-1) {
#else
	if ((seq_fd=open("/dev/sequencer", O_WRONLY, 0))==-1) {
#endif
		return(0);
	}

	if (ioctl (seq_fd, SNDCTL_SEQ_NRSYNTHS, &nrsynths) == -1) {
		return(0);
	}
	if (ioctl (seq_fd, SNDCTL_SEQ_NRMIDIS, &nrmidis) == -1) {
		return(0);
	}
	if (nrmidis > 0) ext_dev = nrmidis - 1; /* 0 for GUS midi interface */

	for (i = 0; i < nrsynths; i++) {
		fm_info.device = i;
		if (ioctl (seq_fd, SNDCTL_SYNTH_INFO, &fm_info) == -1) {
			return(0);
		}

		if (fm_info.synth_type == SYNTH_TYPE_SAMPLE
			&& fm_info.synth_subtype == SAMPLE_TYPE_GUS) {
				gus_dev = i;
				max_cells[i] = fm_info.nr_voices;
		}
		else if (fm_info.synth_type == SYNTH_TYPE_FM) {
				sb_dev = i;
				max_cells[i] = fm_info.nr_voices;
		}
	}
#ifdef GUS
	if (gus_dev >= 0) {
		if (GUS_NUM_VOICES != 24) gus_max_voices(GUS_NUM_VOICES);
		max_cells[gus_dev] = GUS_NUM_VOICES;
	}
#endif
	if (sb_dev >= 0) {
		int n = sb_dev;
		ioctl(seq_fd, SNDCTL_FM_4OP_ENABLE, &n);
		max_cells[sb_dev] = 6;
	}

	if (sb_dev >= 0) for (i = 0; i < max_cells[sb_dev]; i++)
		cell_pitch[sb_dev][i] = -1;
	if (gus_dev >= 0) for (i = 0; i < max_cells[gus_dev]; i++)
		cell_pitch[gus_dev][i] = -1;

	seq_is_open = 1;
	sb_resync();
	return(1);
#else
	seq_is_open = 0;
	return(0);
#endif
}

closesb()
{
#ifndef NOSOUND

	if (!seq_is_open) return(1);
	(void) ioctl(seq_fd, SNDCTL_SEQ_RESET, 0);
	close(seq_fd);
#endif
	gus_dev = -1;
	sb_dev = -1;
	ext_dev = -1;
	seq_is_open = 0;
	tried_it_already = 0;
	return(1);
}

void
sbflush(void)
{
	if (!sbptr) return;

#ifndef NOSOUND
	if (write(seq_fd, sbbuf, sbptr) == -1) {
		perror("/dev/sequencer");
		exit(-1);
	}
#endif

	sbptr=0;
}

void
sbwrite(char *msg)
{
	if (sbptr>400) sbflush();

	memcpy(&sbbuf[sbptr], msg, 4);
	sbptr +=4;
}

void
sqwrite(char *msg)
{
	if (sbptr>400) sbflush();

	memcpy(&sbbuf[sbptr], msg, 8);
	sbptr +=8;
}

struct timeval tv;
struct timezone tz;
unsigned epoch = 0;
void midisync()
{
#ifndef NOSOUND
	unsigned jiffies;

	gettimeofday (&tv, &tz);
	jiffies = tv.tv_sec*100 + tv.tv_usec/10000;
	if (!epoch) epoch = jiffies;
	sb_time = jiffies - epoch;
	jiffies = sb_time;
/*
printf("Time %d:%d\n", (sb_time/100)/60, (sb_time/100)%60);
*/
	jiffies = (jiffies << 8) | SEQ_WAIT;
	sbwrite((char*)&jiffies);
#endif
}

void sb_resync()
{	char buf[4];

	buf[0] = SEQ_SYNCTIMER;
	sbwrite(buf);
	sbflush();
	epoch = sb_time = 0;
}

void midich(char c)
{
#ifndef NOSOUND
	char buf[4];

	if (!opensb()) return;
	if (ext_dev < 0) return;
	buf[0] = SEQ_MIDIPUTC;
	buf[1] = c;
	buf[2] = ext_dev;
	buf[3] = 0;
	sbwrite(buf);
#endif
}
int
find_cell(int card, int pitch)
{
	int i, best = -1, octbest = -1, anote = -1;
	for (i = 0; i < max_cells[card]; i++) {
		if (card == sb_dev && o3_mode && i > 5) break;
		if (cell_pitch[card][i] == pitch) {
			best = i;
			break;
		}
		else if (cell_pitch[card][i] == pitch + 12) octbest = i;
		else if (cell_pitch[card][i] == pitch - 12) octbest = i;
		else if (cell_pitch[card][i] >= 0) anote = i;
	}
	if (best >= 0) return(best);
	if (pitch == -1) return(0);
	if (octbest >= 0) return(octbest);
	if (anote >= 0) return(anote);
	return(0);
}


playsb(chan,pitch,vol,dur,onoff)
{
#ifndef NOSOUND
	int card;
	int other_voice = 0;
	char buf[8];
	unsigned jiffies;
	static int cell = 0;

    if (chan < 1 || chan > 3) return;
    if (!opensb()) return;

    switch(chan) {
	case 1: card = sb_dev; break;
	case 2: card = ext_dev; break;
	case 3: card = gus_dev; break;
    }

    if (card < 0) return;

    if (card == sb_dev) {
#ifdef SBLAST
	if (onoff == -1 && sb_drum_dur) return;
	if (sb_drum_dur) onoff = 0;
	if (sb_drum_note) pitch = sb_drum_note;
	if (sb_transpose) pitch += sb_transpose - 64;
	if (sb_other_voice) other_voice = sb_other_voice;
#endif
#ifdef OPL3
	if (onoff == -1 && o3_drum_dur) return;
	if (o3_drum_dur) onoff = 0;
	if (o3_drum_note) pitch = o3_drum_note;
	if (o3_transpose) pitch += o3_transpose - 64;
	if (o3_other_voice) other_voice = o3_other_voice;
#endif
    }
#ifdef GUS
    else if (card == gus_dev) {
	if (gus_fixed_key) pitch = gus_fixed_key;
    }
#endif
    if (pitch < 0 || pitch > 127) return;

    midisync();

    if (onoff >= 0) {
	if (chan == 2) {
		midich(MIDI_ON_NOTE+0);
		midich(pitch);
		midich(vol);
	}
	else {
		cell = find_cell(card, -1);
		buf[0] = SEQ_EXTENDED;
		buf[1] = SEQ_PGMCHANGE;
		buf[2] = card;
		buf[3] = cell;
#ifdef GUS
		buf[4] = (card==gus_dev)? GUSEDITPROG : SBEDITPROG;
#else
		buf[4] = SBEDITPROG;
#endif
		buf[5] = buf[6] = buf[7] = 0;
		sqwrite(buf);
#ifdef GUS
		if (card == gus_dev) {
			gus_bend(cell);
		}
#endif
#ifdef GUS
    		if (card == gus_dev) {
			/* gus_vol(chan, prog, cell, vol, 0); */
			card_expression(card, 0, cell);
			card_main_volume(card, 0, cell);
    		}
#endif
		buf[0] = SEQ_EXTENDED;
		buf[1] = SEQ_NOTEON;
		buf[2] = card;
		buf[3] = cell;
		buf[4] = cell_pitch[card][cell] = pitch;
		buf[5] = vol;
		buf[6] = buf[7] = 0;
		sqwrite(buf);
		if (card == sb_dev && other_voice) {
			buf[0] = SEQ_EXTENDED;
			buf[1] = SEQ_PGMCHANGE;
			buf[2] = card;
			cell = find_cell(card, -1);
			buf[3] = cell;
			buf[4] = other_voice - 1;
			buf[5] = buf[6] = buf[7] = 0;
			sqwrite(buf);
			buf[0] = SEQ_EXTENDED;
			buf[1] = SEQ_NOTEON;
			buf[2] = card;
			buf[3] = cell;
			buf[4] = cell_pitch[card][cell] = pitch;
			buf[5] = vol;
			buf[6] = buf[7] = 0;
			sqwrite(buf);
		}
	}
    }

    if (!onoff) {
	if (card == ext_dev || card == gus_dev) sb_time += 15 * dur;
#ifdef SBLAST
	else if (sb_drum_dur) sb_time += sb_drum_dur;
#endif
#ifdef OPL3
	else if (o3_drum_dur) sb_time += o3_drum_dur;
#endif
	else if (dur < 4) sb_time += 2 * dur;
	else sb_time += 7 * dur;
	jiffies = sb_time;
	jiffies = (jiffies << 8) | SEQ_WAIT;
	sbwrite((char*)&jiffies);
    }

    if (onoff <= 0) {
	if (chan == 2) {
		midich(MIDI_OFF_NOTE+0);
		midich(pitch);
		midich(vol);
	}
	else {
		cell = find_cell(card, pitch);
		buf[0] = SEQ_EXTENDED;
		buf[1] = SEQ_NOTEOFF;
		buf[2] = card;
		buf[4] = pitch;
		buf[3] = cell;
		buf[5] = vol;
		buf[6] = buf[7] = 0;
		sqwrite(buf);
		cell_pitch[card][cell] = -1;

		if (card == sb_dev && other_voice) {
			buf[0] = SEQ_EXTENDED;
			buf[1] = SEQ_NOTEOFF;
			buf[2] = card;
			buf[4] = pitch;
			cell = find_cell(card, pitch);
			buf[3] = cell;
			buf[5] = vol;
			buf[6] = buf[7] = 0;
			sqwrite(buf);
			cell_pitch[card][cell] = -1;
		}
	}
    }

    sbflush();
#endif
}

#ifdef SBLAST
int
sbvoice(int v, char *buf)
{
#ifndef NOSOUND
	struct sbi_instrument instr;
	int i;

	if (!opensb()) return(-1);
	if (sb_dev < 0) return(-1);
	sbflush();
#ifdef OPL3
	o3_mode = 0;
	o3_drum_note
	= o3_drum_dur
	= o3_transpose
	= o3_other_voice = 0;
#endif
	instr.channel = v;
	instr.device = sb_dev;
	instr.key = FM_PATCH;
	buf[46] |= 0x30;
	for (i=0;i<16;i++) {
		instr.operators[i] = (i+36 < SBVOICESIZE)? buf[i+0x24] : 0;
	}
	if (write(seq_fd, (char*)&instr, sizeof(instr)) == -1) return(-1);
	sb_resync();
	return(0);
#else
	return(-1);
#endif
}
#endif

#ifdef OPL3
int
o3voice(int v, char *buf)
{
/*
differences are that the key field must be initialized to OPL3_PATCH and
the byte offsets between 11 and 21 are used for OPs 3 and 4.
*/
#ifndef NOSOUND
	struct sbi_instrument instr;
	int i, voice_size;

	if (!opensb()) return(-1);
	if (sb_dev < 0) return(-1);
	sbflush();
	o3_mode = 1;
#ifdef SBLAST
	sb_drum_note
	= sb_drum_dur
	= sb_transpose
	= sb_other_voice = 0;
#endif
	instr.channel = v;
	instr.device = sb_dev;
	instr.key = OPL3_PATCH;
	voice_size = 22;
	if ((buf[49]&0x3f) == 0x3f &&
	    (buf[50]&0x3f) == 0x3f ) {
		instr.key = FM_PATCH;
		voice_size = 11;
	}
	else buf[57] |= 0x30;
	buf[46] |= 0x30;
	for (i = 0; i < 32; i++) {
		instr.operators[i] = (i < voice_size)? buf[i+36] : 0;
	}
	if (write(seq_fd, (char*)&instr, sizeof(instr)) == -1) return(-1);
	sb_resync();
	return(0);
#else
	return(-1);
#endif
}
#endif

#ifdef MIDIIN

int midipending()
{	int cnt;
	if (!opensb()) return(0);
	if (ext_dev < 0) return(0);
	if (ioctl(seq_fd, SNDCTL_SEQ_GETINCOUNT, &cnt) == -1) return(0);
	else return(cnt);
}

int midigetc(void)
{	int l, havemidi;
	unsigned char buf[4];

	if (!opensb()) return(0);
	if (ext_dev < 0) return(0);
	havemidi = 0;
	while (!havemidi) {
		if ((l=read(seq_fd, buf, 4))==-1) {
			return(0);
		}
		if (l != 4) {
			fprintf(stderr,"could only read %d bytes\n", l);
			return(0);
		}
		if (buf[0] == SEQ_WAIT) {
/**
			sb_time = buf[1] + (buf[2] << 8) + (buf[3] << 16);
**/
		}
		else if (buf[0] == SEQ_MIDIPUTC) havemidi = 1;
	}
	return(buf[1]);
}

int midigetnb()
{
	if (midipending()) return(midigetc());
	return(-1);
}
/* following 2 routines adapted from midifile.c,
 * by Tim Thompson and M. Czeisperger
 */
static
chanmessage(status,c1,c2)
int status;
int c1, c2;
{
    int chan;

    if (!strcmp(Synthname, "Ultrasound")) chan = 3;
    else chan = 1;

if (c2 < 0 || c2 >127) {
	/* fprintf(stderr, "chanmessage: input vol %d\n", c2);*/
	return;
}

    switch ( status & 0xf0 ) {
    case MIDI_OFF_NOTE:
	playsb(chan,c1,c2,0,-1);
	break;
    case MIDI_ON_NOTE:
	playsb(chan,c1,c2,0,1);
	break;
    case MIDI_POLY_TOUCH:
	break;
    case MIDI_CTRL:
	if (c1 != ALL_NOTES_OFF)	;
	break;
    case MIDI_CH_PROGRAM:
	break;
    case MIDI_TOUCH:
	break;
    case MIDI_BEND:
	break;
    }
}

void midimessage()
{
	/* This array is indexed by the high half of a status byte.  It's */
	/* value is either the number of bytes needed (1 or 2) for a channel */
	/* message, or 0 (meaning it's not  a channel message). */
	static int chantype[] = {
		0, 0, 0, 0, 0, 0, 0, 0,		/* 0x00 through 0x70 */
		2, 2, 2, 2, 1, 1, 2, 0		/* 0x80 through 0xf0 */
	};
	static int c = -1, c1 = -1, c2 = -1;
	static int running = 0;	/* 1 when running status used */
	static int status = 0;		/* status value (e.g. 0x90==note-on) */
	int needed;


	if (c == -1) {
		c = midigetnb();
		if (c == -1) return;
	}
	if ( (c & 0x80) == 0 ) {	 /* running status? */
		if ( status == 0 ) {
			/*mferror("unexpected running status")*/
			c = c1 = c2 = -1;
			return;
		}
		running = 1;
	}
	else {
		status = c;
		running = 0;
	}

	needed = chantype[ (status>>4) & 0xf ];

	if ( needed ) {		/* ie. is it a channel message? */

		if ( running ) c1 = c;
		else if (c1 == -1) {
			c1 = midigetnb();
			if (c1 == -1) return;
		}
		if (c1 & 0x80) {
			c = c1;
			c1 = c2 = -1;
			return;
		}
		if (needed>1) {
			if (c2 == -1) c2 = midigetnb();
			if (c2 == -1) return;
		}
		else c2 = 0;

		if (c2 & 0x80) {
			c = c2;
			c1 = c2 = -1;
			return;
		}

		chanmessage( status, c1, c2 );
		c = c1 = c2 = -1;
	}
	/*else mferror("apparent non-channel message");*/
}

#endif

#ifdef GUS

#include <sys/ultrasound.h>

#define Bit8 0
#define Bit16 4 
#define LoopOff 0
#define LoopOn 8
#define UniDir 0
#define BiDir 16
#define Forw 0
#define Backw 64
#define Up 0
#define Down 64


void
gus_max_voices(int num)
{
	char buf[8];
	buf[0] = SEQ_PRIVATE;
	buf[1] = gus_dev;
	buf[2] = _GUS_NUMVOICES;
	buf[3] = 0;
	*(unsigned short*)&buf[4] = num;
	buf[6] = buf[7] = 0; 
	sqwrite(buf);
}

#define XPN 120
#define MAINV 120
#define GUS_VOLUME 72

#define GUSEXPRAMPMINVOL (11<<8)
#define GUSEXPRAMPMAXVOL (15<<8)
#define GUSMAXVOL (1<<15)

/*
 * Calculate gus volume from note velocity, main volume, expression,
 * and intrinsic patch volume given in patch library.  Expression is
 * multiplied in, so it emphasizes differences in note velocity,
 * while main volume is added in -- I don't know whether this is right,
 * but it seems reasonable to me.  (In the previous stage, main volume
 * controller messages were changed to expression controller messages,
 * if they were found to be used for dynamic volume adjustments, so here,
 * main volume can be assumed to be constant throughout a song.)
 *
 * Intrinsic patch volume is added in, but if over 64 is also multiplied
 * in, so we can give a big boost to very weak voices like nylon
 * guitar and the basses.
 */
unsigned short
gvol(int vel, int mainv, int xpn, int voicev)
{
	int i, m, n, x;

	if (xpn <= 0 || vel == 0) return(0);
/** added **/
	if (xpn < 3) xpn = 3;

	/* A voice volume of 64 is considered neutral, so adjust
	 * the main volume if something other than this neutral
	 * value was assigned in the patch library.
	 */
	x = mainv + 6*(voicev - 64);
	if (x < 0) x = 0;

	/* Boost expression by voice volume above neutral. */
	if (voicev > 65) xpn += (voicev-64)/2;

	/* Combine multiplicative and level components. */
/** changed **/
	/* x = vel*xpn + (voicev/2)*x; */
	x = vel*xpn*2 + (voicev/4)*x;

#ifdef GUS_VOLUME
	/* Further adjustment by installation-specific master
	 * volume control (default 50).
	 */
	x = (x*GUS_VOLUME*GUS_VOLUME)/10000;
#endif

	if (x < 1) return(0);
	else if (x > GUSMAXVOL) x = GUSMAXVOL;

	/* Convert to gus's logarithmic form with 4 bit exponent i
	 * and 8 bit mantissa m.
	 */
	n = x;
	i = 7;
	if (n < 128) {
		while (i > 0 && n < (1<<i)) i--;
	}
	else while (n > 255) {
		n >>= 1;
		i++;
	}
	/* Mantissa is part of linear volume not expressed in
	 * exponent.  (This is not quite like real logs -- I wonder
	 * if it's right.)
	 */
	m = x - (1<<i);

	/* Adjust mantissa to 8 bits. */
	if (m > 0) {
		if (i > 8) m >>= i-8;
		else if (i < 8) m <<= 8-i;
	}

	return((i << 8) + m);
}


void
gus_vol(int chan, int prog, int cell, int vol)
{
	char buf[8];
	int gvolume = getval("gvolume");

	if (vol < 0 || vol > 127) {
		fprintf(stderr, "gus_vol: vol %d out of range\n", vol);
		return;
	}
	buf[0] = SEQ_PRIVATE;
	buf[1] = gus_dev;
	buf[2] = _GUS_VOICEVOL2;
	buf[3] = cell;
	*(unsigned short*)&buf[4] = gvol(vol, MAINV, XPN, gvolume);
	buf[6] = buf[7] = 0; 
	sqwrite(buf);
}


void
gus_bend(int cell)
{
	char buf[8];
	int value = getval("gftune") - 8192;

	buf[0] = SEQ_EXTENDED;
	buf[1] = SEQ_CONTROLLER;
	buf[2] = gus_dev;
	buf[3] = cell;
	buf[4] = CTRL_PITCH_BENDER;
	*(short *)&buf[5] = value;
	buf[7] = 0;
	sqwrite(buf);
}
/*
 * Send expression request to card.
 */
void card_expression(int card, int chan, int cell)
{
	char buf[8];
    int amount = XPN;

    buf[0] = SEQ_EXTENDED;
    buf[1] = SEQ_CONTROLLER;
    buf[2] = card;
    buf[3] = cell;
    buf[4] = CTRL_EXPRESSION;
    *(short *) &buf[5] = amount;
    buf[7] = 0;
    sqwrite(buf);
}
/*
 * Send main volume request to card.
 */
void card_main_volume(int card, int chan, int cell)
{
	char buf[8];
    int amount = MAINV;

    buf[0] = SEQ_EXTENDED;
    buf[1] = SEQ_CONTROLLER;
    buf[2] = card;
    buf[3] = cell;
    buf[4] = CTRL_MAIN_VOLUME;
    *(short *) &buf[5] = amount;
    buf[7] = 0;
    sqwrite(buf);
}

#endif
