/*
 * timidity_dsf.cpp - Timidity DirectShow source filter
 * Based on asap_dsf.cpp - ASAP DirectShow source filter
 *
 * Copyright (C) 2025       Datajake
 * Copyright (C) 2008-2013  Piotr Fusik
 *
 * This file is part of ASAP (Another Slight Atari Player),
 * see http://asap.sourceforge.net
 *
 * ASAP is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * ASAP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ASAP; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <windows.h>
#include <tchar.h>
#include <control.h>
#include <streams.h>
#ifndef UNDER_CE
#include <initguid.h>
#endif
#include <qnetwork.h>

#include "../timidity/timid.h"
#include "../common/registry.h"

static const TCHAR extensions[][6] =
	{ _T(".mid"), _T(".midi"), _T(".kar") };
#define N_EXTS (sizeof(extensions) / sizeof(extensions[0]))
#define EXT_FILTER _T("*.mid;*.midi;*.kar")

#define SZ_TIMID_SOURCE      L"Timidity Source Filter"

static const TCHAR CLSID_TimidSource_str[] = _T("{C919F48D-FC46-4866-85C0-8D6A6D221D84}");
static const GUID CLSID_TimidSource = 
	{ 0xc919f48d, 0xfc46, 0x4866, { 0x85, 0xc0, 0x8d, 0x6a, 0x6d, 0x22, 0x1d, 0x84 } };

class CTimidSourceStream : public CSourceStream, IMediaSeeking
{
	CCritSec cs;
	Timid synth;
	DriverConfig cfg;
	BOOL loaded;
	int channels;
	int bitDepth;
	int duration;
	LONGLONG blocks;

public:

	CTimidSourceStream(HRESULT *phr, CSource *pFilter)
		: CSourceStream(NAME("TimidSourceStream"), phr, pFilter, L"Out"), loaded(FALSE), duration(0)
	{
		CAutoLock lck(&cs);
		memset(&cfg, 0, sizeof(cfg));
#ifdef UNDER_CE
		_tcscpy(cfg.szConfigFile, _T("\\Flash Disk\\gm16\\gm.cfg"));
#else
		_tcscpy(cfg.szConfigFile, _T("C:\\gm16\\gm.cfg"));
#endif
		cfg.nSampleRate = DEFAULT_RATE;
		cfg.nControlRate = CONTROLS_PER_SECOND;
		cfg.nVoices = DEFAULT_VOICES;
		cfg.nAmp = DEFAULT_AMPLIFICATION;
		cfg.fAdjustPanning = TRUE;
		cfg.fMono = FALSE;
		cfg.f8Bit = FALSE;
		cfg.fAntialiasing = FALSE;
		cfg.fPreResample = TRUE;
		cfg.fFastDecay = TRUE;
		cfg.fDynamicLoad = TRUE;
		cfg.nDefaultProgram = DEFAULT_PROGRAM;
		cfg.nDrumChannels = DEFAULT_DRUMCHANNELS;
		cfg.nQuietChannels = 0;
		ReadRegistry(&cfg);
		if (cfg.nSampleRate > MAX_OUTPUT_RATE)
		{
			cfg.nSampleRate = MAX_OUTPUT_RATE;
		}
		else if (cfg.nSampleRate < MIN_OUTPUT_RATE)
		{
			cfg.nSampleRate = MIN_OUTPUT_RATE;
		}
		if (cfg.fMono)
		{
			channels = 1;
		}
		else
		{
			channels = 2;
		}
		if (cfg.f8Bit)
		{
			bitDepth = 8;
		}
		else
		{
			bitDepth = 16;
		}
		memset(&synth, 0, sizeof(synth));
		timid_init(&synth);
		timid_set_sample_rate(&synth, cfg.nSampleRate);
		timid_set_control_rate(&synth, cfg.nControlRate);
		timid_set_max_voices(&synth, cfg.nVoices);
		timid_set_amplification(&synth, cfg.nAmp);
		timid_set_immediate_panning(&synth, cfg.fAdjustPanning);
		timid_set_mono(&synth, cfg.fMono);
		timid_set_antialiasing(&synth, cfg.fAntialiasing);
		timid_set_pre_resample(&synth, cfg.fPreResample);
		timid_set_fast_decay(&synth, cfg.fFastDecay);
		timid_set_dynamic_instrument_load(&synth, cfg.fDynamicLoad);
		timid_set_default_program(&synth, cfg.nDefaultProgram);
		for (int i = 0; i < 16; i++)
		{
			if (cfg.nDrumChannels & (1<<i))
			{
				timid_set_drum_channel(&synth, i, 1);
			}
			else
			{
				timid_set_drum_channel(&synth, i, 0);
			}
			if (cfg.nQuietChannels & (1<<i))
			{
				timid_set_quiet_channel(&synth, i, 1);
			}
			else
			{
				timid_set_quiet_channel(&synth, i, 0);
			}
		}
	}

	~CTimidSourceStream()
	{
		CAutoLock lck(&cs);
		timid_close(&synth);
		memset(&synth, 0, sizeof(synth));
		WriteRegistry(&cfg);
		memset(&cfg, 0, sizeof(cfg));
	}

	DECLARE_IUNKNOWN

	STDMETHODIMP NonDelegatingQueryInterface(REFIID riid, void **ppv)
	{
		if (riid == IID_IMediaSeeking)
			return GetInterface((IMediaSeeking *) this, ppv);
		return CSourceStream::NonDelegatingQueryInterface(riid, ppv);
	}

	BOOL Load(char *filename)
	{
		CAutoLock lck(&cs);
		char szAnsi[MAX_PATH];
		memset(szAnsi, 0, sizeof(szAnsi));
#ifdef _UNICODE
		WideCharToMultiByte(CP_ACP, 0, cfg.szConfigFile, -1, szAnsi, MAX_PATH, NULL, NULL);
#else
		strcpy(szAnsi, cfg.szConfigFile);
#endif
		loaded = timid_load_config(&synth, szAnsi);
		if (!loaded)
			return FALSE;
		loaded = timid_load_smf(&synth, filename);
		if (!loaded)
			return FALSE;
		duration = timid_get_duration(&synth);
		blocks = 0;
		return TRUE;
	}

	HRESULT GetMediaType(CMediaType *pMediaType)
	{
		CheckPointer(pMediaType, E_POINTER);
		CAutoLock lck(&cs);
		if (!loaded)
			return E_FAIL;
		WAVEFORMATEX *wfx = (WAVEFORMATEX *) pMediaType->AllocFormatBuffer(sizeof(WAVEFORMATEX));
		CheckPointer(wfx, E_OUTOFMEMORY);
		pMediaType->SetType(&MEDIATYPE_Audio);
		pMediaType->SetSubtype(&MEDIASUBTYPE_PCM);
		pMediaType->SetTemporalCompression(FALSE);
		pMediaType->SetSampleSize(channels * (bitDepth / 8));
		pMediaType->SetFormatType(&FORMAT_WaveFormatEx);
		wfx->wFormatTag = WAVE_FORMAT_PCM;
		wfx->nChannels = channels;
		wfx->nSamplesPerSec = cfg.nSampleRate;
		wfx->nBlockAlign = channels * (bitDepth / 8);
		wfx->nAvgBytesPerSec = cfg.nSampleRate * wfx->nBlockAlign;
		wfx->wBitsPerSample = bitDepth;
		wfx->cbSize = 0;
		return S_OK;
	}

	HRESULT DecideBufferSize(IMemAllocator *pAlloc, ALLOCATOR_PROPERTIES *pRequest)
	{
		CheckPointer(pAlloc, E_POINTER);
		CheckPointer(pRequest, E_POINTER);
		CAutoLock lck(&cs);
		if (!loaded)
			return E_FAIL;
		if (pRequest->cBuffers == 0)
			pRequest->cBuffers = 2;
		int bytes = AUDIO_BUFFER_SIZE * channels * (bitDepth / 8);
		if (pRequest->cbBuffer < bytes)
			pRequest->cbBuffer = bytes;
		ALLOCATOR_PROPERTIES actual;
		HRESULT hr = pAlloc->SetProperties(pRequest, &actual);
		if (FAILED(hr))
			return hr;
		if (actual.cbBuffer < bytes)
			return E_FAIL;
		return S_OK;
	}

	HRESULT FillBuffer(IMediaSample *pSample)
	{
		CheckPointer(pSample, E_POINTER);
		CAutoLock lck(&cs);
		if (!loaded)
			return E_FAIL;
		BYTE *pData;
		HRESULT hr = pSample->GetPointer(&pData);
		if (FAILED(hr))
			return hr;
		int cbData = pSample->GetSize();
		int audioFormat = 0;
		if (bitDepth == 16)
		{
			audioFormat = AU_SHORT;
		}
		else if (bitDepth == 8)
		{
			audioFormat = AU_CHAR;
		}
		if (timid_play_smf(&synth, audioFormat, pData, cbData / (channels * (bitDepth / 8))) == 0)
			return S_FALSE;
		pSample->SetActualDataLength(cbData);
		LONGLONG startTime = blocks * UNITS / cfg.nSampleRate;
		blocks += cbData / (channels * (bitDepth / 8));
		LONGLONG endTime = blocks * UNITS / cfg.nSampleRate;
		pSample->SetTime(&startTime, &endTime);
		pSample->SetSyncPoint(TRUE);
		return S_OK;
	}

	STDMETHODIMP GetCapabilities(DWORD *pCapabilities)
	{
		CheckPointer(pCapabilities, E_POINTER);
		*pCapabilities = AM_SEEKING_CanSeekAbsolute | AM_SEEKING_CanSeekForwards | AM_SEEKING_CanSeekBackwards | AM_SEEKING_CanGetDuration | AM_SEEKING_CanGetCurrentPos | AM_SEEKING_CanGetStopPos;
		return S_OK;
	}

	STDMETHODIMP CheckCapabilities(DWORD *pCapabilities)
	{
		CheckPointer(pCapabilities, E_POINTER);
		DWORD result = *pCapabilities & (AM_SEEKING_CanSeekAbsolute | AM_SEEKING_CanSeekForwards | AM_SEEKING_CanSeekBackwards | AM_SEEKING_CanGetDuration | AM_SEEKING_CanGetCurrentPos | AM_SEEKING_CanGetStopPos);
		if (result == *pCapabilities)
			return S_OK;
		*pCapabilities = result;
		if (result != 0)
			return S_FALSE;
		return E_FAIL;
	}

	STDMETHODIMP IsFormatSupported(const GUID *pFormat)
	{
		CheckPointer(pFormat, E_POINTER);
		if (IsEqualGUID(*pFormat, TIME_FORMAT_MEDIA_TIME))
			return S_OK;
		return S_FALSE;
	}

	STDMETHODIMP QueryPreferredFormat(GUID *pFormat)
	{
		CheckPointer(pFormat, E_POINTER);
		*pFormat = TIME_FORMAT_MEDIA_TIME;
		return S_OK;
	}

	STDMETHODIMP GetTimeFormat(GUID *pFormat)
	{
		CheckPointer(pFormat, E_POINTER);
		*pFormat = TIME_FORMAT_MEDIA_TIME;
		return S_OK;
	}

	STDMETHODIMP IsUsingTimeFormat(const GUID *pFormat)
	{
		CheckPointer(pFormat, E_POINTER);
		if (IsEqualGUID(*pFormat, TIME_FORMAT_MEDIA_TIME))
			return S_OK;
		return S_FALSE;
	}

	STDMETHODIMP SetTimeFormat(const GUID *pFormat)
	{
		CheckPointer(pFormat, E_POINTER);
		if (IsEqualGUID(*pFormat, TIME_FORMAT_MEDIA_TIME))
			return S_OK;
		return E_INVALIDARG;
	}

	STDMETHODIMP GetDuration(LONGLONG *pDuration)
	{
		CheckPointer(pDuration, E_POINTER);
		*pDuration = duration * (UNITS / MILLISECONDS);
		return S_OK;
	}

	STDMETHODIMP GetStopPosition(LONGLONG *pStop)
	{
		CheckPointer(pStop, E_POINTER);
		*pStop = duration * (UNITS / MILLISECONDS);
		return S_OK;
	}

	STDMETHODIMP GetCurrentPosition(LONGLONG *pCurrent)
	{
		CheckPointer(pCurrent, E_POINTER);
		CAutoLock lck(&cs);
		*pCurrent = timid_get_current_time(&synth) * (UNITS / MILLISECONDS);
		return S_OK;
	}

	STDMETHODIMP ConvertTimeFormat(LONGLONG *pTarget, const GUID *pTargetFormat, LONGLONG Source, const GUID *pSourceFormat)
	{
		CheckPointer(pTarget, E_POINTER);
		CheckPointer(pTargetFormat, E_POINTER);
		CheckPointer(pSourceFormat, E_POINTER);
		CAutoLock lck(&cs);
		if (IsEqualGUID(*pTargetFormat, TIME_FORMAT_SAMPLE) && IsEqualGUID(*pSourceFormat, TIME_FORMAT_MEDIA_TIME))
		{
			*pTarget = timid_millis2samples(&synth, (int) (Source / (UNITS / MILLISECONDS)));
			return S_OK;
		}
		else if (IsEqualGUID(*pTargetFormat, TIME_FORMAT_MEDIA_TIME) && IsEqualGUID(*pSourceFormat, TIME_FORMAT_SAMPLE))
		{
			*pTarget = timid_samples2millis(&synth, (int) Source) * (UNITS / MILLISECONDS);
			return S_OK;
		}
		return E_INVALIDARG;
	}

	STDMETHODIMP SetPositions(LONGLONG *pCurrent, DWORD dwCurrentFlags, LONGLONG *pStop, DWORD dwStopFlags)
	{
		if ((dwCurrentFlags & AM_SEEKING_PositioningBitsMask) == AM_SEEKING_AbsolutePositioning)
		{
			CheckPointer(pCurrent, E_POINTER);
			CAutoLock lck(&cs);
			int position = timid_seek_smf(&synth, (int) (*pCurrent / (UNITS / MILLISECONDS)));
			blocks = 0;
			if ((dwCurrentFlags & AM_SEEKING_ReturnTime) != 0)
				*pCurrent = position * (UNITS / MILLISECONDS);
		}
		if ((dwStopFlags & AM_SEEKING_PositioningBitsMask) == AM_SEEKING_AbsolutePositioning)
		{
			CheckPointer(pStop, E_POINTER);
			if ((dwStopFlags & AM_SEEKING_ReturnTime) != 0)
				*pStop = duration * (UNITS / MILLISECONDS);
		}
		return S_OK;
	}

	STDMETHODIMP GetPositions(LONGLONG *pCurrent, LONGLONG *pStop)
	{
		CheckPointer(pCurrent, E_POINTER);
		CAutoLock lck(&cs);
		*pCurrent = timid_get_current_time(&synth) * (UNITS / MILLISECONDS);
		CheckPointer(pStop, E_POINTER);
		*pStop = duration * (UNITS / MILLISECONDS);
		return S_OK;
	}

	STDMETHODIMP GetAvailable(LONGLONG *pEarliest, LONGLONG *pLatest)
	{
		CheckPointer(pEarliest, E_POINTER);
		*pEarliest = 0;
		CheckPointer(pLatest, E_POINTER);
		*pLatest = duration * (UNITS / MILLISECONDS);
		return S_OK;
	}

	STDMETHODIMP SetRate(double dRate)
	{
		return E_NOTIMPL;
	}

	STDMETHODIMP GetRate(double *dRate)
	{
		return E_NOTIMPL;
	}

	STDMETHODIMP GetPreroll(LONGLONG *pllPreroll)
	{
		CheckPointer(pllPreroll, E_POINTER);
		*pllPreroll = 0;
		return S_OK;
	}

	HRESULT GetMetadata(BOOL copyright, BSTR *pbstr)
	{
		CAutoLock lck(&cs);
		if (!loaded)
			return VFW_E_NOT_FOUND;
		char s[256];
		wchar_t ws[256];
		memset(s, 0, sizeof(s));
		memset(ws, 0, sizeof(ws));
		int len = copyright ? timid_get_song_copyright(&synth, s, 255) : timid_get_song_title(&synth, s, 255);
		if (!len)
			return VFW_E_NOT_FOUND;
		int cch = MultiByteToWideChar(CP_ACP, 0, s, -1, NULL, 0);
		if (MultiByteToWideChar(CP_ACP, 0, s, -1, ws, cch) <= 0)
			return HRESULT_FROM_WIN32(GetLastError());
		*pbstr = SysAllocStringLen(ws, cch - 1);
		CheckPointer(*pbstr, E_OUTOFMEMORY);
		return S_OK;
	}
};

class CTimidSource : public CSource, IFileSourceFilter, IAMMediaContent
{
	CTimidSourceStream *m_pin;
	WCHAR *m_filename;
	CMediaType m_mt;

	CTimidSource(IUnknown *pUnk, HRESULT *phr)
		: CSource(NAME("TimidSource"), pUnk, CLSID_TimidSource), m_pin(NULL), m_filename(NULL)
	{
		m_pin = new CTimidSourceStream(phr, this);
		if (phr != NULL)
			*phr = m_pin == NULL ? E_OUTOFMEMORY : S_OK;
	}

	~CTimidSource()
	{
		if (m_pin != NULL)
			delete m_pin;
		if (m_filename != NULL)
			delete[] m_filename;
	}

public:

	DECLARE_IUNKNOWN

	STDMETHODIMP NonDelegatingQueryInterface(REFIID riid, void **ppv)
	{
		if (riid == IID_IFileSourceFilter)
			return GetInterface((IFileSourceFilter *) this, ppv);
		if (riid == IID_IAMMediaContent)
			return GetInterface((IAMMediaContent *) this, ppv);
		return CSource::NonDelegatingQueryInterface(riid, ppv);
	}

	STDMETHODIMP Load(LPCOLESTR pszFileName, const AM_MEDIA_TYPE *pmt)
	{
		CheckPointer(pszFileName, E_POINTER);
		int cch = lstrlenW(pszFileName) + 1;
		char *filename = new char[cch * 2];
		CheckPointer(filename, E_OUTOFMEMORY);
		if (WideCharToMultiByte(CP_ACP, 0, pszFileName, -1, filename, cch, NULL, NULL) <= 0)
			return HRESULT_FROM_WIN32(GetLastError());
		BOOL ok = m_pin->Load(filename);
		delete[] filename;
		if (!ok)
			return E_FAIL;
		if (m_filename != NULL)
			delete[] m_filename;
		m_filename = new WCHAR[cch];
		CheckPointer(m_filename, E_OUTOFMEMORY);
		CopyMemory(m_filename, pszFileName, cch * sizeof(WCHAR));
		if (pmt == NULL) {
			m_mt.SetType(&MEDIATYPE_Midi);
			m_mt.SetSubtype(&MEDIASUBTYPE_NULL);
		}
		else
			m_mt = *pmt;
		return S_OK;
	}

	STDMETHODIMP GetCurFile(LPOLESTR *ppszFileName, AM_MEDIA_TYPE *pmt)
	{
		CheckPointer(ppszFileName, E_POINTER);
		if (m_filename == NULL)
			return E_FAIL;
		DWORD n = (lstrlenW(m_filename) + 1) * sizeof(WCHAR);
		*ppszFileName = (LPOLESTR) CoTaskMemAlloc(n);
		CheckPointer(*ppszFileName, E_OUTOFMEMORY);
		CopyMemory(*ppszFileName, m_filename, n);
		if (pmt != NULL)
			CopyMediaType(pmt, &m_mt);
		return S_OK;
	}

	STDMETHODIMP GetTypeInfoCount(UINT* pctinfo) { return E_NOTIMPL; }
	STDMETHODIMP GetTypeInfo(UINT itinfo, LCID lcid, ITypeInfo** pptinfo) { return E_NOTIMPL; }
	STDMETHODIMP GetIDsOfNames(REFIID riid, OLECHAR** rgszNames, UINT cNames, LCID lcid, DISPID* rgdispid) { return E_NOTIMPL; }
	STDMETHODIMP Invoke(DISPID dispidMember, REFIID riid, LCID lcid, WORD wFlags, DISPPARAMS* pdispparams,
		VARIANT* pvarResult, EXCEPINFO* pexcepinfo, UINT* puArgErr)
	{
		return E_NOTIMPL;
	}

	STDMETHODIMP get_AuthorName(BSTR* pbstrAuthorName) { return E_NOTIMPL; }
	STDMETHODIMP get_Title(BSTR* pbstrTitle)
	{
		return m_pin->GetMetadata(FALSE, pbstrTitle);
	}

	STDMETHODIMP get_Rating(BSTR* pbstrRating) { return E_NOTIMPL; }
	STDMETHODIMP get_Description(BSTR* pbstrDescription) { return E_NOTIMPL; }
	STDMETHODIMP get_Copyright(BSTR* pbstrCopyright)
	{
		return m_pin->GetMetadata(TRUE, pbstrCopyright);
	}

	STDMETHODIMP get_BaseURL(BSTR* pbstrBaseURL) { return E_NOTIMPL; }
	STDMETHODIMP get_LogoURL(BSTR* pbstrLogoURL) { return E_NOTIMPL; }
	STDMETHODIMP get_LogoIconURL(BSTR* pbstrLogoURL) { return E_NOTIMPL; }
	STDMETHODIMP get_WatermarkURL(BSTR* pbstrWatermarkURL) { return E_NOTIMPL; }
	STDMETHODIMP get_MoreInfoURL(BSTR* pbstrMoreInfoURL) { return E_NOTIMPL; }
	STDMETHODIMP get_MoreInfoBannerImage(BSTR* pbstrMoreInfoBannerImage) { return E_NOTIMPL; }
	STDMETHODIMP get_MoreInfoBannerURL(BSTR* pbstrMoreInfoBannerURL) { return E_NOTIMPL; }
	STDMETHODIMP get_MoreInfoText(BSTR* pbstrMoreInfoText) { return E_NOTIMPL; }

	static CUnknown * WINAPI CreateInstance(IUnknown *pUnk, HRESULT *phr)
	{
		CTimidSource *pNewFilter = new CTimidSource(pUnk, phr);
		if (phr != NULL && pNewFilter == NULL)
			*phr = E_OUTOFMEMORY;
		return pNewFilter;
	}
};

static const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
	&MEDIATYPE_Audio,
	&MEDIASUBTYPE_PCM
};

static const AMOVIESETUP_PIN sudTimidSourceStream =
{
	L"Output",
	FALSE,
	TRUE,
	FALSE,
	FALSE,
	&CLSID_NULL,
	NULL,
	1,
	&sudPinTypes
};

static const AMOVIESETUP_FILTER sudTimidSource =
{
	&CLSID_TimidSource,
	SZ_TIMID_SOURCE,
	MERIT_NORMAL,
	1,
	&sudTimidSourceStream
};

CFactoryTemplate g_Templates[1] =
{
	{
		SZ_TIMID_SOURCE,
		&CLSID_TimidSource,
		CTimidSource::CreateInstance,
		NULL,
		&sudTimidSource
	}
};

int g_cTemplates = 1;

static void WriteTimidValue(LPCTSTR lpSubKey, LPCTSTR value)
{
	HKEY hKey;
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, lpSubKey, 0, KEY_SET_VALUE, &hKey) != ERROR_SUCCESS)
		return;
	if (value != NULL)
		RegSetValueEx(hKey, _T("Timidity"), 0, REG_SZ, (const BYTE *) value, ((DWORD) _tcslen(value) + 1) * sizeof(TCHAR));
	else
		RegDeleteValue(hKey, _T("Timidity"));
	RegCloseKey(hKey);
}

static void RegisterExtension(LPCTSTR lpExtension)
{
#ifdef UNDER_CE
	const TCHAR *fileDescription = _T("audiofile");
#else
	const TCHAR *fileDescription = _T("midfile");
#endif
	const TCHAR *contentType = _T("audio/mid");
	HKEY hKey;
	if (RegOpenKeyEx(HKEY_CLASSES_ROOT, lpExtension, 0, KEY_SET_VALUE, &hKey) == ERROR_SUCCESS) {
		RegCloseKey(hKey);
		return;
	}
	if (RegCreateKeyEx(HKEY_CLASSES_ROOT, lpExtension, 0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL) != ERROR_SUCCESS)
		return;
	RegSetValueEx(hKey, NULL, 0, REG_SZ, (const BYTE *) fileDescription, ((DWORD) _tcslen(fileDescription) + 1) * sizeof(TCHAR));
	RegSetValueEx(hKey, _T("Content Type"), 0, REG_SZ, (const BYTE *) contentType, ((DWORD) _tcslen(contentType) + 1) * sizeof(TCHAR));
	RegCloseKey(hKey);
}

static void RegisterMimeType()
{
	const TCHAR *regKey = _T("MIME\\Database\\Content Type\\audio/mid");
	const TCHAR *playerCLSID = _T("{22D6F312-B0F6-11D0-94AB-0080C74C7E95}");
	const TCHAR *fileExtension = _T(".mid");
	HKEY hKey;
	if (RegOpenKeyEx(HKEY_CLASSES_ROOT, regKey, 0, KEY_SET_VALUE, &hKey) == ERROR_SUCCESS) {
		RegCloseKey(hKey);
		return;
	}
	if (RegCreateKeyEx(HKEY_CLASSES_ROOT, regKey, 0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL) != ERROR_SUCCESS)
		return;
	RegSetValueEx(hKey, _T("Extension"), 0, REG_SZ, (const BYTE *) fileExtension, ((DWORD) _tcslen(fileExtension) + 1) * sizeof(TCHAR));
	RegSetValueEx(hKey, _T("CLSID"), 0, REG_SZ, (const BYTE *) playerCLSID, ((DWORD) _tcslen(playerCLSID) + 1) * sizeof(TCHAR));
	RegCloseKey(hKey);
}

STDAPI DllRegisterServer()
{
	HRESULT hr = AMovieDllRegisterServer2(TRUE);
	if (FAILED(hr))
		return hr;

	RegisterMimeType();

	HKEY hMTKey;
	HKEY hWMPKey;
	if (RegCreateKeyEx(HKEY_CLASSES_ROOT, _T("Media Type\\Extensions"), 0, NULL, 0, KEY_WRITE, NULL, &hMTKey, NULL) != ERROR_SUCCESS)
		return E_FAIL;
	if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\Microsoft\\Multimedia\\WMPlayer\\Extensions"), 0, NULL, 0, KEY_WRITE, NULL, &hWMPKey, NULL) != ERROR_SUCCESS)
		return E_FAIL;
	for (int i = 0; i < N_EXTS; i++) {
		RegisterExtension(extensions[i]);
		HKEY hKey;
		if (RegCreateKeyEx(hMTKey, extensions[i], 0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL) != ERROR_SUCCESS
		 || RegSetValueEx(hKey, _T("Source Filter"), 0, REG_SZ, (const BYTE *) &CLSID_TimidSource_str, sizeof(CLSID_TimidSource_str)) != ERROR_SUCCESS
		 || RegCloseKey(hKey) != ERROR_SUCCESS)
			 return E_FAIL;
		static const DWORD perms = 15;
		static const DWORD runtime = 7;
		if (RegCreateKeyEx(hWMPKey, extensions[i], 0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL) != ERROR_SUCCESS
		 || RegSetValueEx(hKey, _T("Permissions"), 0, REG_DWORD, (const BYTE *) &perms, sizeof(perms)) != ERROR_SUCCESS
		 || RegSetValueEx(hKey, _T("Runtime"), 0, REG_DWORD, (const BYTE *) &runtime, sizeof(runtime)) != ERROR_SUCCESS
		 || RegCloseKey(hKey) != ERROR_SUCCESS)
			 return E_FAIL;
	}
	if (RegCloseKey(hWMPKey) != ERROR_SUCCESS || RegCloseKey(hMTKey) != ERROR_SUCCESS)
		return E_FAIL;
	WriteTimidValue(_T("Software\\Microsoft\\MediaPlayer\\Player\\Extensions\\MUIDescriptions"), _T("Standard MIDI file (smf)"));
	WriteTimidValue(_T("Software\\Microsoft\\MediaPlayer\\Player\\Extensions\\Types"), EXT_FILTER);
	return S_OK;
}

STDAPI DllUnregisterServer()
{
	HKEY hWMPKey;
	HKEY hMTKey;
	WriteTimidValue(_T("Software\\Microsoft\\MediaPlayer\\Player\\Extensions\\MUIDescriptions"), NULL);
	WriteTimidValue(_T("Software\\Microsoft\\MediaPlayer\\Player\\Extensions\\Types"), NULL);
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\Microsoft\\Multimedia\\WMPlayer\\Extensions"), 0, DELETE, &hWMPKey) != ERROR_SUCCESS)
		return E_FAIL;
	if (RegOpenKeyEx(HKEY_CLASSES_ROOT, _T("Media Type\\Extensions"), 0, DELETE, &hMTKey) != ERROR_SUCCESS)
		return E_FAIL;
	for (int i = 0; i < N_EXTS; i++) {
		RegDeleteKey(hWMPKey, extensions[i]);
		RegDeleteKey(hMTKey, extensions[i]);
	}
	RegCloseKey(hMTKey);
	RegCloseKey(hWMPKey);

	return AMovieDllRegisterServer2(FALSE);
}

extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);

BOOL APIENTRY DllMain(HANDLE hInstance, DWORD dwReason, LPVOID lpReserved)
{
	return DllEntryPoint((HINSTANCE) hInstance, dwReason, lpReserved);
}
