/*
Basic digital audio processer
Copyright (C) 2018 Datajake

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include "bitcrush.h"
#include "EAXReverb.h"
#include "filter.h"
#include "LinearResampler.h"
#include "resampler.h"
#include "silence.h"
#include "wav_header.h"
#include "ZOHResampler.h"

static int fileexists(const char * filename)
{
	FILE *check;
	if (check = fopen(filename, "r"))
	{
		fclose(check);
		return 1;
	}
	return 0;
}

int main(int argc, char *argv[])
{
	unsigned int choice;
	unsigned int numsamples;
	unsigned int size;
	signed short *samples;
	FILE *in;
	FILE *out;
	wav_header head;
	if (argc != 3)
	{
		printf("Usage: %s [input].wav [output].wav.\n", argv[0]);
		return 0;
	}
	if (fileexists(argv[1]) == 0)
	{
		printf("%s doesn't exist.\n", argv[1]);
		return 0;
	}
	if (fileexists(argv[2]) == 1)
	{
		printf("%s already exists.\n", argv[2]);
		return 0;
	}
	in = fopen(argv[1], "rb");
	if (in == NULL)
	{
		printf("Failed to open %s.\n", argv[1]);
		return 0;
	}
	fread(&head, sizeof(head), 1, in);
	if (head.fmt_chunk_size != 16)
	{
		printf("%s Isn't a valid PCM WAV file.\n", argv[1]);
		return 0;
	}
	if (head.audio_format != 1)
	{
		printf("%s Isn't a valid PCM WAV file.\n", argv[1]);
		return 0;
	}
	if (head.num_channels != 2)
	{
		printf("%s Isn't a stereo WAV file.\n", argv[1]);
		return 0;
	}
	if (head.bit_depth != 16)
	{
		printf("%s Isn't a 16 bit WAV file.\n", argv[1]);
		return 0;
	}
	if (head.sample_alignment != 4)
	{
		printf("%s Isn't a stereo 16 bit WAV file.\n", argv[1]);
		return 0;
	}
	samples = new short[head.data_bytes / 2];
	numsamples = head.data_bytes / 4;
	fread(samples, 4, numsamples, in);
	fclose(in);
	out = fopen(argv[2], "wb");
	if (out == NULL)
	{
		printf("Failed to open %s.\n", argv[2]);
		return 0;
	}
	fwrite(&head, sizeof(head), 1, out);
	printf("Basic digital audio processer.\n");
	printf("0 = Resample.\n");
	printf("1 = Bit crush.\n");
	printf("2 = Swap stereo channels.\n");
	printf("3 = Mixdown to mono.\n");
	printf("4 = Surround sound (left channel).\n");
	printf("5 = Surround sound (right channel).\n");
	printf("6 = Limit sample values.\n");
	printf("7 = Generate silent waveform same length as input.\n");
	printf("8 = Add reverb.\n");
	printf("9 = Apply filter.\n");
	scanf("%d", &choice);
	if (choice == 0)
	{
		unsigned int quality;
		unsigned int samplerate_new;
		unsigned int numsamples_new;
		signed short *samples_new;
		printf("Enter new sample rate.\n");
		scanf("%d", &samplerate_new);
		numsamples_new = (unsigned int) numsamples / (head.sample_rate / 1000.f) * (samplerate_new / 1000.f) + 0.5f;
		samples_new = new short[numsamples_new*2];
		printf("Enter resampling quality.\n");
		printf("0 = Zero-order hold.\n");
		printf("1 = Linear.\n");
		printf("2 = Sinc.\n");
		scanf("%d", &quality);
		printf("Processing.\n");
		if (quality == 0)
		{
			void *zohresampler;
			zohresampler = ZOHResamplerCreate();
			ZOHResamplerSetup(zohresampler, head.sample_rate, samplerate_new);
			ZOHResamplerProcess(zohresampler, samples, numsamples, samples_new, numsamples_new);
			ZOHResamplerReset(zohresampler);
			ZOHResamplerDestroy(zohresampler);
		}
		if (quality == 1)
		{
			void *linearresampler;
			linearresampler = LinearResamplerCreate();
			LinearResamplerSetup(linearresampler, head.sample_rate, samplerate_new);
			LinearResamplerProcess(linearresampler, samples, numsamples, samples_new, numsamples_new);
			LinearResamplerReset(linearresampler);
			LinearResamplerDestroy(linearresampler);
		}
		if (quality == 2)
		{
			unsigned int samplecount = 0;
			void *resampler;
			resampler = resampler_create();
			resampler_set_rate(resampler, (double)head.sample_rate / (double)samplerate_new);
			for(unsigned int i = 0; i < numsamples_new*2; i+=2)
			{
				sample_t ls, rs;
				for(unsigned int j = 0; j = resampler_get_min_fill(resampler); j++)
				{
					if (samplecount >= numsamples*2)
					{
						samplecount = numsamples*2 - 2;
					}
					resampler_write_pair(resampler, samples[samplecount], samples[samplecount + 1]);
					samplecount += 2;
				}
				resampler_peek_pair(resampler, &ls, &rs);
				resampler_read_pair(resampler, &ls, &rs);
				if ((ls + 0x8000) & 0xFFFF0000) ls = (ls >> 31) ^ 0x7FFF;
				if ((rs + 0x8000) & 0xFFFF0000) rs = (rs >> 31) ^ 0x7FFF;
				samples_new[i] = (short)ls;
				samples_new[i + 1] = (short)rs;
			}
			resampler_clear(resampler);
			resampler_destroy(resampler);
		}
		head.sample_rate = samplerate_new;
		head.byte_rate = 4*samplerate_new;
		fwrite(samples_new, 4, numsamples_new, out);
		delete[] samples_new;
	}
	if (choice == 1)
	{
		unsigned int numbits;
		unsigned int dither;
		unsigned int autodither;
		unsigned int onlyerror;
		bitcrusher bc;
		printf("Enter number of bits.\n");
		scanf("%d", &numbits);
		printf("Enter dither type.\n");
		printf("0 = None.\n");
		printf("1 = Rectangle.\n");
		printf("2 = Triangle.\n");
		printf("3 = Gaussian.\n");
		scanf("%d", &dither);
		printf("Type 1 to not dither silence, or 0 to operate normally.\n");
		scanf("%d", &autodither);
		printf("Type 1 to only output the quantization error, or 0 to operate normally.\n");
		scanf("%d", &onlyerror);
		printf("Processing.\n");
		SetCrushAmount(&bc, numbits);
		if (onlyerror == 1)
		{
			SetOnlyError(&bc, 1);
		}
		if (autodither == 1)
		{
			SetAutoDither(&bc, 1);
		}
#ifdef _DEBUG
		printf("The new bit depth is %d.\n", GetCrushAmount(&bc));
		if (GetOnlyError(&bc) == 1)
		{
			printf("The output is only the quantization error.\n");
		}
		if (GetAutoDither(&bc) == 1)
		{
			printf("Silence isn't dithered.\n");
		}
#endif
		if (dither == 1)
		{
			RectangleDither(&bc, samples, numsamples);
		}
		if (dither == 2)
		{
			TriangleDither(&bc, samples, numsamples);
		}
		if (dither == 3)
		{
			GaussianDither(&bc, samples, numsamples);
		}
		BitCrush(&bc, samples, numsamples);
	}
	if (choice == 2)
	{
		printf("Processing.\n");
		SwapStereo(samples, numsamples);
	}
	if (choice == 3)
	{
		printf("Processing.\n");
		MonoMixdown(samples, numsamples);
	}
	if (choice == 4)
	{
		printf("Processing.\n");
		SurroundSoundLeft(samples, numsamples);
	}
	if (choice == 5)
	{
		printf("Processing.\n");
		SurroundSoundRight(samples, numsamples);
	}
	if (choice == 6)
	{
		unsigned int limit;
		printf("Enter limit.\n");
		scanf("%d", &limit);
		printf("Processing.\n");
		LimitOutput(limit, samples, numsamples);
	}
	if (choice == 7)
	{
		printf("Processing.\n");
		GenerateSilence(samples, numsamples);
	}
	if (choice == 8)
	{
		unsigned int preset;
		unsigned int invertrev;
		unsigned int monorev;
		unsigned int onlyrev;
		unsigned int dither;
		printf("Enter preset number.\n");
		scanf("%d", &preset);
		printf("Type 1 to invert the phase of the reverb, or 0 to keep it as is.\n");
		scanf("%d", &invertrev);
		printf("Type 1 to mixdown the reverb to mono, or 0 to keep it stereo.\n");
		scanf("%d", &monorev);
		printf("Type 1 to only have the reverb, or 0 to add the reverb to the original audio.\n");
		scanf("%d", &onlyrev);
		printf("Type 1 to dither the reverb, or 0 to skip this step.\n");
		scanf("%d", &dither);
		printf("Processing.\n");
		EAXReverb EAX;
		EAX.Init(head.sample_rate);
		EAX.SetPreset(preset);
		if (invertrev == 1)
		{
			EAX.SetInvertReverb(true);
		}
		if (monorev == 1)
		{
			EAX.SetMonoReverb(true);
		}
		if (onlyrev == 1)
		{
			EAX.SetOnlyReverb(true);
		}
		if (dither == 1)
		{
			EAX.SetDither(true);
		}
#ifdef _DEBUG
		printf("The sample rate is %d.\n", EAX.GetRate());
		printf("The name of the selected preset is %s, and its preset number is %d.\n", EAX.GetPresetName(EAX.GetPreset()), EAX.GetPreset());
		if (EAX.GetInvertReverb() == true)
		{
			printf("The reverb is inverted.\n");
		}
		if (EAX.GetMonoReverb() == true)
		{
			printf("The reverb is mono.\n");
		}
		if (EAX.GetOnlyReverb() == true)
		{
			printf("The output is only the reverb.\n");
		}
		if (EAX.GetDither() == true)
		{
			printf("The reverb is dithered.\n");
		}
#endif
		EAX.Generate(samples, numsamples);
		EAX.Close();
	}
	if (choice == 9)
	{
		unsigned int type;
		float freq;
		float res;
		float gain;
		unsigned int dither;
		printf("Enter filter type.\n");
		printf("0 = Lowpass.\n");
		printf("1 = Highpass.\n");
		printf("2 = Bandpass.\n");
		printf("3 = Allpass.\n");
		printf("4 = Notch.\n");
		printf("5 = Peaking.\n");
		printf("6 = Lowshelf.\n");
		printf("7 = Highshelf.\n");
		scanf("%d", &type);
		printf("Enter frequency.\n");
		scanf("%f", &freq);
		printf("Enter resonance.\n");
		scanf("%f", &res);
		printf("Enter gain.\n");
		scanf("%f", &gain);
		printf("Type 1 to dither the filtered audio, or 0 to skip this step.\n");
		scanf("%d", &dither);
		printf("Processing.\n");
		filter FLT;
		FLT.SetType(type);
		FLT.SetFreq(freq);
		FLT.SetRes(res);
		FLT.SetGain(gain);
		if (dither == 1)
		{
			FLT.SetDither(true);
		}
		FLT.Init(head.sample_rate);
#ifdef _DEBUG
		printf("The sample rate is %d.\n", FLT.GetRate());
		printf("The type of filter is %s, and its associated number is %d.\n", FLT.GetFilterName(FLT.GetType()), FLT.GetType());
		printf("The frequency is %f.\n", FLT.GetFreq());
		printf("The resonance is %f.\n", FLT.GetRes());
		printf("The gain is %f.\n", FLT.GetGain());
		if (FLT.GetDither() == true)
		{
			printf("The filtered audio is dithered.\n");
		}
#endif
		FLT.Generate(samples, numsamples);
	}
	if (choice > 0 && choice < 10)
	{
		fwrite(samples, 4, numsamples, out);
	}
	delete[] samples;
	size = ftell(out);
	head.wav_size = size-8;
	head.data_bytes = size-sizeof(head);
	fseek(out, 0, SEEK_SET);
	fwrite(&head, sizeof(head), 1, out);
	fclose(out);
	printf("Done.\n");
	return 0;
}
