/*

    TiMidity -- Experimental MIDI to WAVE converter
    Copyright (C) 1995 Tuukka Toivonen <titoivon@snakemail.hut.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    dos_sb.c, Anthony Cruz (keyboard@engin.umich.edu)

    based off of information in SoundBlaster programming info v0.90,
    sbio.c (Ethan Brodsky) and linux_audio.c (Tuuka Toivonen)
*/

/*
 *   Headers needed to compile under DJGPP
 */
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <string.h>
#include <stdlib.h>
#include "config.h"
#include "controls.h"

#ifdef DIGPAK
#include "h\\keys.h"         // Include #define's for keyboard commands
#include "h\\support.h"      // Include header file for basic support functions.
#include "h\\loader.h"       // Include header for midpak/digpak dynamic loader
#include "h\\midpak.h"       // Include header for link layer to MIDPAK functions.
#include "h\\digplay.h"      // Include header for link layer to DIGPAK functions.
#include "h\\doscalls.h"     // Include header to assembly DOS support functions.
#include "h\\lowmem.h"
static const unsigned int OBUFFERSIZE = 2 * 2304;	// max. 2 * 2304 samples per frame
static const unsigned int MAXCHANNELS = 2;		// max. number of channels

int play_precision =8;
int play_stereo = 0;
int play_sample_rate =44000;
int   ScaleDown=0;
int number_of_channels=2;
short int buffer[2*2304];
short int *bufferp[2];
unsigned int channels=2;
int  CanMap_ConvMem(void);

char   cbuffer[2*2304];
char  *cbufferp[2];
int  *isplaying;
static SNDSTRUC *snd;
#else
#include "mikmod.h"
int play_precision =8;
int play_stereo = 0;
char *myerr;
void write_samples(int lensample);
int have_to_start=1;
#endif
/*
 *   TiMidity header files
 */
#include "config.h"
#include "output.h"
#include "controls.h"

/*
 *   Routines needed by TiMidity
 */
static int open_output(void);
static void close_output(void);
static void output_data(int32 *buf, int32 count);
static void flush_output(void);
static void purge_output(void);
static void write2(char *b, int32 length);

/* export the playback mode */

#define dpm dos_sb_play_mode

PlayMode dpm = {
  DEFAULT_RATE, PE_16BIT|PE_SIGNED,
  -1, {0},
  "SoundBlaster", 'b',
  "SoundBlaster",
  open_output,
  close_output,
  output_data,
  flush_output,
  purge_output
};

/*
 *   TiMidity needed routines
 */
static int open_output(void)
{
    int warnings = 0;
#ifdef DIGPAK
    int i;
    short cap;
    int   do16stereo;
    int max_sample;
    int error;
    char *stereo=getenv("STEREO");
    char *sample=getenv("SAMPLE");
    int dps;
    if( stereo )
	dps=atoi(stereo);
    else
	dps=1;

    dpm.encoding &= ~(PE_BYTESWAP|PE_ULAW);

    if ( !LoadDigPak("SOUNDRV.COM") )
    {
	    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"Failed to load sound driver.\n");
	    exit(1);
    }

    if ( !InitDigPak() )
    {
	    UnLoadDigPak();
	    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"Failed to initialize sound driver.\n");
	    exit(1);
    }

    dpm.encoding ^= PE_16BIT;
    channels = number_of_channels;
    // configure the device:
    play_precision = 8;
    play_stereo = !(dpm.encoding & PE_MONO);

    if (play_stereo && !dps )
    {
	dpm.encoding |= PE_MONO;
	play_stereo=0;
    }

    channels=play_stereo?2:1;

#ifdef DEBUG
    if (!number_of_channels || number_of_channels > MAXCHANNELS)
    {
      cerr << "LinuxObuffer: 0 < number of channels < " << MAXCHANNELS << "!\n";
      exit (1);
    }
#endif
    cap=AudioCapabilities();
    do16stereo=0;

    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "this audiocard : \n" );
    if( cap&PLAYBACK )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    can play audio in the background.\n");
    if( cap&MASSAGE )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    data is massaged.\n");
    if( cap&FIXEDFREQ )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    driver plays at fixed frequency.\n" );
    if( cap&USESTIMER )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    driver uses timer\n");
    if( cap&SHARESTIMER )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    timer can be shared\n" );
    if( cap&STEREOPAN )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    supports stereo panning.\n");
    if( cap&STEREOPLAY )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    supports 8 bit PCM stereo playback.\n" );
    if( cap&AUDIORECORD )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    supports audio recording!\n" );
    if( cap&DMABACKFILL )
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    support DMA backfilling.\n" );
    if( cap&PCM16 )
    {
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    supports 16 bit digital audio.\n" );
	play_precision=16;
    }
    if( cap&PCM16STEREO )
    {
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "    driver support 16 bit digital stereo sound\n" );
	do16stereo=1;
    }
    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"Original frequency : %d Hz\n", dpm.rate );
    if( sample )
	play_sample_rate=atoi(sample);

    if( dpm.rate < play_sample_rate )
	play_sample_rate=dpm.rate;
    else
	dpm.rate=play_sample_rate;

    snd=NULL;
    isplaying=NULL;
    ScaleDown=0;
    if( !snd )
    {
	snd = (SNDSTRUC *) realalloc(sizeof(SNDSTRUC));
	isplaying = (int *) realalloc( 4 );

    }

    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"frequency : %d Hz\n", play_sample_rate );

    setLowMemWord( ((unsigned long)snd)>>4, 10, play_sample_rate );

    error=0;
    if( play_precision==16 )
    {    // mine only does 8 bit
	if( play_stereo )
	{
	    if( do16stereo )
	    {
		error=SetPlayMode( PCM_16_STEREO );
		ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"mode : 16 bit stereo.\n" );
	    }
	    else
	    {
		error=SetPlayMode( PCM_8_STEREO );
		ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"mode : 8 bit stereo.\n" );
		play_precision=8;
	    }
	}
	else
	{
	    error=SetPlayMode( PCM_16_MONO );
	    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"mode : 16 bit mono.\n" );
	}
    }
    else
    {
	if( play_stereo )
	{
	    error=SetPlayMode( PCM_8_STEREO );
	    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"mode : 8 bit stereo.\n" );
	}
	else
	{
	    error=SetPlayMode( PCM_8_MONO );
	    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE,"mode : 8 bit mono.\n" );
	}
    }
    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "return code from setplaymode : %d\n" );
    // I finally know what are the settings.
    if( play_precision== 16 )
    {
	for (i = 0; i < number_of_channels; ++i)
	    bufferp[i] = buffer + i;
    }
    else
    {
	for (i = 0; i < number_of_channels; ++i)
	    cbufferp[i] = cbuffer + i;
    }
#else
    if( !CanMap_ConvMem() )
    {
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Cannot map conventional memory !\n" );
    }
    MD_RegisterDriver(&drv_nos);
    MD_RegisterDriver(&drv_sb);
    MD_RegisterDriver(&drv_ss);

    /*
	    Initialize soundcard parameters.. you _have_ to do this
	    before calling MD_Init(), and it's illegal to change them
	    after you've called MD_Init()
    */

    md_mixfreq      =dpm.rate;                     /* standard mixing freq */
    md_dmabufsize   =8000;                     /* standard dma buf size */
    md_mode=0;
    if( dpm.encoding & PE_16BIT )
	md_mode|=DMODE_16BITS;
    if( !(dpm.encoding & PE_MONO) )
	md_mode|=DMODE_STEREO;
    md_device       =0;                                                     /* standard device: autodetect */

    /*  initialize soundcard */
    if(!MD_Init())
    {
//	    printf("Driver error: %s.\n",myerr);
	    return 0;
    }
/*
    printf("Using %s for %d bit %s %s sound at %u Hz\n\n",
		    md_driver->Name,
		    (md_mode&DMODE_16BITS) ? 16:8,
		    (md_mode&DMODE_INTERP) ? "interpolated":"normal",
		    (md_mode&DMODE_STEREO) ? "stereo":"mono",
		    md_mixfreq);
*/
    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Using %s for %d bit %s %s sound at %u Hz\n\n",
		    md_driver->Name,
		    (md_mode&DMODE_16BITS) ? 16:8,
		    (md_mode&DMODE_INTERP) ? "interpolated":"normal",
		    (md_mode&DMODE_STEREO) ? "stereo":"mono",
		    md_mixfreq);

    dpm.encoding &= ~(PE_BYTESWAP|PE_ULAW);
    if( !(md_mode&DMODE_16BITS) )
    {
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Sample width adjusted to %d bits", 8 );
	dpm.encoding ^= PE_16BIT;
	warnings=1;
    }

    if (dpm.encoding & PE_16BIT)
    {
	dpm.encoding |= PE_SIGNED;
	play_precision=16;
    }
    else
    {
	dpm.encoding &= ~PE_SIGNED;
	play_precision=8;
    }

//    play_stereo = !(dpm.encoding & PE_MONO);

    if ( !(md_mode&DMODE_STEREO) )
    {
	dpm.encoding |= PE_MONO;
	play_stereo=0;
    }
    else
	play_stereo=1;
    if( dpm.rate > md_mixfreq )
    {
	dpm.rate=md_mixfreq;
	ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Output rate adjusted to %d Hz", dpm.rate);
	warnings=1;
    }
    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Sample width  %d bits", play_precision );
    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Stereo play %d", play_stereo );
    ctl->cmsg(CMSG_WARNING, VERB_VERBOSE, "Sample rate %d", dpm.rate );
#endif
    return warnings;
}

#ifndef DIGPAK
int lensample=0;
char *test=NULL;
GJ_SilenceBytes(char *buf,int aant)
{
	if(play_precision==16)
		memset(buf,0,aant);
	else
		memset(buf,0x80,aant);
}

int GJ_WriteBytes( char *buf, int aant )
{
    static int last=0;
    if( test==NULL ) return 0;

    if( last+aant>lensample )
    {
	int h1=lensample-last;
	memcpy( buf, test+last, h1 );
	last=aant-h1;
	if( last>lensample) last=lensample;
	memcpy( buf+h1, test, last);
	last=0;
    }
    else
    {
	memcpy( buf, test+last, aant );
	last+=aant;
    }
    if( last>=lensample ) last=0;

    return aant;
}

#endif

static void output_data(int32 *buff32, int32 length)
{
#ifdef DIGPAK
    int i;
    int channel=0;
    short int *buff=(short int *)buff32;
    if (!(dpm.encoding & PE_MONO)) length*=2; /* Stereo samples */

    s32tos16l(buff32, length); /* Little-endian data */

    if( play_precision==16 )
    {

	short int *ptr=buffer;
	for( i=0; i<length; i++ )
	{
	    *ptr=*buff;
	    buff++;
	    ptr++;
	}
	{
	write2( (char *)buffer, length*2 );
	}

/*
	write2( (char *)buff, length*2 );
*/
    }
    else
    {
	char *ptr=cbuffer;
	for( i=0; i<length; i++ )
	{
	    *ptr = (((unsigned short)*buff)>>8)+128;
	    buff++;
	    ptr++;
	}

	write2( cbuffer, length );
    }
#else
    if( have_to_start )
    {
	MD_PlayStart();
	have_to_start=0;
    }

  if (!(dpm.encoding & PE_MONO)) length*=2; /* Stereo samples */

  if (dpm.encoding & PE_16BIT)
    {
      /* Convert data to signed 16-bit PCM */
      s32tos16(buff32, length);
      lensample=length*2;
      test=buff32;
      MD_write_samples(lensample);
//      while ((-1==write(dpm.fd, buf, count * 2)) && errno==EINTR) ;
    }
  else
    {
      /* Convert to 8-bit unsigned and write out. */
      s32tou8(buff32, length);
      lensample=length;
      test=buff32;
      MD_write_samples(lensample);
//      while ((-1==write(dpm.fd, buf, count)) && errno==EINTR)	;
    }
#endif
}

static void write2(char *b, int32 length)
{
#ifdef DIGPAK
    static unsigned char _cbuffer[0x10000];

    static int cur_len=0;
    static char *soundbuffer=NULL;
    static char *soundbuffer2=NULL;
    static int   CurrBuff=0;
    static int   CheckLen=0x8000;

    if( length==-1 )
    {
	cur_len=0;
	return 0;
    }
    if ( !soundbuffer )
    {
	soundbuffer = (char *)realalloc(65535);
	if( getenv("buffer" ) )
	    CheckLen=atoi( getenv("buffer") );
    }
    if( !soundbuffer2 )
    {
	soundbuffer2=(char *)realalloc(65535);
    }
    if( b )
    {
	memcpy( ((char *)_cbuffer)+cur_len, b, length );
	cur_len+=length;
    }
    if( cur_len > CheckLen || !b )
    {
	int teller;
	if( CurrBuff )
	{
	    copyToLow( ((unsigned long)soundbuffer)>>4, 0 , _cbuffer, cur_len );
	    setLowMemDword( ((unsigned long)snd)>>4, 0, (unsigned long)RealPtr(soundbuffer));
	}
	else
	{
	    copyToLow( ((unsigned long)soundbuffer2)>>4, 0 , _cbuffer, cur_len );
	    setLowMemDword( ((unsigned long)snd)>>4, 0, (unsigned long)RealPtr(soundbuffer2));
	}

	// WaitSound(); // Wait for previous sound to complete.
	// setLowMemDword( ((unsigned long)snd)>>4, 0, (unsigned long)RealPtr(soundbuffer));
	setLowMemWord( ((unsigned long)snd)>>4, 4, cur_len );

	WaitSound(); // Wait for previous sound to complete.
	DigPlay(snd); // Massage and Play the sound effect.
	if( !b )
	    WaitSound();
	cur_len=0;
	if( CurrBuff )
	    CurrBuff=0;
	else
	    CurrBuff=1;
    }
#endif
}

static void close_output(void)
{
    flush_output();
    output_data( NULL, 0 );
#ifdef USE_DIGPAK
    StopSound(); // Wait for previous sound to complete.
    UnLoadDigPak();
    RemoveVectorLoader();
#else
    MD_PlayStop();          /* stop playing */
#endif
}

static void purge_output(void)
{
    flush_output();
}

static void flush_output(void)
{
#ifdef DIGPAK
    write2( "", -1 );
#else
    MD_PLAYSTOP();
    have_to_start=1;
#endif
}

