/*

    TiMidity -- Experimental MIDI to WAVE converter
    Copyright (C) 1995 Tuukka Toivonen <titoivon@snakemail.hut.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   common.c

   */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <errno.h>
#include "config.h"
#include "common.h"
#include "output.h"
#include "controls.h"

char *program_name, current_filename[1024];

FileType *ftypes;

#ifdef DEFAULT_PATH
    /* The paths in this list will be tried whenever we're reading a file */
    static PathList defaultpathlist={DEFAULT_PATH,0};
    static PathList *pathlist=&defaultpathlist; /* This is a linked list */
#else
    static PathList *pathlist=0;
#endif

/* Try to open a file for reading. If the filename ends in one of the 
   defined compressor extensions, pipe the file through the decompressor */
static FILE *try_to_open(char *name, int decompress, int noise_mode)
{
  FILE *fp;
  int compressed = 0;

  fp=fopen(name, "rb"); /* First just check that the file exists */

  if (!fp)
    return 0;

#ifdef DECOMPRESSOR_LIST
  if (decompress)
    {
      int l,el;
      static char *decompressor_list[] = DECOMPRESSOR_LIST, **dec;
      char tmp[1024], tmp2[1024], *cp, *cp2, tmpbuffer[81];
      /* Check if it's a compressed file */ 
      l=strlen(name);
      for (dec=decompressor_list; *dec; dec+=2)
	{
	  el=strlen(*dec);
	  if ((el>=l) || (strcmp(name+l-el, *dec)))
	    continue;

	  /* Yes. Close the file, open a pipe instead. */
	  fclose(fp);

	  /* Quote some special characters in the file name */
	  cp=name;
	  cp2=tmp2;
	  while (*cp)
	    {
	      switch(*cp)
		{
#ifndef DOS_COMPILE
		case '\'':
		case '\\':
		case '`':
		case '!':
		case '&':
#endif /* DOS_COMPILE */
		case '"':
		case ';':
		  *cp2++='\\';
		}
	      *cp2++=*cp++;
	    }
	  *cp2=0;
	  compressed = 1;
#ifndef DOS_COMPILE
	  sprintf(tmp, *(dec+1), tmp2);
	  fp=popen(tmp, "r");
#else
	  sprintf(tmp, *(dec+1), tmp2, "*.mid");
	  fp=popen(tmp, "rb");
	  fgets(tmpbuffer, 80, fp);
	  while(strcmp(strtok(tmpbuffer, " \t"), "Inflating:")) {
	    fgets(tmpbuffer, 80, fp);
	  }
	  fgets(tmpbuffer, 80, fp);
#endif  /* DOS_COMPILE */
	  break;
	}
    }
#endif  /* DECOMPRESSOR_LIST */
  add_filetype(fp, compressed);
  return fp;
}

/* This is meant to find and open files for reading, possibly piping
   them through a decompressor. */
FILE *open_file(char *name, int decompress, int noise_mode)
{
  FILE *fp;
  PathList *plp=pathlist;
  int l;

  if (!name || !(*name))
    {
      ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "Attempted to open nameless file.");
      return 0;
    }

  /* First try the given name */

  strncpy(current_filename, name, 1023);
  current_filename[1023]='\0';

  ctl->cmsg(CMSG_INFO, VERB_DEBUG, "Trying to open %s", current_filename);
  if ((fp=try_to_open(current_filename, decompress, noise_mode)))
    return fp;

  if (noise_mode && (errno != ENOENT))
    {
      ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "%s: %s", 
	   current_filename, sys_errlist[errno]);
      return 0;
    }
 
#ifndef DOS_COMPILE 
  if (name[0] != '/')
#else
  if (name[0] != '\\' && name[1] != ':')
#endif
    while (plp)  /* Try along the path then */
      {
	*current_filename=0;
	l=strlen(plp->path);
	if(l)
	  {
	    strcpy(current_filename, plp->path);
#ifndef DOS_COMPILE
	    if(current_filename[l-1]!='/')
	      strcat(current_filename, "/");
#else
	    if(current_filename[l-1]!='\\')
	      strcat(current_filename, "\\");
#endif
	  }
	strcat(current_filename, name);
	ctl->cmsg(CMSG_INFO, VERB_DEBUG, "Trying to open %s", current_filename);
	if ((fp=try_to_open(current_filename, decompress, noise_mode)))
	  return fp;
	if (noise_mode && (errno != ENOENT))
	  {
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "%s: %s", 
		 current_filename, sys_errlist[errno]);
	    return 0;
	  }
	plp=plp->next;
      }
  
  /* Nothing could be opened. */

  *current_filename=0;
  
  if (noise_mode>=2)
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "%s: %s", name, sys_errlist[errno]);
  
  return 0;
}

/* This closes files opened with open_file */
void close_file(FILE *fp)
{
#ifdef DECOMPRESSOR_LIST
  FileType *ftype = return_filetype(fp);
  if (ftype -> compress == 1)
    pclose(fp);
  else
#endif
    fclose(fp);
}

/* This is meant for skipping a few bytes in a file or fifo. */
void skip(FILE *fp, size_t len)
{
  size_t c;
  char tmp[1024];
  while (len>0)
    {
      c=len;
      if (c>1024) c=1024;
      len-=c;
      if (c!=fread(tmp, 1, c, fp))
	ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "%s: skip: %s",
	     current_filename, sys_errlist[errno]);
    }
}

/* This'll allocate memory or die. */
void *safe_malloc(size_t count)
{
  void *p;
  if (count > (1<<21))
    {
      ctl->cmsg(CMSG_FATAL, VERB_NORMAL, 
	   "Strange, I feel like allocating %d bytes. This must be a bug.",
	   count);
    }
  else if ((p=malloc(count)))
    return p;
  else
    ctl->cmsg(CMSG_FATAL, VERB_NORMAL, "Sorry. Couldn't malloc %d bytes.", count);

  ctl->close();
  play_mode->purge_output();
  play_mode->close_output();
  exit(10);
}

/* This adds a directory to the path list */
void add_to_pathlist(char *s)
{
  PathList *plp=safe_malloc(sizeof(PathList));
  strcpy((plp->path=safe_malloc(strlen(s)+1)),s);
  plp->next=pathlist;
  pathlist=plp;
}

void add_filetype(FILE *fp, int compressed) {
  FileType *new, *curr = ftypes;

  /* Allocate FileType and initialize */
  new = (FileType *) malloc (sizeof(FileType));
  new -> next = NULL;
  new -> file = fp -> _file;
  new -> compress = compressed;

  /* Add to list */
  if (curr == NULL)
    ftypes = new;
  else {
    while (curr -> next != NULL) curr = curr -> next;
    curr -> next = new;
  }
}

FileType *return_filetype(FILE *fp) {
  FileType *curr = ftypes, *other = ftypes -> next;

  /* Check to see if head is the ftype to be removed */
  if (ftypes -> file == fp -> _file) {
    ftypes = ftypes -> next;
    return curr;
  }
  /* otherwise do this */
  else {
    while ((other != NULL) && (other -> file != fp -> _file)) {
      curr = other;
      other = other -> next;
    }
    curr -> next = other -> next;
    return other;
  }
}
