/*

    TiMidity -- Experimental MIDI to WAVE converter
    Copyright (C) 1995 Tuukka Toivonen <titoivon@snakemail.hut.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    dos_sb.c, Anthony Cruz (keyboard@engin.umich.edu)

    based off of information in SoundBlaster programming info v0.90,
    sbio.c (Ethan Brodsky) and linux_audio.c (Tuuka Toivonen)


    plain and Pro SB support, Denis Sablic (klint@fly.cc.fer.hr)

*/


/*
 *   Headers needed to compile under DJGPP
 */
#include <stdio.h>
#include <string.h>
#include <go32.h>
#include <dos.h>
#include <dpmi.h>
#include <string.h>
#include <malloc.h>
#include <pc.h>
/*
 *   TiMidity header files
 */
#include "config.h"
#include "output.h"
#include "controls.h"

/*
 *   Useful DSP definitions
 */
#define DSP_RESET     (baseio+0x6)
#define DSP_READ      (baseio+0xa)
#define DSP_WRITE     (baseio+0xc)
#define DSP_DATA      (baseio+0xe)
#define DSP_IRQ8      DSP_DATA
#define DSP_IRQ16     (baseio+0xf)

#define MIXER_INDEX   (baseio+0x4)
#define MIXER_DATA    (baseio+0x5)

/*
 *   Other useful definitions
 */
#define lo(x)    (unsigned char) ((x) & 0x0ff)
#define hi(x)    (unsigned char) ((x) >> 8)

#define BUFFERS 5

/*
 *   Internal routines for DMA & SB card
 */
void fill_buffer(int block, int32 *src, int32 length);
void acknowledge_interrupt(void);
void compute_variables(void);
void allocate_buffers(int32 count);
void install_handler(void);
void interrupt_handler(void);
void startio(void);
int reset_dsp(void);
void write_dsp(unsigned char data);
int get_sb_environment(void);
unsigned int get_dsp_version(void);
unsigned char read_dsp(void);

/*
 *   Routines needed by TiMidity
 */
static int open_output(void);
static void close_output(void);
static void output_data(int32 *buf, int32 count);
static void flush_output(void);
static void purge_output(void);

/* export the playback mode */

#define dpm dos_sb_play_mode

PlayMode dpm = {
  DEFAULT_RATE, PE_16BIT|PE_SIGNED,
  -1, {0},
  "SoundBlaster", 'b',
  "SoundBlaster",
  open_output,
  close_output,
  output_data,
  flush_output,
  purge_output
};

/*
 *   GLOBAL Variables
 *
 *   This is usually a bad thing to do, but I didn't feel like
 *   passing all the arguments to each function.
 */

char *sb_models[] = {"SoundBlaster 2.0","SoundBlaster Pro",
                     "SoundBlaster Pro 2","SoundBlaster 16",
                     "SoundBlaster 16 Value Edition or AWE 32"};

unsigned int baseio = 0xffff;     /* SB base I/O port */
unsigned char dma8 = 0xf;         /* SoundBlaster low dma channel */
unsigned char dma16 = 0xf;        /* SoundBlaster high dma channel */
unsigned char dma = 0xf;          /* DMA channel SoundBlaster will use */
unsigned int sb_dsp = 0xffff;     /* SB dsp version */
unsigned char sb_irq = 0xf;       /* SB IRQ */
unsigned int sb_int = 0xffff;     /* SB Interrupt vector */
unsigned char fill_ok = 0;        /* Flag for filling buffers */
unsigned char curblock = BUFFERS - 1;  /* The buffer block to fill */
unsigned char prime_buffers = BUFFERS; /* Flag to fill buffers before playing */

_go32_dpmi_seginfo OldIRQ;        /* Old IRQ segment info */
_go32_dpmi_seginfo NewIRQ;        /* Interrupt handler */
_go32_dpmi_seginfo DOSmem;        /* DOS (conventional) memory */

unsigned char buf_page;           /* DMA buffer page */
unsigned int buf_ofs;             /* DMA buffer offset */
int buf_length;                   /* DMA buffer length */
int block_length;                 /* DMA block length = 1/2 buffer length */
char *sb_buf[BUFFERS];                  /* DMA buffer pointer */

int32 last_count;                 /* Keeps track of data sizes */

typedef struct {
  int mask, ff, mode, base, count, page;
  char disable, enable, type;
} DMA_STRUCT;

DMA_STRUCT ctrl;                  /* Hold DMA controller program info */

/*
 *   TiMidity needed routines
 */
static int open_output(void)
{
  int warnings = 0;

  if (!get_sb_environment()) {
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
              "Could not get BLASTER environment");
    return -1;
  }

  dpm.encoding &= ~(PE_BYTESWAP|PE_ULAW);

  if (!reset_dsp()) {
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
              "Error resetting %s DSP", dpm.name);
    return -1;
  }

  if ((sb_dsp = get_dsp_version()) == 0xffff) {
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
              "Could not detect %s hardware", dpm.name);
    return -1;
  }
  else if (sb_dsp < 0x200) {
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
              "DOS_SB.C only supports SB 2.0 cards and up");
    return -1;
  }

  if (dma16 == 0xf && dma8 == 0xf) {
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
              "DMA channels not detected in environment");
    return -1;
  }
  else if (dma16 == 0xf && (dpm.encoding & PE_16BIT)) {
    ctl->cmsg(CMSG_WARNING, VERB_NORMAL,
              "16-bit DMA not detected...switching to 8-bit");
    dpm.encoding ^= PE_16BIT;
    warnings = 1;
  }
  else if (dma8 == 0xf && !(dpm.encoding & PE_16BIT)) {
    ctl->cmsg(CMSG_WARNING, VERB_NORMAL,
              "8-bit DMA not detected...switching to 16-bit");
    dpm.encoding |= PE_16BIT;
    warnings = 1;
  }

  ctl->cmsg(CMSG_INFO, VERB_NORMAL, sb_models[sb_dsp<0x300?0:sb_dsp==0x300?1:sb_dsp<0x400?2:sb_dsp<0x40c?3:4]);
  ctl->cmsg(CMSG_INFO, VERB_NORMAL, "found and used ...\n");

  if ((sb_dsp < 0x400) && (dpm.encoding & PE_16BIT)) dpm.encoding ^= PE_16BIT;

  dma = (dpm.encoding & PE_16BIT) ? dma16 : dma8;

  compute_variables();
  install_handler();
  return warnings;
}

static void output_data(int32 *buf, int32 count)
{
  int32 timeout = 1000000;
  if (!(dpm.encoding & PE_MONO)) count *= 2;     /* stereo samples */
  if (dpm.encoding & PE_16BIT)
    s32tos16(buf, count);
  else 
  	if (sb_dsp < 0x400) {
	  s32tou8(buf, count);
	  timeout <<= 1;
	}
    else
      s32tos8(buf, count);
  if (prime_buffers) {
    if (prime_buffers == BUFFERS) {
      allocate_buffers(count);
    }
    fill_buffer(BUFFERS - prime_buffers, buf, count);
    prime_buffers--;  fill_ok = 0;
    if (!prime_buffers) startio();
    last_count = count;
  }
  else {
    while (!fill_ok && timeout--);
    curblock=(curblock+1)%BUFFERS;
    if (fill_ok) fill_ok--;
    fill_buffer(curblock, buf, count);

    /* Reached the last buffer */
    if (count != last_count) {
      fill_ok = 0;
      prime_buffers = BUFFERS;
      curblock = BUFFERS - 1;
    }
  }
  last_count = count;
  if (timeout <= 0) {
    acknowledge_interrupt();
  }
}

static void close_output(void) {
  disable();
  _go32_dpmi_set_protected_mode_interrupt_vector(sb_int, &OldIRQ);
  enable();
  outportb(0x20, 0x20);
  outportb(0xa0, 0x20);
  flush_output();
}

static void purge_output(void) {
  flush_output();
  fill_ok = 0;
  prime_buffers = BUFFERS;
  curblock = BUFFERS - 1;
}

static void flush_output(void) {
  /* Free memory */
  _go32_dpmi_free_dos_memory(&DOSmem);

  /* Reset DSP */
  if (sb_dsp < 0x400) {
    reset_dsp();
    write_dsp(0xd3);
    return;
  }
  if (dpm.encoding & PE_16BIT) {
    write_dsp(0xd9);
  }
  else {
    write_dsp(0xda);
  }
}
/* =============================================================== */

/*
 *   Internal functions for driver
 */
int get_sb_environment(void)
{
  char *t, *blaster;

  t = getenv("BLASTER");
  if (!t) return 0;

  blaster = strdup(t);

  t = strtok(blaster, " \t");
  while(t) {
    switch(t[0]) {
      case 'a':
      case 'A':
        sscanf(t+1, "%x", &baseio); break;
      case 'i':
      case 'I':
        sb_irq = atoi(t+1); break;
      case 'd':
      case 'D':
        dma8 = atoi(t+1); break;
      case 'h':
      case 'H':
        dma16 = atoi(t+1); break;
      default:
        break;
    }
    t = strtok(NULL, " \t");
  }
  free(blaster);

  sb_int = (sb_irq < 8) ? (sb_irq + 0x08) : (sb_irq + 0x70);
  if (sb_irq == 0xf || baseio == 0xffff || (dma8 == 0xf && dma16 == 0xf)) 
    return 0;

  return 1;
}

/*
 *   Low level DSP operations
 */
int reset_dsp(void) {
  outportb(DSP_RESET,1);
  delay(1);
  outportb(DSP_RESET,0);
  delay(1);
  return (((inportb(DSP_DATA) & 0x80) == 0x80 && inportb(DSP_READ) == 0xaa));
}

unsigned char read_dsp(void) {
  while(!(inportb(DSP_DATA) & 0x80));
  return (inportb(DSP_READ));
}

void write_dsp(unsigned char data) {
  while(inportb(DSP_WRITE) & 0x80);
  outportb(DSP_WRITE, data);
}

void write_mixer(unsigned char index, unsigned char data)
{
  outportb(MIXER_INDEX,index);
  outportb(MIXER_DATA,data);
}

unsigned int get_dsp_version(void) {
  unsigned int hi, lo;
  write_dsp(0xe1);
  hi = read_dsp();
  lo = read_dsp();
  return((hi << 8) + lo);
}

/*
 *   Memory routines (Allocation, transfer)
 */
void allocate_buffers(int32 count) {
  unsigned long buf_addr;
  int loop;

  if (dpm.encoding & PE_16BIT) count *= 2;

  DOSmem.size = 65536*3/16;
  if (_go32_dpmi_allocate_dos_memory(&DOSmem)) {
    printf("Unable to allocate DMA blocks\n");
    printf("Maximum allocation size: %ld\n", DOSmem.size);
    exit(1);
  }

  /* Align DMA block */
  (unsigned long) sb_buf[0]  = DOSmem.rm_segment * 16;
  (unsigned long) sb_buf[0] += 0x0000ffffL;
  (unsigned long) sb_buf[0] &= 0xffff0000L;

  for (loop = 1; loop < BUFFERS; loop++) {
    (unsigned long) sb_buf[loop]  = (unsigned long) sb_buf[loop-1] + count;
  }

  buf_addr = (unsigned long) sb_buf[0];

  if (dpm.encoding & PE_16BIT) {
    /* Need all of this in words */
    buf_ofs = (buf_addr >> 1) % 65536;
    buf_length = count / 2 * BUFFERS;
    block_length = count / 2;
  }
  else {
    /* If 8-bit, then do it in bytes */
    buf_ofs = buf_addr % 65536;
    buf_length = count * BUFFERS;
    block_length = count;
  }
  buf_page = buf_addr >> 16;
}

void fill_buffer(int block, int32 *src, int32 length) {
  if (dpm.encoding & PE_16BIT) length *= 2;
  dosmemput(src, length, (unsigned long) sb_buf[block]);
}

/*
 *   Interrupt routines
 */
void interrupt_handler(void)
{
  /* Increment interrupt counts */
  fill_ok++;
  
  acknowledge_interrupt();
}

void acknowledge_interrupt(void)
{
  /* Acknowledge SoundBlaster card */
  if (dpm.encoding & PE_16BIT)
	inportb(DSP_IRQ16);
  else
	inportb(DSP_IRQ8);

  /* Acknowledge the interrupt */
  outportb(0xa0, 0x20);
  outportb(0x20, 0x20);
}

void install_handler(void) {
  unsigned char irq_disable;
  unsigned char irq_enable;
  unsigned char irq_mask;

  irq_disable = 1 << (sb_irq % 8);
  irq_enable  = ~irq_disable;
  irq_mask = inportb(0x21);

  disable();
  outportb(0x21, irq_mask | irq_disable);
  _go32_dpmi_get_protected_mode_interrupt_vector(sb_int, &OldIRQ);
  NewIRQ.pm_offset = (int) interrupt_handler;
  NewIRQ.pm_selector = _go32_my_cs();
  _go32_dpmi_chain_protected_mode_interrupt_vector(sb_int, &NewIRQ);
  outportb(0x21, irq_mask & irq_enable);
  enable();
}

/*
 *   Other necessary routines
 */
void compute_variables(void) {
  if (dpm.encoding & PE_16BIT) {
    ctrl.mask = 0xd4; ctrl.mode = 0xd6; ctrl.ff = 0xd8;
    ctrl.base = 0xc0 + 4*(dma16-4); ctrl.count = ctrl.base + 2;
    switch(dma) {
      case 4: ctrl.page = 0x8f; break;
      case 5: ctrl.page = 0x8b; break;
      case 6: ctrl.page = 0x89; break;
      case 7: ctrl.page = 0x8a; break;
    }
  }
  else {
    ctrl.mask = 0x0a; ctrl.mode = 0x0b; ctrl.ff = 0x0c;
    ctrl.base = 2 * dma8; ctrl.count = ctrl.base + 1;
    switch(dma) {
      case 0: ctrl.page = 0x87; break;
      case 1: ctrl.page = 0x83; break;
      case 2: ctrl.page = 0x81; break;
      case 3: ctrl.page = 0x82; break;
    }
  }
  ctrl.type = (dma % 4) + 0x58;
  ctrl.disable = (dma % 4) + 0x04;
  ctrl.enable = (dma % 4);
}

void startio(void) {
  unsigned int rate=dpm.rate;

  /* Program the DMA controller */
  outportb(ctrl.mask, ctrl.disable);
  outportb(ctrl.ff, 0);
  outportb(ctrl.mode, ctrl.type);
  outportb(ctrl.base, lo(buf_ofs));
  outportb(ctrl.base, hi(buf_ofs));
  outportb(ctrl.count, lo(buf_length-1));
  outportb(ctrl.count, hi(buf_length-1));
  outportb(ctrl.page, buf_page);
  outportb(ctrl.mask, ctrl.enable);

  /* Program SoundBlaster card */

  write_dsp(0xd1);

  if(sb_dsp<0x400) {
 	if((dpm.encoding & PE_MONO) && sb_dsp>=0x300) 
	  write_mixer(0x0e,0); 
	else { 
	  write_mixer(0x0e,2);
	rate <<= 1;
	}
    if(rate > 44100) rate = 44100;
	write_dsp(0x40);
	write_dsp((unsigned char)(256 - 1000000 / rate));
	write_dsp(0x48);
	write_dsp(lo(block_length-1));
	write_dsp(hi(block_length-1));
	write_dsp(0x90);
  	return;
  }

  write_dsp(0x41);
  write_dsp(hi(dpm.rate));
  write_dsp(lo(dpm.rate));
  if (dpm.encoding & PE_16BIT)
    write_dsp(0xb6);
  else
    write_dsp(0xc6);
  if (dpm.encoding & PE_MONO)
    write_dsp(0x10);
  else
    write_dsp(0x30);
  write_dsp(lo(block_length-1));
  write_dsp(hi(block_length-1));
}
