/*
 *   TIMIDITY non-DMA output device (PC speaker & DAC-LPT ...)
 *
 *   BySte - Marago` Stefano - marago@cselt.stet.it
 *                             Torino - Italia
 *
 *   Based on DOS port (with SB output)
 *   by Anthony Cruz (keyboard@engin.umich.edu)
 *
 *   Thanks to Denis Sablic (klint@fly.cc.fer.hr) for his assistance.
 *
 *   This code can manage all the non DMA devices
 *   by the means of a process that flush output by the
 *   timer interrupt (reprogrammed for high-frequency intervals).
 *   (Also other soundcards could be managed in polling mode...;)
 *
 *   The speaker output is obtained by a PWM modulation.
 *
 */

/*
 * TODO:
 *    - How detect a fatal error (exception) in djgpp?
 *                                   \-> The timer must be resetted to 18.2/s!
 *    - How to call the default timer interrupt a subset of times?
 *    - The speaker click a bit! ...
 *    - Get the lpt-n addres from bios 0:408 ...
 */

/*
 *   Headers needed to compile under DJGPP
 */

#include <stdio.h>
#include <string.h>
#include <go32.h>
#include <dos.h>
#include <dpmi.h>
#include <string.h>
#include <malloc.h>
#include <pc.h>

/*
 *   TiMidity header files
 */

#include "config.h"
#include "output.h"
#include "controls.h"

/*
 *   Routines needed by TiMidity
 */

static int  open_output  ( void );
static void output_data  ( int32 *buf, int32 count );
static void flush_output ( void );
static void purge_output ( void );
static void close_output ( void );

/* internal functions */

static void   install_handler ( void );
static void uninstall_handler ( void );
static void interrupt_handler ( void );

static void new_s32tou8 ( int32 *lp, int32 c );

/* export the playback mode */

#define dpm dos_spkr_play_mode /* new mode ... */

PlayMode dpm = {   /* {_} this parameter is useful for determining spk or lpt (or whatever) output ... */
   9316, PE_MONO, -1, {0}, "PC Speaker & LPT-DAC", 's', "PC Speaker & LPT-DAC",
   open_output, close_output, output_data, flush_output, purge_output
};

/*
 *   Variables (& costants!)
 */

static short int lptn, lpt_addr ; /* selected device output number & address */

#define BUFFERS 7  /* not too many, or else wait between keypress and actions! */

#define timer_int    8                  /* Timer IRQ, re-programmed FAST */
#define hw_timer_irq ( timer_int - 8 )  /* PIC: timer int# */

static unsigned int Timer_FrequencyDivider ; /* default=56 -> 21250 Hz output rate */
#define MainFrequency 1192500                /* old XT 4.77 (other decimals?) MHz / 4 */

volatile /* buf_head is accessed and modified from interrupt routine */
static unsigned char   buf_head = 0 ;           /* Ring for buffer filling   */
volatile /* not really necessary! ... */
static unsigned char   buf_tail = 0 ;           /* 0 .. BUFFERS - 1          */
static unsigned char * buf_pointer[ BUFFERS ] ; /* buffers pointers          */
static unsigned short  buf_length [ BUFFERS ] ; /* length of each one        */

static volatile unsigned char * current_buffer ; /* for optimized calcs in the interrupt routine */
static volatile unsigned short  current_length ;

static _go32_dpmi_seginfo OldIRQ ;  /* Old IRQ segment info */
static _go32_dpmi_seginfo NewIRQ ;  /* Interrupt handler */

static char msg[160] ;
volatile static char nop ; /* used to do nothing! (little pause) */

#define lo(x) (unsigned char) ( (x) & 0x0ff )
#define hi(x) (unsigned char) lo( (x) >> 8 )

/*
 *   TiMidity needed routines
 */

static int open_output( void ) {

   int warnings = 0 ;

   dpm.encoding = PE_MONO ; /* and 8bit, and unsigned */
   Timer_FrequencyDivider = (long) MainFrequency / (long) dpm.rate ;
   dpm.rate = (long) MainFrequency / (long) Timer_FrequencyDivider ; /* correct int approx. */

   lptn = dpm.extra_param[0] ;
   if(  lptn == 0 ) {
      /* speaker output */
      sprintf( msg, " %u Hz - Speaker output - BySte, Stefano Marago` \n", dpm.rate );
   } else {
      /* LPTn output */
      if( lptn == 1 ) lpt_addr = 0x378 ;
      else if( lptn == 2 ) lpt_addr = 0x278 ;
      else if( lptn == 3 ) lpt_addr = 0x3BC ;
      sprintf( msg, " %u Hz - LPT%i (%xh) DAC output - BySte, Stefano Marago` \n", dpm.rate, lptn, lpt_addr );
   }
   ctl->cmsg( CMSG_INFO, VERB_NORMAL, msg );

   install_handler();

   return warnings ;

}

static void output_data( int32 *inbuf, int32 count ) {

   /* add on buffer tail */

   /* convert buffer to 8 bit ... unsigned! */
   s32tou8( inbuf, count );

   /* wait for a free slot */
   while( ( buf_tail + 1 ) % BUFFERS == buf_head ) ;

   /* allocate buffer (and if the buffer is yet allocated free it!) */
   /* (not necessary to free if it has the same dimension of the old one!) */
   if( ( buf_pointer[buf_tail] != NULL ) && ( buf_length[buf_tail] != count ) )
      free( buf_pointer[ buf_tail ] );
   if( ( buf_pointer[buf_tail] == NULL ) || ( buf_length[buf_tail] != count ) ) {
      buf_length [ buf_tail ] = count ;
      buf_pointer[ buf_tail ] = (unsigned char *) malloc( count ) ;
      if( buf_pointer[buf_tail] == NULL ) {
         printf( "Unable to allocate pseudo-DMA memory block \n" );
         exit( 1 );
      }
   }

   memcpy( buf_pointer[buf_tail], inbuf, count ); /* could be done within new_s32tou8() ... */

   /* if not ready in the interrupt routine set the current buffer */
   if( buf_head == buf_tail ) {
      current_buffer = buf_pointer[ buf_head ] ;
      current_length = buf_length[ buf_head ] ;
   }

   /* ok to run: set new tail (the interrupt routine process concurrently!) */
   buf_tail = ( buf_tail + 1 ) % BUFFERS ;

}

/* static void new_s32tou8( int32 *lp , int32 c ) {

   uint8 * cp = ( uint8 * ) lp ;
   int32 l ;

   while( c -- ) {
      l = ( * lp ++ ) >> ( 32 - ( 8 + GUARD_BITS ) ) ;
      if( l > 127 ) l = 127 ;
      else if( l < -128 ) l = -128 ;
      * cp ++ = 128 + l ;
   }

} */

static void close_output( void ) {

   flush_output() ;

   uninstall_handler() ;

}

static void purge_output( void ) {

   flush_output() ;

}

static void flush_output( void ) {

   while( buf_head != buf_tail );

}

/* ===============================================================
 *
 *   Internal functions for driver:
 *
 *   Interrupt routine
 *
 */

/* better or not? ... */
/* static unsigned char precalc[ 256 ] ; */

/* should depend upon FrequencyDivider; (see spline weight functions ;> ...) */
static unsigned char SpeakerXLat[ 256 ] = {
   0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x3F,0x3F,0x3F,0x3F,0x3F,
   0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3E,0x3E,0x3E,0x3E,0x3E,0x3E,0x3E,0x3E,0x3E,0x3E,
   0x3D,0x3D,0x3D,0x3D,0x3D,0x3D,0x3D,0x3D,0x3D,0x3D,0x3C,0x3C,0x3C,0x3C,0x3C,0x3C,
   0x3C,0x3C,0x3C,0x3C,0x3B,0x3B,0x3B,0x3B,0x3B,0x3B,0x3B,0x3B,0x3B,0x3B,0x3A,0x3A,
   0x3A,0x3A,0x3A,0x3A,0x3A,0x3A,0x3A,0x3A,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,
   0x39,0x38,0x38,0x38,0x38,0x38,0x38,0x38,0x38,0x37,0x37,0x37,0x37,0x37,0x36,0x36,
   0x36,0x36,0x35,0x35,0x35,0x35,0x34,0x34,0x34,0x33,0x33,0x32,0x32,0x31,0x31,0x30,
   0x30,0x2F,0x2E,0x2D,0x2C,0x2B,0x2A,0x29,0x28,0x27,0x26,0x25,0x24,0x23,0x22,0x21,
   0x20,0x1F,0x1E,0x1D,0x1C,0x1B,0x1A,0x19,0x18,0x17,0x16,0x15,0x14,0x13,0x12,0x11,
   0x11,0x10,0x10,0x0F,0x0F,0x0E,0x0E,0x0D,0x0D,0x0D,0x0C,0x0C,0x0C,0x0C,0x0B,0x0B,
   0x0B,0x0B,0x0A,0x0A,0x0A,0x0A,0x0A,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x09,0x08,
   0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x07,0x07,0x07,0x07,0x07,0x07,0x07,0x07,
   0x07,0x07,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x05,0x05,0x05,0x05,
   0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
   0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,
   0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01
};

/* interrupt!! */
static void interrupt_handler( void ) {

   /* must be very FAST: it is called many times in a second! */
   /* warning: the old int is not called for now (it should be implemented...) */

   static unsigned short current_sample = 0 ;
   static unsigned char t ; /* (probably) static faster than automatic */

   if( buf_head != buf_tail ) {

      /* get datum from head */
      t = * ( current_buffer + current_sample );
      current_sample ++ ;

      if( lptn == 0 ) {

         outportb( 0x42 , SpeakerXLat[ t ] ); /* alternative! */
         /* outportb( 0x42 , precalc[t] ); */ /* DENIS */
         /* outportb( 0x42 , ( t >> 3 ) + 1 ); /* >> 2 o 3 ? ... */

      } else {

         outportb( /*0x378*/ lpt_addr , t ); /* LPTn DAC */

      }

      if( current_sample >= current_length ) {

         /* current buffer has been flushed out */

         buf_head = ( buf_head + 1 ) % BUFFERS ; /* buf_head must be volatile */

         current_buffer = buf_pointer[ buf_head ] ; /* if it is not ready them will be set from output_data() */
         current_length = buf_length[ buf_head ] ;

         current_sample = 0 ;

      }

   }

   /* acknowledge the interrupt */
   outportb(0x20, 0x20);

   /* int_return ; ... */
   /* should call the original int every 1/18.2 times within a second ! */

}

static void install_handler( void ) {

   unsigned char irq_disable;
   unsigned char irq_enable;
   unsigned char irq_mask;
   int i ;

   #ifdef NEW_DENIS
   int c ;
   for( c = 0 ; c < 256 ; c ++ )
      precalc[ c ] = (unsigned char) ( c * Timer_FrequencyDivider / 256.0 ) + 1 ;
      /* precalc[ c ] = (unsigned char) ( c * Timer_FrequencyDivider / 512.0 ) + 1 ; /* Ste: 512 could be better ... */
   #endif NEW_DENIS

   /* init buffer */
   for( i = 0 ; i < BUFFERS ; i ++ ) buf_pointer[ i ] = NULL ;

   /* init speaker */
   outportb( 0x43 , 0x90 );  /* program timer channel 2 (speaker), mode 2: pulse every N / 1.19M sec.
                             /* 90h -> play with out 42h,MSB; else B4h -> out 42h,LSB & out 42h,MSB */
   outportb( 0x61 ,  inportb( 0x61 ) | 3 ); /* flip-flop speaker on (gate on) */

   /* irq setting */

   irq_disable = 1 << ( hw_timer_irq % 8 );
   irq_enable = ~ irq_disable;
   irq_mask = inportb( 0x21 );

   disable();
   outportb( 0x21, irq_mask | irq_disable ); /* mask int */

   /* set handler */
   _go32_dpmi_get_protected_mode_interrupt_vector( timer_int, &OldIRQ );
   NewIRQ.pm_offset = (int) interrupt_handler ;
   NewIRQ.pm_selector = _go32_my_cs() ;
   _go32_dpmi_allocate_iret_wrapper( &NewIRQ );
   _go32_dpmi_set_protected_mode_interrupt_vector( timer_int, &NewIRQ );

   /* I don't want the normal timer routine to be called so often!
    * _go32_dpmi_chain_protected_mode_interrupt_vector( timer_int, &NewIRQ );
    * (without iret wrapper...) */

   /* set rate: timer channel 0 (timer int), mode 2 (BIOS=3) */
   disable(); /* confirm */
   outportb( 0x43 , 0x34  );
   nop ++ ;   /* little pause */
   outportb( 0x40 , lo( Timer_FrequencyDivider ) );
   nop ++ ;
   outportb( 0x40 , hi( Timer_FrequencyDivider ) );

   outportb( 0x21, irq_mask & irq_enable ); /* int ok */
   enable();

}

static void uninstall_handler( void ) {

   disable();

   /* reset timer channel 0, 18.2/s */
   outportb( 0x43 , 0x34 ); /* or 0x36 ...? */
   nop ++ ;
   outportb( 0x40 , 0 );    /* 18.2 times in a second */
   nop ++ ;
   outportb( 0x40 , 0 );

   /* reset int handler */
   _go32_dpmi_set_protected_mode_interrupt_vector( timer_int, &OldIRQ );
   _go32_dpmi_free_iret_wrapper( &NewIRQ ); /* timer iret */

   enable();

   outportb( 0x43 , 0xB6  ); /* reset timer channel 2 (speaker) */
   nop ++ ;
   outportb( 0x42 , 0 );
   nop ++ ;
   outportb( 0x42 , 0 );

   outportb( 0x61 , inportb( 0x61 ) & 0xFC ); /* flip-flop speaker off (gate off) */

}

