#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bulk.h"

BULK *alloc_bulk(u_int au)
{
  BULK *bulk;

  if ((bulk = (BULK*) malloc(sizeof(BULK))) == NULL)
    return NULL;

  bulk->alloc_unit = au;
  bulk->i = 0;
  bulk->size = 0;
  bulk->data = NULL;

  return bulk;
}

void free_bulk(BULK *bulk)
{
  free(bulk->data);
  free(bulk);
}

BULK *resize_bulk(BULK *bulk, u_int size)
{
  bulk->size = (((bulk->i + size) / bulk->alloc_unit) + 1) * bulk->alloc_unit;

  /*fprintf(stderr,"RE %d %d %02X %d %d\n",bulk->i,bulk->alloc_unit,bulk->data[0],size,bulk->size);*/

  if ((bulk->data = (u_char*) realloc(bulk->data, bulk->size)) == NULL)
    return NULL;

  return bulk;
}

/*
 *  read functions
 */

int read_byte(BULK *bulk, u_char *value)
{
  if (bulk->i + 1 > bulk->size)
    return -1;

  *value = bulk->data[bulk->i++];

  return 1;
}

int read_2bytes(BULK *bulk, u_short *value)
{
  if (bulk->i + 2 > bulk->size)
    return -1;

  *value = bulk->data[bulk->i++];
  *value = (*value << 8) + bulk->data[bulk->i++];

  return 2;
}

int read_4bytes(BULK *bulk, u_int *value)
{
  if (bulk->i + 4 > bulk->size)
    return -1;

  *value = bulk->data[bulk->i++];
  *value = (*value << 8) + bulk->data[bulk->i++];
  *value = (*value << 8) + bulk->data[bulk->i++];
  *value = (*value << 8) + bulk->data[bulk->i++];

  return 4;
}

int read_nbytes(BULK *bulk, u_char **p, int n)
{
  if (bulk->i + n > bulk->size)
    return -1;

  if (p != NULL)
    *p = &bulk->data[bulk->i];
  bulk->i += n;

  return n;
}

int read_file_bulk(FILE *fp, BULK *bulk)
{
  int read_size;

  do {
    if (resize_bulk(bulk, bulk->i + bulk->alloc_unit) == NULL)
      return -1;
    read_size = fread(&bulk->data[bulk->i], 1, bulk->alloc_unit, fp);
    bulk->i += read_size;
  } while (read_size == bulk->alloc_unit);

  bulk->size = bulk->i;

  return bulk->i;
}

/*
 *  write functions
 */

int write_byte(BULK *bulk, u_char value)
{
  if (bulk->i + 1 > bulk->size && resize_bulk(bulk, bulk->i + 1) == NULL)
    return -1;

  bulk->data[bulk->i++] = value;

  return 1;
}

int write_2bytes(BULK *bulk, u_short value)
{
  if (bulk->i + 2 > bulk->size && resize_bulk(bulk, bulk->i + 2) == NULL)
    return -1;

  bulk->data[bulk->i++]  = (value >> 8) & 0xff;
  bulk->data[bulk->i++]  = value & 0xff;

  return 2;
}

int write_4bytes(BULK *bulk, u_int value)
{
  if (bulk->i + 4 > bulk->size && resize_bulk(bulk, bulk->i + 4) == NULL)
    return -1;

  bulk->data[bulk->i++]  = (value >> 24) & 0xff;
  bulk->data[bulk->i++]  = (value >> 16) & 0xff;
  bulk->data[bulk->i++]  = (value >> 8) & 0xff;
  bulk->data[bulk->i++]  = value & 0xff;

  return 4;
}

int write_nbytes(BULK *bulk, u_char *p, int n)
{
  if (bulk->i + n > bulk->size && resize_bulk(bulk, bulk->i + n) == NULL)
    return -1;

  if (p != NULL)
    memcpy(&bulk->data[bulk->i], p, n);
  bulk->i += n;

  return n;
}

int write_file_bulk(FILE *fp, BULK *bulk)
{
  if (fwrite(bulk->data, 1, bulk->i, fp) < bulk->i)
    return -1;

  return bulk->i;
}
