#include <stdio.h>

#include "rcp.h"

int rcp_read_header(BULK *bulk, RCPhd *rcphd)
{
  int i;
  u_char *buf;

  if (read_nbytes(bulk, &buf, 0x206) < 0)
    return -1;

  rcphd->timebase = (buf[0x1e7] << 8) + buf[0x1c0];
  rcphd->play_bias = buf[0x1c5];
  rcphd->bpm = buf[0x1c1];

  for (i = 0; i < 32; i++) {
    if (read_nbytes(bulk, NULL, 16) < 0)
      return -1;
  }

  for (i = 0; i < 8; i++) {
    if (read_nbytes(bulk, &rcphd->usrex[i].memo, 24) < 0
	|| read_nbytes(bulk, &rcphd->usrex[i].data, 24) < 0)
	return -1;
    /*fprintf(stderr,"EX %d: %24s\n",i,rcphd->usrex[i].memo);*/
  }
  /*fprintf(stderr,"TB %d\n",rcphd->timebase);*/

  return 0x586;
}

int rcp_read_track(BULK *bulk, RCPhd *rcphd, PACKET *packet)
{
  int i;
  int start_sum = 0;
  int running = 0;
  int sp = 0;
  int step = 0;
  int same_measure = 0;
  u_char c;
  u_char gts[128];
  u_char *buf;
  u_char ch, key_plus, st_plus, mode;
  u_char rolex[24] = { 0xf0, 0x0a, 0x41, 0, 0, 0x12 };
  u_char running_buf[1024];
  u_int tick = 0;
  u_int toberead;
  u_int track_top = bulk->i;
  struct { u_int i; int count; u_int toberead; } stack[16];

  track_top = bulk->i;
  if (read_nbytes(bulk, &buf, 2) < 0)
    return -1;

  toberead = ((buf[1] << 8) | buf[0]);
  if (toberead <= 2)
    return -1;
  toberead -= 2;

  if (read_nbytes(bulk, &buf, 42) < 0)
    return -1;

  toberead -= 42;
  ch = buf[2];
  key_plus = (buf[3] & 0x80) ? 0 : buf[3];
  st_plus = buf[4];  /* $BIi$N(BST+$B$OL5;k(B! */
  mode = buf[5];
  tick = st_plus;

#ifdef DEBUG
  fprintf(stderr,"TBR:%d/i:%d ch:%d mode:%d key+:%02X st+:%d\n"
	  ,toberead,bulk->i,ch,mode,key_plus,st_plus);
#endif

  if (ch == 0xff || (mode & 1)) {
    if (read_nbytes(bulk, NULL, toberead) < 0)
      return -1;
    return bulk->i - track_top;
  }

#ifdef DEBUG
  fprintf(stderr,"[%.36s]\n",&buf[6]);
#endif

  ch &= 0xf;
  for (i = 0; i < sizeof gts; i++)
    gts[i] = 0;
  
  while (toberead >= 4) {
    int len = 0;
    u_char event[256];
    u_char cmd, st, gt, vel;

    if (read_nbytes(bulk, &buf, 4) < 0)
      return -1;

    toberead -= 4;
    cmd = buf[0]; st = buf[1]; gt = buf[2]; vel = buf[3];

#ifdef DEBUG
    fprintf(stderr,"%d:%d [%d] %02X %02X %02X %02X\n",tick,step,toberead,cmd,st,gt,vel);
#endif

    if (running) {
      if (cmd == 0xf7) {
	running_buf[running++] = gt;
	if (gt != 0xf7)
	  running_buf[running++] = vel;

	continue;  /* goto next event */

      } else {
	int len = 0;
	u_char cmd = running_buf[0];
	u_char gt = running_buf[1];
	u_char vel = running_buf[2];

	if (running_buf[running - 1] != 0xf7)
	  running_buf[running++] = 0xf7;  /* needed?? */

	packet = get_insert_point(packet, tick);

	switch (cmd) {
	case RCP_CHEX:
	  for (i = 0; i < running; i++) {
	    c = running_buf[i + 3];
	    /*fprintf(stderr,"%02x ",c);*/
	    if (c == 0xf7)
	      break;
	    switch (c) {
	    case 0x80: event[len++] = gt; break;
	    case 0x81: event[len++] = vel; break;
	    case 0x82: event[len++] = ch; break;
	    case 0x83: start_sum = len; break;
	    case 0x84:
	      event[len] = rolex_check_sum(&event[start_sum], len - start_sum);
	      len++;
	      break;
	    default:
	      event[len++] = c;
	      break;
	    }
	  }

	  event[len++] = 0xf7;

#ifdef DEBUG
	  fprintf(stderr,"CHEX %d: ",len);
	  for (i = 0; i < len; i++) fprintf(stderr,"%02X ", event[i]);
	  fprintf(stderr,"\n");
#endif
	  write_byte(packet->bulk, 0xf0);
	  write_varlen(packet->bulk, len);
	  break;
	
	default:
	  break;
	}

	running = 0;
	if ((packet = add_event(packet, tick, event, len)) == NULL)
	  return -1;
      }
    }

    if (cmd < 0x80) {  /* note */
      if (gt != 0) {  /* not rest */
	u_char note = (cmd + key_plus) & 0x7f;

	if (!gts[note]) {  /* note on */
	  event[0] =  0x90 | ch; event[1] = note; event[2] = vel;
	  len = 3;
	}
	gts[note] = gt;
      }

    } else {  /* not note */
      switch (cmd) {
      case 0x90: case 0x91: case 0x92: case 0x93: 
      case 0x94: case 0x95: case 0x96: case 0x97: 
	event[0] = 0xf0;
	len = 2;

	for (i = 0; i < 24; i++) {
	  c = rcphd->usrex[cmd & 0xf].data[i];

	  /*fprintf(stderr,"%02x ",c);*/
	  if (c == 0xf7)
	    break;
	  switch (c) {
	  case 0x80: event[len++] = gt; break;
	  case 0x81: event[len++] = vel; break;
	  case 0x82: event[len++] = ch; break;
	  case 0x83: start_sum = len; break;
	  case 0x84:
	    event[len] = rolex_check_sum(&event[start_sum], len - start_sum);
	    len++;
	    break;
	  default:
	    event[len++] = c;
	    break;
	  }
	}
	event[len++] = 0xf7;
	event[1] = len - 2;

#ifdef DEBUG
	fprintf(stderr,"USREX %d: ",len);
	for (i = 0; i < len; i++) fprintf(stderr,"%02X ",event[i]);
	fprintf(stderr,"\n");
#endif
	break;
      
      case RCP_SAME_MEASURE:
	/*fprintf(stderr,"%d [ SM %d %d\n",sp,offset,st+((gt&3)<<8));*/

	if (!same_measure) {  /* push env */
	  same_measure = 1;
	  stack[sp].toberead = toberead;
	  stack[sp].i = bulk->i;
	  sp++;
	}
	if (step == 0) {  /* jump! */
	  u_int temp = track_top + ((gt & 0xfc) | (vel << 8));
	  toberead = bulk->i - temp;
	  bulk->i = temp;
	  step = 0; st = 0;
	  break;
	}
	/* fall into next case RCP_MEASURE_END */

      case RCP_MEASURE_END:
	if (same_measure) {  /* pop env */
	  same_measure = 0;
	  --sp;
	  /*fprintf(stderr,"%d ] ME\n",sp);*/
	  bulk->i = stack[sp].i;
	  toberead = stack[sp].toberead;
	}
	step = 0; st = 0;
	break;

      case RCP_LOOP_START:  /* push env */
	/*fprintf(stderr,"%d [ LS\n",sp);*/
	stack[sp].i = bulk->i;
	stack[sp].count = -1;
	stack[sp].toberead = toberead;
	sp++;
	step = 0; st = 0;
	break;

      case RCP_LOOP_END:
	if (stack[sp - 1].count == -1)
	  stack[sp - 1].count = (st == 0 || st == 0xff) ? 0 : st - 1;

	if (stack[sp - 1].count > 0) {  /* pop env */
	  stack[sp - 1].count--;
	  toberead = stack[sp - 1].toberead;
	  bulk->i = stack[sp - 1].i;

	} else {
	  sp--;
	  /*fprintf(stderr,"%d ] LE %d\n", sp,stack[sp].count);*/
	}
	step = 0; st = 0;
	break;

      case RCP_CHEX:
	running_buf[running++] = cmd;
	running_buf[running++] = gt;
	running_buf[running++] = vel;
	break;

      case RCP_ROLDEV:
	rolex[3] = gt; rolex[4] = vel;
	break;

      case RCP_ROLBASE:
	rolex[6] = gt; rolex[7] = vel;
	break;

      case RCP_ROLPARA:
	rolex[8] = gt; rolex[9] = vel;
	rolex[10] = rolex_check_sum(&rolex[6], 4);
	rolex[11] = 0xf7;
#ifdef DEBUG
	fprintf(stderr,"SYSEX ");
	for (i = 0; i < 12; i++) fprintf(stderr,"%02X ",rolex[i]);
	fprintf(stderr,"\n");
#endif
	len = 12;
	memcpy(event, rolex, len);
	break;

      case RCP_CH_CHANGE:
	ch = gt - 1;
	break;

      case RCP_TEMPO:
	event[0] = 0xff; event[1] = 0x51; event[2] = 0x03;
	{   
	  u_int tempo = 60 * 1e+6 / (rcphd->bpm * gt / 64);
	  
	  /*fprintf(stderr,"tempo %d %d : %d %ld\n",gt,vel,rcphd->bpm,tempo);*/
	  event[3] = (tempo >> 16) & 0xff;
	  event[4] = (tempo >> 8) & 0xff;
	  event[5] = tempo & 0xff;
	  len = 6;
	}
	break;
	
      case RCP_BANK_PGM:
	event[0] = 0xb0 | ch; event[1] = 0; event[2] = vel; event[3] = 0;
	event[4] = event[0]; event[5] = 0x20; event[6] = 0; event[7] = 0;
	event[8] = 0xc0 | ch; event[9] = gt;
	len = 10;
	break;

      case RCP_KEY_PRESSURE:
	event[0] = 0xa0 | ch; event[1] = gt; event[2] = vel;
	len = 3;
	break;

      case RCP_CH_PRESSURE:
	event[0] = 0xd0 | ch; event[1] = gt;
	len = 2;
	break;

      case RCP_PGM_CHANGE:
	event[0] = 0xc0 | ch; event[1] = gt;
	len = 2;
	break;

      case RCP_CTL_CHANGE: case RCP_PITCH_BEND:
	event[0] = ((cmd << 4) & 0xff) | ch; event[1] = gt; event[2] = vel;
	len = 3;
	break;

      case RCP_TRACK_END:
	event[0] = 0xff; event[1] = 0x2f; event[2] = 0;
	len = 3;
	st = 0;
	break;

      case RCP_COMMENT: case RCP_KEY_CHANGE:
	st = 0;
	break;

      default:
	break;
      }
    }

    if (len > 0) {
      if ((packet = add_event(packet, tick, event, len)) == NULL)
	return -1;
    }

    event[0] =  0x90 | ch; event[2] = 0;
    for (i = 0; i < sizeof gts; i++) {
      if (gts[i] == 0)
	continue;

      if (gts[i] > st) {
	gts[i] -= st;

      } else {
	event[1] = i;
	if (add_event(packet, tick + gts[i], event, 3) == NULL)
	  return -1;
	gts[i] = 0;
      }
    }

    if (st != 0) {
      step++;
      tick += st;
    }
  }

  return bulk->i - track_top;
}

int rcp_read(BULK *bulk, RCPhd *rcphd, PACKET *heads[])
{
  int i;
  u_char event[6];
  int tempo, size;

  if (rcp_read_header(bulk, rcphd) < 0)
    return -1;
  
  event[0] = 0xff; event[1] = 0x51; event[2] = 0x03;
  tempo = 60 * 1e+6 / rcphd->bpm;

  /*fprintf(stderr,"bias %d tempo %d %d\n",
    rcphd->play_bias,rcphd->bpm,tempo);*/

  event[3] = (tempo >> 16) & 0xff;
  event[4] = (tempo >> 8) & 0xff;
  event[5] = tempo & 0xff;
  heads[0] = alloc_packet(0);
  if (add_event(heads[0], 0, event, 6) == NULL)
    return -1;

  for (i = 1; i < 64; i++) {
    if ((heads[i] = alloc_packet(0)) == NULL
	|| (size = rcp_read_track(bulk, rcphd, heads[i])) < 0)
      break;

    /*fprintf(stderr,"TRACK %d [%d] \n",i,size);*/

    if (size <= 48)  /* track has no valid data */
      heads[i] = NULL;
  }

  return i;
}

u_char rolex_check_sum(u_char mes[], int len)
{
  int i;
  u_char mm, ll;

  /*fprintf(stderr,"CS[ ");*/
  mm = ll = 0;
  for (i = 0; i < len; i++) {
    /*fprintf(stderr,"%02X ",mes[i]);*/
    if (mes[i] + ll > 0x80) {
      mm++;
      ll += mes[i] - 0x80;

    } else {
      ll += mes[i];
    }
  }
  /*fprintf(stderr,"]CS\n");*/

  return 0x80 - ll;
}
