#include <stdio.h>
#include <string.h>

#include "smf.h"

int read_varlen(BULK *bulk, u_int *value)
{
  u_char c;
  int n = 0;

  if (read_byte(bulk, &c) < 0)
    return SMF_EOF;

  *value = c;
  n++;

  if (c & 0x80) {
    *value &= 0x7f;

    do {
      if (read_byte(bulk, &c) < 0)
	return SMF_EOF;

      *value = (*value << 7) + (c & 0x7f);
      n++;
    } while (c & 0x80);
  }

  return n;
}

int smf_read_type_length(BULK *bulk, char **type, u_int *length)
{
  if (read_nbytes(bulk, (u_char**) type, 4) < 0
      || read_4bytes(bulk, length) < 0)
    return SMF_EOF;

  return 8;
}

int smf_read_header(BULK *bulk, MThd *mthd)
{
  if (read_2bytes(bulk, &mthd->format) < 0
      || read_2bytes(bulk, &mthd->ntracks) < 0
      || read_2bytes(bulk, &mthd->division) < 0)
    return SMF_EOF;

  return 6;
}

int smf_read_event(BULK *bulk, EVENT *event)
{
  int n;
  u_int top = bulk->i;
  static u_char running_status = 0;
  static EVENT_TYPE prev_type = ERROR;
  static int nbytes[] = {
    0, 0, 0, 0, 0, 0, 0, 0,
    2, 2, 2, 2, 1, 1, 2, 0
  };

  if (read_byte(bulk, &event->status) < 0)
    return SMF_EOF;

  if (event->status & 0x80) {
    running_status = event->status;

  } else {  /* running status */
    if (prev_type != MIDI)
      return SMF_ILLRUN;

    event->status = running_status;
    bulk->i--;  /* unread 1byte */
  }

/*fprintf(stderr,"s %d\n",event->status);*/

  n = nbytes[(event->status >> 4) & 0xf];

  if (n > 0) {  /* midi event */
    prev_type = event->type = MIDI;
    event->length = n;

    /*printf("midi %02x %02x %02x %d\n",event->status, event->data[0],event->data[1],event->length);*/

    if (read_nbytes(bulk, &event->data, event->length) < 0)
      return SMF_EOF;

  } else {
    switch (event->status) {
    case 0xff:  /* meta event*/
      prev_type = event->type = META;
      if (read_byte(bulk, &event->status) < 0
	  || read_varlen(bulk, &event->length) < 0
	  || read_nbytes(bulk, &event->data, event->length) < 0)
	return SMF_EOF;

      break;

    case 0xf0: case 0xf7:  /* system exclusive */
      prev_type = event->type = SYSEX;
      if (read_varlen(bulk, &event->length) < 0
	  || read_nbytes(bulk, &event->data, event->length) < 0)
	return SMF_EOF;

      break;

    default:
      prev_type = event->type = ERROR;
      return SMF_ILLSTAT;
    }
  }

  return bulk->i - top;
}

int smf_read_track(BULK *bulk, PACKET *head)
{
  char *type;
  int i;
  int n;
  u_char buf[3];
  u_int tick = 0;
  u_int delta_time;
  u_int toberead;
  u_int top = bulk->i;
  EVENT event;
  PACKET *packet = head;

  if (smf_read_type_length(bulk, &type, &toberead) < 0
      || strncmp(type, "MTrk", 4) != 0)
    return SMF_EOF;

  /*fprintf(stderr,"1to0 %4s size %d\n",type,toberead);*/
    
  while (toberead > 0) {
    if ((n = read_varlen(bulk, &delta_time)) < 0)
      return SMF_EOF;
    toberead -= n;

    tick += delta_time;
    /* fprintf(stderr,"%d %d\n",tick, delta_time);*/

    if ((n = smf_read_event(bulk, &event)) < 0)
      return SMF_EOF;
    toberead -= n;

    switch (event.type) {
    case MIDI:
      buf[0] = event.status;
      /* fprintf(stderr,"%d %02X\n",tick, event.status);*/
      for (i = 0; i < event.length; i++)
	buf[i + 1] = event.data[i];

      if ((packet = add_event(packet, tick, buf, event.length + 1)) == NULL)
	return SMF_MEM;
      break;

    case SYSEX:
      if ((packet = add_event(packet, tick, &bulk->data[bulk->i - n], n))
	  == NULL)
	return SMF_MEM;
      break;

    case META:
      if (event.status == 0x2f)  /* end of track */
	break;
      
      if ((packet = add_event(packet, tick, &bulk->data[bulk->i - n], n))
	  == NULL)
	return SMF_MEM;
      break;

    default:
      return SMF_ILLSTAT;
    }
  }

  return bulk->i - top;
}

int write_varlen(BULK *bulk, u_int value)
{
  int n;
  u_int buf;

  buf = value & 0x7f;
  while ((value >>= 7) > 0) {
    buf <<= 8;
    buf |= 0x80;
    buf += value & 0x7f;
  }

  for (n = 0; ; n++) {
    if (write_byte(bulk, buf & 0xff) < 0)
      return SMF_MEM;

    if (buf & 0x80)
      buf >>= 8;
    else
      break;
  }

  return n;
}

int smf_write_header(BULK *bulk, MThd *mthd)
{
  if (write_nbytes(bulk, "MThd", 4) < 0
      || write_4bytes(bulk, 6) < 0
      || write_2bytes(bulk, mthd->format) < 0
      || write_2bytes(bulk, mthd->ntracks) < 0
      || write_2bytes(bulk, mthd->division) < 0)
    return SMF_MEM;

  return 14;
}

int smf_write_track_header(BULK *bulk, u_int toberead)
{
  if (write_nbytes(bulk, "MTrk", 4) < 0
      || write_4bytes(bulk, toberead) < 0)
    return SMF_MEM;

  return 8;
}

int smf_write_event(BULK *bulk, EVENT *event)
{
  u_int top = bulk->i;

  switch (event->type) {
  case MIDI:
    if (write_byte(bulk, event->status) < 0
	|| write_nbytes(bulk, event->data, event->length) < 0)
      return SMF_MEM;

    break;

  case SYSEX:
    if (write_byte(bulk, event->status) < 0
	&& write_varlen(bulk, event->length) < 0
	&& write_nbytes(bulk, event->data, event->length) < 0)
      return SMF_MEM;

    break;

  case META:
    if (write_byte(bulk, 0xff) < 0
	&& write_byte(bulk, event->status) < 0
	&& write_nbytes(bulk, event->data, event->length) < 0)
      return SMF_MEM;

    break;

  default:
    break;
  }

  return bulk->i - top;
}

int smf_write_track(BULK *bulk, PACKET *head)
{
  PACKET *packet;
  u_int prev_tick = 0;
  u_int track_size;
  u_int top = bulk->i;
  u_int temp;

  if (write_nbytes(bulk, NULL, 8) < 0)  /* dummy */
    return SMF_MEM;

  for (packet = head; packet != NULL; packet = packet->next) {
    /*fprintf(stderr,"%ld %02X\n",packet->tick,packet->bulk->data[0]);*/

    if (packet->bulk->i != 0) {
      if (write_varlen(bulk, packet->tick - prev_tick) < 0
	  || write_nbytes(bulk, packet->bulk->data, packet->bulk->i - 1) < 0)
	/* -1 to ignore last byte */
	return SMF_MEM;
    }
    prev_tick = packet->tick;
  }

  temp = bulk->i;
  track_size = bulk->i - top - 8;  /* 8 is lenth of "MTrk" + track size */

  bulk->i = top;
  if (smf_write_track_header(bulk, track_size) < 0)
    return SMF_MEM;

  bulk->i = temp;

  return 0;
}

int smf_write(BULK *bulk, MThd *mthd, PACKET *heads[])
{
  int i;
  u_int top = bulk->i;
  
  if (smf_write_header(bulk, mthd) < 0)
    return SMF_MEM;

  for (i = 0; i < mthd->ntracks; i++) {
    if (heads[i] == NULL)
      continue;

    if (smf_write_track(bulk, heads[i]) < 0)
      return SMF_MEM;
  }

  return  bulk->i - top;
}
