#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>
#include <unistd.h>
#include <sys/soundcard.h>

#include "smf.h"
#include "error.h"

int main(int, char*[]);
int play(BULK*);
void midi_sync(u_int, u_int, u_short);
void sbwrite(char*);
void seqbuf_dump(void);

SEQ_DEFINEBUF(2048);
int seqfd;

static int opt_debug = 0;
static int opt_device = 0;

int main(int argc, char *argv[])
{
  int i;
  BULK *infile = alloc_bulk(4096);
  FILE *fp;

  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-')
      continue;

    switch (argv[i][1]) {
    case '\0':
      continue;

    case 'd':
      opt_device = atoi(&argv[i][2]);
      break;

    case 'D':
      opt_debug = 1;
      break;

    default:
      break;
    }

    argv[i] = NULL;
  }

  fp = NULL;
  for (i = 1; i < argc; i++) {
    if (argv[i] == NULL)
      continue;

    if (strcmp(argv[i], "-") == 0) {
      fp = stdin;

    } else {
      if ((fp = fopen(argv[i], "r"))  == NULL) {
	perror(argv[i]);
      }
    }

    if (read_file_bulk(fp, infile) < 0) 
      die(1, "%s: %s\n", argv[0], argv[i]);
  }

  if (fp == NULL) {
    if (read_file_bulk(stdin, infile) < 0) 
      die(1, "%s: stdin\n", argv[0]);
  }

  infile->i = 0;  /* seek(0) */

  if ((seqfd = open("/dev/sequencer", O_WRONLY, 0)) == -1) {
    perror("/dev/sequencer");
    exit(-1);
  }

  play(infile);
  seqbuf_dump();

  return 0;
}

int play(BULK *bulk)
{
  char *type;
  int i, n;
  int device = opt_device;
  MThd mthd;
  EVENT event;
  u_int tempo = 500000;
  u_int toberead;
  u_int tick = 0;
  u_int delta_time;

  while (smf_read_type_length(bulk, &type, &toberead) >= 0) {
    if (opt_debug)
      fprintf(stderr,"PLAY %.4s %d/%d/%d\n",type,bulk->i,toberead,bulk->size);

    if (strncmp(type, "MThd", 4) == 0) {
      if (smf_read_header(bulk, &mthd) < 0)
	return -1;

      tick = 0;
      tempo = 500000;
      midi_sync(tick, tempo, mthd.division);

    {  /* $B$3$l$O$I$3$KCV$/$N$,@5$7$$$N$+!#!#!#(B */
      char buf[4];
      buf[0] = SEQ_SYNCTIMER;
      sbwrite(buf);
    }

    } else if (strncmp(type, "MTrk", 4) == 0) {
      while (toberead > 0) {
	if ((n = read_varlen(bulk, &delta_time)) < 0)
	  return -1;

	/*fprintf(stderr,"%ld\n",delta_time);*/

	toberead -= n;
	tick += delta_time;

	if ((n = smf_read_event(bulk, &event)) < 0)
	  return -1;
	toberead -= n;

	if (opt_debug) {
	  fprintf(stderr, "%d: ", tick);

	  switch (event.type) {
	  case MIDI: fprintf(stderr, "MIDI "); break;
	  case SYSEX: fprintf(stderr, "SYSEX "); break;
	  case META: fprintf(stderr, "META FF "); break;
	  default: break;
	  }

	  fprintf(stderr, "%02X (%d)", event.status, event.length);
	  for(i = 0; i < event.length; i++)
	    fprintf(stderr, " %02X", event.data[i]);

	  fputc('\n', stderr);
	}

	midi_sync(tick, tempo, mthd.division);

	switch (event.type) {
	case MIDI:
	  SEQ_MIDIOUT(device, event.status);
	  for (i = 0; i < event.length; i++)
	    SEQ_MIDIOUT(device, event.data[i]);
	  break;

	case SYSEX:
	  if (event.status == 0xf0)
	    SEQ_MIDIOUT(device, event.status);

	  for (i = 0; i < event.length; i++)
	    SEQ_MIDIOUT(device, event.data[i]);
	  break;

	case META:
	  if (event.status == 0x51) {  /* tempo */
	    tempo = event.data[0];
	    tempo = (tempo << 8) | event.data[1];
	    tempo = (tempo << 8) | event.data[2];
	  }
	  break;

	default:
	  return -1;
	}
      }

    } else {
      warn("unknown chunk type '%.4s'\n", type);
      if (read_nbytes(bulk, NULL, toberead) < 0)
	return -1;
    }
  }

  return 0;
}

void sbwrite(char *msg)
{
  _SEQ_NEEDBUF(4);
  _seqbuf[_seqbufptr + 0] = msg[0];
  _seqbuf[_seqbufptr + 1] = msg[1];
  _seqbuf[_seqbufptr + 2] = msg[2];
  _seqbuf[_seqbufptr + 3] = msg[3];
  _SEQ_ADVBUF(4);
}

void midi_sync(u_int tick, u_int tempo, u_short division)
{
  double delta_time;
  u_int diff;
  u_int temp;
  u_int wait;
  static double curr_time = 0;
  static u_int prev_tick = 0;
  static u_int prev_tempo = 500000;

  /*fprintf(stderr,"%ld ",tick);*/

  if (tick == 0) {
    prev_tick = 0;
    curr_time = 0;
    prev_tempo = 500000;
    return;
  }

  diff = tick - prev_tick;
  delta_time = tempo * diff;
  wait = (u_int) 100 * delta_time / 1e+6 / division;

  if (tempo != prev_tempo || (diff > 0 && wait > 0)) {
    prev_tick = tick;
    prev_tempo = tempo;

    /*fprintf(stderr,"!\n%d: ",tempo);*/

    curr_time += delta_time;

    if (wait > 0) {
      temp = (u_int) 100 * curr_time / 1e+6 / division;
      temp = (temp << 8) | TMR_WAIT_ABS;
      sbwrite((char*) &temp);
    }
  }
}

void seqbuf_dump(void)
{
  if (_seqbufptr) {
    if (write(seqfd, _seqbuf, _seqbufptr) == -1) {
      perror ("write /dev/sequencer");
      exit (-1);
    }
  }
  _seqbufptr = 0;
}
