/*
 * colalign.c
 * Bob Crispen (crispen@hiwaay.net)
 *
 * This routine is suitable for aligning the colons in one paragraph:
 *
 *    XX : Integer;
 *    Y : Integer;
 *    ZZZZZ : Integer;
 *
 * becomes
 *
 *    XX    : Integer;
 *    Y     : Integer;
 *    ZZZZZ : Integer;
 *
 * This is exceptionally useful in prettifying declarations in Ada,
 * and pretty well useless for anything else.
 *
 * In the vi editor (and some others?) you can assign this to a function
 * key in your .exrc file:
 *
 * map ^V<your fkey> !} d:\usr\bin\colalign^M
 *
 * DANGER: if you have a paragraph >MAX_LINES long (a paragraph is
 * defined in vi as ending when a blank line is found), this will
 * undoubtedly crash.
 *
 * In a small declarative region, e.g.:
 *
 * procedure foo is
 *    X : Integer;
 *    YYY : Integer;
 * begin
 *
 * you can insert a blank line temporarily before the "begin", run this,
 * and then delete the blank line.
 *
 * XXX If you find yourself doing a lot of record rep clauses, you can
 * hack this so it aligns "at"s.
 *
 * This software is distributed according to the terms of the Gnu
 * General Public License.
*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define MAX_CHARS 200
#define MAX_LINES 1000
#define NO_COLON -1
main ( )
{
    int c, i, j;
    char lines[MAX_LINES][MAX_CHARS];
    int colon_position[MAX_LINES];
    int lastchar[MAX_LINES];
    char linebuf[MAX_CHARS*2];
    int colon_offset;
    int line_remaining;
    int charptr = 0;
    int lineptr = 0;
    int maxcolon = 0;

    for (i=0; i<MAX_LINES; i++)
	colon_position[i] = 0;

    /*
     * Find:
     *   the position of the colon (if any) in each line
     *   the position of the last char in each line
     *   the rightmost colon in the paragraph
     */
    while ((c=getc(stdin)) != EOF) {
	lines[lineptr][charptr++] = c;
	if (c=='\n') {
    	    lines[lineptr][charptr] = '\0';
	    lastchar[lineptr] = charptr;
    	    colon_position[lineptr] = find_colon(lines[lineptr]);
	    if (colon_position[lineptr] > maxcolon)
		maxcolon = colon_position[lineptr];
	    charptr = 0;
	    lineptr++;
	}
    }

    /*
     * Shift all colons to the position of the rightmost, but leave
     * lines without colons alone.
     */
    for (i=0; i<lineptr; i++) {
	memset(linebuf, 0, sizeof(linebuf));
	if (colon_position[i] == NO_COLON) {
	    strcpy (linebuf, lines[i]);
	} else {
	    colon_offset = maxcolon - colon_position[i];
	    line_remaining = lastchar[i] - colon_position[i];
	    if (colon_offset) {
		memcpy(lines[i], linebuf, colon_position[i]);
		for (j=0; j<colon_offset; j++)
		    linebuf[colon_position[i]+j] = ' ';
		for (j=0; j<line_remaining; j++)
		    linebuf[maxcolon+j] = lines[i][j+colon_position[i]];
	    } else {
		strcpy (linebuf, lines[i]);
	    }
	}
	for (j=0; (c = linebuf[j]) != '\0'; j++)
	    putc (c, stdout);
    }

    exit(0);
}

/*
 * Find the position of the colon (if any) in the current line
 */
int find_colon (char *buf)
{
    int i;
    char c;
    for (i=0; (c = buf[i]) != '\0'; i++) {
	if (c == ':')
	    return(i);
    }
    return(NO_COLON);
}
