/*
 * pnmtopix
 *
 * Converts a pnm file (binary format ppm file, as produced by
 * netpbm) into a VRML 2.0 PixelTexture listing
 *
 * by Bob Crispen
 * 5 March 1997
 *
 * Modified 18 June 1997 -- now takes P5 (greyscale) images
 *
 * For information on netpbm, including source code, look in
 * ftp://wuarchive.wustl.edu/graphics/graphics/packages/NetPBM
 *
 * NOTE: Even though this is written under DOS, it outputs and
 * expects Unix-style newlines (0x0a), not DOS-style (0d0a).
 *
 * Options:
 *
 * NOTE: All options flags may be spelled out in full as in the
 * usage line, or may be abbreviated as shown here.
 *
 *  -o outfile		Specifies output filename. Under some
 *   			shells (e.g., Cygnus bash) you must specify
 *   			output this way because redirecting stdout
 *   			generates CR characters.  For this reason,
 *   			you must specify an input filename as well,
 *   			though naturally on Unix systems you can
 *   			redirect standard input.
 *
 *  -v			Generates a stand-alone VRML 2.0 file, complete
 *   			with headers and a Box node on which the
 *   			texture is displayed.
 *
 *  -t 0xnnnnnn 0xnn	All pixels with the color value (in hex)
 *   			in the first argument are given a (hex)
 *   			transparency value as in the second argument
 *
 *  -d 0xnn		Sets default transparency to the (hex)
 *			parameter specified (otherwise 0xff).
 *
 *  -l			Sets leading zeros (e.g., 0x0000ff instead
 *			of 0xff) on -- default: off
 */
#include <stdio.h>

/*
 * If you're compiling this on a system with DOS newline conventions
 * (e.g., Win95) compile it -DMSDOS
 */

#ifdef MSDOS
#define READOPTS "rb"
#define WRITEOPTS "wb"
#else
#define READOPTS "r"
#define WRITEOPTS "w"
#endif

static char *xusage = "\n\t[-output outfile]\n\t[-vrmlheader]\n\t[-leadingzeros]\n\t[-transparent 0xnnnnnn 0xnn]\n\t[-defaulttransparency 0xnn]\n\t[infile]\n(the first letter of an option name will do)";
static char *progname;

main (int argc, char **argv)
{
    FILE *in = stdin;
    FILE *out = stdout;
    char hdrbuf[1000];
    unsigned char *buf;
    int c;
    long bufsize;
    long i, j, indx;
    int argn=1;
    unsigned long count = 0;
    int width=0;
    int height=0;
    unsigned long transparent=0;
    unsigned long thiscolor=0;
    unsigned char xamount=0;
    unsigned char defxamount=0xff;
    int vrmlheader=0;
    int leadingzeros=0;
    int colorgif=0;
    int greygif=0;

    /*
     * Get options from the caller
     */
    progname = argv[0];
    while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
	if (!strncmp(argv[argn], "-output", 2)) {
	    out = fopen(argv[++argn], "wb");
	} else if (!strncmp(argv[argn], "-leadingzeros", 2)) {
	    leadingzeros = 1;
	} else if (!strncmp(argv[argn], "-vrmlheader", 2)) {
	    vrmlheader = 1;
	} else if (!strncmp(argv[argn], "-defaulttransparency", 2)) {
	    sscanf(argv[++argn], "%x", &defxamount);
	} else if (!strncmp(argv[argn], "-transparent", 2)) {
	    sscanf(argv[++argn], "%x", &transparent);
	    sscanf(argv[++argn], "%x", &xamount);
	} else {
	    usage();
	}
	++argn;
    }
    if (argn != argc)
	in = fopen(argv[argn++], "rb");
    if (argn != argc)
	usage();

    /*
     * Read in the header lines
     */
    memset(hdrbuf, 0, sizeof(hdrbuf));
    fscanf(in, "%s\n", hdrbuf);
    colorgif = !strcmp(hdrbuf, "P6");
    greygif = !strcmp(hdrbuf, "P5");
    if (!colorgif && !greygif) {
	fprintf(stderr, "Don't recognize file type\n");
	exit(-1);
    }

    if (vrmlheader) {
	fprintf(out, "#VRML V2.0 utf8\nGroup {\nchildren [\n");
	fprintf(out, "Shape {\nappearance Appearance {\n");
	fprintf(out, "material Material {\n");
	fprintf(out, "diffuseColor 1 0 0\n}\n");
    }

    memset(hdrbuf, 0, sizeof(hdrbuf));
    fscanf(in, "%s\n", hdrbuf);
    sscanf(hdrbuf, "%d", &width);
    fprintf(out, "texture PixelTexture {\n");
    fprintf(out, "  image %s ", hdrbuf);
    memset(hdrbuf, 0, sizeof(hdrbuf));
    fscanf(in, "%s\n", hdrbuf);
    sscanf(hdrbuf, "%d", &height);
    if (colorgif)
	fprintf(out, "%s 4\n", hdrbuf);
    else
	fprintf(out, "%s 2\n", hdrbuf);
    fscanf(in, "%s\n", hdrbuf);	/* Discard the number of colors */

    /*
     * Read the whole data section of the input file into
     * the buffer
     */
    if (colorgif)
	bufsize=width*height*3;
    else
	bufsize=width*height;
    if ((buf = (char *)malloc(bufsize)) <= 0) {
	fprintf(stderr, "Malloc failed\n");
	exit(-1);
    }
    i=0;
    while (((c=getc(in)) != EOF) && (i<bufsize))
	buf[i++] = c;

    /*
     * The size of the data in the file, once we're past the
     * header, should be exactly width * height
     * Let's make sure it is.
     */
    if (i<bufsize) {
	fprintf(stderr, "File too small\n");
	fprintf(stderr, "i=%d bufsize=%d\n", i, bufsize);
	exit(-1);
    }
    if ((c=getc(in)) != EOF) {
	fprintf(stderr, "File too big\n");
	exit(-1);
    }
    close(in);

    /*
     * Haul the data out of the buffer and print it, starting
     * in the bottom left hand corner
     */
    count = 0;
    for (i=height-1; i>=0; --i) {
	if (colorgif) {
	    for (j=0; j<width*3; j++) {
		indx = i*width*3 + j;
		if (count++ % 3 == 0)
		    thiscolor =  buf[indx];
		else
		    thiscolor = (thiscolor <<8) + buf[indx];
		if (count % 3 == 0) {
		    if (leadingzeros)
			fprintf(out, "0x%6.6x", thiscolor);
		    else
			fprintf(out, "0x%x", thiscolor);
		    if (thiscolor == transparent)
			fprintf(out, "%2.2x ", xamount);
		    else
			fprintf(out, "%2.2x ", defxamount);
		}
		if (count % 18 == 0)
		    fprintf(out, "\n");
	    }
	} else {
	    for (j=0; j<width; j++) {
		indx = i*width + j;
		thiscolor = buf[indx];
		if (leadingzeros)
		    fprintf(out, "0x%2.2x", thiscolor);
		else
		    fprintf(out, "0x%x", thiscolor);
		if (thiscolor == transparent)
		    fprintf(out, "%2.2x ", xamount);
		else
		    fprintf(out, "%2.2x ", defxamount);
		if (++count % 8 == 0)
		    fprintf(out, "\n");
	    }
	}
    }
    fprintf(out, "}\n");

    if (vrmlheader)
	fprintf(out, "}\ngeometry Box { }\n}\n]\n}\n");

    close(out);
    exit(0);
}

usage()
{
    fprintf(stderr, "usage: %s %s\n", progname, xusage);
    exit(-1);
}
