/*
* Name: dem2wrl.c
* Based on dem2tga.c (c) 1997 Jon Larimer
* Date: 24 August 1997
*
* This is a program to convert USGS Digital Elevation Model (DEM) data
* into a VRML ElevationGrid in a VRML .wrl file.  This program is public
* domain, do whatever you want with it, except claim it as your own or
* sell it.  If you find a way to make money from my work, I want a piece
* of the action. -- Jon Larimer
*
* All rights assigned to Jon Larimer. -- Bob Crispen
*
* V1.0 24 August 1997:	Initial release
*
* V1.1 25 August 1997:	Rotated and flipped the output image so north
*			is up and west is left (formerly matched the tga
*			output of dem2tga which was wrong).
*
* V1.2 27 August 1998:	Code from Pioneer Joel to write color field,
*			added ability to stream, -color and -crease
*			options.
*
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef MSDOS
#define WRITEMODE "wb"
#else
#define WRITEMODE "w"
#endif

/*
* Write the parts of the VRML file before the numbers
*/
writeheader(FILE *wrlfile, int r, int c, int e, int m, int s, float a)
{
    int spacing = e*s/50;
    int center = c*spacing/2;

    fprintf(wrlfile, "#VRML V2.0 utf8\n");
    fprintf(wrlfile, "Group {\n");
    fprintf(wrlfile, "  children [\n");
    fprintf(wrlfile, "    Viewpoint {\n");
    fprintf(wrlfile, "      description \"Far Away\"\n");
    fprintf(wrlfile, "      position 0 %d %d\n", m*2, center*3);
    fprintf(wrlfile, "    }\n");
    fprintf(wrlfile, "    Viewpoint {\n");
    fprintf(wrlfile, "      description \"Overhead\"\n");
    fprintf(wrlfile, "      position 0 %d 0\n", center*3);
    fprintf(wrlfile, "      orientation -1 0 0 1.570795\n");
    fprintf(wrlfile, "    }\n");
    fprintf(wrlfile, "    Viewpoint {\n");
    fprintf(wrlfile, "      description \"On the Edge\"\n");
    fprintf(wrlfile, "      position 0 %d %d\n", m, center);
    fprintf(wrlfile, "    }\n");
    fprintf(wrlfile, "    Viewpoint {\n");
    fprintf(wrlfile, "      description \"In the Middle\"\n");
    fprintf(wrlfile, "      position 0 %d %d\n", m, 0);
    fprintf(wrlfile, "    }\n");
    fprintf(wrlfile, "    Transform {\n");
    fprintf(wrlfile, "      translation %d 0 %d\n", -center, -center);
    fprintf(wrlfile, "      children [\n");
    fprintf(wrlfile, "\tShape {\n");
    fprintf(wrlfile, "\t  appearance Appearance {\n");
    fprintf(wrlfile, "\t    material Material {\n");
    fprintf(wrlfile, "\t      diffuseColor 0 1 0.5\n");
    fprintf(wrlfile, "\t    }\n");
    fprintf(wrlfile, "\t  }\n");
    fprintf(wrlfile, "\t  geometry ElevationGrid {\n");
    fprintf(wrlfile, "\t    xDimension %d\n", c);
    fprintf(wrlfile, "\t    zDimension %d\n", r);
    fprintf(wrlfile, "\t    xSpacing %d\n", spacing);
    fprintf(wrlfile, "\t    zSpacing %d\n", spacing);
    fprintf(wrlfile, "\t    creaseAngle %f\n", a);
    fprintf(wrlfile, "\t    height [\n");
    fprintf(wrlfile, "\t      ");
}

/*
* End height field, begin color field
*/
writecolor(FILE *wrlfile, int color_on)
{
    fprintf(wrlfile, "\n\t    ]\n");
    if (color_on) {
	fprintf(wrlfile, "\t    color Color {\n");
	fprintf(wrlfile, "\t      color [\n");
	fprintf(wrlfile, "\t\t");
    }
}

/*
* Finish off the VRML file
*/
writetrailer(FILE *wrlfile, int color_on)
{
    if (color_on) {
	fprintf(wrlfile, "\n\t      ]\n");
	fprintf(wrlfile, "\t    }\n");
    }
    fprintf(wrlfile, "\t  }\n");
    fprintf(wrlfile, "\t}\n");
    fprintf(wrlfile, "      ]\n");
    fprintf(wrlfile, "    }\n");
    fprintf(wrlfile, "  ]\n");
    fprintf(wrlfile, "}\n");
}

/*
* Get the next integer from the DEM file
*/
int nextint(FILE * fptr)
{
    char ch;
    int i;
    char in[64];

    while(isspace(ch = fgetc(fptr))) { }
    i=0;
    in[0] = ch;
    while(!isspace(ch = fgetc(fptr))) { i++; in[i] = ch; }
    in[i+1] = '\0';
    return(atoi(in));
} 

/*
* Get the next floating point number from the DEM file
*/
double nextfloat(FILE * fptr)
{
    char ch;
    int i;
    char in[64];
  
    while(isspace(ch = fgetc(fptr))) { }
    i=0;
    in[0] = ch;
    while(!isspace(ch = fgetc(fptr))) { 
	i++; 
	if(ch == 'D') { ch = 'E'; }
	in[i] = ch; 
    }
    in[i+1] = '\0';
    atof(in);
}

/*
* Complain about the user's command line
*/
usage()
{
    fprintf(stderr, "usage: dem2wrl [-invert] [-skip n] [-crease n] [-color] [demfile] [wrlfile]\n");
    fprintf(stderr, "  invert : inverts heights\n");
    fprintf(stderr, "  skip   : the number of rows and columns to skip\n");
    fprintf(stderr, "  crease : creaseAngle value (default = 0)\n");
    fprintf(stderr, "  color  : generates a color field (default: no color field)\n");
    fprintf(stderr, "  demfile: the DEM to convert (default: stdin)\n");
    fprintf(stderr, "  wrlfile: the VRML file (default: stdout)\n");
    exit(1);
}

main(int argc, char **argv)
{
    FILE * demfile;
    FILE * wrlfile;
    int i, j;
    float k;
    int skip = 1;
    int skip_given = 0;
    int maxelev, minelev;
    int rows, columns;
    int elevrange;
    int r, c;
    int cur_r, cur_c;
    char ch;
    char name[144];
    int argn = 1;
    int printrow, printcol;
    long eprinted=0;
    long cprinted=0;
    int *buf;
    long bufptr=0;
    char wrlfilename[400];
    char demfilename[400];
    int wrlrows;
    int wrlcols;
    int inverting = 0;
    float crease = 0.0;
    float t;
    int color_on = 0;
  
    fprintf(stderr, "\t--------------------------------------------\n");
    fprintf(stderr, "\tdem2wrl v1.2 by Bob Crispen and Pioneer Joel\n");
    fprintf(stderr, "\t      dem2tga v1.0 (c) 1997 Jon Larimer\n");
    fprintf(stderr, "\t        This program is public domain\n");
    fprintf(stderr, "\t--------------------------------------------\n");

    /*
    * Get options from the caller
    */
    demfile = stdin;
    wrlfile = stdout;
    strcpy(demfilename, "stdin");
    strcpy(wrlfilename, "stdout");
    while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
	if (!strncmp(argv[argn], "-invert", 2)) {
	    inverting = 1;
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-skip", 2)) {
	    sscanf(argv[++argn], "%d", &skip);
	    skip_given = 1;
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-crease", 3)) {
	    sscanf(argv[++argn], "%d", &crease);
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-color", 3)) {
	    color_on = 1;
	    ++argn;
	    continue;
	}
	usage();
    }

    if (argn != argc) {
	strcpy(demfilename, argv[argn]);
	if((demfile = fopen(argv[argn++], "r")) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) {
	strcpy(wrlfilename, argv[argn]);
	if ((wrlfile = fopen(argv[argn++], WRITEMODE)) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) usage();
  
    /****** DEM TYPE A RECORDS ******/
    /* get the name field (144 characters) */
    for(i=0; i < 144; i++) { name[i] = fgetc(demfile); }
    name[i+1] = '\0';
    /* clean off the whitespace at the end */
    for(i=strlen(name)-2; i > 0; i--) {
	if(!isspace(name[i])) {
	    i=0;
	} else {
	    name[i] = '\0';
	}
    }
    fprintf(stderr, "Quad name field: %s\n", name);
    /* don't need the next 19 items for anything */
    for(i=0; i < 19; i++)
	(void)nextint(demfile);
    fprintf(stderr, "Units code (ground planametric coordinates): %i\n",
	nextint(demfile));
    fprintf(stderr, "Units code (elevation coordinates): %i\n",
	nextint(demfile));
    (void)nextint(demfile);
    fprintf(stderr, "Ground coordinates of 4 corners of DEM: (in arc-seconds) \n");
    for(i=1; i<=4; i++)
	fprintf(stderr, "  %.4f  %.4f\n",
	    nextfloat(demfile), nextfloat(demfile));
    minelev = (int)nextfloat(demfile);
    maxelev = (int)nextfloat(demfile);
    elevrange = maxelev - minelev;
    fprintf(stderr, "Elevation: min: %d, max: %d, range: %d\n",
	minelev, maxelev, elevrange);

    /* don't need the next 3 items */
    for(i=1; i<3; i++)
	(void)nextint(demfile);
    cur_r = nextint(demfile);
    columns = nextint(demfile);

    /* DEM TYPE B RECORDS */
    cur_c = 1;
    /* we need to see how many rows of data before we write the header */
    cur_r = nextint(demfile);
    cur_c = nextint(demfile);
    rows = nextint(demfile);
    fprintf(stderr, "Skipping %d rows and columns", skip);
    wrlrows = rows/skip;
    wrlcols = columns/skip;
    fprintf(stderr, " to produce a %d x %d ElevationGrid\n",
	wrlrows, wrlcols);

    /*
    * Allocate memory for the DEM file data
    */
    fprintf(stderr, "Allocating memory to store %d DEM data points...",
	wrlrows*wrlcols);
    fflush(stderr);
    if ((buf = (int *)malloc((wrlrows*wrlcols)*sizeof(int))) <= 0) {
	fprintf(stderr, "failed\n");
	exit(-1);
    }
    fprintf(stderr, "OK\n");

    fprintf(stderr, "Reading in DEM data from %s...", demfilename);
    fflush(stderr);

    /*
    * now ready to write the row and column data
    */
    while(cur_c < columns) {
	printcol = ((cur_c % skip) == 0);
	(void)nextint(demfile); (void)nextint(demfile);
	(void)nextint(demfile); (void)nextint(demfile);
	(void)nextint(demfile); (void)nextint(demfile);
	for(i=1; i<=rows; i++) { 
	    printrow = ((i % skip) == 0);
	    if (inverting)
		j = maxelev - nextint(demfile);
	    else
		j = nextint(demfile) - minelev; 
	    if (printrow && printcol)
		buf[bufptr++] = j;
	}
	cur_r = nextint(demfile);
	cur_c = nextint(demfile);
	rows = nextint(demfile);
    }
    fclose(demfile);
    fprintf(stderr, "OK\n");
    
    /*
    * Write ElevationGrid data from the DEM array to the VRML file
    */
    fprintf(stderr, "Writing ElevationGrid to %s...", wrlfilename);
    fflush(stderr);
    writeheader(wrlfile, wrlrows, wrlcols, elevrange, maxelev, skip, crease);
    for (i=wrlcols-1; i>=0; --i) {
	for (j=0; j<wrlrows; j++) {
	    eprinted++;
	    fprintf(wrlfile, "%d", buf[(j*wrlcols)+i]);
	    if (eprinted % 10)
		fprintf(wrlfile, " ");
	    else
		fprintf(wrlfile, "\n\t      ");
	}
    }
    writecolor(wrlfile, color_on);

    /*
    * Write color field data from the DEM array to the VRML file
    */
    if (color_on) {
	fprintf(stderr, "OK\n");
    	fprintf(stderr, "Writing color field to %s...", wrlfilename);
	for (i=wrlcols-1; i>=0; --i) {
	    for (j=0; j<wrlrows; j++) {
		cprinted++;
		t = buf[(j*wrlcols)+i]/(float)elevrange;
		if (t < 0.0) t = 0.0;
		if (t > 1.0) t = 1.0;
		if (t==0) {
		    fprintf(wrlfile, "0 0.02 0.8 ");
		} else if (t < 0.5) {
		    fprintf(wrlfile, "%1.3f %1.3f 0 ",
			t*0.02,
			0.1+(t*0.8));
		} else {
		    fprintf(wrlfile, "%1.3f %1.3f %1.3f ",
			t*0.6,
			0.45-(t*0.05),
			0.05+(t*0.15));
		}
		if (cprinted % 3)
		    fprintf(wrlfile, " ");
		else
		    fprintf(wrlfile, "\n\t\t");
	    }
	}
    }

    free(buf);
    writetrailer(wrlfile, color_on);
    fclose(wrlfile);
    fprintf(stderr, "OK\n");
    fprintf(stderr, "ElevationGrid has %d Y coordinates\n", eprinted);
    fprintf(stderr, "color field has %d color values\n", cprinted);
    exit(0);
}
