/*
 * pbm2wrl
 * by Bob Crispen
 * 7 April 1998
 *
 * Called: see usage() or execute "pbm2wrl -?"
 *
 * Converts a pbm file (as produced by netpbm) into a VRML 2.0
 * ElevationGrid.  Currently reads:
 *	P2: greyscale ASCII PGM
 *	P3: color ASCII PPM, but only if colors = 255
 *	P5: greyscale binary PNM
 *	P6: color binary PNM
 *
 * For information on netpbm, including source code, look in
 * ftp://wuarchive.wustl.edu/graphics/graphics/packages/NetPBM
 *
 * This program is subject to the Gnu General Public License.
 * See ftp://prep.ai.mit.edu/pub/gnu/ for details.
 *
 * Hints:
 *	(1) pbm2wrl will turn DOS-style CR-LF into Unix-style LF.
 *	    If you don't want this to happen, redirect to the
 *	    output file instead of naming it:
 *		pbm2wrl -v foo.pnm >foo2.wrl
 *
 * Version: see usage()
 */
#include <stdio.h>
#include <math.h>

#ifdef MSDOS
#define WRITEMODE "wb"
#else
#define WRITEMODE "w"
#endif


main (int argc, char **argv)
{
    FILE *in = stdin;
    FILE *out = stdout;
    char hdrbuf[1000];
    unsigned char *buf;
    enum filetypes { None, GreyPNM, ColorPNM, PGM, PPM };
    enum filetypes filetype = None;
    int c;
    long bufsize;
    unsigned long i, j, k, indx;
    int argn=1;
    int bufptr=0;
    int width=0;
    int height=0;
    int invert=0;
    int precision=3;
    int precision2=3;
    int precisionspecified=0;
    double xspacing=10.0;
    double yscale=1.0;
    double zspacing=10.0;
    double thisheight=0.0;
    int vrmlheader=0;
    int ncolors=255;
    double vpx=0.0;
    double vpy=0.0;
    double vpz=0.0;
    unsigned long count=0;
    int numbersperline=8;
    unsigned char maxht=0;

    /*
    * Get options from the caller
    */
    while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
	if (!strncmp(argv[argn], "-vrmlheader", 2)) {
	    vrmlheader = 1;
	} else if (!strncmp(argv[argn], "-invert", 2)) {
	    invert = 1;
	} else if (!strncmp(argv[argn], "-precision", 2)) {
	    precisionspecified = 1;
	    sscanf(argv[++argn], "%d", &precision);
	} else if (!strncmp(argv[argn], "-xspacing", 2)) {
	    sscanf(argv[++argn], "%lf", &xspacing);
	} else if (!strncmp(argv[argn], "-yscale", 2)) {
	    sscanf(argv[++argn], "%lf", &yscale);
	} else if (!strncmp(argv[argn], "-zspacing", 2)) {
	    sscanf(argv[++argn], "%lf", &zspacing);
	} else {
	    usage(argv[0]);
	}
	++argn;
    }
    if (argn != argc) {
	if((in = fopen(argv[argn++], "r")) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) {
	if ((out = fopen(argv[argn++], WRITEMODE)) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    fclose(in);
	    exit(1);
	}
    }
    if (argn != argc) usage(argv[0]);

    /*
    * Read in the file type (1st 2 characters of file)
    */
    getline(in, hdrbuf, sizeof(hdrbuf));
    if (!strncmp(hdrbuf, "P2", 2)) filetype = PGM;
    if (!strncmp(hdrbuf, "P3", 2)) filetype = PPM;
    if (!strncmp(hdrbuf, "P5", 2)) filetype = GreyPNM;
    if (!strncmp(hdrbuf, "P6", 2)) filetype = ColorPNM;
    if (filetype == None) {
	fprintf(stderr, "Don't recognize file type\n");
	fclose(in);
	exit(-1);
    }

    /*
    * Other header data: width, height, number of colors
    */
    getline(in, hdrbuf, sizeof(hdrbuf));
    if (hdrbuf[0] == '#') getline(in, hdrbuf, sizeof(hdrbuf));
    sscanf(hdrbuf, "%d", &width);
    sscanf(hdrbuf, "%d", &height);
    getline(in, hdrbuf, sizeof(hdrbuf));
    if (hdrbuf[0] == '#') getline(in, hdrbuf, sizeof(hdrbuf));
    sscanf(hdrbuf, "%d", &ncolors);
    if (((filetype == PPM) || (filetype == ColorPNM)) && (ncolors > 255)) {
	fprintf(stderr, "Can't do color files yet\n");
	fclose(in);
	exit(-1);
    }

    /*
    * Compute the viewpoint
    */
    vpx = width * xspacing * 0.5;
    vpy = height * zspacing * -0.5;
    if (vpx > -vpy) {
	vpz = (vpx * 2.6) + (yscale * 255.0);
    } else {
	vpz = (vpy * -2.6) + (yscale * 255.0);
    }

    /*
    * Print the VRML file header if requested
    */
    if (vrmlheader) {
	fprintf(out, "#VRML V2.0 utf8\n");
	fprintf(out, "# Converted by pbm2wrl v1.0 by Bob Crispen <crispen@home.hiwaay.net>\n");
	fprintf(out, "Group {\nchildren [\n");
	fprintf(out, "NavigationInfo {\n");
	fprintf(out, "type [ \"EXAMINE\", \"ANY\" ]\n");
	fprintf(out, "}\n");
	fprintf(out, "# Note: world rather than viewpoint is rotated\n");
	fprintf(out, "Viewpoint {\n");
	fprintf(out, "position ");
	precision2 = (int)log10(vpx) + 1;
	printnum (vpx, out, precision2);
	fprintf(out, " ");
	precision2 = (int)log10(-vpy) + 1;
	printnum (vpy, out, precision2);
	fprintf(out, " ");
	precision2 = (int)log10(vpz) + 1;
	printnum (vpz, out, precision2);
	fprintf(out, "\n");
	fprintf(out, "}\n");
	fprintf(out, "Transform {\n");
	fprintf(out, "rotation 1 0 0 1.57\n");
	fprintf(out, "children [\n");
	fprintf(out, "Shape {\nappearance Appearance {\n");
	fprintf(out, "material Material {\n");
	fprintf(out, "diffuseColor 1 0 0\n}\n}\n");
    }
    
    /*
    * Print the header for the ElevationGrid
    */
    fprintf(out, "geometry ElevationGrid {\n");
    fprintf(out, "xDimension %d\n", width);
    fprintf(out, "xSpacing %lf\n", xspacing);
    fprintf(out, "zDimension %d\n", height);
    fprintf(out, "zSpacing %lf\n", zspacing);
    fprintf(out, "height [\n");

    /*
    * Read the whole data section of the input file into
    * the buffer
    */
    bufsize=width*height;
    if ((buf = (char *)malloc(bufsize)) <= 0) {
	fprintf(stderr, "Malloc failed\n");
	fclose(in);
	exit(-1);
    }
    i=0;

    /*
    * Binary greymap -- note: sometimes giftopnm will create a file
    * that's too short.  Pad with zeros if it does.
    */
    switch (filetype) {
    case GreyPNM:
	while (((c=getc(in)) != EOF) && (i<bufsize)) {
	    buf[i++] = c;
	    if (buf[i-1] > maxht) maxht = buf[i-1];
	}
	if (i<bufsize) {
	    for (;i<bufsize; i++)
		buf[i]=0;
	}
	if ((c=getc(in)) != EOF) {
	    fprintf(stderr, "File too big\n");
	    fclose(in);
	    exit(-1);
	}
	break;
    /*
    * Binary colormap
    */
    case ColorPNM:
	k=0;
	while (((c=getc(in)) != EOF) && (i<bufsize)) {
	    if (++k%3 == 2)
		buf[i++] = c;
	    if (buf[i-1] > maxht) maxht = buf[i-1];
	}
	if (i<bufsize) {
	    for (;i<bufsize; i++)
		buf[i]=0;
	}
	if ((c=getc(in)) != EOF) {
	    fprintf(stderr, "File too big\n");
	    fclose(in);
	    exit(-1);
	}
	break;
    /*
    * ASCII Greymap
    */
    case PGM:
	while (i < bufsize) {
	    if ((j = fscanf(in, "%d", &buf[i++])) == EOF) {
		fprintf(stderr, "File too small\n");
		fclose(in);
		exit(-1);
	    }
	}
	break;
    /*
    * ASCII Color map
    */
    case PPM:
	k=0;
	while (i < bufsize) {
	    if ((j = fscanf(in, "%d", &c)) == EOF) {
		fprintf(stderr, "File too small %d %d\n", bufsize, i);
		fclose(in);
		exit(-1);
	    }
	    if (++k%3 == 2)
		buf[i++] = c;
	}
	break;
    default:
	fprintf(stderr, "Bad filetype\n");
	exit(-1);
    }
    fclose(in);

    /*
    * Compute the precision if it hasn't been specified
    */
    if (!precisionspecified) {
	precision = (int)log10(maxht * yscale) + 1;
	if (precision < 3) precision = 3;
    }
    numbersperline = 80/(precision+2);

    /*
    * Haul the data out of the buffer and print it, starting
    * in the bottom left hand corner
    */
    for (i=0; i<height; i++) {
	for (j=0; j<width; j++) {
	    indx = i*width + j;
	    if (invert)
		thisheight = yscale * (255-buf[indx]);
	    else
		thisheight = yscale * buf[indx];
	    printnum (thisheight, out, precision);
	    if (++count % numbersperline == 0)
		fprintf(out, "\n");
	    else
		fprintf(out, " ");
	}
    }
    fprintf(out, "]}\n");
    if (vrmlheader)
	fprintf(out, "}]}]}\n");

    fclose(out);
    exit(0);
}

/*
* Notify of usage
*/
usage(char *name)
{
    fprintf(stderr, "\n%s - image to ElevationGrid converter\n", name);
    fprintf(stderr, "Version 1.0, 8 April 1998\n");
    fprintf(stderr, "Copyleft 1998 by Bob Crispen <crispen@home.hiwaay.net>\n");
    fprintf(stderr, "usage: %s [args] [infile] [outfile]\n", name);
    fprintf(stderr, "  Where args are any of the following:\n");
    fprintf(stderr, "   -i[nvert] : Blacker areas are higher (default: whiter=higher)\n");
    fprintf(stderr, "   -v[rmlheader] : Generates VRML 2.0 headers (default: ElevationGrid only)\n");
    fprintf(stderr, "   -p[recision] n : Digits of precision (default: computed, >= 3)\n");
    fprintf(stderr, "   -x[spacing] n.n : xSpacing (default: 10.0)\n");
    fprintf(stderr, "   -z[spacing] n.n : zSpacing (default: 10.0)\n");
    fprintf(stderr, "   -y[scale] n.n : multiply each Y value by n.n (default: 1.0)\n");
    exit(-1);
}

/*
* Get a line of input
*/
getline (FILE *f, char *buf, int len)
{
    int c;
    int bufptr=0;

    memset(buf, 0, len);
    while ((c=getc(f)) !=EOF) {
	buf[bufptr++] = c;
	if (bufptr >= len) {
	    fprintf(stderr, "Line too big\n");
	    exit(-1);
	}
	if (c == '\n') {
	    buf[bufptr] = '\0';
	    return;
	}
    }
    buf[bufptr] = '\0';
}

/*
* Print a floating point number with the specified number of
* digits of precision to the specified output device
*/
printnum (double n, FILE *f, int precision)
{
    char thisnum[100];
    char format[100];
    double g;

    memset(thisnum, 0, sizeof(thisnum));
    memset(format, 0, sizeof(format));
    sprintf(format, "%%2.%de", precision-1);
    sprintf(thisnum, format, n);
    sscanf(thisnum, "%lf", &g);
    sprintf(thisnum, "%lg", g);
    fprintf(f, "%s", thisnum);
}
