/*
* vwaif
* by Bob Crispen
* crispen@hiwaay.net
*
* Called: see usage() or execute "vwaif -?"
*
* Purpose: reduces the number of significant digits in numerical
* data in VRML files.
*
* This program is subject to the Gnu General Public License.
* See ftp://prep.ai.mit.edu/pub/gnu/ for details.
*
* Hints:
*	(1) Vwaif doesn't care whether the input file is a VRML
*	    file.  This means that it can be used on any text file
*	    containing numerical data.  It also means it can be
*	    used on individual nodes or even parts of nodes to
*	    reduce unnecessary detail appropriately.  It may
*	    make sense to reduce some nodes to 4 significant
*	    digits and others to 2 significant digits.
*
*	(2) Vwaif will turn DOS-style CR-LF into Unix-style LF.
*	    If you don't want this to happen, redirect to the
*	    output file instead of naming it:
*		vwaif -3 foo.wrl >foo2.wrl
*
*	(3) As part of its radical diet plan, vwaif eliminates
*	    leading and trailing whitespace (blanks and tabs),
*	    commas, and extra spaces inside lines.  Disabling this
*	    may be offered as a future enhancement if there's a
*	    demand.
*
* Version: 1.1, 11 December 1997
*
* Known bugs:
*	(1) No checking is done for comments or quotes.  Numbers
*	    in comments and quotes will be modified.
*
*	(2) Sometimes rounding is a bad idea: the year 1997, if
*	    used in a date, will be turned into 2000.
*/

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#ifdef MSDOS
#define WRITEMODE "wb"
#else
#define WRITEMODE "w"
#endif

#define MAXBUFSIZE 1000
#define BLANK " \t\r\n,"

main(int argc, char **argv)
{
    FILE *infile;
    FILE *outfile;
    int argn = 1;
    char infilename[400];
    char outfilename[400];
    char format[100];
    char thisnum[100];
    char temptok[MAXBUFSIZE];
    int precision=3;
    int c, i;
    char buf[MAXBUFSIZE];
    int bufptr=0;
    int lineptr=0;
    char *token;
    char *pbuf;
    double f, g;
    double maxnz = 0.0;
    int noleadingzero = 0;

    infile = stdin;
    outfile = stdout;
    strcpy(infilename, "stdin");
    strcpy(outfilename, "stdout");
    
    /* Get user args */
    while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
	if (isdigit(argv[argn][1])) {
	    sscanf(&argv[argn][1], "%d", &precision);
	    if (precision == 0)
		usage(argv[0]);
	    /* Account for "--" from old DOS habits */
	    if (precision < 0)
		precision = -precision;
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-maxnonzero", 2)) {
	    sscanf(argv[++argn], "%lf", &maxnz);
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-noleadingzero", 2)) {
	    noleadingzero = 1;
	    ++argn;
	    continue;
	}
	usage(argv[0]);
    }
		    
    if (argn != argc) {
	strcpy(infilename, argv[argn]);
	if((infile = fopen(argv[argn++], "r")) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) {
	strcpy(outfilename, argv[argn]);
	if ((outfile = fopen(argv[argn++], WRITEMODE)) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) usage(argv[0]);
    
    sprintf(format, "%%2.%de", precision-1);

    /* Process input file */
    while ((c=getc(infile)) != EOF) {
	buf[bufptr++] = c;
	if (bufptr >= MAXBUFSIZE-1) {
	    fprintf(stderr, "Line %d too long\n", lineptr+1);
	    fclose(outfile);
	    fclose(infile);
	    exit(-1);
	}
	/* Process one line at a time */
	if (c == '\n') {
	    buf[bufptr]='\0';
	    lineptr++;
	    pbuf=buf;
	    while (token=strtok(pbuf, BLANK)) {
		if (pbuf == NULL)
		    fprintf(outfile, " ");
		pbuf=NULL;
		/* Get rid of leading jammed quotes, brackets & braces */
		c = token[0];
		if (((c == '"') || (c == '{') || (c == '[')) &&
		    (strlen(token) > 1))
		{
		    putc(c, outfile);
		    token = &token[1];
		}
		/*
		* There's a silly problem with floating point conversions:
		* "NaN" and "Inf" are considered valid floating point
		* numbers!  First, convert to uppercase so we catch all
		* the possibilities.
		*/
		memset(temptok, 0, sizeof(temptok));
		strcpy(temptok, token);
		for (i=0; i<strlen(temptok); i++) {
		    if (islower(temptok[i]))
			temptok[i] = toupper(temptok[i]);
		}
		/* In case we get a jammed trailing character */
		c = token[strlen(token)-1];
		/* If it's a number, convert it down and print it */
		if ((sscanf(token, "%lg", &f)) &&
		    (strncmp(temptok, "INF", 3)) &&
		    (strncmp(temptok, "NAN", 3)))
		{
		    sprintf(thisnum, format, f);
		    sscanf(thisnum, "%lf", &g);
		    if (fabs(g) < maxnz)
			g = 0;
		    sprintf(thisnum, "%lg", g);
		    if (noleadingzero && (fabs(g) < 1.0) && (g != 0)) {
			if (g < 0)
			   thisnum[1] = '-';
			fprintf(outfile, "%s", &thisnum[1]);
		    } else {
			fprintf(outfile, "%s", thisnum);
		    }
		    /* Print trailing jammed quotes, brackets & braces */
		    if (((c == '"') || (c == '}') || (c == ']')) &&
			(strlen(token) > 1)) {
			putc(c, outfile);
		    }
		/* Not a number: just print it */
		} else {
		    fprintf(outfile, "%s", token);
		}
	    }
	    fprintf(outfile, "\n");
	    bufptr=0;
	}
    }
    fclose(outfile);
    fclose(infile);
    exit(0);
}

usage(char *name)
{
    fprintf(stderr, "usage: %s [args] [infile] [outfile]\n", name);
    fprintf(stderr, "  Where args are any of the following:\n");
    fprintf(stderr, "  -n :  number of significant digits to keep (default: 3)\n");
    fprintf(stderr, "  -n[oleadingzero] : '0.1' changes to '.1'\n");
    fprintf(stderr, "  -m[axnonzero] n : All numbers whose absolute value\n");
    fprintf(stderr, "     is below n will be set to zero (default: 0).\n");
    fprintf(stderr, "Version 1.1, 11 December 1997\n");
    exit(-1);
}
