/*
 * wireframe 
 * by Bob Crispen
 * crispen@hiwaay.net
 *
 * Called: wireframe [infile] [outfile]
 *
 * Turns an IndexedFaceSet into an IndexedLineSet
 *
 * If you stick a VRML 2.0 Shape node into a file (the infile) and run
 * wireframe, naming an outfile (e.g. wireframe foo foo.wrl), it will
 * generate an equavalent Shape node with the IndexedFaceSet turned into
 * an IndexedLineSet and the diffuseColor changed to an emissiveColor.
 *
 * You can also get input from stdin and send to stdout, but note that
 * some shells (e.g., DOS) will add the carriage return character.
 *
 * Built in limits: this program assumes that the IFS you give it:
 *	(a) has fewer than 10,000 points
 *	(b) has no line longer than 10,000 characters
 *	(c) has no polygon with more than 10,000 points
 * For files that don't meet these criteria, change MAXSIZE and recompile.
 *
 */
#include <stdio.h>
#include <string.h>

#ifdef MSDOS
#define OUTMODE "wb"
#else
#define OUTMODE "w"
#endif

#define MAXSIZE 10000

struct edge {
  int from;
  int to;
} edges[MAXSIZE] ;
int edgeCount = 0;

main (int argc, char **argv)
{
    FILE *infile = stdin;
    FILE *outfile = stdout;
    char buf[MAXSIZE];
    int thisFace[MAXSIZE];
    int i, c, bufptr=0;
    char *token, *tokptr;
    int incoords = 0;
    int coordCount = 0;
    
    void insertEdge(int from, int to);
    void printEdges(FILE *outfile);
    void usage(char *me);

    if (argc > 1) {
	if (argv[1][0] == '-') usage(argv[0]);
	if ((infile = fopen(argv[1], "r")) == NULL) {
	    fprintf (stderr, "can't open %s\n", argv[1]);
	    exit(-1);
	}
    }
    if (argc > 2) {
	if (argv[2][0] == '-') usage(argv[0]);
	if ((outfile = fopen(argv[2], OUTMODE)) == NULL) {
	    fprintf (stderr, "can't open %s\n", argv[2]);
	    fclose (infile);
	    exit(-1);
	}
    }
    if (argc > 3) 
	usage(argv[0]);

    while ((c=getc(infile)) != EOF) {
	buf[bufptr++] = c;
	if (c == '\n') {
	    buf[bufptr] = '\0';
	    tokptr = buf;
	    while((token = strtok(tokptr, " \r\t\0")) != NULL) {
		tokptr = NULL;
		if (incoords) {
		    if (token[0] == ']') {
			incoords = 0;
			printEdges(outfile);
			fprintf(outfile, "]\n");
		    } else {
			if (token[0] == '[') {
			    fprintf(outfile, "%s\n", token);
			} else if (token[0] != '\n') {
			    sscanf(token, "%d", &thisFace[coordCount]);
			    if (thisFace[coordCount] == -1) {
				for (i=0; i<coordCount-1; i++) {
				    insertEdge(thisFace[i], thisFace[i+1]);
				}
				insertEdge(thisFace[0], thisFace[coordCount-1]);
				coordCount = 0;
			    } else {
				++coordCount;
			    }
			}
		    }
		} else { /* Not in the coords yet */

		    /* Enter coordIndex processing */
		    if (!strcmp(token, "coordIndex")) {
			fprintf(outfile, "coordIndex ");
			incoords = 1;

		    /* Replace IndexedFaceSet with IndexedLineSet */
		    } else if (!strcmp(token, "IndexedFaceSet")) {
			fprintf(outfile, "IndexedLineSet ");

		    /* The emissiveColor is what counts on an ILS */
		    } else if (!strcmp(token, "diffuseColor")) {
			fprintf(outfile, "emissiveColor ");

		    /* Don't duplicate the emissiveColor */
		    } else if (!strcmp(token, "emissiveColor")) {
			token = strtok(tokptr, " \r\t\0");

		    /* IFS Fields that aren't in the IndexedLineSet */
		    } else if (!strcmp(token, "normal")) {
			while ((c=getc(infile)) != '}');
			bufptr = 0;
			memset(buf, 0, sizeof(buf));
			tokptr = buf;
		    } else if (!strcmp(token, "texCoord")) {
			while ((c=getc(infile)) != '}');
			bufptr = 0;
			memset(buf, 0, sizeof(buf));
			tokptr = buf;
		    } else if (!strcmp(token, "ccw")) {
			token = strtok(tokptr, " \r\t\0");
		    } else if (!strcmp(token, "convex")) {
			token = strtok(tokptr, " \r\t\0");
		    } else if (!strcmp(token, "creaseAngle")) {
			token = strtok(tokptr, " \r\t\0");
		    } else if (!strcmp(token, "normalIndex")) {
			while ((c=getc(infile)) != ']');
			bufptr = 0;
			memset(buf, 0, sizeof(buf));
			tokptr = buf;
		    } else if (!strcmp(token, "normalPerVertex")) {
			token = strtok(tokptr, " \r\t\0");
		    } else if (!strcmp(token, "solid")) {
			token = strtok(tokptr, " \r\t\0");
		    } else if (!strcmp(token, "texCoordIndex")) {
			while ((c=getc(infile)) != ']');
			bufptr = 0;
			memset(buf, 0, sizeof(buf));
			tokptr = buf;

		    } else if (!strcmp(token, "\n")) {
			fprintf(outfile, "\n");
		    } else {
			fprintf(outfile, "%s ", token);
		    }
		}
	    }
	    bufptr=0;
	}
    }

    fclose(outfile);
    fclose(infile);
}

/*
* Compare two edge structs
*/
int compareEdge(struct edge *this, struct edge *that)
{
    if (this->from < that->from)
	return -1;
    if (this->from > that->from)
	return 1;
    if (this->to < that->to)
	return -1;
    if (this->to > that->to)
	return 1;
    return 0;
}

/*
* See if this edge is already in our edges array.  If not, insert it.
*/
void insertEdge(int from, int to) {

    struct edge newEdge;
    int i;

    if (from < to) {
	newEdge.from = from;
	newEdge.to = to;
    } else {
	newEdge.from = to;
	newEdge.to = from;
    }
    if (bsearch(&newEdge, edges, edgeCount, sizeof (struct edge), compareEdge))
	return;

    edges[edgeCount].from = newEdge.from;
    edges[edgeCount++].to = newEdge.to;

    qsort(edges, edgeCount, sizeof(struct edge), compareEdge);
}

/*
* Print all the edges (the array should already be sorted)
*/
void printEdges(FILE *outfile)
{
    int i;

    for (i=0; i<edgeCount-1; i++) {
	fprintf (outfile, "\t%d %d -1,\n", edges[i].from, edges[i].to);
    }
    fprintf (outfile, "\t%d %d -1\n", edges[edgeCount-1].from,
	edges[edgeCount-1].to);
}

/*
* Tell the user the calling conventions for this program
*/
void usage(char *me) {
  fprintf (stderr, "Usage: %s [infile] [outfile]\n", me);
  exit(-1);
}
