// Grid.cpp : implementation file
//


#include "stdafx.h"
#include "resource.h"

#include "Grid.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

const int WIDTH = 26;
const int ROWS = 9;

const int NORMAL = 0;
const int MARKED = 1;
const int DOOMED = 2;

const int FIRST_MOVE = 0;
const int SECOND_MOVE = 1;
const int MOVING = 2;

const int OFFSET_X = 5;
const int OFFSET_Y = 34;


/////////////////////////////////////////////////////////////////////////////
// Grid

Grid::Grid()
{
	CTime time;
	srand(::GetTickCount());

	pieces.SetSize(81);	
	open.SetSize(40);	
	closed.SetSize(40);	
	path.SetSize(40);	

	start = new Space;
	target = new Space;

	Reset();
}

Grid::~Grid()
{
	delete start;
	delete target;	
	CleanUp();
}

/////////////////////////////////////////////////////////////////////////////
// Grid message handlers

void Grid::AddPieces(int number)
{
	int index, colorIndex;
	int have = 0;
	int freeSize = free_spaces.GetSize(); 	
	
	Piece *piece;
	Space *space;
	
	while(have < number)
	{				
		if(freeSize == 0)	//grid is full!							
			break;
							
		index = rand() % freeSize;										

		space = (Space*)free_spaces.ElementAt(index); 
						
		if(number > 3)	//first time		
		{			
			colorIndex = rand() % 7;
			piece = new Piece(space->x, space->y, colorIndex);				
		}		

		else					
			piece = new Piece(space->x, space->y, depot[have]->colorIndex);			
		
		pieces.Add(piece);			
		free_spaces.RemoveAt(index);			
		have++;				
		freeSize--;
	} 		

	//fill up the dots depot for the next move
				
	colorIndex = rand() % 7;	
	depot[0] = new Piece(161, 5, colorIndex);		

	colorIndex = rand() % 7;	
	depot[1] = new Piece(187, 5, colorIndex);	

	colorIndex = rand() % 7;	
	depot[2] = new Piece(213, 5, colorIndex);			
}


void Grid::Reset()
{
	int x, y;
	freeSize = 81;		
	score = bonus = 0;
	
	finishedWay = moveAborted = passage = getReady = false;
	gameState = 0;

	CleanUp();
	
	spaces.RemoveAll();
	free_spaces.RemoveAll();
	pieces.RemoveAll();
	open.RemoveAll();
	closed.RemoveAll();
	path.RemoveAll();

	for(int i=0; i<81; i++)
	{	
		x = (i % ROWS * WIDTH) + OFFSET_X;		
		y = (i / ROWS * WIDTH) + OFFSET_Y;			
		
		Space *s = new Space(x, y);	
		Space *s1 = new Space(x, y);	
		
		spaces.Add(s);
		free_spaces.Add(s1);		
	}
	
	AddPieces(5);
}

bool Grid::IsOccupied(Space *space)
{	    
    int piecesSize = pieces.GetSize(); 
    
    for(int i=0; i<piecesSize; i++)
	{	
		Piece *p = (Piece*)pieces.ElementAt(i);
		
		if(p->x == space->x && p->y == space->y)
			return true;
	}	
	
	return false;	
}


Space *Grid::GetSpace(int x, int y)
{    
	Space *space;
	
    for(int i=0; i<81; i++)
	{	
		space = (Space*)spaces.ElementAt(i);

		if(space->rect.PtInRect(CPoint(x, y)))		
			return space;
	}
	return NULL;	
}

Space *Grid::GetFreeSpace(int x, int y)
{    
	Space *space;
	
	int freeSize = free_spaces.GetSize();

    for(int i=0; i<freeSize; i++)
	{	
		space = (Space*)free_spaces.ElementAt(i);

		if(space->x == x && space->y == y)		
			return space;
	}
	return NULL;
}

bool Grid::PenDown(int x, int y)
{
	Space *space;
	Piece *p;
	
	int i;
	int piecesSize = pieces.GetSize(); 
		
	if(gameState == FIRST_MOVE)
    {       		
		PrepareNextMove();
		found = false;
    
		for(i=0; i<piecesSize; i++)
		{	
			p = (Piece*)pieces.ElementAt(i);
			
			if(p->clickedAt(x, y))
			{
				p->state = MARKED;
				markedIndex = i;
				start = GetSpace(x, y);	
				found = true;
				
				break;
			}
		}		
	}			
	else if(gameState == SECOND_MOVE)
    {		
		space = GetSpace(x, y);
    	moveAborted = false;    				
    	
    	for(i=0; i<piecesSize; i++)
		{	
			p = (Piece*)pieces.ElementAt(i);
			
			if(p->clickedAt(x, y))	//second click was at another dot, not a free space
			{
				moveAborted = true;					
								
				Piece *p1 = (Piece*)pieces.ElementAt(markedIndex);
				p1->state = NORMAL;
							    
			    p->state = MARKED;						
				markedIndex = i;
				start = GetSpace(x, y);		
				gameState = FIRST_MOVE;

				break;
			}
		}				

		if(!moveAborted)
		{	
			target = space;	
			space->marked = true;				
			markTarget = true;					
			
			if(FindPath())		
			{						
				passage = true;
				gameState = MOVING;
				p = (Piece*)pieces.ElementAt(markedIndex);				
				p->state = NORMAL;     	
				int pathSize = path.GetSize();
				waypoint = pathSize-1;	
				
				return true;	//get that piece moving!
			}
			else
			{
				p = (Piece*)pieces.ElementAt(markedIndex);				
				p->state = NORMAL;     	
				passage = false;			
			}
		}			
    }
    	
    return false;	
}

void Grid::PrepareNextMove()
{
	int piecesSize = pieces.GetSize(); 
    int i;

	Piece *p;;	
	Space *s;

    for(i=0; i<piecesSize; i++)
	{	
		p = (Piece*)pieces.ElementAt(i);
		p->state = NORMAL;
	}

	for(i=0; i<81; i++)
	{	
		s = (Space*)spaces.ElementAt(i);
		s->marked = false;
	}

	open.RemoveAll();
	closed.RemoveAll();
	path.RemoveAll();
	
	targetFound = false;	
	waypoint = 0;		
}

Piece *Grid::GetNeighbor(Piece *p, int dir)
{
	Piece *piece;	
	int piecesSize = pieces.GetSize();    
	int E = 2;    
    int S = 4;    
	int NE = 6;
    int SE = 8;

	for(int i=0; i<piecesSize; i++)
	{	
		piece = (Piece*)pieces.ElementAt(i);

		if(piece->colorIndex == p->colorIndex)
		{
			if(dir == E)	//East neigbor
				if(piece->y == p->y && piece->x == p->x + WIDTH)
					return piece;

			if(dir == S)	//South neigbor
				if(piece->y == p->y + WIDTH && piece->x == p->x)
					return piece;

			if(dir == SE)	//Southeast neigbor
				if(piece->y == p->y + WIDTH && piece->x == p->x + WIDTH)
					return piece;

			if(dir == NE)	//Northeast neigbor
				if(piece->y == p->y - WIDTH && piece->x == p->x + WIDTH)
					return piece;
		}
	}		

	return NULL;
}

Space *Grid::GetNeighbor(Space *s, int dir)
{
	if(s == NULL)
		return NULL;

	Space *space = NULL;	
	Piece *piece = NULL;	
	
	int piecesSize = pieces.GetSize();    
	int closedSize = closed.GetSize();
    
	int i;
	int xx = 0;
	int yy = 0;

	int W = 1;
    int E = 2;
	int N = 3;
    int S = 4;    
	int NW = 5;
    int NE = 6;
	int SW = 7;
    int SE = 8;	

	if(dir==W)	//West
    {
    	xx = s->x-WIDTH;
		yy = s->y;
	}
	if(dir==E)	//East
    {
    	xx = s->x+WIDTH;
		yy = s->y;
	}
	if(dir==N)	//North
    {
    	xx = s->x;
		yy = s->y-WIDTH;
	}
	if(dir==S)	//South
    {
    	xx = s->x;
		yy = s->y+WIDTH;
	}
	if(dir==NW)	//NW
    {
    	xx = s->x-WIDTH;
		yy = s->y-WIDTH;
	}
	if(dir==NE)	//NE
    {
    	xx = s->x+WIDTH;
		yy = s->y-WIDTH;
	}
	if(dir==SW)	//SW
    {
    	xx = s->x-WIDTH;
		yy = s->y+WIDTH;
	}
	if(dir==SE)	//SE
    {
    	xx = s->x+WIDTH;
		yy = s->y+WIDTH;
	}

	//is space already occupied?				
    for(i=0; i<piecesSize; i++)
	{
		piece = (Piece*)pieces.ElementAt(i);		
		if(piece->x == xx && piece->y == yy)	
			return NULL;					
    }

	//is it just below or above an occupied field (no cutting corners!)?				
    for(i=0; i<piecesSize; i++)
	{	
		piece = (Piece*)pieces.ElementAt(i);		
		if(dir == SW && ((piece->x == xx && piece->y == yy-WIDTH) || (piece->x == xx+WIDTH && piece->y == yy)))
			return NULL;					
		if(dir == SE && ((piece->x == xx && piece->y == yy-WIDTH) || (piece->x == xx-WIDTH && piece->y == yy)))	
			return NULL;					
		if(dir == NW && ((piece->x == xx && piece->y == yy+WIDTH) || (piece->x == xx+WIDTH && piece->y == yy)))	
			return NULL;					
		if(dir == NE && ((piece->x == xx && piece->y == yy+WIDTH) || (piece->x == xx-WIDTH && piece->y == yy)))	
			return NULL;					
    }

	//is this space already in the "closed" list?				
    for(i=0; i<closedSize; i++)
	{	
		space = (Space*)closed.ElementAt(i);		
		if(space->x == xx && space->y == yy)	
			return NULL;					
    }

	//no, it isn't - continue...
	for(i=0; i<81; i++)		
	{
		space = (Space*)spaces.ElementAt(i);	    
	    if(space->x == xx && space->y == yy)
	    	return space;
	}	 	    	
	
	return NULL;
}

bool Grid::FindPath()
{	
	int piecesSize = pieces.GetSize();    
	
	boolean included;
	Piece *p = (Piece*)pieces.ElementAt(markedIndex);	

	Space *space;
	Space *neighbor;
	Space *op;	
   
	//find Space with start dot coordinates
    for(int i=0; i<81; i++)
	{	
		space = (Space*)spaces.ElementAt(i);
		
		if(p->x == space->x && p->y == space->y)
		{
			open.Add(space);			
			break;
		}
	}
	
	int loops = 0;
		
	//loop until target space is found		
	while(!targetFound)
	{			
		loops++;

		if(loops > 80)
			return false;

		int checked = 1;

		//add all 8 neighbors to the "open" list, if not occupied or already in the "closed" list
		for(int j=1; j<9; j++)
		{				
			neighbor = GetNeighbor(space, j);			
			
			if(neighbor != NULL) 			
			{	
				//add it to the "open" list, if it isn't already there:					
				int openSize = open.GetSize();    
				included = false;
				
				//check all spaces in the "open" list
				for(int k=0; k<openSize; k++)			
				{
					op = (Space*)open.ElementAt(k);					
					
					//it's already in there...
					if(op->x == neighbor->x && op->y == neighbor->y)
					{							
						included = true;	
						
						//Is this a better (cheaper) path to the target?
						//If so, recalculate cost for this square, reset its parent									
						if(op->wayCost < space->wayCost)
						{
							CalculateCost(op, j);
							op->parent = space;
						}						
					}
				}

		
				//It's not in there, so add it, set it's parent and calculate it's cost:
				if(!included)
				{							
					CalculateCost(neighbor, j);
					neighbor->parent = space;
					open.Add(neighbor);		
					
					if(target->x == neighbor->x && target->y == neighbor->y)			
					{						
						targetFound = true;	//exit loop	
						closed.Add(neighbor);						
						break;
					}					
				}					
			}				
		}		
					
		//find index of space item
		int openSize = open.GetSize();  
		for(int k=0; k<openSize; k++)			
		{
			op = (Space*)open.ElementAt(k);	
			if(op->IsEqual(space))
			{
				open.RemoveAt(k);
				//open.FreeExtra();
				break;
			}
		}				
			
		closed.Add(space);					
		
		//get space with the cheapest cost out of "open" list for the next loop
		space = GetCheapest();	
	}
				
	CalculatePath();			
		
	return true;
}

Space *Grid::GetCheapest()
{
	Space *space;	
	Space *cheapestSpace;	

	int openSize = open.GetSize();
	int cheapest = 1000;

    for(int i=0; i<openSize; i++)
	{
		space = (Space*)open.ElementAt(i);

		if(space->totalCost < cheapest)
		{
			cheapest = space->totalCost;
			cheapestSpace = space;
		}
	}
	
    return cheapestSpace;	
}

void Grid::CalculateCost(Space *s, int dir)
{
	int wayCost, distanceCost;	
    	    	
    if(dir > 4)
    	wayCost = 9;	//you can experiment with these values, to get the best paths...
    else
    	wayCost = 10;
    	
    distanceCost = abs(target->x/WIDTH - s->x/WIDTH) + abs(target->y/WIDTH - s->y/WIDTH);
    	
    s->SetCost(wayCost, distanceCost);
}

void Grid::CalculatePath()
{
	Space *s;	
	Space *nextSpace = target;
	
	boolean found = false;
	int closedSize = closed.GetSize();

	path.Add(target);
    	
    while(!found)
    {		
    	for(int i=0; i<closedSize; i++)
		{	
			s = (Space*)closed.ElementAt(i);
			
			if(s->IsEqual(nextSpace->parent))					
			{
				path.Add(s);	
				
				if(s->x == start->x && s->y == start->y)
					found = true;
				else						
					nextSpace = s;
			}
    	}  
    }			
}

void Grid::Move()
{
	Space *s;	
	int piecesSize = pieces.GetSize();    

	Piece *piece = (Piece*)pieces.ElementAt(markedIndex);

	s = (Space*)path.ElementAt(waypoint);
	piece->MoveTo(s->x, s->y);

	waypoint--;

	if(waypoint < 0)	//finished way	
	{
		s = GetFreeSpace(target->x, target->y);			
		s->x = start->x;
		s->y = start->y;
		
	    finishedWay = true;	    
	}	
}

bool Grid::SetDoomed(int ticker)
{
	int i, delay = 4, points = 0;
    int piecesSize = pieces.GetSize();    				

	Piece *p;
    
	CPtrArray temp;
   
    if(ticker == 0)	//set dots doomed state
	{			
	    for(i=0; i<piecesSize; i++)
		{
			p = (Piece*)pieces.ElementAt(i);				
		
			if(!p->alive)		
			{		
				p->state = DOOMED;					
				points ++;
			}
		}	
		
		bonus = points-4;
		score += points * bonus;			
	}
	
	//if(bonus > 1)
	//	delay = 8;
	
	if(ticker > delay)			//remove dots
	{
		for(i=0; i<piecesSize; i++)
		{
			p = (Piece*)pieces.ElementAt(i);			
		
			if(p->alive)
				temp.Add(p);
			else				
				free_spaces.Add(new Space(p->x, p->y));
		}	
	
		pieces.RemoveAll();
		pieces.Copy(temp);			
		bonus = 0;
		
		return true;
	}
		
	return false;	
}

bool Grid::CheckForRows(bool single)
{
	int i, j, rowLength;
    int piecesSize = pieces.GetSize();    

    Piece *p1, *p2;
	CPtrArray *row;
	CPtrArray temp;
	
	int ROWS = 9;    
	int ROW = 5;
   
    int E = 2;    
    int S = 4;    
	int NE = 6;
    int SE = 8;
    int diffX, diffY;
    bool singleTest;
    
    int limitX = 109;
	int limitY = 138;
       
	for(i=0; i<piecesSize; i++)
	{
		p1 = (Piece*)pieces.ElementAt(i);			
		
		singleTest = single ? (target->y == p1->y) : true;
		
		//find horizontal rows 
		if(p1->x <= limitX && singleTest)
		{			
			row = FindRow(p1, E);
			rowLength = row->GetSize()+1;
		
			if(rowLength >= ROW)
			{					
				temp.Add(p1);	
			
				for(j=0; j<rowLength-1; j++)
				{
					p2 = (Piece*)row->ElementAt(j);		

					temp.Add(p2);						
				}
			}
		}				
					
		singleTest = single ? (target->x == p1->x) : true;

		
		//find vertical rows
		if(p1->y <= limitY && singleTest)
		{
			row = FindRow(p1, S);
			rowLength = row->GetSize()+1;
		
			if(rowLength >= ROW)
			{					
				if(!IsInside(temp, p1))				
					temp.Add(p1);	
			
				for(j=0; j<rowLength-1; j++)
				{
					p2 = (Piece*)row->ElementAt(j);	

					if(!IsInside(temp, p2))				
						temp.Add(p2);					
				}
			}
		}
		
		diffX = abs(target->x - p1->x);
    	diffY = abs(target->y - p1->y);
		singleTest = single ? (diffX == diffY) : true;
    
		//find diagonal rows: -> SE
		if(p1->x <= limitX && p1->y <= limitY && singleTest)
		{
			row = FindRow(p1, SE);
			rowLength = row->GetSize()+1;
		
			if(rowLength >= ROW)
			{	
				if(!IsInside(temp, p1))				
					temp.Add(p1);	
			
				for(j=0; j<rowLength-1; j++)
				{
					p2 = (Piece*)row->ElementAt(j);	

					if(!IsInside(temp, p2))				
						temp.Add(p2);					
				}
			}
		}
					
		//find diagonal rows: -> NE
		if(p1->x <= limitX && p1->y >= limitY && singleTest)
		{
			row = FindRow(p1, NE);
			rowLength = row->GetSize()+1;
		
			if(rowLength >= ROW)
			{	
				if(!IsInside(temp, p1))				
					temp.Add(p1);	
			
				for(j=0; j<rowLength-1; j++)
				{
					p2 = (Piece*)row->ElementAt(j);	

					if(!IsInside(temp, p2))				
						temp.Add(p2);					
				}
			}
		}			
		
	}	
		
	//now we have those to be removed in the "temp" list - mark them for removal:				
	int tempSize = temp.GetSize();
	
	for(i=0; i<piecesSize; i++)
	{
		p1 = (Piece*)pieces.ElementAt(i);	
			
		for(j=0; j<tempSize; j++)
		{
			p2 = (Piece*)temp.ElementAt(j);
			
			if(p1->x == p2->x && p1->y == p2->y)
				p1->alive = false;
		}	
	}	
	
	if(tempSize >= ROW)
		return true;
		
	return false;	
}

CPtrArray *Grid::FindRow(Piece *p, int dir)
{
	Piece *piece;
	CPtrArray *temp = new CPtrArray;
	
    piece = GetNeighbor(p, dir);
				
	while(true)
	{
		if(piece == NULL)	
			break;
			
		temp->Add(piece);
		piece = GetNeighbor(piece, dir);				
	}
	
	return temp;
}

bool Grid::IsInside(CPtrArray &ar, Piece *p)
{
	int arraySize = ar.GetSize();  
	Piece *piece;

	for(int i=0; i<arraySize; i++)			
	{
		piece = (Piece*)ar.ElementAt(i);	

		if(piece->IsEqual(p))		
			return true;		
	}
	return false;
}

void Grid::CleanUp()
{
	if(depot[0] != NULL)
		delete depot[0];
	if(depot[1] != NULL)
		delete depot[1];
	if(depot[2] != NULL)
		delete depot[2];

	int i;

	for(i=0; i<spaces.GetSize(); i++)
		delete (Space*)spaces.ElementAt(i); 

	for(i=0; i<free_spaces.GetSize(); i++)
		delete (Space*)free_spaces.ElementAt(i); 

	for(i=0; i<pieces.GetSize(); i++)
		delete (Piece*)pieces.ElementAt(i); 
}
