#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <commctrl.h>
#include <commdlg.h>
#include <mmsystem.h>
#include "hooklib.h"
#include "ttsapi.h"
#include "resource.h"

#define HK_READACTIVE 1001
#define HK_READFOCUSED 1002
#define HK_READVERBOSE 1003
#define HK_READTITLE 1004
#define HK_READSTATUS 1005
#define HK_READDESKTOP 1006
#define HK_REPEAT 1007
#define HK_REINST 1008
#define HK_SETTINGS 1009
#define HK_HIDE 1010
#define HK_RELOAD 1011
#define HK_RESTART 1012

HINSTANCE hInst = NULL;
LPTTS_HANDLE_T engine = NULL;
HANDLE port = NULL;
DWORD written = 0;
TCHAR inifile[MAX_PATH] = {'\0'};

LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM); // Window procedure.
BOOL CALLBACK DlgProc(HWND, UINT, WPARAM, LPARAM); // Dialog procedure.
UINT CALLBACK OFNHookProc(HWND, UINT, WPARAM, LPARAM); // File dialog hook procedure.

BOOL StartupSpeech(void);
void SetupSpeech(void);
void SpeakText(CHAR *text);
void InterruptSpeech(void);
void ShutdownSpeech(void);
void DoSomethingSpecial(void);

#ifdef UNICODE
int WINAPI wWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PWSTR szCmdLine, int iCmdShow)
#else
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PSTR szCmdLine, int iCmdShow)
#endif
{
	TCHAR szAppName[] = TEXT("RetroReader32");
	HWND hwnd;
	HMENU hMenu;
	WNDCLASS wndclass;
	MSG msg;
	ZeroMemory(&wndclass, sizeof(wndclass));
	ZeroMemory(&msg, sizeof(msg));

	hInst = hInstance;

	InitCommonControls();

	wndclass.style = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc = WndProc;
	wndclass.hInstance = hInst;
	wndclass.hIcon = LoadIcon(hInst, MAKEINTRESOURCE(IDI_ICON));
	wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH) GetStockObject(WHITE_BRUSH);
	wndclass.lpszClassName = szAppName;
	RegisterClass(&wndclass);
	hwnd = CreateWindow(szAppName, TEXT("ROAR"), WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInst, NULL);
	// Set always-on-top, ignore the rest.
	SetWindowPos(hwnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);

	if(lstrlen(szCmdLine))
	{ // Get the INI file path from the command line.
		lstrcpy(inifile, szCmdLine);
	} // if
	else
	{ // Construct the INI file path.
		if(GetCurrentDirectory(MAX_PATH, inifile))
		lstrcat(inifile, TEXT("\\ROAR.ini"));
	} // else if

	// Initialize the speech engine.
	if(!StartupSpeech())
	return 0;
	// Special handling for a particular Windows version.
	DoSomethingSpecial();

	// Load speech parameters.
	SetupSpeech();

	// Register hotkeys.
	RegisterHotKey(hwnd, HK_READACTIVE, MOD_CONTROL | MOD_SHIFT, 0x57);
	RegisterHotKey(hwnd, HK_READFOCUSED, MOD_CONTROL | MOD_SHIFT, VK_SPACE);
	RegisterHotKey(hwnd, HK_READVERBOSE, MOD_CONTROL | MOD_SHIFT, VK_BACK);
	RegisterHotKey(hwnd, HK_READTITLE, MOD_CONTROL | MOD_SHIFT, 0x54);
	RegisterHotKey(hwnd, HK_READSTATUS, MOD_CONTROL | MOD_SHIFT, 0x42);
	RegisterHotKey(hwnd, HK_READDESKTOP, MOD_CONTROL | MOD_SHIFT, 0x44);
	RegisterHotKey(hwnd, HK_REPEAT, MOD_CONTROL | MOD_SHIFT, 0x4c);
	RegisterHotKey(hwnd, HK_REINST, MOD_CONTROL | MOD_SHIFT, 0x52);
	RegisterHotKey(hwnd, HK_SETTINGS, MOD_CONTROL | MOD_SHIFT, 0x53);
	RegisterHotKey(hwnd, HK_HIDE, MOD_CONTROL | MOD_SHIFT, 0x48);
	RegisterHotKey(hwnd, HK_RELOAD, MOD_CONTROL | MOD_SHIFT, 0x45);
	RegisterHotKey(hwnd, HK_RESTART, MOD_CONTROL | MOD_SHIFT, 0x41);

	// Add items to system menu.
	hMenu = GetSystemMenu(hwnd, FALSE);
	if(hMenu)
	{
		TCHAR text[MAX_PATH] = {'\0'};
		AppendMenu(hMenu, MF_SEPARATOR, 0, NULL);
		LoadString(hInst, IDS_MENUITEM1, text, MAX_PATH);
		AppendMenu(hMenu, MF_STRING, IDM_CONFIGURE, text);
		LoadString(hInst, IDS_MENUITEM2, text, MAX_PATH);
		AppendMenu(hMenu, MF_STRING, IDM_ABOUT, text);
	} // if

	// Install hooks.
	if(! hooklibInitialize(hwnd))
	{ // If setting up the hooks failed.
		MessageBox(NULL, TEXT("Error installing a hook."), TEXT("Error"), MB_ICONERROR);
		return GetLastError();
	} // if

	// Speak a startup message.
	if(GetPrivateProfileInt(TEXT("Settings"), TEXT("SpeakStartup"), 1, inifile))
	SpeakText("ROAR for Windows initialized. [:sync]");

	ShowWindow(hwnd, iCmdShow);
	UpdateWindow(hwnd);

	// The message loop:
	while(GetMessage(&msg, NULL, 0, 0))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	} // while

	// Clean up.
	hooklibCleanup();
	UnregisterHotKey(hwnd, HK_RESTART);
	UnregisterHotKey(hwnd, HK_RELOAD);
	UnregisterHotKey(hwnd, HK_HIDE);
	UnregisterHotKey(hwnd, HK_SETTINGS);
	UnregisterHotKey(hwnd, HK_REINST);
	UnregisterHotKey(hwnd, HK_REPEAT);
	UnregisterHotKey(hwnd, HK_READDESKTOP);
	UnregisterHotKey(hwnd, HK_READSTATUS);
	UnregisterHotKey(hwnd, HK_READTITLE);
	UnregisterHotKey(hwnd, HK_READVERBOSE);
	UnregisterHotKey(hwnd, HK_READFOCUSED);
	UnregisterHotKey(hwnd, HK_READACTIVE);
	ShutdownSpeech();
	UnregisterClass(szAppName, hInst);
	return msg.wParam;
} // winmain

LRESULT CALLBACK WndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	COPYDATASTRUCT* copyData = NULL; // For IPC with hooklib.dll.
	TCHAR caption[MAX_PATH] = {'\0'};
	TCHAR text[MAX_PATH] = {'\0'};
	switch(message)
	{
	case WM_COPYDATA: // A message from hooklib.dll - speak it.
		copyData = (COPYDATASTRUCT*) lParam;
		if(copyData->dwData) InterruptSpeech();
		SpeakText((PCHAR) copyData->lpData);
		SpeakText("[:sync]");
		return TRUE;
	case WM_SYSCOMMAND: // Additional system menu commands.
		switch(LOWORD(wParam))
		{
		case IDM_CONFIGURE: // Bring up Settings dialog.
			DialogBox(hInst, MAKEINTRESOURCE(IDD_DIALOG), hwnd, (DLGPROC)DlgProc);
			return 0;
		case IDM_ABOUT: // Bring up About box.
			LoadString(hInst, IDS_ABOUTCAP, caption, MAX_PATH);
			LoadString(hInst, IDS_ABOUTTXT, text, MAX_PATH);
			MessageBox(hwnd, text, caption, MB_ICONINFORMATION);
			return 0;
		} // switch
		break;
	case WM_HOTKEY: // Process hotkey messages.
		switch(LOWORD(wParam))
		{
		case HK_READACTIVE: // Read currently active window.
			hooklibReadActive();
			return 0;
		case HK_READFOCUSED: // Read currently focused control.
			hooklibReadFocused(FALSE);
			return 0;
		case HK_READVERBOSE: // Read currently focused control, verbose.
			hooklibReadFocused(TRUE);
			return 0;
		case HK_READTITLE: // Read title of currently active window.
			hooklibReadTitle();
			return 0;
		case HK_READSTATUS: // Read status bar of currently active window.
			hooklibReadStatus();
			return 0;
		case HK_READDESKTOP: // Read desktop window.
			hooklibReadDesktop();
			return 0;
		case HK_REPEAT: // Repeat last speech.
			hooklibRepeat();
			return 0;
		case HK_REINST: // Reinstall hooks.
			InterruptSpeech();
			hooklibCleanup();
			if(hooklibInitialize(hwnd))
			SpeakText("Hooks reinstalled.");
			else
			SpeakText("Failed to reinstall hooks.");
			SpeakText("[:sync]");
			return 0;
		case HK_SETTINGS: // Bring up the Settings dialog from anywhere in Windows.
			DialogBox(hInst, MAKEINTRESOURCE(IDD_DIALOG), NULL, (DLGPROC)DlgProc);
			return 0;
		case HK_HIDE: // Hide or show ROAR window.
			if(IsWindowVisible(hwnd))
			ShowWindow(hwnd, SW_HIDE);
			else
			ShowWindow(hwnd, SW_RESTORE);
			return 0;
		case HK_RELOAD: // Reload speech settings.
			SetupSpeech();
			SpeakText("Speech settings reloaded. [:sync]");
			return 0;
		case HK_RESTART: // Restart speech engine.
			InterruptSpeech();
			ShutdownSpeech();
			StartupSpeech();
			SetupSpeech();
			SpeakText("Speech engine restarted. [:sync]");
			return 0;
		} // switch
		break;
	case WM_CLOSE: // Ask the user if they want to quit.
		LoadString(hInst, IDS_CLOSECAP, caption, MAX_PATH);
		LoadString(hInst, IDS_CLOSETXT, text, MAX_PATH);
		if(MessageBox(hwnd, text, caption, MB_ICONEXCLAMATION | MB_YESNO) == IDYES)
		DestroyWindow(hwnd);
		return 0;
	case WM_DESTROY: // App quitted.
		PostQuitMessage(0);
		return 0;
	} // switch
	return DefWindowProc(hwnd, message, wParam, lParam);
} // window procedure

BOOL CALLBACK DlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	UINT i;
	TTS_CAPS_T caps;
	OPENFILENAME ofn;
	TCHAR filter[MAX_PATH] = {'\0'};
	TCHAR filename[MAX_PATH] = {'\0'};
	TCHAR title[MAX_PATH] = {'\0'};
	TCHAR caption[MAX_PATH] = {'\0'};
	TCHAR text[MAX_PATH] = {'\0'};
	ZeroMemory(&caps, sizeof(caps));
	ZeroMemory(&ofn, sizeof(ofn));
	switch(message)
	{
	case WM_INITDIALOG:
		for(i = 0; i < waveOutGetNumDevs(); i++)
		{
			WAVEOUTCAPS waveoutcaps;
			ZeroMemory(&waveoutcaps, sizeof(waveoutcaps));
			waveOutGetDevCaps(i, &waveoutcaps, sizeof(waveoutcaps));
			SendDlgItemMessage(hWnd, IDC_DEVICE, CB_INSERTSTRING, i, (LPARAM)waveoutcaps.szPname);
		}
		SendDlgItemMessage(hWnd, IDC_VOLUME, TBM_SETRANGE, 0, MAKELONG(0, 100));
		SendDlgItemMessage(hWnd, IDC_VOLUME, TBM_SETPAGESIZE, 0, 10);
		for(i = 0; i < WENDY + 1; i++)
		{
			LoadString(hInst, IDS_SPEAKER1 + i, text, MAX_PATH);
			SendDlgItemMessage(hWnd, IDC_SPEAKER, CB_INSERTSTRING, i, (LPARAM)text);
		}
		TextToSpeechGetCaps(&caps);
		SendDlgItemMessage(hWnd, IDC_RATE, TBM_SETRANGE, 0, MAKELONG(caps.dwMinimumSpeakingRate, caps.dwMaximumSpeakingRate));
		SendDlgItemMessage(hWnd, IDC_RATE, TBM_SETPAGESIZE, 0, 20);
		SendDlgItemMessage(hWnd, IDC_USERDICT, EM_LIMITTEXT, MAX_PATH-1, 0);
		SendDlgItemMessage(hWnd, IDC_DEVICE, CB_SETCURSEL, GetPrivateProfileInt(TEXT("Settings"), TEXT("AudioDevice"), 0, inifile), 0);
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("OwnDevice"), 0, inifile))
		{
			CheckDlgButton(hWnd, IDC_OWN, BST_CHECKED);
		}
		SendDlgItemMessage(hWnd, IDC_VOLUME, TBM_SETPOS, 0, GetPrivateProfileInt(TEXT("Settings"), TEXT("Volume"), 100, inifile));
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("HardwareMode"), 0, inifile))
		{
			CheckDlgButton(hWnd, IDC_HARDWARE, BST_CHECKED);
		}
		SetDlgItemInt(hWnd, IDC_PORT, GetPrivateProfileInt(TEXT("Settings"), TEXT("PortNumber"), 1, inifile), 0);
		SendDlgItemMessage(hWnd, IDC_SPEAKER, CB_SETCURSEL, GetPrivateProfileInt(TEXT("Settings"), TEXT("Speaker"), PAUL, inifile), 0);
		SendDlgItemMessage(hWnd, IDC_RATE, TBM_SETPOS, 0, GetPrivateProfileInt(TEXT("Settings"), TEXT("Rate"), 180, inifile));
		ZeroMemory(text, sizeof(text));
		GetPrivateProfileString(TEXT("Settings"), TEXT("UserDictionary"), NULL, text, MAX_PATH, inifile);
		SetDlgItemText(hWnd, IDC_USERDICT, text);
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("CharacterEcho"), 0, inifile))
		{
			CheckDlgButton(hWnd, IDC_ECHOCHAR, BST_CHECKED);
		}
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("CommandEcho"), 0, inifile))
		{
			CheckDlgButton(hWnd, IDC_ECHOCMD, BST_CHECKED);
		}
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("ModifierEcho"), 0, inifile))
		{
			CheckDlgButton(hWnd, IDC_ECHOMOD, BST_CHECKED);
		}
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("SpeakShortcuts"), 1, inifile))
		{
			CheckDlgButton(hWnd, IDC_SHORTCUTS, BST_CHECKED);
		}
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("SpeakEditChange"), 0, inifile))
		{
			CheckDlgButton(hWnd, IDC_EDIT, BST_CHECKED);
		}
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("SpeakStartup"), 1, inifile))
		{
			CheckDlgButton(hWnd, IDC_STARTUP, BST_CHECKED);
		}
		return TRUE;
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_USERDICTB:
			LoadString(hInst, IDS_DICFLT, filter, MAX_PATH);
			LoadString(hInst, IDS_DICCAP, caption, MAX_PATH);
			ofn.lStructSize = sizeof(ofn);
			ofn.hwndOwner = hWnd;
			ofn.hInstance = hInst;
			ofn.lpstrFilter = filter;
			ofn.nFilterIndex = 1;
			ofn.lpstrDefExt = TEXT("DIC");
			ofn.lpstrFile = filename;
			ofn.nMaxFile = MAX_PATH;
			ofn.lpstrFileTitle = title;
			ofn.nMaxFileTitle = MAX_PATH;
			ofn.lpstrTitle = caption;
			ofn.Flags = OFN_ENABLEHOOK | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_LONGNAMES | OFN_NOCHANGEDIR;
			ofn.lpfnHook = (LPOFNHOOKPROC)OFNHookProc;
			GetDlgItemText(hWnd, IDC_USERDICT, filename, MAX_PATH);
			if(GetOpenFileName(&ofn))
			{
				SetDlgItemText(hWnd, IDC_USERDICT, filename);
			}
			return TRUE;
		case IDOK:
		case IDC_APPLY:
			wsprintf(text, TEXT("%d"), SendDlgItemMessage(hWnd, IDC_DEVICE, CB_GETCURSEL, 0, 0));
			WritePrivateProfileString(TEXT("Settings"), TEXT("AudioDevice"), text, inifile);
			if(IsDlgButtonChecked(hWnd, IDC_OWN))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("OwnDevice"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("OwnDevice"), TEXT("0"), inifile);
			}
			wsprintf(text, TEXT("%d"), SendDlgItemMessage(hWnd, IDC_VOLUME, TBM_GETPOS, 0, 0));
			WritePrivateProfileString(TEXT("Settings"), TEXT("Volume"), text, inifile);
			if(IsDlgButtonChecked(hWnd, IDC_HARDWARE))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("HardwareMode"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("HardwareMode"), TEXT("0"), inifile);
			}
			GetDlgItemText(hWnd, IDC_PORT, text, MAX_PATH);
			WritePrivateProfileString(TEXT("Settings"), TEXT("PortNumber"), text, inifile);
			wsprintf(text, TEXT("%d"), SendDlgItemMessage(hWnd, IDC_SPEAKER, CB_GETCURSEL, 0, 0));
			WritePrivateProfileString(TEXT("Settings"), TEXT("Speaker"), text, inifile);
			wsprintf(text, TEXT("%d"), SendDlgItemMessage(hWnd, IDC_RATE, TBM_GETPOS, 0, 0));
			WritePrivateProfileString(TEXT("Settings"), TEXT("Rate"), text, inifile);
			GetDlgItemText(hWnd, IDC_USERDICT, text, MAX_PATH);
			WritePrivateProfileString(TEXT("Settings"), TEXT("UserDictionary"), text, inifile);
			if(IsDlgButtonChecked(hWnd, IDC_ECHOCHAR))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("CharacterEcho"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("CharacterEcho"), TEXT("0"), inifile);
			}
			if(IsDlgButtonChecked(hWnd, IDC_ECHOCMD))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("CommandEcho"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("CommandEcho"), TEXT("0"), inifile);
			}
			if(IsDlgButtonChecked(hWnd, IDC_ECHOMOD))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("ModifierEcho"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("ModifierEcho"), TEXT("0"), inifile);
			}
			if(IsDlgButtonChecked(hWnd, IDC_SHORTCUTS))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("SpeakShortcuts"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("SpeakShortcuts"), TEXT("0"), inifile);
			}
			if(IsDlgButtonChecked(hWnd, IDC_EDIT))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("SpeakEditChange"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("SpeakEditChange"), TEXT("0"), inifile);
			}
			if(IsDlgButtonChecked(hWnd, IDC_STARTUP))
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("SpeakStartup"), TEXT("1"), inifile);
			}
			else
			{
				WritePrivateProfileString(TEXT("Settings"), TEXT("SpeakStartup"), TEXT("0"), inifile);
			}
			SetupSpeech();
			if(LOWORD(wParam) == IDOK)
			{
				EndDialog(hWnd, TRUE);
			}
			return TRUE;
		case IDCANCEL:
			EndDialog(hWnd, FALSE);
			return TRUE;
		}
	}
	return FALSE;
}

UINT CALLBACK OFNHookProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	return 0;
}

BOOL StartupSpeech()
{
	BOOL NoSoundCard = FALSE;
	if(!waveOutGetNumDevs())
	{ // If there isn't a sound card installed in the system, beep through the PC speaker.
		Beep(512,256);
		NoSoundCard = TRUE;
	} // if
	if(GetPrivateProfileInt(TEXT("Settings"), TEXT("HardwareMode"), 0, inifile) || NoSoundCard)
	{ // Attempt to open a serial port.
		int PortNumber = GetPrivateProfileInt(TEXT("Settings"), TEXT("PortNumber"), 1, inifile);
		TCHAR PortString[MAX_PATH] = {'\0'};
		DCB SerialParams;
		COMMTIMEOUTS timeouts;
		ZeroMemory(&SerialParams, sizeof(SerialParams));
		ZeroMemory(&timeouts, sizeof(timeouts));
		wsprintf(PortString, TEXT("COM%d"), PortNumber);
		port = CreateFile(PortString, GENERIC_READ|GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
		if(port == INVALID_HANDLE_VALUE)
		{ // If opening up the serial port failed.
			TCHAR temp[MAX_PATH] = {'\0'};
			wsprintf(temp, TEXT("Error opening COM%d."), PortNumber);
			MessageBox(NULL, temp, TEXT("Error"), MB_ICONERROR);
			port = NULL;
			return FALSE;
		} // if
		SerialParams.DCBlength = sizeof(SerialParams);
		if(GetCommState(port, &SerialParams))
		{ // Set serial parameters.
			SerialParams.BaudRate = CBR_9600;
			SerialParams.ByteSize = 8;
			SerialParams.StopBits = ONESTOPBIT;
			SerialParams.Parity = NOPARITY;
			SetCommState(port, &SerialParams);
		} // if
		timeouts.ReadIntervalTimeout = MAXDWORD;
		SetCommTimeouts(port, &timeouts);
	} // if
	else
	{ // Attempt to load the software synth.
		UINT device;
		DWORD flags;
		MMRESULT EngineError;
		device = GetPrivateProfileInt(TEXT("Settings"), TEXT("AudioDevice"), WAVE_MAPPER, inifile);
		flags = REPORT_OPEN_ERROR;
		if(GetPrivateProfileInt(TEXT("Settings"), TEXT("OwnDevice"), 0, inifile))
		{ // This will tie up single channel sound cards if this is enabled.
			flags |= OWN_AUDIO_DEVICE;
		} // if
		EngineError = TextToSpeechStartup(NULL, &engine, device, flags);
		if(EngineError)
		{ // If starting up the speech engine failed.
			TCHAR temp[MAX_PATH] = {'\0'};
			wsprintf(temp, TEXT("Error starting speech engine. Error code: %d"), EngineError);
			MessageBox(NULL, temp, TEXT("Error"), MB_ICONERROR);
			return FALSE;
		} // if
	} // else if
	return TRUE;
}

TCHAR *SpeakerStrings[9] =
{
	TEXT("[:np]"),
	TEXT("[:nb]"),
	TEXT("[:nh]"),
	TEXT("[:nf]"),
	TEXT("[:nd]"),
	TEXT("[:nk]"),
	TEXT("[:nu]"),
	TEXT("[:nr]"),
	TEXT("[:nw]")
};

void SetupSpeech()
{
	TCHAR text[MAX_PATH] = {'\0'};
#ifdef UNICODE
	CHAR ansitext[MAX_PATH] = {'\0'};
#endif
	int speaker = GetPrivateProfileInt(TEXT("Settings"), TEXT("Speaker"), PAUL, inifile)%9;
	int rate = GetPrivateProfileInt(TEXT("Settings"), TEXT("Rate"), 180, inifile);
	int volume = GetPrivateProfileInt(TEXT("Settings"), TEXT("Volume"), 100, inifile);
	wsprintf(text, TEXT("%s [:ra %d] [:volume set %d] [:sync]"), SpeakerStrings[speaker], rate, volume);
	InterruptSpeech();
#ifdef UNICODE
	WideCharToMultiByte(CP_ACP, 0, text, -1, ansitext, MAX_PATH, NULL, NULL);
	SpeakText(ansitext);
#else
	SpeakText(text);
#endif
	if(engine)
	{
		ZeroMemory(text, sizeof(text));
		GetPrivateProfileString(TEXT("Settings"), TEXT("UserDictionary"), NULL, text, MAX_PATH, inifile);
		TextToSpeechUnloadUserDictionary(engine);
		if(lstrlen(text))
		{
#ifdef UNICODE
			WideCharToMultiByte(CP_ACP, 0, text, -1, ansitext, MAX_PATH, NULL, NULL);
			TextToSpeechLoadUserDictionary(engine, ansitext);
#else
			TextToSpeechLoadUserDictionary(engine, text);
#endif
		}
	}
	hooklibEchoChars(GetPrivateProfileInt(TEXT("Settings"), TEXT("CharacterEcho"), 0, inifile));
	hooklibEchoCommands(GetPrivateProfileInt(TEXT("Settings"), TEXT("CommandEcho"), 0, inifile));
	hooklibEchoModifiers(GetPrivateProfileInt(TEXT("Settings"), TEXT("ModifierEcho"), 0, inifile));
	hooklibSpeakShortcuts(GetPrivateProfileInt(TEXT("Settings"), TEXT("SpeakShortcuts"), 1, inifile));
	hooklibSpeakEditChange(GetPrivateProfileInt(TEXT("Settings"), TEXT("SpeakEditChange"), 0, inifile));
}

void SpeakText(CHAR *text)
{
	if(engine)
	{
		TextToSpeechSpeak(engine, text, TTS_NORMAL);
	}
	if(port)
	{
		int len = lstrlenA(text);
		WriteFile(port, text, len+1, &written, NULL);
	}
}

void InterruptSpeech()
{
	if(engine)
	{
		TextToSpeechReset(engine, FALSE);
	}
	if(port)
	{
		WriteFile(port, "\003", 1, &written, NULL);
	}
}

void ShutdownSpeech()
{
	if(engine)
	{
		TextToSpeechSync(engine);
		TextToSpeechUnloadUserDictionary(engine);
		TextToSpeechReset(engine, TRUE);
		TextToSpeechShutdown(engine);
		engine = NULL;
	}
	if(port)
	{
		CloseHandle(port);
		port = NULL;
	}
}

void DoSomethingSpecial()
{
	OSVERSIONINFO VersionInfo;
	ZeroMemory(&VersionInfo, sizeof(VersionInfo));
	VersionInfo.dwOSVersionInfoSize = sizeof(VersionInfo);
	GetVersionEx(&VersionInfo);
	if(VersionInfo.dwMajorVersion == 4 && VersionInfo.dwMinorVersion == 90 && VersionInfo.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS)
	{ // For the unsuspecting user who happens to be running Windows Me, i'm sorry I have to do this.
		if(engine)
		{ // The user will have to suffer through a message every time the program starts. After all, this is a free program, and you get what you pay for.
			TextToSpeechSpeak(engine, "[:nh :dv la 100] You are at an extremely high risk of crashes, and are proceeding at your own risk. You have been warned, and we can not be held responsible for any loss of data or crashes that may arise under these conditions!", TTS_FORCE);
			TextToSpeechSync(engine);
			TextToSpeechReset(engine, TRUE);
		} // if
		else if(port)
		{ // I don't care if you don't have a soundcard. It's hell either way!
			Beep(800,15000);
		} // else if
	} // if
}
