#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <commctrl.h>
#include "hooklib.h"

// Shared data section that's global to all DLL instances.
typedef struct
{
	HHOOK hKbdHook; // KBD-hook handle.
	HHOOK hMouseHook; // Mouse-hook handle.
	HHOOK hCbtHook; // CBT-hook handle.
	HHOOK hCallprocHook; // Callwndproc-hook handle.
	HWND hObserver; // Main app to be notified of events.
	HWND hActive; // Last activated window.
	HWND hFocused; // Last focused window.
	BOOL CharEcho; // Character echo.
	BOOL CommandEcho; // Command key echo.
	BOOL ModifierEcho; // Modifier key echo.
	BOOL SpeakShortcuts; // Speak shortcut keys.
	BOOL SpeakEditChange; // Speak on edit control change.
	TCHAR repeatBuf[MAX_TEXT_LENGTH]; // Buffer for storing last speech.
	TCHAR textOut[MAX_TEXT_LENGTH]; // Output buffer: message to observer.
	TCHAR textBuf[MAX_TEXT_LENGTH]; // Scratch-buffer for appending.
#ifdef UNICODE
	CHAR textOutAnsi[MAX_TEXT_LENGTH]; // Buffer sent to observer in unicode version.
#endif
} SharedMemory;

HINSTANCE hDllInstance = NULL; // Current DLL instance.
LPVOID lpvMem = NULL; // Pointer to shared memory.
HANDLE hMapObject = NULL; // Handle to file mapping.

BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
	BOOL fInit = TRUE;
	switch(fdwReason)
	{
	case DLL_PROCESS_ATTACH:
		hDllInstance = hinstDLL; // Save current instance.
		hMapObject = CreateFileMapping(INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE, 0, sizeof(SharedMemory), TEXT("HookMap")); // Allocate shared memory.
		if(!hMapObject) return FALSE;
		if(GetLastError() == ERROR_ALREADY_EXISTS) fInit = FALSE;
		lpvMem = MapViewOfFile(hMapObject, FILE_MAP_WRITE, 0, 0, 0); // Map shared memory to current process.
		if(!lpvMem) return FALSE;
		if(fInit) ZeroMemory(lpvMem, sizeof(SharedMemory));
		break;
	case DLL_PROCESS_DETACH:
		UnmapViewOfFile(lpvMem); // Unmap shared memory from current process.
		CloseHandle(hMapObject); // Deallocate shared memory.
		break;
	}
	return TRUE;
} // main

// VirtualAllocEx and VirtualFreeEx aren't implemented on Windows NT 3.51, so the following define enables NT 3.51 API compatibility.
//#define NT351COMPAT
LPVOID WINAPI VirtualAllocExStub(HANDLE hProcess, LPVOID lpAddress, DWORD dwSize, DWORD flAllocationType, DWORD flProtect)
{
#ifndef NT351COMPAT
	return VirtualAllocEx(hProcess, lpAddress, dwSize, flAllocationType, flProtect);
#else
	return VirtualAlloc(lpAddress, dwSize, flAllocationType, flProtect);
#endif
}

BOOL WINAPI VirtualFreeExStub(HANDLE hProcess, LPVOID lpAddress, DWORD dwSize, DWORD dwFreeType)
{
#ifndef NT351COMPAT
	return VirtualFreeEx(hProcess, lpAddress, dwSize, dwFreeType);
#else
	return VirtualFree(lpAddress, dwSize, dwFreeType);
#endif
}

LRESULT CALLBACK kbdHookFunction(int ncode, WPARAM wParam, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(ncode >= 0)
	{ // Pass on the message unmodified.
		if(ncode == HC_ACTION)
		{ // Handle keypresses.
			handleKeys(wParam, lParam);
		} // if
	} // if
	return CallNextHookEx(sm->hKbdHook, ncode, wParam, lParam);
} // kbdHookFunction

LRESULT CALLBACK mouseHookFunction(int ncode, WPARAM wParam, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(ncode >= 0)
	{ // Pass on the message unmodified.
		if(ncode == HC_ACTION)
		{ // Handle mouse events.
			switch(wParam)
			{ // Hack to interrupt speech on mouse down.
			case WM_NCLBUTTONDOWN:
			case WM_NCRBUTTONDOWN:
			case WM_NCMBUTTONDOWN:
			case WM_LBUTTONDOWN:
			case WM_RBUTTONDOWN:
			case WM_MBUTTONDOWN:
				lstrcpy(sm->textOut, TEXT(""));
				notifyObserver(TRUE);
				break;
			} // switch
		} // if
	} // if
	return CallNextHookEx(sm->hMouseHook, ncode, wParam, lParam);
} // mouseHookFunction

LRESULT CALLBACK cbtHookFunction(int ncode, WPARAM wParam, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(ncode >= 0)
	{ // Pass on the message unmodified.
		if(ncode == HCBT_SETFOCUS)
		{ // Handle focuschange.
			HWND hWindow = (HWND) wParam;
			sm->hFocused = hWindow;
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			if(hWindow == sm->hObserver)
			{ // If this is the observer, simply announce the window title.
				SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textOut);
			} // if
			else
			{ // Process normally.
				handleControls(hWindow, TRUE);
			} // else if
			notifyObserver(TRUE);
		} // if
		else if(ncode == HCBT_ACTIVATE)
		{ // Get the children of the window being activated.
			HWND hWindow = (HWND) wParam;
			sm->hActive = hWindow;
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			handleCommon(hWindow);
			EnumChildWindows(hWindow, enumerateDirectDescendants, (LPARAM) hWindow);
			notifyObserver(TRUE);
		} // else if
	} // if
	return CallNextHookEx(sm->hCbtHook, ncode, wParam, lParam);
} // cbtHookFunction

LRESULT CALLBACK callprocHookFunction(int ncode, WPARAM wParam, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(ncode >= 0)
	{ // Can process the message.
		CWPSTRUCT *cwp = (CWPSTRUCT*) lParam;
		if(cwp->message == WM_COMMAND && HIWORD(cwp->wParam) == BN_CLICKED)
		{ // On button click.
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			handleControls((HWND) cwp->lParam, FALSE);
			notifyObserver(TRUE);
		} // if
		else if(cwp->message == WM_COMMAND && HIWORD(cwp->wParam) == CBN_SELCHANGE)
		{ // On combo box selection change.
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			handleControls((HWND) cwp->lParam, FALSE);
			notifyObserver(TRUE);
		} // else if
		else if(cwp->message == WM_COMMAND && HIWORD(cwp->wParam) == LBN_SELCHANGE)
		{ // On list box selection change.
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			handleControls((HWND) cwp->lParam, FALSE);
			notifyObserver(TRUE);
		} // else if
		else if(cwp->lParam == (LPARAM)sm->hFocused && cwp->message == WM_COMMAND && HIWORD(cwp->wParam) == EN_CHANGE)
		{ // On edit change, and only when in focus.
			if(sm->SpeakEditChange)
			{ // Only speak if the user enables this functionality, as it can get annoying with long passages of text.
				ZeroMemory(sm->textOut, sizeof(sm->textOut));
				handleControls((HWND) cwp->lParam, FALSE);
				notifyObserver(TRUE);
			} // if
		} // else if
		else if(cwp->message == WM_HSCROLL || cwp->message == WM_VSCROLL)
		{ // On scroll.
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			handleControls((HWND) cwp->lParam, FALSE);
			notifyObserver(TRUE);
		} // else if
		else if(cwp->message == WM_MENUSELECT)
		{ // On menu selection change.
			ZeroMemory(sm->textOut, sizeof(sm->textOut));
			handleMenus(cwp->wParam, cwp->lParam);
			notifyObserver(TRUE);
		} // else if
		else if(cwp->message == WM_NOTIFY)
		{ // On notify, used by common controls.
			handleNotifications(cwp->lParam);
		} // else if
	} // if
	return CallNextHookEx(sm->hCallprocHook, ncode, wParam, lParam);
} // callprocHookFunction

BOOL CALLBACK enumerateDescendants(HWND hwnd, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(IsWindowVisible(hwnd))
	{ // Only if it's visible.
		handleCommon(hwnd);
		if(!IsWindowEnabled(hwnd))
		lstrcat(sm->textOut, TEXT(" disabled"));
		lstrcat(sm->textOut, TEXT(" [:sync] "));
	} // if
	return TRUE;
} // enumerateDescendants

BOOL CALLBACK enumerateDirectDescendants(HWND hwnd, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(GetParent(hwnd) == ((HWND) lParam))
	{ // Only if it's a direct descendant.
		handleControls(hwnd, TRUE);
		if(!IsWindowEnabled(hwnd))
		lstrcat(sm->textOut, TEXT(" disabled"));
		lstrcat(sm->textOut, TEXT(" [:sync] "));
	} // if
	return TRUE;
} // enumerateDirectDescendants

void notifyObserver(BOOL interrupt)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	COPYDATASTRUCT data;
	// Make a copy of textOut for repeat functionality.
	if(lstrlen(sm->textOut)) lstrcpy(sm->repeatBuf, sm->textOut);
	// Send the text in textOut to main window.
	if(interrupt)
	data.dwData = 1;
	else
	data.dwData = 0;
	data.cbData = lstrlen(sm->textOut) + 1;
#ifdef UNICODE
	WideCharToMultiByte(CP_ACP, 0, sm->textOut, -1, (LPSTR)sm->textOutAnsi, MAX_TEXT_LENGTH, NULL, NULL);
	data.lpData = (PVOID) sm->textOutAnsi;
#else
	data.lpData = (PVOID) sm->textOut;
#endif
	SendMessage(sm->hObserver, WM_COPYDATA, (WPARAM) hDllInstance, (LPARAM) &data);
} // notifyObserver

void filterShortcut()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	TCHAR tempBuf[MAX_PATH] = {'\0'};
	TCHAR shortcut[2] = {'\0'};
	int len = lstrlen(sm->textBuf);
	int count = 0;
	int i;
	if(len >= MAX_PATH) return;
	for(i = 0; i < len; i++)
	{
		if(sm->textBuf[i] == '&')
		{
			shortcut[0] = sm->textBuf[i+1];
			continue;
		}
		tempBuf[count] = sm->textBuf[i];
		count++;
	}
	if(count != len)
	{
		lstrcpy(sm->textBuf, tempBuf);
		if(sm->SpeakShortcuts)
		{
			lstrcat(sm->textBuf, TEXT(" shortcut "));
			lstrcat(sm->textBuf, shortcut);
		}
	}
}

void handleControls(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	handleCommon(hWindow);
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	GetClassName(hWindow, sm->textBuf, MAX_PATH);
	if(lstrcmp(sm->textBuf, TEXT("Button")) == 0)
	handleButtons(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("ComboBox")) == 0)
	handleComboBox(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("Edit")) == 0)
	handleEdit(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("ListBox")) == 0)
	handleListBox(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("MDIClient")) == 0)
	handleMDIClient(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("ScrollBar")) == 0)
	handleScrollBar(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("Static")) == 0)
	handleStaticText(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("SysAnimate32")) == 0)
	handleAnimation(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("ComboBoxEx32")) == 0)
	handleComboBoxEx(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("SysDateTimePick32")) == 0)
	handleDateTimeControl(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysHeader32")) == 0)
	handleHeaderControl(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("msctls_hotkey32")) == 0)
	handleHotKeyControl(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysIPAddress32")) == 0)
	handleIPAddress(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysListView32")) == 0)
	handleListView(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysMonthCal32")) == 0)
	handleMonthCalControl(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysPager")) == 0)
	handlePagerControl(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("msctls_progress32")) == 0)
	handleProgressBar(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("ReBarWindow32")) == 0)
	handleReBar(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("msctls_updown32")) == 0)
	handleSpinControl(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("msctls_statusbar32")) == 0)
	handleStatusBar(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysTabControl32")) == 0)
	handleTabControl(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("ToolbarWindow32")) == 0)
	handleToolBar(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("tooltips_class32")) == 0)
	handleToolTip(hWindow);
	else if(lstrcmp(sm->textBuf, TEXT("msctls_trackbar32")) == 0)
	handleTrackBar(hWindow, verbose);
	else if(lstrcmp(sm->textBuf, TEXT("SysTreeView32")) == 0)
	handleTreeView(hWindow, verbose);
} // handleControls

#define NUM_CLASSES 25

const TCHAR *ClassStrings[NUM_CLASSES] =
{
	TEXT("Button"),
	TEXT("ComboBox"),
	TEXT("Edit"),
	TEXT("ListBox"),
	TEXT("MDIClient"),
	TEXT("ScrollBar"),
	TEXT("Static"),
	TEXT("SysAnimate32"),
	TEXT("ComboBoxEx32"),
	TEXT("SysDateTimePick32"),
	TEXT("SysHeader32"),
	TEXT("msctls_hotkey32"),
	TEXT("SysIPAddress32"),
	TEXT("SysListView32"),
	TEXT("SysMonthCal32"),
	TEXT("SysPager"),
	TEXT("msctls_progress32"),
	TEXT("ReBarWindow32"),
	TEXT("msctls_updown32"),
	TEXT("msctls_statusbar32"),
	TEXT("SysTabControl32"),
	TEXT("ToolbarWindow32"),
	TEXT("tooltips_class32"),
	TEXT("msctls_trackbar32"),
	TEXT("SysTreeView32")
};

void handleCommon(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int i;
	BOOL found = FALSE;
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	GetClassName(hWindow, sm->textBuf, MAX_PATH);
	for(i = 0; i < NUM_CLASSES; i++)
	{
		if(lstrcmp(sm->textBuf, ClassStrings[i]) == 0)
		{
			found = TRUE;
			break;
		}
	}
	if(!found)
	{ // Generic processing.
		if(lstrcmp(sm->textBuf, TEXT("#32770")) == 0)
		lstrcat(sm->textOut, TEXT("dialog"));
		else if(lstrcmp(sm->textBuf, TEXT("#32769")) == 0)
		lstrcat(sm->textOut, TEXT("desktop window"));
		else if(lstrcmp(sm->textBuf, TEXT("#32768")) == 0)
		lstrcat(sm->textOut, TEXT("menu"));
		else
		lstrcat(sm->textOut, sm->textBuf);
		lstrcat(sm->textOut, TEXT(" "));
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
		filterShortcut();
		lstrcat(sm->textOut, sm->textBuf);
		lstrcat(sm->textOut, TEXT(" [:sync] "));
	} // if
} // handleCommon

void handleButtons(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD s = GetWindowLong(hWindow, GWL_STYLE) & 0xFL; // All but the 4 LSB.
	BOOL isCheckable = TRUE; // SHould query whether it's checked.
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
	filterShortcut();
	lstrcat(sm->textOut, sm->textBuf);
	switch(s)
	{
	case BS_PUSHBUTTON:
		isCheckable = FALSE;
		if(verbose) lstrcat(sm->textOut, TEXT(" push button"));
		break;
	case BS_DEFPUSHBUTTON:
		isCheckable = FALSE;
		if(verbose) lstrcat(sm->textOut, TEXT(" default push button"));
		break;
	case BS_GROUPBOX:
		isCheckable = FALSE;
		if(verbose) lstrcat(sm->textOut, TEXT(" group box"));
		break;
	case BS_USERBUTTON:
	case BS_OWNERDRAW:
		isCheckable = FALSE;
		if(verbose) lstrcat(sm->textOut, TEXT(" customized button"));
		break;
	case BS_CHECKBOX:
	case BS_AUTOCHECKBOX:
		if(verbose) lstrcat(sm->textOut, TEXT(" check box"));
		break;
	case BS_3STATE:
	case BS_AUTO3STATE:
		if(verbose) lstrcat(sm->textOut, TEXT(" 3-state check box"));
		break;
	case BS_RADIOBUTTON:
	case BS_AUTORADIOBUTTON:
		if(verbose) lstrcat(sm->textOut, TEXT(" radio button"));
		break;
	} // switch

	if(isCheckable == TRUE)
	{
		switch(SendMessage(hWindow, BM_GETCHECK, 0, 0))
		{
		case BST_CHECKED:
			lstrcat(sm->textOut, TEXT(" selected"));
			break;
		case BST_INDETERMINATE:
			lstrcat(sm->textOut, TEXT(" partially selected"));
			break;
		case BST_UNCHECKED:
			lstrcat(sm->textOut, TEXT(" unselected"));
			break;
		} // switch
	} // if
} // handleButtons

void handleComboBox(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD s;
	int current;
	s = GetWindowLong(hWindow, GWL_STYLE);
	if(verbose)
	{
		lstrcat(sm->textOut, TEXT("combo box "));
		if(s & CBS_OWNERDRAWFIXED || s & CBS_OWNERDRAWVARIABLE)
		lstrcat(sm->textOut, TEXT("owner-drawn "));
	}
	current = SendMessage(hWindow, CB_GETCURSEL, 0, 0);
	if(current == CB_ERR)
	{
		lstrcat(sm->textOut, TEXT("nothing selected"));
	}
	else
	{
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, CB_GETLBTEXT, current, (LPARAM)sm->textBuf);
		lstrcat(sm->textOut, sm->textBuf);
		if(verbose) lstrcat(sm->textOut, TEXT(" selected"));
	}
} // handleComboBox

void handleEdit(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD s;
	int length;
	s = GetWindowLong(hWindow, GWL_STYLE);
	if(verbose)
	{
		lstrcat(sm->textOut, TEXT("edit "));
		if(s & ES_MULTILINE)
		lstrcat(sm->textOut, TEXT("multi line "));
		if(s & ES_PASSWORD)
		lstrcat(sm->textOut, TEXT("protected "));
		if(s & ES_READONLY)
		lstrcat(sm->textOut, TEXT("read only "));
	}
	if(!(s & ES_PASSWORD))
	{ // We can't be exposing passwords, can we?
		length = SendMessage(hWindow, WM_GETTEXTLENGTH, 0, 0);
		if(length >= (MAX_TEXT_LENGTH-lstrlen(sm->textOut)))
		{ // Report to the user that there isn't enough room in the buffer, and play a beep.
			lstrcat(sm->textOut, TEXT("not enough room in buffer"));
			MessageBeep(0xffffffff);
		} // if
		else if(length == 0)
		{ // Report to the user that the edit field is blank.
			lstrcat(sm->textOut, TEXT("blank"));
		} // else if
		else
		{ // get the actual text
			ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
			SendMessage(hWindow, WM_GETTEXT, length+1, (LPARAM)sm->textBuf);
			lstrcat(sm->textOut, sm->textBuf);
		} // else if
	} // if
} // handleEdit

void handleListBox(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD s;
	int current;
	s = GetWindowLong(hWindow, GWL_STYLE);
	if(verbose)
	{
		lstrcat(sm->textOut, TEXT("list box "));
		if(s & LBS_OWNERDRAWFIXED || s & LBS_OWNERDRAWVARIABLE)
		lstrcat(sm->textOut, TEXT("owner-drawn "));
	}
	current = SendMessage(hWindow, LB_GETCURSEL, 0, 0);
	if(current == LB_ERR)
	{
		lstrcat(sm->textOut, TEXT("nothing selected"));
	}
	else
	{
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, LB_GETTEXT, current, (LPARAM)sm->textBuf);
		lstrcat(sm->textOut, sm->textBuf);
		if(verbose) lstrcat(sm->textOut, TEXT(" selected"));
	}
} // handleListBox

void handleMDIClient(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(verbose) lstrcat(sm->textOut, TEXT("MDI client "));
	handleControls((HWND)SendMessage(hWindow, WM_MDIGETACTIVE, 0, 0), verbose);
} // handleMDIClient

void handleScrollBar(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD s;
	int current;
	s = GetWindowLong(hWindow, GWL_STYLE);
	if(verbose)
	{
		if(s & SBS_HORZ)
		lstrcat(sm->textOut, TEXT("horizontal "));
		if(s & SBS_VERT)
		lstrcat(sm->textOut, TEXT("vertical "));
		lstrcat(sm->textOut, TEXT(" scroll bar "));
	}
	current = SendMessage(hWindow, SBM_GETPOS, 0, 0);
	wsprintf(sm->textBuf, TEXT("%d"), current);
	lstrcat(sm->textOut, sm->textBuf);
} // handleScrollBar

void handleStaticText(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
	filterShortcut();
	lstrcat(sm->textOut, sm->textBuf);
} // handleStaticText

void handleAnimation(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	lstrcat(sm->textOut, TEXT("animation"));
} // handleAnimation

void handleComboBoxEx(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	lstrcat(sm->textOut, TEXT("extended combo box"));
} // handleComboBoxEx

void handleDateTimeControl(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD ProcessId;
	if(verbose) lstrcat(sm->textOut, TEXT("date time control "));
	GetWindowThreadProcessId(hWindow, &ProcessId);
	if(ProcessId == GetCurrentProcessId())
	{
		SYSTEMTIME curtime;
		ZeroMemory(&curtime, sizeof(curtime));
		if(SendMessage(hWindow, DTM_GETSYSTEMTIME, 0, (LPARAM)&curtime) == GDT_VALID)
		{
			TCHAR timeformat[MAX_PATH] = {'\0'};
			TCHAR dateformat[MAX_PATH] = {'\0'};
			GetTimeFormat(LOCALE_SYSTEM_DEFAULT, 0, &curtime, NULL, timeformat, MAX_PATH);
			GetDateFormat(LOCALE_SYSTEM_DEFAULT, 0, &curtime, NULL, dateformat, MAX_PATH);
			wsprintf(sm->textBuf, TEXT("%s on %s"), timeformat, dateformat);
			lstrcat(sm->textOut, sm->textBuf);
		}
	}
	else
	{
		HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
		if(hProcess)
		{
			SYSTEMTIME *pTime = (SYSTEMTIME*)VirtualAllocExStub(hProcess, NULL, sizeof(SYSTEMTIME), MEM_COMMIT, PAGE_READWRITE);
			if(pTime)
			{
				if(SendMessage(hWindow, DTM_GETSYSTEMTIME, 0, (LPARAM)pTime) == GDT_VALID)
				{
					SYSTEMTIME curtime;
					TCHAR timeformat[MAX_PATH] = {'\0'};
					TCHAR dateformat[MAX_PATH] = {'\0'};
					ZeroMemory(&curtime, sizeof(curtime));
					ReadProcessMemory(hProcess, pTime, &curtime, sizeof(curtime), NULL);
					GetTimeFormat(LOCALE_SYSTEM_DEFAULT, 0, &curtime, NULL, timeformat, MAX_PATH);
					GetDateFormat(LOCALE_SYSTEM_DEFAULT, 0, &curtime, NULL, dateformat, MAX_PATH);
					wsprintf(sm->textBuf, TEXT("%s on %s"), timeformat, dateformat);
					lstrcat(sm->textOut, sm->textBuf);
				}
				VirtualFreeExStub(hProcess, pTime, 0, MEM_RELEASE);
			}
			CloseHandle(hProcess);
		}
	}
} // handleDateTimeControl

void handleHeaderControl(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int count;
	int i;
	if(verbose)
	{
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
		filterShortcut();
		lstrcat(sm->textOut, sm->textBuf);
		lstrcat(sm->textOut, TEXT(" header control "));
	}
	count = SendMessage(hWindow, HDM_GETITEMCOUNT, 0, 0);
	for(i = 0; i < count; i++)
	{
		DWORD ProcessId;
		GetWindowThreadProcessId(hWindow, &ProcessId);
		if(ProcessId == GetCurrentProcessId())
		{
			HDITEM item;
			ZeroMemory(&item, sizeof(item));
			item.mask = HDI_TEXT;
			item.pszText = sm->textBuf;
			item.cchTextMax = MAX_PATH;
			ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
			if(SendMessage(hWindow, HDM_GETITEM, i, (LPARAM)&item))
			{
				lstrcat(sm->textOut, sm->textBuf);
				if(i < (count-1)) lstrcat(sm->textOut, TEXT(", "));
			}
		}
		else
		{
			HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
			if(hProcess)
			{
				HDITEM *pHdItem = (HDITEM*)VirtualAllocExStub(hProcess, NULL, sizeof(HDITEM), MEM_COMMIT, PAGE_READWRITE);
				LPTSTR pText = (LPTSTR)VirtualAllocExStub(hProcess, NULL, sizeof(TCHAR)*MAX_PATH, MEM_COMMIT, PAGE_READWRITE);
				if(pHdItem && pText)
				{
					HDITEM item;
					ZeroMemory(&item, sizeof(item));
					item.mask = HDI_TEXT;
					item.pszText = pText;
					item.cchTextMax = MAX_PATH;
					WriteProcessMemory(hProcess, pHdItem, &item, sizeof(item), NULL);
					if(SendMessage(hWindow, HDM_GETITEM, i, (LPARAM)pHdItem))
					{
						ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
						ReadProcessMemory(hProcess, pText, sm->textBuf, sizeof(TCHAR)*MAX_PATH, NULL);
						lstrcat(sm->textOut, sm->textBuf);
						if(i < (count-1)) lstrcat(sm->textOut, TEXT(", "));
					}
					VirtualFreeExStub(hProcess, pText, 0, MEM_RELEASE);
					VirtualFreeExStub(hProcess, pHdItem, 0, MEM_RELEASE);
				}
				CloseHandle(hProcess);
			}
		}
	}
} // handleHeaderControl

void handleHotKeyControl(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int hotkey;
	BYTE kbstate[256] = {'\0'};
	if(verbose) lstrcat(sm->textOut, TEXT("hot key field "));
	hotkey = SendMessage(hWindow, HKM_GETHOTKEY, 0, 0);
	if(hotkey)
	{
		if(HIBYTE(hotkey) & HOTKEYF_CONTROL)
		lstrcat(sm->textOut, TEXT("control "));
		if(HIBYTE(hotkey) & HOTKEYF_ALT)
		lstrcat(sm->textOut, TEXT("alt "));
		if(HIBYTE(hotkey) & HOTKEYF_SHIFT)
		lstrcat(sm->textOut, TEXT("shift "));
		if(HIBYTE(hotkey) & HOTKEYF_EXT)
		lstrcat(sm->textOut, TEXT("extended "));
		GetKeyboardState(kbstate);
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
#ifdef UNICODE
		ToUnicode(LOBYTE(hotkey), 0, kbstate, sm->textBuf, MAX_PATH, 0);
#else
		ToAscii(LOBYTE(hotkey), 0, kbstate, (WCHAR *)sm->textBuf, 0);
#endif
		lstrcat(sm->textOut, sm->textBuf);
	}
	else
	{
		lstrcat(sm->textOut, TEXT("none"));
	}
} // handleHotKeyControl

void handleIPAddress(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD ProcessId;
	if(verbose) lstrcat(sm->textOut, TEXT("IP address "));
	GetWindowThreadProcessId(hWindow, &ProcessId);
	if(ProcessId == GetCurrentProcessId())
	{
		DWORD address;
		if(SendMessage(hWindow, IPM_GETADDRESS, 0, (LPARAM)&address))
		{
			DWORD fields[4];
			fields[0] = FIRST_IPADDRESS(address);
			fields[1] = SECOND_IPADDRESS(address);
			fields[2] = THIRD_IPADDRESS(address);
			fields[3] = FOURTH_IPADDRESS(address);
			wsprintf(sm->textBuf, TEXT("%d.%d.%d.%d"), fields[0], fields[1], fields[2], fields[3]);
			lstrcat(sm->textOut, sm->textBuf);
		}
		else
		{
			lstrcat(sm->textOut, TEXT("blank"));
		}
	}
	else
	{
		HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
		if(hProcess)
		{
			LPDWORD pAddress = (DWORD*)VirtualAllocExStub(hProcess, NULL, sizeof(DWORD), MEM_COMMIT, PAGE_READWRITE);
			if(pAddress)
			{
				if(SendMessage(hWindow, IPM_GETADDRESS, 0, (LPARAM)pAddress))
				{
					DWORD address;
					DWORD fields[4];
					ReadProcessMemory(hProcess, pAddress, &address, sizeof(address), NULL);
					fields[0] = FIRST_IPADDRESS(address);
					fields[1] = SECOND_IPADDRESS(address);
					fields[2] = THIRD_IPADDRESS(address);
					fields[3] = FOURTH_IPADDRESS(address);
					wsprintf(sm->textBuf, TEXT("%d.%d.%d.%d"), fields[0], fields[1], fields[2], fields[3]);
					lstrcat(sm->textOut, sm->textBuf);
				}
				else
				{
					lstrcat(sm->textOut, TEXT("blank"));
				}
				VirtualFreeExStub(hProcess, pAddress, 0, MEM_RELEASE);
			}
			CloseHandle(hProcess);
		}
	}
} // handleIPAddress

void handleListView(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int current;
	if(verbose)
	{
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
		filterShortcut();
		lstrcat(sm->textOut, sm->textBuf);
		lstrcat(sm->textOut, TEXT(" list view "));
	}
	current = SendMessage(hWindow, LVM_GETNEXTITEM, -1, LVNI_SELECTED);
	if(current >= 0)
	{
		DWORD ProcessId;
		GetWindowThreadProcessId(hWindow, &ProcessId);
		if(ProcessId == GetCurrentProcessId())
		{
			LVITEM item;
			ZeroMemory(&item, sizeof(item));
			item.mask = LVIF_TEXT;
			item.pszText = sm->textBuf;
			item.cchTextMax = MAX_PATH;
			ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
			if(SendMessage(hWindow, LVM_GETITEMTEXT, current, (LPARAM)&item))
			{
				lstrcat(sm->textOut, sm->textBuf);
				if(verbose) lstrcat(sm->textOut, TEXT(" selected"));
			}
		}
		else
		{
			HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
			if(hProcess)
			{
				LVITEM *pLvItem = (LVITEM*)VirtualAllocExStub(hProcess, NULL, sizeof(LVITEM), MEM_COMMIT, PAGE_READWRITE);
				LPTSTR pText = (LPTSTR)VirtualAllocExStub(hProcess, NULL, sizeof(TCHAR)*MAX_PATH, MEM_COMMIT, PAGE_READWRITE);
				if(pLvItem && pText)
				{
					LVITEM item;
					int nRes;
					ZeroMemory(&item, sizeof(item));
					item.mask = LVIF_TEXT;
					item.pszText = pText;
					item.cchTextMax = MAX_PATH;
					WriteProcessMemory(hProcess, pLvItem, &item, sizeof(item), NULL);
					nRes = SendMessage(hWindow, LVM_GETITEMTEXT, current, (LPARAM)pLvItem);
					if(nRes)
					{
						ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
						ReadProcessMemory(hProcess, pText, sm->textBuf, sizeof(TCHAR)*nRes, NULL);
						lstrcat(sm->textOut, sm->textBuf);
						if(verbose) lstrcat(sm->textOut, TEXT(" selected"));
					}
					VirtualFreeExStub(hProcess, pText, 0, MEM_RELEASE);
					VirtualFreeExStub(hProcess, pLvItem, 0, MEM_RELEASE);
				}
				CloseHandle(hProcess);
			}
		}
	}
	else
	{
		lstrcat(sm->textOut, TEXT("nothing selected"));
	}
} // handleListView

void handleMonthCalControl(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD ProcessId;
	if(verbose) lstrcat(sm->textOut, TEXT("month calendar control "));
	GetWindowThreadProcessId(hWindow, &ProcessId);
	if(ProcessId == GetCurrentProcessId())
	{
		SYSTEMTIME curtime;
		ZeroMemory(&curtime, sizeof(curtime));
		if(SendMessage(hWindow, MCM_GETCURSEL, 0, (LPARAM)&curtime))
		{
			ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
			GetDateFormat(LOCALE_SYSTEM_DEFAULT, 0, &curtime, NULL, sm->textBuf, MAX_PATH);
			lstrcat(sm->textOut, sm->textBuf);
		}
	}
	else
	{
		HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
		if(hProcess)
		{
			SYSTEMTIME *pTime = (SYSTEMTIME*)VirtualAllocExStub(hProcess, NULL, sizeof(SYSTEMTIME), MEM_COMMIT, PAGE_READWRITE);
			if(pTime)
			{
				if(SendMessage(hWindow, MCM_GETCURSEL, 0, (LPARAM)pTime))
				{
					SYSTEMTIME curtime;
					ZeroMemory(&curtime, sizeof(curtime));
					ReadProcessMemory(hProcess, pTime, &curtime, sizeof(curtime), NULL);
					ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
					GetDateFormat(LOCALE_SYSTEM_DEFAULT, 0, &curtime, NULL, sm->textBuf, MAX_PATH);
					lstrcat(sm->textOut, sm->textBuf);
				}
				VirtualFreeExStub(hProcess, pTime, 0, MEM_RELEASE);
			}
			CloseHandle(hProcess);
		}
	}
} // handleMonthCalControl

void handlePagerControl(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	lstrcat(sm->textOut, TEXT("pager control"));
} // handlePagerControl

void handleProgressBar(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int current;
	int minrange;
	int maxrange;
	if(verbose) lstrcat(sm->textOut, TEXT("progress bar "));
	current = SendMessage(hWindow, PBM_GETPOS, 0, 0);
	minrange = SendMessage(hWindow, PBM_GETRANGE, TRUE, 0);
	maxrange = SendMessage(hWindow, PBM_GETRANGE, FALSE, 0);
	if(minrange == maxrange)
	current = 100;
	else
	current = (100 * (current - minrange)) / (maxrange - minrange);
	wsprintf(sm->textBuf, TEXT("%d"), current);
	lstrcat(sm->textOut, sm->textBuf);
} // handleProgressBar

void handleReBar(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	lstrcat(sm->textOut, TEXT("cool bar"));
} // handleReBar

void handleSpinControl(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int current;
	if(verbose) lstrcat(sm->textOut, TEXT("spin control "));
	current = SendMessage(hWindow, UDM_GETPOS, 0, 0);
	wsprintf(sm->textBuf, TEXT("%d"), current);
	lstrcat(sm->textOut, sm->textBuf);
} // handleSpinControl

void handleStatusBar(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	if(verbose) lstrcat(sm->textOut, TEXT("status bar "));
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	SendMessage(hWindow, SB_GETTEXT, 0, (LPARAM)sm->textBuf);
	lstrcat(sm->textOut, sm->textBuf);
} // handleStatusBar

void handleTabControl(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	int current;
	if(verbose)
	{
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
		filterShortcut();
		lstrcat(sm->textOut, sm->textBuf);
		lstrcat(sm->textOut, TEXT(" tab control "));
	}
	current = SendMessage(hWindow, TCM_GETCURSEL, 0, 0);
	if(current >= 0)
	{
		DWORD ProcessId;
		GetWindowThreadProcessId(hWindow, &ProcessId);
		if(ProcessId == GetCurrentProcessId())
		{
			TCITEM item;
			ZeroMemory(&item, sizeof(item));
			item.mask = TCIF_TEXT;
			item.pszText = sm->textBuf;
			item.cchTextMax = MAX_PATH;
			ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
			if(SendMessage(hWindow, TCM_GETITEM, current, (LPARAM)&item))
			{
				lstrcat(sm->textOut, sm->textBuf);
			}
		}
		else
		{
			HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
			if(hProcess)
			{
				TCITEM *pTcItem = (TCITEM*)VirtualAllocExStub(hProcess, NULL, sizeof(TCITEM), MEM_COMMIT, PAGE_READWRITE);
				LPTSTR pText = (LPTSTR)VirtualAllocExStub(hProcess, NULL, sizeof(TCHAR)*MAX_PATH, MEM_COMMIT, PAGE_READWRITE);
				if(pTcItem && pText)
				{
					TCITEM item;
					ZeroMemory(&item, sizeof(item));
					item.mask = TCIF_TEXT;
					item.pszText = pText;
					item.cchTextMax = MAX_PATH;
					WriteProcessMemory(hProcess, pTcItem, &item, sizeof(item), NULL);
					if(SendMessage(hWindow, TCM_GETITEM, current, (LPARAM)pTcItem))
					{
						ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
						ReadProcessMemory(hProcess, pText, sm->textBuf, sizeof(TCHAR)*MAX_PATH, NULL);
						lstrcat(sm->textOut, sm->textBuf);
					}
					VirtualFreeExStub(hProcess, pText, 0, MEM_RELEASE);
					VirtualFreeExStub(hProcess, pTcItem, 0, MEM_RELEASE);
				}
				CloseHandle(hProcess);
			}
		}
	}
} // handleTabControl

void handleToolBar(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	lstrcat(sm->textOut, TEXT("tool bar"));
} // handleToolBar

void handleToolTip(HWND hWindow)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
	lstrcat(sm->textOut, sm->textBuf);
} // handleToolTip

void handleTrackBar(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	DWORD s;
	int current;
	s = GetWindowLong(hWindow, GWL_STYLE);
	if(verbose)
	{
		if(s & TBS_HORZ)
		lstrcat(sm->textOut, TEXT("horizontal "));
		if(s & TBS_VERT)
		lstrcat(sm->textOut, TEXT("vertical "));
		lstrcat(sm->textOut, TEXT(" track bar "));
	}
	current = SendMessage(hWindow, TBM_GETPOS, 0, 0);
	wsprintf(sm->textBuf, TEXT("%d"), current);
	lstrcat(sm->textOut, sm->textBuf);
} // handleTrackBar

void handleTreeView(HWND hWindow, BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	HTREEITEM tvhandle = NULL;
	if(verbose)
	{
		ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
		SendMessage(hWindow, WM_GETTEXT, MAX_PATH, (LPARAM)sm->textBuf);
		filterShortcut();
		lstrcat(sm->textOut, sm->textBuf);
		lstrcat(sm->textOut, TEXT(" tree view "));
	}
	tvhandle = (HTREEITEM)SendMessage(hWindow, TVM_GETNEXTITEM, TVGN_CARET, (LPARAM)tvhandle);
	if(tvhandle)
	{
		DWORD ProcessId;
		GetWindowThreadProcessId(hWindow, &ProcessId);
		if(ProcessId == GetCurrentProcessId())
		{
			TVITEM item;
			ZeroMemory(&item, sizeof(item));
			item.mask = TVIF_CHILDREN | TVIF_HANDLE | TVIF_STATE | TVIF_TEXT;
			item.hItem = tvhandle;
			item.stateMask = TVIS_EXPANDED;
			item.pszText = sm->textBuf;
			item.cchTextMax = MAX_PATH;
			ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
			if(SendMessage(hWindow, TVM_GETITEM, 0, (LPARAM)&item))
			{
				lstrcat(sm->textOut, sm->textBuf);
				if(item.cChildren)
				{
					if(item.state & TVIS_EXPANDED)
					lstrcat(sm->textOut, TEXT(" expanded"));
					else
					lstrcat(sm->textOut, TEXT(" collapsed"));
				}
			}
		}
		else
		{
			HANDLE hProcess = OpenProcess(PROCESS_VM_READ | PROCESS_VM_WRITE | PROCESS_VM_OPERATION, FALSE, ProcessId);
			if(hProcess)
			{
				TVITEM *pTvItem = (TVITEM*)VirtualAllocExStub(hProcess, NULL, sizeof(TVITEM), MEM_COMMIT, PAGE_READWRITE);
				LPTSTR pText = (LPTSTR)VirtualAllocExStub(hProcess, NULL, sizeof(TCHAR)*MAX_PATH, MEM_COMMIT, PAGE_READWRITE);
				if(pTvItem && pText)
				{
					TVITEM item;
					ZeroMemory(&item, sizeof(item));
					item.mask = TVIF_CHILDREN | TVIF_HANDLE | TVIF_STATE | TVIF_TEXT;
					item.hItem = tvhandle;
					item.stateMask = TVIS_EXPANDED;
					item.pszText = pText;
					item.cchTextMax = MAX_PATH;
					WriteProcessMemory(hProcess, pTvItem, &item, sizeof(item), NULL);
					if(SendMessage(hWindow, TVM_GETITEM, 0, (LPARAM)pTvItem))
					{
						ReadProcessMemory(hProcess, pTvItem, &item, sizeof(item), NULL);
						ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
						ReadProcessMemory(hProcess, pText, sm->textBuf, sizeof(TCHAR)*MAX_PATH, NULL);
						lstrcat(sm->textOut, sm->textBuf);
						if(item.cChildren)
						{
							if(item.state & TVIS_EXPANDED)
							lstrcat(sm->textOut, TEXT(" expanded"));
							else
							lstrcat(sm->textOut, TEXT(" collapsed"));
						}
					}
					VirtualFreeExStub(hProcess, pText, 0, MEM_RELEASE);
					VirtualFreeExStub(hProcess, pTvItem, 0, MEM_RELEASE);
				}
				CloseHandle(hProcess);
			}
		}
	}
} // handleTreeView

void handleMenus(WPARAM wParam, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	HMENU handle = (HMENU) lParam;
	UINT offset = LOWORD(wParam);
	UINT flags = HIWORD(wParam);
	UINT offsetStyle; // Search by id or index.
	if(flags == 0xFFFF && handle == NULL)
	{ // The menu is closed.
		lstrcat(sm->textOut, TEXT("menu closed"));
		return;
	} // if
	if(flags & MF_POPUP)
	offsetStyle = MF_BYPOSITION; // If it's got a popup, by index then.
	else
	offsetStyle = MF_BYCOMMAND; // By id.
	ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
	GetMenuString(handle, offset, sm->textBuf, MAX_PATH, offsetStyle);
	filterShortcut();
	lstrcat(sm->textOut, sm->textBuf);
	if(flags & MF_GRAYED)
	lstrcat(sm->textOut, TEXT(" grayed"));
	if(flags & MF_DISABLED)
	lstrcat(sm->textOut, TEXT(" disabled"));
	if(flags & MF_CHECKED)
	lstrcat(sm->textOut, TEXT(" checked"));
	if(flags & MF_POPUP)
	lstrcat(sm->textOut, TEXT(" has sub-menu"));
	if(flags & MF_BITMAP)
	lstrcat(sm->textOut, TEXT(" bitmap"));
	if(flags & MF_OWNERDRAW)
	lstrcat(sm->textOut, TEXT(" owner-drawn"));
	if(flags & MF_SYSMENU)
	lstrcat(sm->textOut, TEXT(" in system menu"));
} // handleMenus

void handleNotifications(LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	NMHDR *hdr = (NMHDR *)lParam;
	switch(hdr->code)
	{
	case DTN_DATETIMECHANGE:
	case HDN_ITEMCHANGED:
	case IPN_FIELDCHANGED:
	case LVN_ITEMCHANGED:
	case MCN_SELCHANGE:
	case SBN_SIMPLEMODECHANGE:
	case UDN_DELTAPOS:
	case TCN_SELCHANGE:
	case TTN_SHOW:
	case TVN_ITEMEXPANDED:
	case TVN_SELCHANGED:
		ZeroMemory(sm->textOut, sizeof(sm->textOut));
		handleControls(hdr->hwndFrom, FALSE);
		// Special case for tooltips where we don't want to interrupt speech.
		if(hdr->code == TTN_SHOW)
		notifyObserver(FALSE);
		else
		notifyObserver(TRUE);
		break;
	} // switch
} // handleNotifications

void handleKeys(WPARAM wParam, LPARAM lParam)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	BYTE kbstate[256] = {'\0'};
	GetKeyboardState(kbstate);
	if(!(lParam & (1 << 30)))
	{ // Only if the key wasn't previously down.
		switch(wParam)
		{
		case VK_LBUTTON:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("left mouse button"));
				notifyObserver(TRUE);
			}
			break;
		case VK_RBUTTON:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("right mouse button"));
				notifyObserver(TRUE);
			}
			break;
		case VK_CANCEL:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("break"));
				notifyObserver(TRUE);
			}
			break;
		case VK_MBUTTON:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("middle mouse button"));
				notifyObserver(TRUE);
			}
			break;
		case VK_BACK:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("backspace"));
				notifyObserver(TRUE);
			}
			break;
		case VK_TAB:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("tab"));
				notifyObserver(TRUE);
			}
			break;
		case VK_CLEAR:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("clear"));
				notifyObserver(TRUE);
			}
			break;
		case VK_RETURN:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("enter"));
				notifyObserver(TRUE);
			}
			break;
		case VK_SHIFT:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("shift"));
				notifyObserver(TRUE);
			}
			break;
		case VK_CONTROL:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("control"));
			}
			else
			{ // Hack to interrupt speech.
				lstrcpy(sm->textOut, TEXT(""));
			}
			notifyObserver(TRUE);
			break;
		case VK_MENU:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("alt"));
				notifyObserver(TRUE);
			}
			break;
		case VK_PAUSE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("pause"));
				notifyObserver(TRUE);
			}
			break;
		case VK_CAPITAL:
			lstrcpy(sm->textOut, TEXT("caps lock "));
			// Speak caps lock key state.
			if(kbstate[wParam] & 1)
			lstrcat(sm->textOut, TEXT("on"));
			else
			lstrcat(sm->textOut, TEXT("off"));
			notifyObserver(TRUE);
			break;
		case VK_KANA:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("kana"));
				notifyObserver(TRUE);
			}
			break;
		case VK_JUNJA:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("junja"));
				notifyObserver(TRUE);
			}
			break;
		case VK_FINAL:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("final"));
				notifyObserver(TRUE);
			}
			break;
		case VK_HANJA:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("hanja"));
				notifyObserver(TRUE);
			}
			break;
		case VK_ESCAPE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("escape"));
				notifyObserver(TRUE);
			}
			break;
		case VK_CONVERT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("convert"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NONCONVERT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("non-convert"));
				notifyObserver(TRUE);
			}
			break;
		case VK_ACCEPT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("accept"));
				notifyObserver(TRUE);
			}
			break;
		case VK_MODECHANGE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("mode change"));
				notifyObserver(TRUE);
			}
			break;
		case VK_SPACE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("spacebar"));
				notifyObserver(TRUE);
			}
			break;
		case VK_PRIOR:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("page up"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NEXT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("page down"));
				notifyObserver(TRUE);
			}
			break;
		case VK_END:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("end"));
				notifyObserver(TRUE);
			}
			break;
		case VK_HOME:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("home"));
				notifyObserver(TRUE);
			}
			break;
		case VK_LEFT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("left arrow"));
				notifyObserver(TRUE);
			}
			break;
		case VK_UP:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("up arrow"));
				notifyObserver(TRUE);
			}
			break;
		case VK_RIGHT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("right arrow"));
				notifyObserver(TRUE);
			}
			break;
		case VK_DOWN:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("down arrow"));
				notifyObserver(TRUE);
			}
			break;
		case VK_SELECT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("select"));
				notifyObserver(TRUE);
			}
			break;
		case VK_PRINT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("print"));
				notifyObserver(TRUE);
			}
			break;
		case VK_EXECUTE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("execute"));
				notifyObserver(TRUE);
			}
			break;
		case VK_SNAPSHOT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("print screen"));
				notifyObserver(TRUE);
			}
			break;
		case VK_INSERT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("insert"));
				notifyObserver(TRUE);
			}
			break;
		case VK_DELETE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("delete"));
				notifyObserver(TRUE);
			}
			break;
		case VK_HELP:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("help"));
				notifyObserver(TRUE);
			}
			break;
		case VK_LWIN:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("left windows"));
				notifyObserver(TRUE);
			}
			break;
		case VK_RWIN:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("right windows"));
				notifyObserver(TRUE);
			}
			break;
		case VK_APPS:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("applications"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD0:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 0"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD1:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 1"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD2:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 2"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD3:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 3"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD4:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 4"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD5:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 5"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD6:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 6"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD7:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 7"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD8:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 8"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMPAD9:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad 9"));
				notifyObserver(TRUE);
			}
			break;
		case VK_MULTIPLY:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad multiply"));
				notifyObserver(TRUE);
			}
			break;
		case VK_ADD:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad add"));
				notifyObserver(TRUE);
			}
			break;
		case VK_SEPARATOR:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad seperator"));
				notifyObserver(TRUE);
			}
			break;
		case VK_SUBTRACT:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad subtract"));
				notifyObserver(TRUE);
			}
			break;
		case VK_DECIMAL:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad decimal"));
				notifyObserver(TRUE);
			}
			break;
		case VK_DIVIDE:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("numpad divide"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F1:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f1"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F2:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f2"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F3:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f3"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F4:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f4"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F5:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f5"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F6:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f6"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F7:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f7"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F8:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f8"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F9:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f9"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F10:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f10"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F11:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f11"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F12:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f12"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F13:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f13"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F14:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f14"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F15:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f15"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F16:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f16"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F17:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f17"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F18:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f18"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F19:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f19"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F20:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f20"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F21:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f21"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F22:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f22"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F23:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f23"));
				notifyObserver(TRUE);
			}
			break;
		case VK_F24:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("f24"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NUMLOCK:
			lstrcpy(sm->textOut, TEXT("num lock "));
			// Speak num lock key state.
			if(kbstate[wParam] & 1)
			lstrcat(sm->textOut, TEXT("on"));
			else
			lstrcat(sm->textOut, TEXT("off"));
			notifyObserver(TRUE);
			break;
		case VK_SCROLL:
			lstrcpy(sm->textOut, TEXT("scroll lock "));
			// Speak scroll lock key state.
			if(kbstate[wParam] & 1)
			lstrcat(sm->textOut, TEXT("on"));
			else
			lstrcat(sm->textOut, TEXT("off"));
			notifyObserver(TRUE);
			break;
		case VK_LSHIFT:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("left shift"));
				notifyObserver(TRUE);
			}
			break;
		case VK_RSHIFT:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("right shift"));
				notifyObserver(TRUE);
			}
			break;
		case VK_LCONTROL:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("left control"));
			}
			else
			{ // Hack to interrupt speech.
				lstrcpy(sm->textOut, TEXT(""));
			}
			notifyObserver(TRUE);
			break;
		case VK_RCONTROL:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("right control"));
			}
			else
			{ // Hack to interrupt speech.
				lstrcpy(sm->textOut, TEXT(""));
			}
			notifyObserver(TRUE);
			break;
		case VK_LMENU:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("left alt"));
				notifyObserver(TRUE);
			}
			break;
		case VK_RMENU:
			if(sm->ModifierEcho)
			{
				lstrcpy(sm->textOut, TEXT("right alt"));
				notifyObserver(TRUE);
			}
			break;
		case VK_ATTN:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("attention"));
				notifyObserver(TRUE);
			}
			break;
		case VK_CRSEL:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("crop selection"));
				notifyObserver(TRUE);
			}
			break;
		case VK_EXSEL:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("extend selection"));
				notifyObserver(TRUE);
			}
			break;
		case VK_EREOF:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("erase end of file"));
				notifyObserver(TRUE);
			}
			break;
		case VK_PLAY:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("play"));
				notifyObserver(TRUE);
			}
			break;
		case VK_ZOOM:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("zoom"));
				notifyObserver(TRUE);
			}
			break;
		case VK_NONAME:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("no name"));
				notifyObserver(TRUE);
			}
			break;
		case VK_PA1:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("PA1"));
				notifyObserver(TRUE);
			}
			break;
		case VK_OEM_CLEAR:
			if(sm->CommandEcho)
			{
				lstrcpy(sm->textOut, TEXT("OEM clear"));
				notifyObserver(TRUE);
			}
			break;
		default:
			// Generic processing.
			if(sm->CharEcho && !kbstate[VK_CONTROL])
			{ // Echo characters only when control isn't held down.
				ZeroMemory(sm->textBuf, sizeof(sm->textBuf));
#ifdef UNICODE
				if(ToUnicode(wParam, lParam, kbstate, sm->textBuf, MAX_PATH, 0) > 0)
#else
				if(ToAscii(wParam, lParam, kbstate, (WCHAR *)sm->textBuf, 0) > 0)
#endif
				{ // Only speak if conversion was successfull.
					lstrcpy(sm->textOut, sm->textBuf);
					notifyObserver(TRUE);
				} // if
			} // if
			break;
		} // switch
	} // if
} // handleKeys

EXPORT BOOL CALLBACK hooklibInitialize(HWND hWnd)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	// There musn't be any former hooks by this app. Ensure single-instance only!
	if(sm->hObserver || sm->hKbdHook || sm->hMouseHook || sm->hCbtHook || sm->hCallprocHook)
	return FALSE;
	sm->hObserver = hWnd;
	sm->hKbdHook = SetWindowsHookEx(WH_KEYBOARD, kbdHookFunction, hDllInstance, 0);
	sm->hMouseHook = SetWindowsHookEx(WH_MOUSE, mouseHookFunction, hDllInstance, 0);
	sm->hCbtHook = SetWindowsHookEx(WH_CBT, cbtHookFunction, hDllInstance, 0);
	sm->hCallprocHook = SetWindowsHookEx(WH_CALLWNDPROC, callprocHookFunction, hDllInstance, 0);
	return sm->hObserver && sm->hKbdHook && sm->hMouseHook && sm->hCbtHook && sm->hCallprocHook ? TRUE : FALSE;
} // hooklibInitialize

EXPORT void CALLBACK hooklibCleanup()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	// Unhooking always returns true.
	UnhookWindowsHookEx(sm->hKbdHook);
	UnhookWindowsHookEx(sm->hMouseHook);
	UnhookWindowsHookEx(sm->hCbtHook);
	UnhookWindowsHookEx(sm->hCallprocHook);
	sm->hObserver = NULL;
	sm->hKbdHook = NULL;
	sm->hMouseHook = NULL;
	sm->hCbtHook = NULL;
	sm->hCallprocHook = NULL;
} // hooklibCleanup

EXPORT void CALLBACK hooklibReadActive()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	ZeroMemory(sm->textOut, sizeof(sm->textOut));
	EnumChildWindows(sm->hActive, enumerateDirectDescendants, (LPARAM) sm->hActive);
	notifyObserver(TRUE);
} // hooklibReadActive

EXPORT void CALLBACK hooklibReadFocused(BOOL verbose)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	ZeroMemory(sm->textOut, sizeof(sm->textOut));
	handleControls(sm->hFocused, verbose);
	notifyObserver(TRUE);
} // hooklibReadFocused

EXPORT void CALLBACK hooklibReadTitle()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	ZeroMemory(sm->textOut, sizeof(sm->textOut));
	handleCommon(sm->hActive);
	notifyObserver(TRUE);
} // hooklibReadTitle

EXPORT void CALLBACK hooklibReadStatus()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	HWND hStatus;
	ZeroMemory(sm->textOut, sizeof(sm->textOut));
	hStatus = FindWindowEx(sm->hActive, NULL, TEXT("msctls_statusbar32"), NULL);
	if(hStatus)
	{
		if(SendMessage(hStatus, WM_GETTEXTLENGTH, 0, 0))
		handleStatusBar(hStatus, FALSE);
		else
		lstrcat(sm->textOut, TEXT("status bar empty"));
	}
	else
	{
		lstrcat(sm->textOut, TEXT("no status bar found"));
	}
	notifyObserver(TRUE);
} // hooklibReadStatus

EXPORT void CALLBACK hooklibReadDesktop()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	ZeroMemory(sm->textOut, sizeof(sm->textOut));
	handleCommon(GetDesktopWindow());
	EnumWindows(enumerateDescendants, (LPARAM) NULL);
	notifyObserver(TRUE);
} // hooklibReadDesktop

EXPORT void CALLBACK hooklibRepeat()
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	lstrcpy(sm->textOut, sm->repeatBuf);
	notifyObserver(TRUE);
} // hooklibRepeat

EXPORT void CALLBACK hooklibEchoChars(BOOL enable)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	sm->CharEcho = enable;
} // hooklibEchoChars

EXPORT void CALLBACK hooklibEchoCommands(BOOL enable)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	sm->CommandEcho = enable;
} // hooklibEchoCommands

EXPORT void CALLBACK hooklibEchoModifiers(BOOL enable)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	sm->ModifierEcho = enable;
} // hooklibEchoModifiers

EXPORT void CALLBACK hooklibSpeakShortcuts(BOOL enable)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	sm->SpeakShortcuts = enable;
} // hooklibSpeakShortcuts

EXPORT void CALLBACK hooklibSpeakEditChange(BOOL enable)
{
	SharedMemory *sm = (SharedMemory*)lpvMem;
	sm->SpeakEditChange = enable;
} // hooklibSpeakEditChange
