/*C_HEADER_FILE****************************************************************
FILE			:	BuildFileSystemTables.cpp
DESC			:	
TABS			:	3
OWNER			:	
DATE CREATED	:	

(C) Copyright 2005 All rights reserved.
This is an unpublished work, and is confidential and proprietary: 
technology and information of fonix corporation.  No part of this
code may be reproduced, used or disclosed without written consent of 
fonix corporation in each and every instance.

  $Date:  $
  $Revision:  $

*END_HEADER*******************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "FileSystemData.h"

#ifndef bool
#define bool char
#endif

typedef struct _tagTableData
{
	char	*sTableName;		// Data array name for later identification
	unsigned short
			wBitFlag;
	unsigned int		
			nValues,			// Number of values in array
			offset;				// offset of file system table, used for testing
	int		nElementSize;		// size of each element in the array
	bool	bSigned;			// flag to indicate signed or unsigned
	bool	bGTrieNode;			// flag to indicate that array type is a GTrieNode
	void	*pValues;			// pointer to array of data
	struct _tagTableData *pNext;// pointer to next table

}TABLE_DATA;

/*FUNCTION_HEADER**********************
 * NAME:    ;ByteSwap32
 * DESC:    Swap the bytes in a 4-byte integer
 * IN:      i - 32-bit (4-byte) integer to swap bytes in
 * OUT:     n/a
 * RETURN:  i with bytes swapped
 * NOTES:   Unlike big/little-to-little/big-Endian routines,
              this one swaps the bytes without checking to make
              sure it needs to be done.  (This is about 15% faster)
 *END_HEADER***************************/
static unsigned int ByteSwap32(unsigned int i)
{
   return (0x000000FF & i >> 24) | (0x00FF0000 & i << 8) | (0x0000FF00 & i >> 8) | (0xFF000000 & i << 24);
}

/*FUNCTION_HEADER**********************
 * NAME:    ;ByteSwap16
 * DESC:    Swap the bytes in a 2-byte integer
 * IN:      i - 16-bit (2-byte) integer to swap bytes in
 * OUT:     n/a
 * RETURN:  i with bytes swapped
 * NOTES:   Unlike big/little-to-little/big-Endian routines,
              this one swaps the bytes without checking to make
              sure it needs to be done.  (This is about 15% faster)
 *END_HEADER***************************/
static unsigned short ByteSwap16(unsigned short i)
{
   return ((unsigned short) 0x00FFU & i >> 8) | ((unsigned short) 0xFF00U & i << 8);
}


bool ParseString(char *sBuf, char *sString, int nMax)
{
	char *pc1, *pc2;

	if( (pc1 = strchr(sBuf, '"')) != NULL )
	{
		pc1++;
		if( (pc2 = strchr(pc1, '"')) != NULL )
		{
			*pc2 = 0x00;
			if( pc2-pc1 < nMax )
			{
				strcpy(sString, pc1);
			}
			else
			{
				strncpy(sString, pc1, nMax-2);
				sString[nMax-1] = 0x00;
			}

			// Check to see if there are any other strings in sBuf
			pc2++;
			if( (pc1 = strchr(pc2, '"')) != NULL )
				sBuf = pc1;

			return 1;
		}
	}

	return 0;
}

/*FUNCTION_HEADER**********************
 * NAME:	;ReadGTrieNode
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	Sample data lines

	{0x0000,    0,    1,   53,   14,   10,    0,    0,    0,  128 }, 
	{"s/s/",    0,    0,   54,   15,   21,    0,    0,    0,    0 }, 

 *END_HEADER***************************/
bool ReadGTrieNode(char *sBuf, GTRIENODE *pGTrieNode)
{
	char *pc, *pc2;
	int	 ret;

	if( (pc = strstr(sBuf, "{0x0000,")) != NULL )
	{
		pGTrieNode->sName[0] = 0x00;
	}
	else if( (pc = strstr(sBuf, "{\"")) != NULL )
	{
		pc += 2;
		pc2 = pc;
		while( *pc2 != '\"' )
			pc2++;
		*pc2 = 0x00;
		strcpy(pGTrieNode->sName, strdup(pc));
		*pc2 = '\"';
	}

	while( *pc != ' ' )
		pc++;
	while( *pc == ' ' )
		pc++;

	ret = sscanf(pc, "%d, %d, %d, %d, %d, %d, %d, %d, %d",
		&pGTrieNode->alt,
		&pGTrieNode->what,
		&pGTrieNode->right,
		&pGTrieNode->index,
		&pGTrieNode->typeinfo,
		&pGTrieNode->info.status,
		&pGTrieNode->info.counter,
		&pGTrieNode->info.counterB,
		&pGTrieNode->info.parameter);

	if( ret == 9 )
		return 1;
	else
	{
		printf("Error: Didn't read all values in GTrieNode!\n");
		return 0;
	}

}

/*FUNCTION_HEADER**********************
 * NAME:	;WriteGTrieNode
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void WriteGTrieNode(FILE *fp, GTRIENODE *pGTrieNode, unsigned int *piOffset, bool bBigEndian)
{
	char c=0x00;
	unsigned short wValue;
	unsigned int curpos,
				 offset=0;

	if( pGTrieNode->sName[0] == 0x00 )
	{
		offset = 0;
		fwrite(&offset, sizeof(int), 1, fp);
	}
	else
	{
		offset = *piOffset;
		if( bBigEndian )	
			offset = ByteSwap32(offset);
		fwrite(&offset, sizeof(int), 1, fp);			// write offset to data
		curpos = ftell(fp);								// save current file location
		fseek(fp, *piOffset, SEEK_SET);					// seek to data location
		fwrite(pGTrieNode->sName, sizeof(char), strlen(pGTrieNode->sName), fp);	// write data
		fwrite(&c, sizeof(char), 1, fp);				// write null terminating character
		*piOffset += strlen(pGTrieNode->sName) + 1;		// update offset for next data string
		fseek(fp, curpos, SEEK_SET);					// seek back to table data
	}

	wValue = pGTrieNode->alt;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->what;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->right;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->index;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->typeinfo;
	fwrite(&wValue, sizeof(char), 1, fp);
	
	wValue = pGTrieNode->info.status;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->info.counter;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->info.counterB;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);
	
	wValue = pGTrieNode->info.parameter;
	if( bBigEndian )	wValue = ByteSwap16(wValue);
	fwrite(&wValue, sizeof(U16), 1, fp);

}

/*FUNCTION_HEADER**********************
 * NAME:	;StripLeadingStuff
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
char *StripLeadingStuff(char *sBuf)
{
	char *pc;

	for( pc=sBuf; pc && *pc && *pc == ' '; pc++ )	// Skip leading spaces
		;

	if( strstr(pc, "/*") == pc )					// Look for a leading comment 
	{
		if( (pc = strstr(pc, "*/")) )				// Find closing comment characters
		{
			pc += 2;
			while( pc && *pc && *pc == ' ' )		// Skip leading spaces after comment
				pc++;
		}
	}

	return pc;

}

/*FUNCTION_HEADER**********************
 * NAME:	;ReadValuesFromBuf
 * DESC: 	Read a hand full of values from a line of text
 * IN:		sBuf - the line of text
			pValue - pointer to an array to hold the data
			iSize - the number of bytes per element in the array
			nMax - the maximum number of elements that the array can hold
			bSigned - flag for signed/unsigned data
 * OUT:		data in pValue
 * RETURN:	number of data values read
 * NOTES:	
 *END_HEADER***************************/
int ReadValuesFromBuf(char *sBuf, void *pValue, int iSize, int nMax, bool bSigned)
{
	char *pc;
	int	 iBase=10;
	int	 i;

	if( (pc = StripLeadingStuff(sBuf)) == NULL )
		return 0;

	if( strstr(sBuf, "0x") )
		iBase = 16;

	for( i=0; pc && *pc != 0x00 && *pc != '\n' && *pc != '\r' && i<nMax; i++ )
	{
		if( bSigned )
		{
			switch(iSize)
			{
				case 1:	((char *)pValue)[i]		= (char)strtol(pc, &pc, iBase);		break;
				case 2: ((short *)pValue)[i]	= (short)strtol(pc, &pc, iBase);	break;
				case 4: ((int *)pValue)[i]		= (int)strtol(pc, &pc, iBase);		break;
			}
		}
		else
		{
			switch(iSize)
			{
				case 1:	((unsigned char *)pValue)[i]	= (unsigned char)strtol(pc, &pc, iBase);	break;
				case 2: ((unsigned short *)pValue)[i]	= (unsigned short)strtol(pc, &pc, iBase);	break;
				case 4: ((unsigned int *)pValue)[i]		= (unsigned int)strtol(pc, &pc, iBase);		break;
			}
		}

		if( *pc == 0x00 || *pc == '\n' )
		{
			i++;									// Increment to a count instead of an index
			break;
		}

		pc++;										// Move past the stop character

		while( *pc == ' ' )							// Skip spaces
			pc++;
	}

	return i;

}

#define MAX_VALUES 128
/*FUNCTION_HEADER**********************
 * NAME:	;ReadTablesInFile
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
TABLE_DATA *ReadTablesInFile(char *sFile)
{
	bool	bReading=0,
			bDebug=0;
	char	sBuf[MAX_VALUES];
	char	*pc1, *pc2;
	unsigned short 
			wBitFlag;
	int		i, j, 
			nValues,
			nElementSize;
	char	pValues[MAX_VALUES];
	
	FILE	*fp;
	TABLE_DATA 
			*pTableHead=NULL, *pTbl=NULL;
	
	if( sFile == NULL ||
		(fp = fopen(sFile, "rt")) == NULL )
	{
		printf("Error: Can't open file %s\n", sFile);
		return NULL;
	}

	while( fgets(sBuf, 128, fp) )
	{
		if( sBuf[0] == 0x00 || 
			sBuf[0] == '\n' ||
			sBuf[0] == '\r' ||
			sBuf[0] == ';' ||
			(sBuf[0] == '/' && sBuf[1] == '/') )
			continue;

		if( !bDebug && !bReading &&
			((pc2 = strstr(sBuf, "[] = {")) != NULL ||
			 (pc2 = strstr(sBuf, "] = {")) != NULL ||
			 (pc2 = strstr(sBuf, "[]={")) != NULL ||
			 (pc2 = strstr(sBuf, "[]= {")) ) &&
			 strchr(sBuf, '}') == NULL )
		{												// Found the start of a table
			nElementSize = 0;			// Initialize to 0 so we can tell later if a known type was found
			wBitFlag = 0;

			// Check for a known type
			if( strstr(sBuf, "U8") ||
				strstr(sBuf, "GERLETTER") )
			{
				wBitFlag |= FSD_BYTES1;
				nElementSize = sizeof(char);
			}
			else if( strstr(sBuf, "U16") )
			{
				wBitFlag |= FSD_BYTES2;
				nElementSize = sizeof(short);
			}
			else if( strstr(sBuf, "U32") )
			{
				wBitFlag |= FSD_BYTES4;
				nElementSize = sizeof(int);
			}
			else if( strstr(sBuf, "const char *") )
			{
				wBitFlag |= FSD_SIGNED;
				wBitFlag |= FSD_STRING;
				nElementSize = FSD_MAXSTRING;
			}
			else if( strstr(sBuf, "CONST listtable") )
			{
				wBitFlag |= FSD_SIGNED;
				wBitFlag |= FSD_ITLIST;
				nElementSize = 3 * sizeof(char);
			}
			else if( strstr(sBuf, "CONST ruletable") )
			{
				wBitFlag |= FSD_SIGNED;
				wBitFlag |= FSD_ITRULE;
				nElementSize = 4 * sizeof(char);
			}
			else if( strstr(sBuf, "char") )
			{
				if( strstr(sBuf, "unsigned") == NULL )
				{
					wBitFlag |= FSD_SIGNED;
				}

				wBitFlag |= FSD_BYTES1;
				nElementSize = sizeof(char);
			}
			else if( strstr(sBuf, "short") )
			{
				if( strstr(sBuf, "unsigned") == NULL )
					wBitFlag |= FSD_SIGNED;
				wBitFlag |= FSD_BYTES2;
				nElementSize = sizeof(short);
			}
			else if( strstr(sBuf, "int") )
			{
				if( strstr(sBuf, "unsigned") == NULL )
					wBitFlag |= FSD_SIGNED;
				wBitFlag |= FSD_BYTES4;
				nElementSize = sizeof(int);
			}
			else if( strstr(sBuf, "GTrieNode") )
			{
				wBitFlag |= FSD_GTRIE;
				nElementSize = sizeof(GTRIENODE);
			}

			if( nElementSize > 0 )
			{
				bReading=1;									// Set state to bReading == true

				if( pTableHead == NULL )
				{
					pTableHead = pTbl = (TABLE_DATA *)calloc(1, sizeof(TABLE_DATA));
				}
				else
				{
					pTbl->pNext = (TABLE_DATA *)calloc(1, sizeof(TABLE_DATA));
					pTbl = pTbl->pNext;
				}

				if( wBitFlag & FSD_SIGNED )
					pTbl->bSigned = 1;
				pTbl->nElementSize = nElementSize;
				pTbl->wBitFlag = wBitFlag;

				// Get the table name
				while( *pc2 != '[' )						// Backup to the beginning array character '['
					pc2--;
				while( *(pc2-1) == ' ' && pc2 > sBuf )		// Skip any trailing spaces as we start backing up.
					pc2--;
				pc1 = pc2-1;								// Find the beginning of the table name
				while( *pc1 != ' ' && pc1 >= sBuf)			// Backup until we find the beginning of the name
					pc1--;
				if( pc1 > sBuf )
				{
					char c = *pc2;

					pc1++;									// advance from the space before the name to the beginning of the name
					*pc2 = 0x00;							// null terminate the name
					pTbl->sTableName = strdup(pc1);			// copy the name
					*pc2 = c;

					for( pc1=pTbl->sTableName; *pc1; pc1++ )
						*pc1 = toupper(*pc1);

					printf("\tFound Table: %s\n", pTbl->sTableName);
				}
			}
			else
			{
				bReading = 0;
			}

		}
		else if( strstr(sBuf, "#ifdef DEB") ||				// Don't include these sections
				 strstr(sBuf, "#if 0") )
		{
			bDebug = 1;
			bReading = 0;
		}
		else if( strstr(sBuf, "};") ||
				 strstr(sBuf, "} ;") )
		{
			bReading = 0;
		}
		else if( strstr(sBuf, "#endif") )
		{
			bDebug = 0;
			bReading = 0;
		}
		else if( bReading )
		{

			if( pTbl->wBitFlag & FSD_GTRIE )
			{
				GTRIENODE GTrieNode;
				if( ReadGTrieNode(sBuf, &GTrieNode) )
				{
					pTbl->pValues = (void *)realloc(pTbl->pValues, (pTbl->nValues+1)*pTbl->nElementSize);
					((GTRIENODE *)pTbl->pValues)[pTbl->nValues] = GTrieNode;
					pTbl->nValues++;
				}

			}
			else if( pTbl->wBitFlag & FSD_STRING )
			{
				char sString[FSD_MAXSTRING],
					 *pc=sBuf;

				while( ParseString(pc, sString, FSD_MAXSTRING) )
				{
					pTbl->pValues = (char *)realloc(pTbl->pValues, (pTbl->nValues+1)*pTbl->nElementSize);

					strcpy( (((char *)pTbl->pValues) + (pTbl->nValues*pTbl->nElementSize)), sString);
					pTbl->nValues++;
				}
			}
			else if( pTbl->wBitFlag & FSD_ITLIST )
			{

			}
			else if( pTbl->wBitFlag & FSD_ITRULE )
			{

			}
			else
			{
				nValues = ReadValuesFromBuf(sBuf, pValues, pTbl->nElementSize, 
						MAX_VALUES/pTbl->nElementSize, pTbl->bSigned);

				if( nValues > 0 )
				{
					pTbl->pValues = (void *)realloc(pTbl->pValues, (pTbl->nValues+nValues)*pTbl->nElementSize);
					switch(pTbl->nElementSize)
					{
						case 1:
							for(i=pTbl->nValues, j=0; j < nValues; i++, j++)
							{
								if( pTbl->bSigned )
									((char *)pTbl->pValues)[i] = ((char *)pValues)[j];
								else
									((unsigned char *)pTbl->pValues)[i] = ((unsigned char *)pValues)[j];
							}
							break;
						case 2:
							for(i=pTbl->nValues, j=0; j < nValues; i++, j++)
							{
								if( pTbl->bSigned )
									((short *)pTbl->pValues)[i] = ((short *)pValues)[j];
								else
									((unsigned short *)pTbl->pValues)[i] = ((unsigned short *)pValues)[j];
							}
							break;
						case 4:
							for(i=pTbl->nValues, j=0; j < nValues; i++, j++)
							{
								if( pTbl->bSigned )
									((int *)pTbl->pValues)[i] = ((int *)pValues)[j];
								else
									((unsigned int *)pTbl->pValues)[i] = ((unsigned int *)pValues)[j];

							}
							break;

					}
					pTbl->nValues += nValues;
				}

			}	// if !bGTrieNode

		}	// if bReading
	}

	fclose(fp);

	return pTableHead;
}

/*FUNCTION_HEADER**********************
 * NAME:	;WriteCompositeTable
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void WriteCompositeTable(char *sOutFile, char *sListName, TABLE_DATA *pTable, bool bBigEndian)
{
	char		sHeader[_MAX_PATH],
				sSrc[_MAX_PATH],
				*pc;
	char		Byte1;
	short		Byte2;
	int			Byte4;
	unsigned char
				uByte1;
	unsigned short
				uByte2;
	unsigned int
				i,
				uByte4,
				offset,
				uiDataOffset;
	void		*pValue;
	FILE		*fp,
				*fpHeader,
				*fpSrc;
	TABLE_DATA	*pTbl;

	if( sOutFile == NULL || pTable == NULL )
		return;

	if( (fp = fopen(sOutFile, "wb")) == NULL )
	{
		printf("Error: Can't open file %s\n", sOutFile);
		return;
	}

	// Create the filename for the header file where the offsets will be stored
	if( (pc = strrchr(sOutFile, '\\')) != NULL )
	{
		char c = *pc;
		*pc = 0x00;
		sprintf(sHeader, "%s\\%sFileSystemTable.h", sOutFile, sListName);
		sprintf(sSrc, "%s\\%sFileSystemTable.c", sOutFile, sListName);
		*pc = c;
	}
	else
	{
		sprintf(sHeader, "%sFileSystemTable.h", sListName);
		sprintf(sSrc, "%sFileSystemTable.c", sListName);
	}
	if( (fpHeader = fopen(sHeader, "wt")) == NULL )
	{
		fclose(fp);
		printf("Error: Couldn't open header file %s\n", sHeader);
		return;
	}

	// Write out the file header
	fprintf(fpHeader, 
		"/*H_HEADER_FILE***************************************************************\n"
		"FILE			: %sFileSystemTable.h\n"
		"DESC			: \n"
		"TABS			: 4\n"
		"OWNER			: Fonix\n"
		"DATE CREATED	: %s\n\n"
		"(C) Copyright 2005 All rights reserved.\n"
		"This is an unpublished work, and is confidential and proprietary: \n"
		"technology and information of fonix corporation.  No part of this\n"
		"code may be reproduced, used or disclosed without written consent of\n" 
		"fonix corporation in each and every instance.\n"
		"*END_HEADER******************************************************************/\n"
		"#ifndef	%sFILESYSTEMDATA_H\n"
		"#define	%sFILESYSTEMDATA_H\n\n", 
		sListName, 
		__DATE__,
		sListName,
		sListName);

	// Write the Table enums used to access the array of offsets
	fprintf(fpHeader, "\t// The following enum provides a method for accessing the data array.\n");
	fprintf(fpHeader, "\ttypedef enum {\n");
	for(pTbl=pTable; pTbl; pTbl=pTbl->pNext)
	{
		if( pTbl == pTable )
			fprintf(fpHeader, "\t\t%s = 0,\n", pTbl->sTableName);	// Write out table name enums (1st)
		else
			fprintf(fpHeader, "\t\t%s,\n", pTbl->sTableName);		// (all others)
	}
	fprintf(fpHeader, "\t\t%s_NUM_TABLES\n\t}%s_FS_TABLES;\n", 
		sListName, sListName);
	fprintf(fpHeader, "\n#endif\n");
	fclose(fpHeader);


	// Write out the offsets array
	if( (fpSrc = fopen(sSrc, "wt")) == NULL )
	{
		fclose(fp);
		printf("Error: Couldn't open source file %s\n", sSrc);
		return;
	}

	fprintf(fpSrc, 
		"/*C_HEADER_FILE****************************************************************\n"
		"FILE			:	%sFileSystemTable.c\n"
		"DESC			:	This file includes an array of offsets to the German File System dictionary.\n"
		"					See %sFileSystemDataTable.h for the table name enumeration.\n"
		"TABS			:	4\n"
		"OWNER			:	Fonix\n"
		"DATE CREATED	:	%s\n\n"
		"(C) Copyright 2005 All rights reserved.\n"
		"This is an unpublished work, and is confidential and proprietary: \n"
		"technology and information of fonix corporation.  No part of this\n"
		"code may be reproduced, used or disclosed without written consent of\n" 
		"fonix corporation in each and every instance.\n\n"
		"*END_HEADER*******************************************************************/\n"
		"#include \"%sFileSystemTable.h\"\n\n",
		sListName, 
		sListName, 
		__DATE__,
		sListName);

	fprintf(fpSrc, "\n\n//  This is an array of offsets into the File System tables.\n");
	fprintf(fpSrc, "const unsigned int ui%sTableOffsets[] = {\n", sListName);

	offset = 0;
	for(pTbl=pTable; pTbl; pTbl=pTbl->pNext)
	{
		pTbl->offset = offset;										// Store for later testing
		if( pTbl->pNext == NULL )
			fprintf(fpSrc, "\t%d\n};\n", offset);					// Write offsets to header file
		else
			fprintf(fpSrc, "\t%d,\n", offset);


		// Write the number of values in this table
		uByte4 = pTbl->nValues;
		if( bBigEndian )
			uByte4 = ByteSwap32(uByte4);
		fwrite(&uByte4, sizeof(unsigned int), 1, fp);				// Write number of values in table

		// Write bit flag with nBytes and Signed bits set
		uByte2 = pTbl->wBitFlag;
		if( bBigEndian )
			uByte2 = ByteSwap16(uByte2);
		fwrite(&uByte2, sizeof(short), 1, fp);

		// Write the values
		for(i=0; i<pTbl->nValues; i++)
		{
			switch(pTbl->nElementSize)
			{
				case 1:	
					if( pTbl->bSigned )
					{
						Byte1 = ((char *)pTbl->pValues)[i];
						pValue = &Byte1;
					}
					else
					{
						uByte1 = ((unsigned char *)pTbl->pValues)[i];
						pValue = &uByte1;
					}
					fwrite(pValue, pTbl->nElementSize, 1, fp);
					break;

				case 2:
					if( pTbl->bSigned )
					{
						Byte2 = ((short *)pTbl->pValues)[i];
						if( bBigEndian )
							Byte2 = ByteSwap16(Byte2);
						pValue = &Byte2;
					}
					else
					{
						uByte2 = ((unsigned short *)pTbl->pValues)[i];
						if( bBigEndian ) 
							uByte2 = ByteSwap16(uByte2);
						pValue = &uByte2;
					}
					fwrite(pValue, pTbl->nElementSize, 1, fp);
					break;
				
				case 4:
					if( pTbl->bSigned )
					{
						Byte4 = ((int *)pTbl->pValues)[i];
						if( bBigEndian ) 
							Byte4 = ByteSwap32(Byte4);
						pValue = &Byte4;
					}
					else
					{
						uByte4 = ((unsigned int *)pTbl->pValues)[i];
						if( bBigEndian ) 
							uByte4 = ByteSwap32(uByte4);
						pValue = &uByte4;
					}
					fwrite(pValue, pTbl->nElementSize, 1, fp);
					break;

				default:
					if( pTbl->wBitFlag & FSD_GTRIE )
					{
						if( i == 0 )
						{
							// structure size on disk = sizeof(uiDataOffset) + sizeof(U16)*8 values + sizeof(typeinfo)
							unsigned int uiSzOnDisk = sizeof(int) + sizeof(U16)*8 + sizeof(char);
							// Beginning of string data
							uiDataOffset = offset + sizeof(int) + sizeof(char) + pTbl->nValues*uiSzOnDisk;
						}
						WriteGTrieNode(fp, &((GTRIENODE *)pTbl->pValues)[i], &uiDataOffset, bBigEndian);
					}
					else if( pTbl->wBitFlag & FSD_STRING )
					{

					}
					break;

			}	// switch


		}	// for all values in table

		// Get the offset to the next table
		offset = ftell(fp);

	}	// for all tables

	fclose(fp);
	fclose(fpSrc);

}	// WriteCompositeTable


/*FUNCTION_HEADER**********************
 * NAME:	;
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void TestGTrieTable(char *sFile)
{
	int		Value;
	char	sBuf[128];
	unsigned int
			uiIndex;
	FILE	*fp;
	GTRIENODE GTrieNode;

	if( (fp = fopen(sFile, "rb")) == NULL )
		return;

	printf("\nEnter an index value: ");
	while( gets(sBuf) )		// Test the newly built LTree
	{
		if( !sBuf[0] )		// get out if nothing was entered
			break;
		uiIndex = atoi(sBuf);
		Value = 0;
		if( GetGTrieNode(fp, &GTrieNode, uiIndex, 0) == 1 )
		{
			printf("Found GTrieNode: Name %s, alt %d, what %d, right %d, index %d, typeinfo %d\n",
				GTrieNode.sName[0]==0x00 ? "none" : GTrieNode.sName,
				GTrieNode.alt,
				GTrieNode.what,
				GTrieNode.right,
				GTrieNode.index,
				(int)GTrieNode.typeinfo);
		}
		else
		{
			printf("Error: reading value\n");
		}

		printf("Enter index: ");
	}


}


/*FUNCTION_HEADER**********************
 * NAME:	;
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
bool FindTableOffset(char *sTable, TABLE_DATA *pTable, unsigned int *puiTableOffset)
{
	TABLE_DATA *pTbl;

	for(pTbl=pTable; pTbl; pTbl=pTbl->pNext)
	{
		if( stricmp(sTable, pTbl->sTableName) == 0 )
		{
			*puiTableOffset = pTbl->offset;
			return 1;
		}
	}

	return 0;
}

/*FUNCTION_HEADER**********************
 * NAME:	;TestCompositeTable
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void TestCompositeTable(char *sFile, TABLE_DATA *pTable)
{
	char	sTable[_MAX_PATH];
	int		nSize,
			Value,
			bSigned;
	char	sBuf[128];
	unsigned int
			uiTableOffset,
			uiIndex;
	FILE	*fp;

	if( (fp = fopen(sFile, "rb")) == NULL )
		return;

	printf("\nEnter the name of a table to test: ");
	while( gets(sTable) )
	{
		if( !sTable[0] )
			break;

		if( FindTableOffset(sTable, pTable, &uiTableOffset) == 0 )
		{
			printf("Warning: Unknown table %s\n", sTable);
			continue;
		}

		nSize = GetBytesPerValue(fp, &bSigned, uiTableOffset);

		printf("\nEnter an index value: ");
		while( gets(sBuf) )		// Test the newly built LTree
		{
			if( !sBuf[0] )		// get out if nothing was entered
				break;
			uiIndex = atoi(sBuf);
			Value = 0;
			if( GetTableValue(fp, &Value, uiIndex, uiTableOffset) == 1 )
			{
				printf("Value: %d (Hex: %x)\n", 
					bSigned?Value:(unsigned int)Value, 
					bSigned?Value:(unsigned int)Value);
			}
			else
			{
				printf("Error: reading value\n");
			}

			printf("Enter index: ");
		}

		printf("\nEnter the name of a table to test: ");

	}

}


/*FUNCTION_HEADER**********************
 * NAME:	;BuildCompositeTable
 * DESC: 	
 * IN:		sFileList - file containing a list of filenames
			sOutFile - filename for the output file
			bTest - flag to test dictionaries along the way
			bBigEndian - flag to write out all multi-byte items in big-endian format
 * OUT:		
 * RETURN:	1 on success, 0 on failure
 * NOTES:	
 *END_HEADER***************************/
int BuildCompositeTable(char *sFileList, char *sOutFile, bool bTest, bool bBigEndian)
{
	char sInFile[_MAX_PATH],
		 sListName[_MAX_PATH],
		 *pc;
	int	 nTables=0;
	FILE *fp;
	TABLE_DATA *pTable;
	
	if( (fp = fopen(sFileList, "rt")) == NULL )
	{
		printf("Error: Can't open file %s\n", sFileList);
		return 0;
	}

	// Create a list name that will be used in creating the output files
	if( (pc = strrchr(sFileList, '\\')) != NULL )
	{
		pc++;
		strcpy(sListName, pc);
	}
	else
	{
		strcpy(sListName, sFileList);
	}
	if( (pc = strrchr(sListName, '.')) != NULL )
		*pc = 0x00;

	while( fgets(sInFile, _MAX_PATH, fp) )					// Iterate through all files in the list
	{
		if( *sInFile == 0x00 || *sInFile == ';' )			// Make sure we have a filename
			continue;

		if( (pc = strchr(sInFile, '\n')) != NULL )			// Strip any \n ending characters
			*pc = 0x00;

		printf("Loading %s\n", sInFile);
		if( (pTable = ReadTablesInFile(sInFile)) != NULL )
		{
			WriteCompositeTable(sOutFile, sListName, pTable, bBigEndian);

			if( bTest )
			{
				if( pTable->bGTrieNode )
					TestGTrieTable(sOutFile);
				else
					TestCompositeTable(sOutFile, pTable);
			}
		}

	}	// while more files in the list


	fclose(fp);

	return 1;
}


/*FUNCTION_HEADER**********************
 * NAME:	;main
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int main(int argc, char* argv[])
{
	char *sInFile=NULL,
		 *sOutFile=NULL;
	bool bTest=0,
		 bItalianRules=0;
	bool bBigEndian=0;
	int	 i;

	// Parse the command line arguments to get the input and output filenames
	for( i=1; i<argc; i++)
	{
		if( argv[i][0] == '-' )
		{
			switch( argv[i][1] )
			{
				case 'B':
				case 'b':
					bBigEndian = 1;
					break;

				case 'i':	
					sInFile = argv[++i];		// Get the input (raw text dictionary) filename
					break;

				case 'I':
					bItalianRules = 1;
					break;

				case 'O':
				case 'o':	
					sOutFile = argv[++i];		// Get the output (file system letter tree) filename
					break;

				case 'T':
				case 't':
					bTest = 1;
					break;

			}
		}
	}

	if( sInFile == NULL || sOutFile == NULL )
	{
		printf("Usage: %s -i <infile> -o <outfile> [-T] [-B]\n", argv[0]);
		printf("\t<infile> - input file\n");
		printf("\t<output> - output file system file\n");
		printf("\t-T - test\n");
		printf("\t-B - output in big-endian format\n");
		exit(0);
	}

	// Read the raw text dictionary file
	printf("Reading: %s ...\n", sInFile);
	BuildCompositeTable(sInFile, sOutFile, bTest, bBigEndian);

	return 1;
}

