/*C_HEADER_FILE****************************************************************
FILE			:	Kana2Rom.c
DESC			:	convert Hiragana or Katakana to Romaji
TABS			:	4
OWNER			:	
DATE CREATED:	

(C) Copyright 2005 All rights reserved.
This is an unpublished work, and is confidential and proprietary: 
technology and information of fonix corporation.  No part of this
code may be reproduced, used or disclosed without written consent of 
fonix corporation in each and every instance.

  $Date:  $
  $Revision:  $

*END_HEADER*******************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include "Kana2Rom.h"

#define NRKANA 250*2   /*  size of the romaji to kana tables*/
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
//unsigned char KANATAB[NRKANA*2][7];

// Hiragana and Katakana to Romaji conversion table (EUC encoding)
unsigned char *KANATAB[] = {
	//$HIRAGANA TABLE   (don't delete this line)
	"_", "",
	".", "",
	"n'i", "",
	"n'a", "",
	"n'ya", "",
	"n'o", "",
	"n'yu", "",
	"n'e", "",
	"n'yo", "",
	"n'u", "",
	"a_", "",
	"ii", "",
	"u_", "",
	"e_", "",
	"o_", "",
	"a", "",
	"i", "",
	"u", "",
	"e", "",
	"ye", "",
	"o", "",
	"ka", "",
	"ga", "",
	"kya", "",
	"kyu_", "夦",
	"kyu", "",
	"kye", "",
	"kyo_", "礦",
	"kyo", "",
	"gya", "",
	"gyu_", "夦",
	"gyu", "",
	"gye", "",
	"gyo_", "礦",
	"gyo", "",
	"gi", "",
	"ki", "",
	"ku_", "",
	"ku", "",
	"gu_", "",
	"gu", "",
	"ke_", "",
	"ke", "",
	"ge_", "",
	"ge", "",
	"ko_", "",
	"ko", "",
	"go_", "",
	"go", "",
	"sa", "",
	"za", "",
	"sha", "",
	"shu_", "夦",
	"shu", "",
	"she", "",
	"sho_", "礦",
	"sho", "",
	"ja", "",
	"ju_", "夦",
	"ju", "",
	"je", "",
	"jo_", "礦",
	"jo", "",
	"ji", "",
	"shi", "",
	"su_", "",
	"su", "",
	"zu_", "",
	"zu", "",
	"zi", "",
	"se_", "",
	"se", "",
	"ze", "",
	"so_", "",
	"so", "",
	"zo_", "",
	"zo", "",
	"ta", "",
	"da", "",
	"cha", "",
	"chu_", "夦",
	"chu", "",
	"che", "",
	"cho_", "礦",
	"cho", "",
	"dja", "¤",
	"dju_", "¤夦",
	"dju", "¤",
	"dje", "¤",
	"djo_", "¤礦",
	"dji", "",
	"chi", "",
	"tchi", "ä",
	"tcha", "ä",
	"tchu", "ä",
	"tche", "ä",
	"tcho", "ä",
	"*", "",				//     Looks funny, but it is swapped to a double consonant
	"tsu_", "Ĥ",
	"tsu", "",
	"tsa", "Ĥ",
	"tse", "Ĥ",
	"tso", "Ĥ",
	"dzu_", "Ť",
	"dzu", "",
	"te_", "Ƥ",
	"ti", "Ƥ",
	"tya", "Ƥ",
	"tyu", "Ƥ",
	"tyo", "Ƥ",
	"te", "",
	"di", "Ǥ",
	"dya", "Ǥ",
	"dyu", "Ǥ",
	"dye", "Ǥ",
	"dyo", "Ǥ",
	"de", "",
	"to_", "Ȥ",
	"to", "",
	"do_", "ɤ",
	"do", "",
	"du", "ɤ",
	"na", "",
	"nya", "ˤ",
	"nyu_", "ˤ夦",
	"nyu", "ˤ",
	"nye", "ˤ",
	"nyo_", "ˤ礦",
	"nyo", "ˤ",
	"ni", "",
	"nu_", "̤",
	"nu", "",
	"ne", "",
	"no_", "Τ",
	"no", "",
	"ha", "",
	"ba", "",
	"pa", "",
	"hya", "Ҥ",
	"hyu_", "Ҥ夦",
	"hyu", "Ҥ",
	"hye", "Ҥ",
	"hyo_", "Ҥ礦",
	"hyo", "Ҥ",
	"bya", "Ӥ",
	"byu_", "Ӥ夦",
	"byu", "Ӥ",
	"bye", "Ӥ",
	"byo_", "Ӥ礦",
	"byo", "Ӥ",
	"pya", "Ԥ",
	"pyu_", "Ԥ夦",
	"pyu", "Ԥ",
	"pye", "Ԥ",
	"pyo_", "Ԥ礦",
	"pyo", "Ԥ",
	"bi", "",
	"hi", "",
	"pi", "",
	"fu_", "դ",
	"fu", "",
	"fa", "դ",
	"fi", "դ",
	"fe", "դ",
	"fo", "դ",
	"fya", "դ",
	"fyu", "դ",
	"fye", "դ",
	"fyo", "դ",
	"bu_", "֤",
	"bu", "",
	"pu_", "פ",
	"pu", "",
	"he", "",
	"be", "",
	"pe", "",
	"ho_", "ۤ",
	"ho", "",
	"bo_", "ܤ",
	"bo", "",
	"po", "",
	"ma", "",
	"mya", "ߤ",
	"myu_", "ߤ夦",
	"myu", "ߤ",
	"mye", "ߤ",
	"myo_", "ߤ礦",
	"myo", "ߤ",
	"mi", "",
	"mu_", "द",
	"mu", "",
	"me_", "ᤨ",
	"me", "",
	"mo_", "⤦",
	"mo", "",
	"xya", "",
	"ya", "",
	"xyu", "",
	"yu_", "椦",
	"yu", "",
	"xyo", "",
	"yo_", "褦",
	"yo", "",
	"ra", "",
	"rya", "",
	"ryu_", "夦",
	"ryu", "",
	"rye", "ꤧ",
	"ryo_", "礦",
	"ryo", "",
	"ri", "",
	"ru_", "뤦",
	"ru", "",
	"re_", "줨",
	"re", "",
	"ro_", "",
	"ro", "",
	"xwa", "",
	"wa", "",
	"wi", "",
	"we", "",
	"wo", "",
	"n", "",
//$KATAKANA  (DON'T delete this line!)
	"n'i", "",
	"n'a", "",
	"n'ya", "",
	"n'o", "",
	"n'yu", "",
	"n'e", "",
	"n'yo", "",
	"n'u", "",
	"a_", "",
	"u_", "",
	"e_", "",
	"o_", "",
	"_", "",
	".", "",
	"xa", "",
	"a", "",
	"xi", "",
	"i", "",
	"xu", "",
	"u", "",
	"xe", "",
	"e", "",
	"ye", "",
	"xo", "",
	"o", "",
	"ka", "",
	"ga", "",
	"kya", "",
	"kyu_", "奦",
	"kyu", "",
	"kye", "",
	"kyo_", "祦",
	"kyo", "",
	"ki", "",
	"gya", "",
	"gyu_", "奦",
	"gyu", "",
	"gye", "",
	"gyo_", "祦",
	"gyo", "",
	"gi", "",
	"ku_", "",
	"ku", "",
	"gu_", "",
	"gu", "",
	"ke", "",
	"ge", "",
	"ko_", "",
	"ko", "",
	"go_", "",
	"go", "",
	"sa", "",
	"za", "",
	"sha", "",
	"shu_", "奦",
	"shu", "",
	"she", "",
	"sho_", "祦",
	"sho", "",
	"shi", "",
	"ja", "",
	"ju_", "奦",
	"ju", "",
	"je", "",
	"jo_", "祦",
	"jo", "",
	"ji", "",
	"su_", "",
	"su", "",
	"zu_", "",
	"zu", "",
	"zi", "",
	"se_", "",
	"se", "",
	"ze", "",
	"so_", "",
	"so", "",
	"zo_", "",
	"zo", "",
	"ta", "",
	"da", "",
	"cha", "",
	"chu_", "奦",
	"chu", "",
	"che", "",
	"cho_", "祦",
	"cho", "",
	"chi", "",
	"dja", "¥",
	"dju_", "¥奦",
	"dju", "¥",
	"dje", "¥",
	"djo_", "¥祦",
	"djo", "¥",
	"dji", "",
	"tchi", "å",
	"tcha", "å",
	"tchu", "å",
	"tche", "å",
	"tcho", "å",
	"*", "",
	"dzu_", "",
	"dzu", "",
	"tsu_", "ĥ",
	"tsu", "",
	"tsa", "ĥ",
	"tse", "ĥ",
	"tso", "ĥ",
	"ti", "ƥ",
	"tya", "ƥ",
	"tyu", "ƥ",
	"tyo", "ƥ",
	"te", "",
	"di", "ǥ",
	"dya", "ǥ",
	"dyu_", "ǥ奦",
	"dyu", "ǥ",
	"dye", "ǥ",
	"dyo_", "ǥ祦",
	"dyo", "ǥ",
	"de", "",
	"to_", "ȥ",
	"to", "",
	"du", "ɥ",
	"do_", "ɥ",
	"do", "",
	"na", "",
	"nya", "˥",
	"nyu_", "˥奦",
	"nyu", "˥",
	"nye", "˥",
	"nyo_", "˥祦",
	"nyo", "˥",
	"ni", "",
	"nu_", "̥",
	"nu", "",
	"ne", "",
	"no_", "Υ",
	"no", "",
	"ha", "",
	"ba", "",
	"pa", "",
	"hya", "ҥ",
	"hyu_", "ҥ奦",
	"hyu", "ҥ",
	"hye", "ҥ",
	"hyo_", "ҥ祦",
	"hyo", "ҥ",
	"hi", "",
	"bya", "ӥ",
	"byu_", "ӥ奦",
	"byu", "ӥ",
	"bye", "ӥ",
	"byo_", "ӥ祦",
	"byo", "ӥ",
	"bi", "",
	"pya", "ԥ",
	"pyu_", "ԥ奦",
	"pyu", "ԥ",
	"pye", "ԥ",
	"pyo_", "ԥ祦",
	"pyo", "ԥ",
	"pi", "",
	"fu_", "ե",
	"fa", "ե",
	"fi", "ե",
	"fe", "ե",
	"fo", "ե",
	"fya", "ե",
	"fyu", "ե",
	"fye", "ե",
	"fyo", "ե",
	"fu", "",
	"bu_", "֥",
	"bu_", "֥",
	"bu", "",
	"pu", "",
	"he", "",
	"be", "",
	"pe", "",
	"ho_", "ۥ",
	"ho", "",
	"bo_", "ܥ",
	"bo", "",
	"po_", "ݥ",
	"po", "",
	"ma", "",
	"mya", "ߥ",
	"myu_", "ߥ奦",
	"myu", "ߥ",
	"mye", "ߥ",
	"myo_", "ߥ祦",
	"myo", "ߥ",
	"mi", "",
	"mu_", "०",
	"mu", "",
	"me_", "ᥨ",
	"me", "",
	"mo_", "⥦",
	"mo", "",
	"xya", "",
	"ya", "",
	"xyu", "",
	"yu_", "楦",
	"yu", "",
	"xyo", "",
	"yo_", "襦",
	"yo", "",
	"ra", "",
	"rya", "",
	"ryu_", "奦",
	"ryu", "",
	"rye", "ꥧ",
	"ryo_", "祦",
	"ryo", "",
	"ri", "",
	"ru_", "륦",
	"ru", "",
	"re_", "쥨",
	"re", "",
	"ro_", "",
	"ro", "",
	"xwa", "",
	"wa", "",
	"wi", "",
	"we", "",
	"wo", "",
	"n", "",
	"vu", "",
	"va", "",
	"vi", "",
	"ve", "",
	"vo", "",
	"xka", "",
	"xke", "",
	"", ""
};



/*FUNCTION_HEADER**********************
 * NAME:	;sjis2jis
 * DESC: 	convert ShiftJIS to JIS
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void sjis2jis(int *p1,int *p2)   /* Ken Lunde's routine  */
{
    register unsigned char c1 = *p1;
    register unsigned char c2 = *p2;
    register int adjust = c2 < 159;
    register int rowOffset = c1 < 160 ? 112 : 176;
    register int cellOffset = adjust ? (31 + (c2 > 127)) : 126;
	
    *p1 = ((c1 - rowOffset) << 1) - adjust;
    *p2 -= cellOffset;
}


/*FUNCTION_HEADER**********************
 * NAME:	;FixSJIS
 * DESC: 	convert any SJIS to EUC in a string
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:
	http://www.roguewave.com/support/docs/sourcepro/stdlibug/23-3.html
	http://lfw.org/text/jp.html
	Any byte having a value in the range 0x81-9F or 0xE0-EF is assumed to be the first byte of a 
	two-byte character from the set JIS X 0208-1990. The second byte must have a value in the 
	range 0x40-7E or 0x80-FC.	
 *END_HEADER***************************/
void FixSJIS(unsigned char *jline)
{
	int i,p1,p2;
	
	for( i=0; i < (int)strlen(jline); i++)
	{
		p1 = jline[i];							// Get first byte
		if( p1 < 127 )							// Check if in Ascii range
			continue;							// only increment i by 1 if ascii

		p2 = jline[i+1];						// Get second byte

		if( ((0x81 <= p1 && p1 <= 0x9f) ||		// Check 1st byte for a valid sjis range
			 (0xe0 <= p1 && p1 <= 0xef)) && 
			((0x40 <= p2 && p2 <= 0x7e) ||		// Check 2nd byte for a valid sjis range
			 (0x80 <= p2 && p2 <= 0xfc)) )
		{
			sjis2jis(&p1, &p2);					// convert sjis to jis
			p1 += 128;							// convert jis to euc
			p2 += 128;
			jline[i] = p1;						// store in output
			jline[i+1] = p2;
		}
		i++;									// increment i by 2 if jis
	}
}

/*FUNCTION_HEADER**********************
 * NAME:	;GetEUC
 * DESC: 	ensures that any JIS in the string is EUC
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:Based on the GetEUC in JREADER, this routine examines
	the string "Instr" and converts any JIS or SJIS to EUC.
	The result is placed in the specified string.	
 *END_HEADER***************************/
void GetEUC(unsigned char *eucline, unsigned char *Instr, int bForceEUC)
{
	int i,j,SI,J212,J212sw;

	J212 = FALSE;
	eucline[0] = '\0';
	SI = FALSE;
	j = 0;
	for( i=0; i < (int)strlen(Instr); i++)
	{
		if((Instr[i]==0x1b)&&(Instr[i+1]=='$')&&(Instr[i+2]=='(')&&(Instr[i+3]=='D'))	// JIS X 0212-1990 (ISO-2022-JP-1)
		{
			SI = TRUE;
			J212 = TRUE;
			J212sw = 0;
			i+=3;
			continue;
		}
		if((Instr[i]==0x1b)&&(Instr[i+1]=='$')&&(Instr[i+2]=='B'))	// JIS X 0208-1983 (ISO-2022-JP)
		{
			SI = TRUE;
			J212 = FALSE;
			i+=2;
			continue;
		}
		if((Instr[i]==0x1b)&&(Instr[i+1]=='$')&&(Instr[i+2]=='@'))	// JIS X 0208-1978 (ISO-2022-JP)
		{
			SI = TRUE;
			J212 = FALSE;
			i+=2;
			continue;
		}
		if((Instr[i]==0x1b)&&(Instr[i+1]=='(')&&(Instr[i+2]=='J'))	// JIS X 0201-Roman (legacy) (ISO-2022-JP)
		{
			SI = FALSE;
			J212 = FALSE;
			i+=2;
			continue;
		}
		if((Instr[i]==0x1b)&&(Instr[i+1]=='(')&&(Instr[i+2]=='B'))	// Ascii characters (ISO-2022-JP)
		{
			SI = FALSE;
			J212 = FALSE;
			i+=2;
			continue;
		}
		if (Instr[i] == '\0')
			break;
		if( SI )
		{
			if (J212 && (J212sw == 0)) 
				eucline[j++] = 0x8f;
			eucline[j] = Instr[i] | 0x80;
			J212sw = (J212sw+1) % 2;
		}
		else
		{
			eucline[j] = Instr[i];
		}
		j++;
		eucline[j] = '\0';
	}

	/*  fix up SHIFT-JIS, if present  */
	if( !bForceEUC ) 
		FixSJIS(eucline);
}


/*FUNCTION_HEADER**********************
 * NAME:	;LoadKana
 * DESC: 	Load the kana conversion table if not using the
			one in memory
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void LoadKana(char *sControlFile, int bForceEUC)
{
	int i,
		ih,
		mode;
	FILE 
		*fp;
	unsigned char 
		LKin[80],
		LKinraw[80],
		*ptr;
	
	for (i = 0; i < NRKANA*2; i++) 
		strcpy(KANATAB[i]," ");
	
	if( (fp=fopen(sControlFile,"r")) == NULL )
	{
		printf("Unable to open: %s\n", sControlFile);
		exit(1);
	}
	mode = 0;
	while( fgets(LKinraw,79,fp) )
	{
		LKin[strlen(LKinraw)-1] = '\0';
		GetEUC(LKin,LKinraw, bForceEUC);

		if(LKin[0] == '#')				// Comment line
			continue;

		if( (LKin[0] == '$')&&			// Reading hirigana table
			((LKin[1]|0x20) == 'h'))
		{
			mode = 0;
			ih = 0;
			continue;
		}

		if( (LKin[0] == '$') &&			// Reading katakana table
			((LKin[1]|0x20) == 'k'))
		{
			mode = 1;
			ih = 0;
			continue;
		}

		ptr = (unsigned char *)strtok(LKin," \t");
		strcpy(KANATAB[mode*NRKANA+ih*2+1], ptr);
		ptr = (unsigned char *)strtok(NULL," \t\r\n");
		strcpy(KANATAB[mode*NRKANA+ih*2], ptr);
		ih++;
		if(ih == NRKANA) 
			printf("Too many romaji table entries!\n");
	}
	fclose(fp);
}

/*FUNCTION_HEADER**********************
 * NAME:	;addch
 * DESC: 	add a character
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void addch(unsigned char *targ, unsigned char ch)
{
	unsigned char ctemp[2];
	
	ctemp[0] = ch;
	ctemp[1] = 0;
	strcat(targ, ctemp);
}

/*FUNCTION_HEADER**********************
 * NAME:	;process_kana
 * DESC: 	convert kana to romaji
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void process_kana(unsigned char *kstr, unsigned char *outstr)
{
	int hit,i,k;
	unsigned char roma[256];
	
	if (strlen(kstr) == 0) return;
	
	roma[0] = 0;
	for (i = 0; i < (int)strlen(kstr); )
	{
		hit = FALSE;
		for (k = 0; k < NRKANA && KANATAB[k*2][0] != 0x00; k++)
		{
			if(strncmp(kstr+i, KANATAB[k*2+1], strlen(KANATAB[k*2+1])) == 0)
			{
				strcat(roma,KANATAB[k*2]);
				i = i+strlen(KANATAB[k*2+1]);
				hit = TRUE;
				break;
			}
		}
		if (!hit)
		{
			strcat(roma,"#");
			i++;
		}
	}
	if( roma[strlen(roma)-1] == '*' )						/* fix up trailing small tsu   */
	{
		i = strlen(roma)-1;
		roma[i++] = 't';
		roma[i++] = '-';
		roma[i] = 0;
	}
	if( kstr[0] == 0xa5 )									// Make Katakana uppercase
	{
		for( i=0; i < (int)strlen(roma); i++ )
		{
			if( 'a' <= roma[i] && roma[i] <= 'z' ) 
				roma[i] = 'A' + (roma[i]-'a');
		}
	}
	for( i = 0; i < (int)strlen(roma); i++ )
	{
		if (roma[i] == '*') 
			roma[i] = roma[i+1];
	}
	strcat(outstr,roma);
	kstr[0] = 0;
}

/*FUNCTION_HEADER**********************
 * NAME:	;jis2sjis
 * DESC: 	convert jis to sjis encoding
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	(from Ken Lunde)
 *END_HEADER***************************/
void jis2sjis(unsigned char *p1,unsigned char *p2)
{
    register unsigned char c1 = *p1;
    register unsigned char c2 = *p2;
    register int rowOffset = c1 < 95 ? 112 : 176;
    register int cellOffset = c1 % 2 ? 31 + (c2 > 95) : 126;

    *p1 = ((c1 + 1) >> 1) + rowOffset;
    *p2 = c2 + cellOffset;
}

/*FUNCTION_HEADER**********************
 * NAME:	;Kana2Rom
 * DESC: 	Convert kana to romaji by calling process_kana
 * IN:		sRaw - raw euc kana text
			ok_mark - okurigana marker . - or (
			bForceEUC - 
 * OUT:		sRom - romaji translation
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void Kana2Rom(char *sRaw, char *sRom, char ok_mark, int bForceEUC)
{
	unsigned char 
		sEuc[256],
		kstr[256];
	int j,
		OkActive=FALSE;

	GetEUC(sEuc, sRaw, bForceEUC);
	kstr[0] = 0;
	sRom[0] = 0;
	for (j = 0; j < (int)strlen(sEuc); j++)
	{
		if (sEuc[j] < 128)
		{
			process_kana(kstr, sRom);
			if ((sEuc[j] == '.') && (sRom[strlen(sRom)-1] == '-'))
			{
				sRom[strlen(sRom)-1] = 0;
			}
			if (sEuc[j] == '.') 
			{
				sEuc[j] = ok_mark;
				OkActive = TRUE;
			}
			else
			{
				if( OkActive && ok_mark == '(' ) 
					addch(sRom,')');
				OkActive = FALSE;
			}
			addch(sRom,sEuc[j]);
			continue;
		}
		if ((sEuc[j] == 0xa1) && (sEuc[j+1] == 0xbc))
		{
			addch(kstr,sEuc[j]);
			addch(kstr,sEuc[j+1]);
			j++;
			continue;
		}
		if (sEuc[j] == 0x8f)
		{
			process_kana(kstr, sRom);
			if( OkActive && ok_mark == '(' ) 
				addch(sRom,')');
			OkActive = FALSE;
			addch(sRom,sEuc[j]);
			addch(sRom,sEuc[j+1]);
			addch(sRom,sEuc[j+2]);
			j+=2;
			continue;
		}
		if ((sEuc[j] < 0xa4) || (sEuc[j] > 0xa5))
		{
			process_kana(kstr, sRom);
			if (OkActive && ok_mark == '(') 
				addch(sRom,')');
			OkActive = FALSE;
			addch(sRom,sEuc[j]);
			addch(sRom,sEuc[j+1]);
			j++;
			continue;
		}
		addch(kstr,sEuc[j]);
		addch(kstr,sEuc[j+1]);
		j++;
		continue;
	}
	process_kana(kstr, sRom);
	if (OkActive && ok_mark == '(') 
		addch(sRom,')');
}

/*FUNCTION_HEADER**********************
 * NAME:	;Kana2RomW
 * DESC: 	Convert kana to romaji by calling process_kana
 * IN:		wsRaw - raw euc kana text in wide character format
			ok_mark - okurigana marker . - or (
			bForceEUC - 
 * OUT:		sRom - romaji translation
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void Kana2RomW(wchar_t *wsRaw, char *sRom, char ok_mark, int bForceEUC)
{
	char 
		*pc,
		sRaw[512];
	wchar_t
		*pwc;

	for(pwc=wsRaw, pc=sRaw; *pwc != 0x00; pwc++)
	{
		*pc = (*pwc & 0xFF00) >> 8;
		pc++;
		*pc = *pwc & 0x00FF;
		pc++;
	}
	*pc = 0x00;		// Null terminate
	Kana2Rom(sRaw, sRom, ok_mark, bForceEUC);
}

#ifdef KANA2ROM_PROJECT
/*FUNCTION_HEADER**********************
 * NAME:	;KOut
 * DESC: 	output kana/kanji according to selected mode
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void KOut(unsigned char *sout,unsigned char *oline, int omode)
{
	int i,op,JISin;
	unsigned char c1,c2;

	op = 0;
	JISin = FALSE;
	for (i = 0; i < (int)strlen(sout); i++)
	{
		c1 = sout[i]; 
		c2 = sout[i+1];
		if (c1 < 127)
		{
			if (JISin)
			{
				oline[op++] = 0x1b;
				oline[op++] = '(';
				oline[op++] = 'B';
				JISin = FALSE;
				oline[op] = 0;
			}
			oline[op] = c1; oline[op+1] = 0;
			op++;
			continue;
		}
		switch(omode)
		{
		case 'j' :  /* JIS (default)  */
			if (c1 == 0x8f)
			{
				if (JISin)
				{
					oline[op++] = 0x1b;
					oline[op++] = '(';
					oline[op++] = 'B';
					JISin = FALSE;
					oline[op] = 0;
				}
				oline[op++] = 0x1b;
				oline[op++] = '$';
				oline[op++] = '(';
				oline[op++] = 'D';
				oline[op++] = c2&0x7f;
				oline[op++] = sout[i+2]&0x7f;
				oline[op++] = 0x1b;
				oline[op++] = '(';
				oline[op++] = 'B';
				oline[op] = 0;
				i+=2;
				break;
			}
			if (JISin)
			{
				oline[op++] = c1&0x7f;
				oline[op++] = c2&0x7f;
			}
			else
			{
				oline[op++] = 0x1b;
				oline[op++] = '$';
				oline[op++] = 'B';
				oline[op++] = c1&0x7f;
				oline[op++] = c2&0x7f;
				JISin = TRUE;
			}
			i++;
			break;

		case 'e' : /* EUC (internal format) */
			if (c1 == 0x8f)
			{
				oline[op++] = c1;
				oline[op++] = c2;
				oline[op++] = sout[i+2];
				oline[op] = 0;
				i+=2;
				break;
			}
			oline[op++] = c1;
			oline[op++] = c2;
			oline[op] = 0;
			i++;
			break;

		case 's' : /* Shift-JIS */
			if (c1 == 0x8f)
			{
				c1 = 0xa2; 
				c2 = 0xa1;	/* a star in SJIS as there's no JIS212*/
				i++;
			}
			c1 -= 128; c2 -= 128;
			jis2sjis(&c1,&c2);
			oline[op++] = c1;
			oline[op++] = c2;
			oline[op] = 0;
			i++;
			break;
		}
	}
	if (JISin)
	{
		oline[op++] = 0x1b;
		oline[op++] = '(';
		oline[op++] = 'B';
		oline[op] = 0;
		JISin = FALSE;
	}
}

/*FUNCTION_HEADER**********************
 * NAME:	;print_ins
 * DESC: 	Print usage help
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void print_ins()
{
	printf("usage: kana2rom -i <input file> -o <output file> <options>\n ");
	printf("  options:   -E force EUC input mode (input has JIS212)\n");
	printf("             -Os output in Shift-JIS\n");
	printf("             -Oj output in JIS\n");
	printf("             -c <control-file>\n");
	printf("             -h force okurigana to hyphen\n");
	printf("             -p force okurigana to parentheses\n");
	printf("             -H these instructions\n");
}

/*FUNCTION_HEADER**********************
 * NAME:	;main
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int main(int argc, unsigned char **argv)
{
	char 
		sInFile[_MAX_PATH],
		sOutFile[_MAX_PATH],
		sControlFile[_MAX_PATH] = {0x00};	// = {"kanarom.cnv"};
	unsigned char 
		instr_raw[256],
		outstr[256],
		outstr2[256],
		ok_mark = '.';
	int i,
		omode='e',
		bForceEUC=FALSE;
	FILE 
		*fi,
		*fo;
	
	if (argc < 3)
	{
		print_ins();
		exit(0);
	}
	else
	{
		printf("KANA2ROM: kana to romaji conversion program, J.W. Breen, 1996\n\n");
		printf("          modified R.B. Moncur, 2005\n\n");
	}

	for( i=1; i<argc; i++ )
	{
		if( argv[i][0] == '-' )
		{
			switch( argv[i][1] )
			{
				case 'H':								// Help
					print_ins();
					exit(0);
					
				case 'O':								// Output mode
					if ((argv[i][2] == 'j') || 
						(argv[i][2] == 's'))
					{
						omode = argv[i][2];
						break;
					}
					else
					{
						printf("Illegal output mode!\n");
						exit(1);
					}
					
				case 'E':								// EUC input
					bForceEUC = TRUE;
					break;
					
				/*
					Okurigana are kana suffixes following kanji stems in Japanese written words. 
					Generally used to inflect an adjective or verb, okurigana can indicate tense 
					(past or present/future), affirmative/negative meaning, or grammatical politeness, 
					among many other functions. In modern usage, okurigana are almost invariably 
					written with hiragana; katakana were also commonly used in the past.
				*/
				case 'h':								// force okurigana to hyphen
					ok_mark = '-';
					break;
					
				case 'p':								// force okurigana to parenthesis
					ok_mark = '(';
					break;
					
				case 'c':								// Control file
					i++;
					strcpy(sControlFile, argv[i]);
					printf("Using user-specified control file: %s\n", sControlFile);
					break;

				case 'i':
					i++;
					strcpy(sInFile, argv[i]);
					break;

				case 'o':
					i++;
					strcpy(sOutFile, argv[i]);
					break;
					
				default:
					printf("Illegal option\n");
					print_ins();
					exit(1);
			}
			
		}
		
	}

	printf("Input File: %s\n", sInFile);			// Open input file
	if( (fi= fopen(sInFile,"r")) == NULL )
	{
		printf("\nCannot open %s\n", sInFile);
		exit(1);
	}

	printf("Output File: %s\n", sOutFile);			// Open output file
	if( (fo = fopen(sOutFile, "w")) == NULL )
	{
		printf("\nCannot open %s\n", sOutFile);
		exit(1);
	}

	if( *sControlFile )
		LoadKana(sControlFile, bForceEUC);

	while( fgets(instr_raw,255,fi) )
	{
		instr_raw[strlen(instr_raw)-1] = '\0';		// replace line feed with null terminator

		Kana2Rom(instr_raw, outstr, ok_mark, bForceEUC);

		KOut(outstr, outstr2, omode);
		fprintf(fo,"%s\n",outstr2);
	}
	fclose(fi);
	fclose(fo);
}
#endif	// #ifdef KANA2ROM_PROJECT