/*C_HEADER_FILE****************************************************************
FILE			:	JpDict.c
DESC			:	
TABS			:	3
OWNER			:	
DATE CREATED	:	22 Nov 2005

(C) Copyright 2005 All rights reserved.
This is an unpublished work, and is confidential and proprietary: 
technology and information of fonix corporation.  No part of this
code may be reproduced, used or disclosed without written consent of 
fonix corporation in each and every instance.

  $Date:  $
  $Revision:  $

*END_HEADER*******************************************************************/

#include "BuildJpDic.h"
#include "Kana2Rom.h"
#include "CodePageConv.h"
#include "onealignW.h"
#include <assert.h>

int gnBranches = 0;				// This is just a debug tool


JP_WORD_TYPE gptJpWordType[] = {
	{"abbr",	JP_ABBR},			// abbreviation
	{"adj",		JP_ADJ},			// adjective (keiyoushi)
	{"adv",		JP_ADV_ADV},		// adverb (fukushi)
	{"adv-n",	JP_ADV_N},			// adverbial noun
	{"adj-na",	JP_ADJ_NA},			// adjectival nouns or quasi-adjectives (keiyoudoushi)
	{"adj-no",	JP_ADJ_NO},			// nouns which may take the genitive case particle "no"
	{"adj-pn",	JP_ADJ_PN},			// pre-noun adjectival (rentaishi)
	{"adj-s",	JP_ADJ_S},			// special adjective (e.g. ookii)
	{"adj-t",	JP_ADJ_T},			// "taru" adjective
	{"arch",	JP_MISC_ARCH},		// archaism
	{"ateji",	JP_MISC_ATEJI},		// ateji reading of the kanji
	{"aux",		JP_AUX_WORD},		// auxiliary word or phrase
	{"aux-v",	JP_AUX_VERB},		// auxiliary verb
	{"conj",	JP_CONJ},			// conjunction
	{"col",		JP_MISC_COL},		// colloquialism 
	{"exp",		JP_MISC_COL},		// Expressions (phrases, clauses, etc.)
	{"ek",		JP_MISC_OTHER},		// exclusively kanji, rarely just in kana
	{"fam",		JP_MISC_COL},		// familiar language 
	{"fem",		JP_MISC_FEM},		// female term or language
	{"gikun",	JP_MISC_GIKUN},		// gikun (meaning) reading
	{"gram",	JP_MISC_GRAM},		// grammatical term
	{"hon",		JP_MISC_HON},		// honorific or respectful (sonkeigo) language 
	{"hum",		JP_MISC_HON},		// humble (kenjougo) language 
	{"id",		JP_MISC_COL},		// idiomatic expression 
	{"int",		JP_MISC_INTJ},		// interjection (kandoushi)
	{"iK",		JP_MISC_IRREG},		// word containing irregular kanji usage
	{"ik",		JP_MISC_IRREG},		// word containing irregular kana usage
	{"io",		JP_MISC_IRREG},		// irregular okurigana usage
	{"MA",		JP_MISC_MA},		// martial arts term
	{"male",	JP_MISC_MALE},		// male term or language
	{"m-sl",	JP_MISC_COL},		// manga slang
	{"n",		JP_NOUN_CMN},		// noun (common) (futsuumeishi)
	{"n-adv",	JP_NOUN_ADV},		// adverbial noun (fukushitekimeishi)
	{"n-t",		JP_NOUN_TMP},		// noun (temporal) (jisoumeishi)
	{"n-suf",	JP_NOUN_SUF},		// noun, used as a suffix
	{"n-pref",	JP_NOUN_PREF},		// noun, used as a prefix
	{"neg",		JP_MISC_OTHER},		// negative (in a negative sentence, or with negative verb)
	{"neg-v",	JP_VERB_MISC},		// negative verb (when used with)
	{"num",		JP_MISC_NUMB},		// number, numeric
	{"obs",		JP_MISC_OLD},		// obsolete term
	{"obsc",	JP_MISC_OLD},		// obscure term
	{"oK",		JP_MISC_OLD},		// word containing out-dated kanji 
	{"ok",		JP_MISC_OLD},		// out-dated or obsolete kana usage
	{"pol",		JP_MISC_HON},		// polite (teineigo) language 
	{"pref",	JP_PREF},			// prefix 
	{"prt",		JP_PART},			// particle
	{"qv",		JP_MISC_OTHER},		// quod vide (see another entry)
	{"sl",		JP_MISC_COL},		// slang
	{"suf",		JP_SUFX},			// suffix 
	{"uK",		JP_MISC_OTHER},		// word usually written using kanji alone 
	{"uk",		JP_MISC_KANA},		// word usually written using kana alone 
	{"v1",		JP_VERB_V1},		// Ichidan verb
	{"v5",		JP_VERB_MISC},		// Godan verb (not completely classified)
	{"v5u",		JP_VERB_V5U},		// Godan verb with `u' ending
	{"v5u-s",	JP_VERB_MISC},		// Godan verb with `u' ending - special class
	{"v5k",		JP_VERB_V5K},		// Godan verb with `ku' ending
	{"v5g",		JP_VERB_V5G},		// Godan verb with `gu' ending
	{"v5s",		JP_VERB_V5S},		// Godan verb with `su' ending
	{"v5t",		JP_VERB_V5T},		// Godan verb with `tsu' ending
	{"v5n",		JP_VERB_V5N},		// Godan verb with `nu' ending
	{"v5b",		JP_VERB_V5B},		// Godan verb with `bu' ending
	{"v5m",		JP_VERB_V5M},		// Godan verb with `mu' ending
	{"v5r",		JP_VERB_V5R},		// Godan verb with `ru' ending
	{"v5k-s",	JP_VERB_V5KS},		// Godan verb - Iku/Yuku special class
	{"v5aru",	JP_VERB_V5ARU},		// Godan verb - -aru special class
	{"v5uru",	JP_VERB_MISC},		// Godan verb - Uru old class verb (old form of Eru)
	{"vi",		JP_VERB_MISC},		// intransitive verb 
	{"vs",		JP_NOUN_SURU},		// noun or participle which takes the aux. verb suru
	{"vs-i",	JP_VERB_MISC},		// suru verb - irregular
	{"vs-s",	JP_VERB_VSS},		// suru verb - special class
	{"vz",		JP_VERB_VZ},		// zuru verb - (alternative form of -jiru verbs)
	{"vk",		JP_VERB_VK},		// Kuru verb - special class
	{"vt",		JP_VERB_MISC},		// transitive verb
	{"vulg",	JP_MISC_VULG},		// vulgar expression or word 
	{"X",		JP_MISC_VULG},		// rude or X-rated term (not displayed in educational software)
	{"", 0}
};

int gnJpWordTypes = (sizeof(gptJpWordType) / sizeof(JP_WORD_TYPE)) -1;	

/*
						EUC Encoding		Unicode
	Ascii				0x0021 - 0x007E		

	Punctuation			0xA1A0 - 0xA1DF

	Roman characters	0xA3A0 - 0xA3FF

	Half-width katakana	0x8EA1 - 0x8EDF		0xFF66 - 0xFF9F

	Hiragana			0xA4A1 - 0xA4F3		0x3041 - 0x3093

	Katakana			0xA5A1 - 0xA5F6		0x30A1 - 0x30F6
	
	Kanji				0xB0A0 - 0xFFE0		0x4E00 - 0x9FBF

 */


// =====================================================================
//	The following functions are used to load a raw text dictionary into
//  memory, sort it so that you can find words by doing a binary search
// =====================================================================

/*FUNCTION_HEADER**********************
 * NAME:	;FreeDictionary
 * DESC: 	Free memory used by a dictionary
 * IN:		pDict - pointer to dictionary data
			nEntries - number of entries in pDict
 * OUT:		none
 * RETURN:	none
 * NOTES:	
 *END_HEADER***************************/
void FreeDictionary(JPDICTPTR pDict, int nEntries)
{
	int i;

	if( pDict == NULL )
		return;

	for(i=0; i<nEntries; i++)
	{
		if( pDict[i].wsKana )		free(pDict[i].wsKana);
		if( pDict[i].wsWord )		free(pDict[i].wsWord);
		if( pDict[i].wsKanaAlign )	free(pDict[i].wsKanaAlign);
		if( pDict[i].wsWordAlign )	free(pDict[i].wsWordAlign);
		if( pDict[i].sWordTypes )	free(pDict[i].sWordTypes);
	}

	free(pDict);
}

/*FUNCTION_HEADER**********************
 * NAME:	;DictWordCmp
 * DESC: 	Compare two words in the dictionary
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int DictWordCmp( const void *arg1, const void *arg2 )
{
	JPDICT p1 = *(JPDICTPTR )arg1,
		 p2 = *(JPDICTPTR )arg2;

   /* Compare all of both strings: */
   return _wcsicmp( p1.wsWord, p2.wsWord );
}


/*FUNCTION_HEADER**********************
 * NAME:	;SortDictionary
 * DESC: 	Sort the in-memory dictionary
 * IN:		pDict, nEntries
 * OUT:		data in pDict is sorted so we can do bsearch on it
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void SortDictionary(JPDICTPTR pDict, int nEntries)
{
	qsort(pDict, nEntries, sizeof(JPDICT), DictWordCmp);
}

unsigned int FindTagBitMask(char *sStr)
{
	char 
		sTmp[64],
		*pc;
	int i;
	unsigned int
		iBitMask=0;

	strcpy(sTmp, sStr);
	pc = strtok(sTmp, ",");
	while( pc )
	{
		for(i=0; i<gnJpWordTypes; i++ )
		{
			if( strcmp(pc, gptJpWordType[i].psWordType) == 0 )
			{
				iBitMask |= gptJpWordType[i].uiBitMask;
				break;
			}
		}
		pc = strtok(NULL, ",");
	}

	return iBitMask;
}


/*FUNCTION_HEADER**********************
 * NAME:	;ParseDictEntry
 * DESC: 	Parse a dictionary entry
 * IN:		sBuf - the raw data to parse
 * OUT:		parsed data is put in pDict
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int ParseDictEntry(char *sBuf, JPDICTPTR pDict)
{
	unsigned char 
		*pc=sBuf,
		*pc2;
	wchar_t 
		wsTmp[128];
	int	i,
		iBitFlag;

	if( sBuf == NULL )
		return 0;

	// skip any leading spaces
	while( *pc == ' ' )
		pc++;

	// find end of word
	for( pc2=pc; pc2 && *pc2 != ' ' && *pc2 != '\n' && *pc2 != 0x00; pc2++ )
		;
	if( *pc2 != ' ' )	
		return 0;
	*pc2 = 0x00;							// Null terminate end of word
	
	// Find Kanji/Kana word and convert to wide character string
	for(i=0; i<128 && pc < pc2; i++, pc+=2 )
	{
		wsTmp[i] = (*pc<<8) + (*(pc+1));
	}
	wsTmp[i] = 0x00;						// Null terminate
	pDict->wsWord = wcsdup(wsTmp);			// Copy word
	*pc2 = ' ';
	StrKata2Hira(pDict->wsWord);			// Convert any Katakana symbols to Hiragana

	// find kana: [<kana>] and convert to wide character string
	if( (pc = strchr(sBuf, '[')) )
	{
		pc++;
		if( (pc2 = strchr(sBuf, ']')) )
		{
			*pc2 = 0x00;							// null terminate the kana
			for(i=0; i<128 && pc < pc2; i++, pc+=2 )
			{
				wsTmp[i] = (*pc<<8) + (*(pc+1));
			}
			wsTmp[i]=0x00;
			pDict->wsKana = wcsdup(wsTmp);
			*pc2 = ']';								// put the kana closing bracket back in place
			StrKata2Hira(pDict->wsKana);
		}
	}

	// find tags
	pc2++;
	while( (pc = strchr(pc2, '('))  )			// Find an opening parenthesis
	{
		pc++;
		if( (pc2 = strchr(pc, ')')) )			// Find the matching closing parenthesis
		{
			*pc2 = 0x00;
			if( pc2-pc < 60 )
			{
				iBitFlag = FindTagBitMask(pc);	// Get the Parts-of-Speech bit flag
			}

			if( iBitFlag != 0 )
			{
				pDict->iBitFlags |= iBitFlag;
				pDict->sWordTypes = strdup(pc);			// Store the text POS data also
			}

			*pc2 = ')';
			pc2++;									// Get ready for the next one
		}
		else
			break;
	}

	return 1;
}

/*FUNCTION_HEADER**********************
 * NAME:	;LoadDictionary
 * DESC: 	Load a dictionary from file
 * IN:		sInFile - dictionary filename
 * OUT:		pnEntries - number of entires in the dictionary
 * RETURN:	pointer to the dictionary on success, NULL on failure
 * NOTES:	
 *END_HEADER***************************/
JPDICTPTR LoadDictionary(char *sInFile, int *pnEntries)
{
	char sBuf[512];
	int	 i,
		 nEntries=0;
	JPDICTPTR pDict;
	FILE *fp;

	// Open the text dictionary for read
	if( (fp = fopen(sInFile, "rb")) == NULL )
	{
		printf("Error: Can't open dictionary %s\n", sInFile);
		return NULL;
	}

	// Count the number of entries in the dictionary
	fgets(sBuf, 512, fp);		// Read header line
	while( fgets(sBuf, 512, fp) )
	{
		nEntries++;
	}

	// Allocate memory to hold all entries
	if( (pDict = (JPDICTPTR )calloc(nEntries, sizeof(JPDICT))) == NULL )
	{
		printf("Error: Out of memory in LoadDictionary\n");
		return NULL;
	}

	// Reset back to the beginning of the file
	fseek(fp, 0L, SEEK_SET);

	// Read the dictionary
	fgets(sBuf, 512, fp);		// Read header line
	for( i=0; i<nEntries && fgets(sBuf, 512, fp); )
	{
		ParseDictEntry(sBuf, &pDict[i++]);
	}

	if( pnEntries )
		*pnEntries = nEntries;

	fclose(fp);

	return pDict;
}

/* ================================================================

	Functions to build and use an in-memory Letter-Tree dictionary

   ================================================================ */

/*FUNCTION_HEADER**********************
 * NAME:	;GetNewBranch
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
BRANCHPTR GetNewBranch(void)
{
	BRANCHPTR pBranch;

	if( (pBranch = (BRANCHPTR )calloc(1, sizeof(BRANCH))) == NULL )
		return NULL;

	gnBranches++;

	return pBranch;
}

/*FUNCTION_HEADER**********************
 * NAME:	;SymbolStructCmp
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int SymbolStructCmp(const void *arg1, const void *arg2)
{
	SYMBOL s1 = *(SYMBOLPTR)arg1,
		   s2 = *(SYMBOLPTR)arg2;

	if( s1.wcSym > s1.wcSym ) 
		return 1;
	else if( s1.wcSym < s2.wcSym )
		return -1;
	else 
		return 0;

}

/*FUNCTION_HEADER**********************
 * NAME:	;SortBranchBySymbolStruct
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void SortBranchBySymbolStruct(BRANCHPTR pBr)
{
	qsort(pBr->pSyms, pBr->nSym, sizeof(SYMBOL), SymbolStructCmp);
}

/*FUNCTION_HEADER**********************
 * NAME:	;GetBranchSymbolIndex
 * DESC: 	Get the index of a symbol on a branch
 * IN:		pBr - pointer to the branch
			wcSym - the symbol to find
 * OUT:		none
 * RETURN:	index of symbol on success, -1 on failure
 * NOTES:	
 *END_HEADER***************************/
int GetBranchSymbolIndex(BRANCHPTR pBr, wchar_t wcSym)
{
	int i;
	for(i=0; i<pBr->nSym; i++)
	{
		if( pBr->pSyms[i].wcSym == wcSym )
			return i;
	}
	return -1;
}

/*FUNCTION_HEADER**********************
 * NAME:	;AddSymbolStructToBranch
 * DESC: 	Add a new symbol to a branch of the LTree
 * IN:		pBr - pointer to the branch where the symbol is to be added
			wcSym - the new symbol
 * OUT:		none
 * RETURN:	index to symbol on success, -1 on failure
 * NOTES:	
 *END_HEADER***************************/
int AddSymbolStructToBranch(BRANCHPTR pBr, wchar_t wcSym)
{
	int nSym;

	if( pBr == NULL )
		return -1;

	pBr->nSym++;
	nSym = pBr->nSym;
	
	if( (pBr->pSyms = (SYMBOLPTR)realloc(pBr->pSyms, nSym*sizeof(SYMBOL))) == NULL )
		return -1;

	pBr->pSyms[nSym-1].wcSym = wcSym;
	pBr->pSyms[nSym-1].pNext = NULL;
	pBr->pSyms[nSym-1].wsData = NULL;

//	SortBranchBySymbolStruct(pBr);

	return GetBranchSymbolIndex(pBr, wcSym);
}

/*FUNCTION_HEADER**********************
 * NAME:	;AddWordToLTree
 * DESC: 	Add a new word to the letter-tree
 * IN:		sWord - the new word
			sData - output data for sWord, typically the pronunciation
			pLTree - pointer to the LTree being built
			bLowerCase - flag to build dictionary with lower case letters only
 * OUT:		none
 * RETURN:	none
 * NOTES:	
 *END_HEADER***************************/
void AddWordToLTree(wchar_t *wsWord, wchar_t *wsData, BRANCHPTR pLTree)
{
	wchar_t *pc;
	int	 iSym;
	BRANCHPTR pBr;

	if( wsWord == NULL )
		return;

	for(pc=wsWord, pBr=pLTree; *pc; pc++)
	{
		if( (iSym = GetBranchSymbolIndex(pBr, *pc)) < 0)		// get the symbol index
		{
			iSym = AddSymbolStructToBranch(pBr, *pc);			// add a new symbol to this branch
		}

		if( *(pc+1) == 0x00 )									// this is the last symbol of the word
		{
			if( !pBr->pSyms[iSym].wsData )
				pBr->pSyms[iSym].wsData = wcsdup(wsData);			// add data at last symbol
			return;
		}
		else if( pBr->pSyms[iSym].pNext == NULL )				// add a new branch for the next symbol
		{
			pBr->pSyms[iSym].pNext = GetNewBranch();
		}
		pBr = pBr->pSyms[iSym].pNext;							// Follow the branch to next symbol
	}
}


/*FUNCTION_HEADER**********************
 * NAME:	;BuildLetterTree
 * DESC: 	Build in-memory Letter-Tree version of dictionary
 * IN:		pDict - in-memory dictionary sorted alphabetically
			nEntries - number of entries in dictionary
			bLowerCase - flag to create dictionary with lower case letters only
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
BRANCHPTR BuildLetterTree(JPDICTPTR pDict, int nEntries)
{
	wchar_t wsData[512];
	int	i;
	BRANCHPTR pLTree;

	if( !pDict || nEntries <= 0 )
		return NULL;

	pLTree = GetNewBranch();

	for( i=0; i<nEntries; i++ )
	{
		if( pDict[i].wsKana )
		{
			swprintf(wsData, L"0x%08x,%s", pDict[i].iBitFlags, pDict[i].wsKana);
			AddWordToLTree(pDict[i].wsWord, wsData, pLTree);
		}
	}

	return pLTree;
}

/*FUNCTION_HEADER**********************
 * NAME:	;FreeLTree
 * DESC: 	Release memory used by LTree
 * IN:		pLTree - pointer to head of LTree
 * OUT:		none
 * RETURN:	none
 * NOTES:	This function is recursive
 *END_HEADER***************************/
void FreeLTree(BRANCHPTR pLTree)
{
	int i;

	for( i=0; i<pLTree->nSym; i++)
	{
		if( pLTree->pSyms[i].pNext )				// Make sure that all of the next pointers are free
		{
			FreeLTree(pLTree->pSyms[i].pNext);		// Recurse down into the tree
		}

		if( pLTree->pSyms[i].wsData )				// Make sure that all of the data space is free
		{
			free(pLTree->pSyms[i].wsData);
			pLTree->pSyms[i].wsData = NULL;
		}
	}
	free(pLTree->pSyms);
	free(pLTree);									// Free this branch
}

#if 0
/*FUNCTION_HEADER**********************
 * NAME:	;SizeOfData
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
unsigned int SizeOfData(wchar_t *wsData)
{
	wchar_t *pc1, *pc2;
	unsigned int nSize = 0;

	if( wsData )
	{
		nSize += sizeof(long);							// Bit Flag 1
		if( (pc1 = wcschr(wsData, L',')) != NULL )
		{
			pc1++;
			if( (pc2 = wcschr(pc1, L',')) != NULL )
			{
				*pc2 = 0x00;
				nSize += wcslen(pc1) + 1;				// Pronunciation 1 + NULL
				*pc2 = L',';
				
				pc2 += 12;								// Skip bit flag
				nSize += sizeof(long);					// Bit Flag 2
				nSize += wcslen(pc2) + 1;				// Pronunciation 2 + NULL
			}
			else
			{
				nSize += wcslen(pc1) + 1;				// Pronunciation 1 + NULL
			}
		}
	}

	return nSize;
}

/*FUNCTION_HEADER**********************
 * NAME:	;GetLTreeSize
 * DESC: 	Recursively sum the size of the LTree
 * IN:		pLTree - pointer to head of LTree
 * OUT:		none
 * RETURN:	size of LTree
 * NOTES:	
 *END_HEADER***************************/
int GetLTreeSize(BRANCHPTR pLTree)
{
	int i,
		nSize=0;

	for( i=0; i<pLTree->nSym; i++)
	{
		if( pLTree->pSyms[i].pNext )
			nSize += GetLTreeSize(pLTree->pSyms[i].pNext);	// Recurse down into the tree

		if( pLTree->pSyms[i].wsData )						// Get data size
		{
			nSize += SizeOfData(pLTree->pSyms[i].wsData);
		}
	}

	// Size of this branch on disk		nSym + (nSym * (wcSym+iNext+iData))
	nSize += sizeof(int) + pLTree->nSym * (sizeof(wchar_t) + sizeof(int) + sizeof(int));

	gnBranches++;

	return nSize;
}

/*FUNCTION_HEADER**********************
 * NAME:	;GetLTreeSizeWithoutData
 * DESC: 	Calculate the size of the LTree without
			the output data or in other words where is
			the end of the LTree and beginning of where I
			can start to store data in the File System version
 * IN:		pLTree
 * OUT:		none
 * RETURN:	size of LTree minus the data
 * NOTES:	This function is recursive
 *END_HEADER***************************/
int GetLTreeSizeWithoutData(BRANCHPTR pLTree)
{
	int i,
		nSize=0;

	for( i=0; i<pLTree->nSym; i++)
	{
		if( pLTree->pSyms[i].pNext )
			nSize += GetLTreeSizeWithoutData(pLTree->pSyms[i].pNext);	// Recurse down into the tree
	}

	// Size of this branch on disk		nSym + (nSym * (wcSym+iNext+iData))
	nSize += sizeof(char) + pLTree->nSym * (sizeof(char) + sizeof(int) + sizeof(int));

	gnBranches++;

	return nSize;
}

/*FUNCTION_HEADER**********************
 * NAME:	;GetDataSizeInLTree
 * DESC: 	Calculate how much output data is stored in the LTree
 * IN:		pLTree - pointer to a branch in the pLTree.  This should be 
			the head branch for the first call.
 * OUT:		none
 * RETURN:	size of data stored in LTree
 * NOTES:	This function is recursive
 *END_HEADER***************************/
unsigned int GetDataSizeInLTree(BRANCHPTR pLTree)
{
	int i,
		nSize=0;

	for( i=0; i<pLTree->nSym; i++)
	{
		if( pLTree->pSyms[i].pNext )
			nSize += GetDataSizeInLTree(pLTree->pSyms[i].pNext);	// Recurse down into the tree

		if( pLTree->pSyms[i].wsData )						// Get data size
		{
			if( strcmp(pLTree->pSyms[i].wsData, "1") != 0 )
				nSize += SizeOfData(pLTree->pSyms[i].wsData);
		}
	}

	gnBranches++;

	return nSize;
}
#endif

/*FUNCTION_HEADER**********************
 * NAME:	;FindWordInLTree
 * DESC: 	Traverse the LTree to find a word
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
wchar_t *FindWordInLTree(wchar_t *wsWord, BRANCHPTR pLTree)
{
	wchar_t *pc;
	int	 iSym;
	BRANCHPTR pBr;

	for(pc=wsWord, pBr=pLTree; *pc && pBr; pc++)
	{
		if( (iSym = GetBranchSymbolIndex(pBr, *pc)) >= 0)	// get the symbol index
		{
			if( *(pc+1) == 0x00 )					// last symbol of word
				return pBr->pSyms[iSym].wsData;		// return data
			else
				pBr = pBr->pSyms[iSym].pNext;		// Follow the branch
		}
	}

	return NULL;
}

/*FUNCTION_HEADER**********************
 * NAME:	;IsUnicode
 * DESC: 	look for FEFF bytes at the beginning of file
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int IsUnicode(FILE *fp)
{
	unsigned char c1, c2;

	fseek(fp, 0, SEEK_SET);
	fread(&c1, sizeof(char), 1, fp);
	fread(&c2, sizeof(char), 1, fp);

	fseek(fp, 0, SEEK_SET);

	if( (c1 == 0xff && c2 == 0xfe) ||
		(c1 == 0xfe && c2 == 0xff) )
		return 1;	// TRUE
	else
		return 0;	// FALSE
}

/*FUNCTION_HEADER**********************
 * NAME:	;IsBytesSwapped
 * DESC: 	Check to see if the bytes are swapped when reading
			data (big-endian vs little-endian machine.
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
BOOL IsBytesSwapped(FILE *fp)
{
	unsigned char c1, c2, c3, c4;
	wchar_t wc;

	fseek(fp, 0, SEEK_SET);
	fread(&c1, sizeof(char), 1, fp);	// read 2 one byte items
	fread(&c2, sizeof(char), 1, fp);

	fseek(fp, 0, SEEK_SET);
	fread(&wc, sizeof(wchar_t), 1, fp);	// read 1 two byte item
	c3 = (wc & 0xff00) >> 8;
	c4 = (wc & 0x00ff);

	fseek(fp, 0, SEEK_SET);				// Seek back to beginning

	if( c1 == 0xff && c2 == 0xfe )
		return 0;						// Unicode bytes are preswapped
	else if( c1 == c3 && c2 == c4 )		// Does the byte order match
		return 0;
	else
		return 1;						// No, they're swapped

}

#ifdef UNDER_CONSTRUCTION
typedef struct _KRToken		// Kanji Reading token
{
	unsigned int
		iKanjiIndex;
	KANJIDIC
		*pKanji;
	KANJIREADING
		*pReading;
	wchar_t
		*pwcKanaStart;
	unsigned int
		nReadingLen;

	struct _KRToken
		*pNextMatch[10],
		*pPrev;

}KRTOKEN;

typedef struct _KRTokenList
{
	KRTOKEN *pKRToken;
	struct _KRTokenList *pNext;
}KRTOKENLIST;

/*FUNCTION_HEADER**********************
 * NAME:	;AlignKanjiWithReadings
 * DESC: 	
 * IN:		wsWord - A word segment containing 1 or more kanji
			wsKana - The kana segment that matches wsWord
			ppKanji - pointer to kanji dictionary
			nKanji - number of kanji in ppKanji
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
KRTOKEN *AlignKanjiWithReadings(wchar_t *wsWord, wchar_t *wsKana, KANJIDIC **ppKanji, unsigned int nKanji)
{
	wchar_t
		*pwc;
	unsigned int
		i,
		nKanjiToUse=0;
	KANJIDIC 
		*ppKanjiToUse[128];
	KANJIREADING
		*pReading;
	KRTOKEN
		*pToken;
	KRTOKENLIST
		*pTokenListHead=NULL,
		*pTokenList;

	// Let's get the information for each kanji used in this word
	for( pwc=wsWord; *pwc; pwc++ )
	{
		if( *pwc != L'#' )		// All characters should be either '#' or kanji by the time that we get to here
		{
			ppKanjiToUse[nKanjiToUse++] = FindKanji(*pwc, ppKanji, nKanji);
		}
	}

	// Match readings for first kanji
	// Allocate token for each reading that finds a match at the beginning of the kana segment
	// Move to the next kanji
	// Search for reading that matches starting after the first kanji reading(s)
	for( i=0; i<nKanjiToUse; i++ )
	{
		if( i==0 )
		{
			// Find all matching readings that align with the beginning of the kana segment
			for(pReading=ppKanjiToUse[i]->pReading; pReading; pReading)
			{
				if( (pwc = wcsstr(wsKana, pReading->wsReading)) == wsKana ) 
				{
					if( pTokenListHead == NULL )
					{
						pTokenListHead = pTokenList = (KRTOKENLIST *)calloc(1, sizeof(KRTOKENLIST));
					}
					else
					{
						pTokenList->pNext = (KRTOKENLIST *)calloc(1, sizeof(KRTOKENLIST));
						pTokenList = pTokenList->pNext;
					}
					pTokenList->pKRToken = pToken = (KRTOKEN *)calloc(1, sizeof(KRTOKEN));
					pToken->pKanji = ppKanjiToUse[i];
					pToken->iKanjiIndex = i;
					pToken->pReading = pReading;
					pToken->pwcKanaStart = pwc;
					pToken->nReadingLen = wcslen(pReading->wsReading);

				}
			}
		}
		else
		{
			for( pTokenList = pTokenListHead; pTokenList; pTokenList=pTokenList->pNext )
			{
				pToken = pTokenList->pKRToken;
				for( pToken=pTokenList->pKRToken; pToken; pToken=pToken->pNext )
				{
					if( pToken->pNext == NULL && 
						pToken->iKanjiIndex == i-1 )
					{
						for(pReading=ppKanjiToUse[i]->pReading; pReading; pReading)
						{
							wchar_t *pwcStart = pToken->pwcKanaStart + pToken->nReadingLen
							if( wcsstr(pwcStart, pReading->wsReading) == pwcStart )
							{
									
							}
						}
					}
				}
			}

				if( (pwc = wcsstr(wsKana, pReading->wsReading)) == wsKana )
				{
					if( pTokenListHead == NULL )
					{
						pTokenListHead = pTokenList = (KRTOKENLIST *)calloc(1, sizeof(KRTOKENLIST));
					}
					else
					{
						pTokenList->pNext = (KRTOKENLIST *)calloc(1, sizeof(KRTOKENLIST));
						pTokenList = pTokenList->pNext;
					}
					pTokenList->pKRToken = pToken = (KRTOKEN *)calloc(1, sizeof(KRTOKEN));
					pToken->pKanji = ppKanjiToUse[i];
					pToken->i = i;
					pToken->pReading = pReading;
					pToken->pwcKanaStart = pwc;
					pToken->nReadingLen = wcslen(pReading->wsReading);
				}
			}
		}
	}


}
#endif

/*FUNCTION_HEADER**********************
 * NAME:	;MatchWordsWithKanji
 * DESC: 	This function attempts to match a hiragana reading from each kanji
			character with the kanji and hiragana of a dictionary word.  From this
			alignment we should be able to determine the context of each kanji character
			required for the various readings.  In other words we're trying to come up
			with a way to map each kanji character to the appropriate hiragana translation.
 * IN:		pDict - pointer to a list of japanese words.  Each word may contain
				hiragana and kanji.  The members wsWordAlign and wsKanaAlign contain
				the best possible alignment of the hiragana only with the word.
			nEntries - number of words in pDict.
			ppKanji - array of pointers to kanji character dictionary.  Each kanji 
				character contains a list of possible hiragana readings.
			nKanji - number of kanji characters in ppKanji
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void MatchWordsWithKanji(JPDICTPTR pDict, unsigned int nEntries, KANJIDIC **ppKanji, unsigned int nKanji)
{
	wchar_t
		*pwcStart,		// pointer to start of string segment containing kanji
		*pwcEnd,		// pointer to end of string segment containing kanji
		wsKanjiSeg[128],// string segment containing only kanji
		wsKanaSeg[128],	// matching string segment containing only kana
		*pwcKanaStart;	// pointer to start of string segment containing kana
	char
		bIsKanji;		// Flag to indicate that Kanji characters were found in this string
	unsigned int
		i,
		nStrKanji;		// Number of Kanji in this string

	for( i=0; i<nEntries; i++ )		// Search all dictionary entries 
	{
		if( pDict[i].wsWordAlign == NULL || pDict[i].wsKanaAlign == NULL )
			continue;

		// Don't mess with this word if it has any characters that are not 
		// # or Hiragana or Kanji
		for( pwcStart=pDict[i].wsWordAlign,bIsKanji=0; *pwcStart; pwcStart++ )
		{
			if( !(*pwcStart == L'#' || IsEucKanji(*pwcStart) || IsEucHira(*pwcStart)) )
			{
				bIsKanji=1;
				break;
			}
		}
		if( bIsKanji )
			continue;

		// Check each character in the word to see if it is a Kanji
		for( pwcStart=pDict[i].wsWordAlign; *pwcStart; pwcStart++ )		
		{
			bIsKanji = 0;
			if( *pwcStart == L'#' || (bIsKanji = IsEucKanji(*pwcStart)) )	// Find any Kanji in word
			{
				nStrKanji = bIsKanji ? 1 : 0;
				for( pwcEnd=pwcStart+1; *pwcEnd; pwcEnd++ )	// Find the end of a potential series of Kanji
				{
					bIsKanji = 0;
					if( *pwcEnd != L'#' && !(bIsKanji = IsEucKanji(*pwcEnd)) )
						break;
					nStrKanji += bIsKanji ? 1 : 0;
				}
				if( nStrKanji == 0 )
				{
					pwcStart = pwcEnd;		// No kanji just # symbols
					continue;
				}
				
				// Copy the matching Kanji and Kana segments
				wcsncpy(wsKanjiSeg, pwcStart, (pwcEnd-pwcStart));
				wsKanjiSeg[(pwcEnd-pwcStart)] = 0x00;		// Null terminate
				pwcKanaStart = pDict[i].wsKanaAlign + (pwcStart-pDict[i].wsWordAlign);
				wcsncpy(wsKanaSeg, pwcKanaStart, (pwcEnd-pwcStart));
				wsKanaSeg[(pwcEnd-pwcStart)] = 0x00;		// Null terminate


				///AlignKanjiWithReadings(wsKanjiSeg, wsKanaSeg, ppKanji, nKanji);

				if( *pwcEnd == 0x00 )
					break;
				else
					pwcStart = pwcEnd;
			}
		}
	}

}

void RemoveSpacesW(wchar_t *wsString)
{
	wchar_t *pwc1, *pwc2;

	for(pwc1=pwc2=wsString; *pwc1; pwc1++)	// For all characters in wsString
	{
		if( *pwc1 != L' ' )		// Skip spaces
		{
			*pwc2 = *pwc1;		// Copy characters
			pwc2++;
		}
	}
	*pwc2 = 0x00;	// NULL terminate
}

/*FUNCTION_HEADER**********************
 * NAME:	;AlignKanjiAndKana
 * DESC: 	Align the kana characters as closely as possible with the 
			Word containing kanji
 * IN:		pDict - pointer to dictionary word array
			nEntries - number of words in pDict
 * OUT:		
 * RETURN:	none
 * NOTES:	
 *END_HEADER***************************/
void AlignKanjiAndKana(JPDICTPTR pDict, unsigned int nEntries)
{
	wchar_t
		wc,
		wsUtfWord[128],
		wsUtfKana[128],
		**psKanji,
		**psKana;
	int
		nKanji,
		nKana,
		iInsertions,
		iDeletions,
		iCorrect,
		iSubstitutions;
	unsigned int
		i, j;
	FILE *fp;

	if( (fp = fopen("AlignKanjiAndKana.utxt", "wb")) == NULL )
		return;

	psKanji = (wchar_t **)calloc(100, sizeof(wchar_t *));
	psKana = (wchar_t **)calloc(100, sizeof(wchar_t *));
	for(i=0; i<100; i++)
	{
		psKanji[i] = (wchar_t *)calloc(5, sizeof(wchar_t));
		psKana[i] = (wchar_t *)calloc(5, sizeof(wchar_t));
	}

	wc = 0xfeff;									// Unicode header
	fwrite(&wc, sizeof(wchar_t), 1, fp);
	for( i=0; i<nEntries; i++ )						// Search all dictionary entries 
	{
		if( pDict[i].wsWord == NULL || pDict[i].wsKana == NULL )
		{
			continue;
		}

		for(j=0; j < wcslen(pDict[i].wsWord); j++)
		{
			psKanji[j][0] = pDict[i].wsWord[j];
			psKanji[j][1] = 0x00;
		}
		nKanji = j;

		for(j=0; j < wcslen(pDict[i].wsKana); j++)
		{
			psKana[j][0] = pDict[i].wsKana[j];
			psKana[j][1] = 0x00;
		}
		nKana = j;

		MatchStringsW(psKanji, nKanji, psKana, nKana, 
                  &iInsertions, &iDeletions, &iCorrect, &iSubstitutions,
                  &pDict[i].wsWordAlign, &pDict[i].wsKanaAlign);

		RemoveSpacesW(pDict[i].wsWordAlign);
		RemoveSpacesW(pDict[i].wsKanaAlign);
		
		// Convert to unicode for printout
		wcscpy(wsUtfWord, pDict[i].wsWordAlign);
		wcscpy(wsUtfKana, pDict[i].wsKanaAlign);
		e2u(wsUtfWord);						
		e2u(wsUtfKana);
		fwprintf(fp, L"[%d]\t%s\n\t%s\n\n", i, wsUtfWord, wsUtfKana);

	}

	for(i=0; i<100; i++)
	{
		free(psKanji[i]);
		free(psKana[i]);
	}
	free(psKanji);
	free(psKana);


	fclose(fp);
}


/*FUNCTION_HEADER**********************
 * NAME:	;TestLTree
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
void TestLTree(BRANCHPTR pLTree)
{
	char 
		sFile[_MAX_PATH],
		sRom[256];
	unsigned char
		bUnicode,
		bSwapBytes;
	wchar_t 
		*pwc,
		*wsData,
		*wsKana,
		wsBuf[256];
	FILE 
		*fp;

	printf("Enter a file name: ");
	while( gets(sFile) )		// Test the newly built LTree
	{
		if( !sFile[0] )			// get out if nothing was entered
			break;

		if( (fp = fopen(sFile, "rb")) == NULL )
		{
			printf("File: %s could not be opened\n", sFile);
			printf("\nFile: ");
			continue;
		}

		bUnicode = IsUnicode(fp);
		bSwapBytes = IsBytesSwapped(fp);

		while( fgetws(wsBuf, 255, fp) )
		{
			if( (pwc = wcschr(wsBuf, L'\n')) != NULL )
				*pwc = 0x00;

			if( wsBuf[0] == 0xfeff )
				wcscpy(wsBuf, &wsBuf[1]);

			if( bSwapBytes )
			{
				unsigned char c1, c2;
				for( pwc=wsBuf; *pwc; pwc++ )
				{
					c1 = ((*pwc) & 0xff00) >> 8;
					c2 = ((*pwc) & 0x00ff);
					*pwc = (c2 << 8) + c1;
				}
			}

			if( bUnicode )
			{
				for( pwc=wsBuf; *pwc; pwc++ )
				{
					if( (UNI_HIRAGANA_MIN <= *pwc && *pwc <= UNI_HIRAGANA_MAX) ||
						(UNI_KATAKANA_MIN <= *pwc && *pwc <= UNI_KATAKANA_MAX) ||
						(UNI_STDCJK_MIN <= *pwc && *pwc <= UNI_STDCJK_MAX) )
					{
						*pwc = Convert(*pwc, eUtf16, eEuc);
					}
				}
			}

			if( (wsData = FindWordInLTree(wsBuf, pLTree)) != NULL )
			{
				if( (wsKana = wcschr(wsData, L',')) )
				{
					wsKana++;
					Kana2RomW(wsKana, sRom, '(', FALSE);
					printf("Found %s\n", sRom);
				}
			}
			else
			{
				sRom[0] = 0x00;
				Kana2RomW(wsBuf, sRom, '(', FALSE);
				if( sRom[0] )
					printf("Found %s\n", sRom);
				else
					printf("Not found\n");
			}
		}

		fclose(fp);
		
		printf("\nFile: ");
	}

}

/*FUNCTION_HEADER**********************
 * NAME:	;WriteDictionaryWordList
 * DESC: 	Output a list of Japanese words from the dictionary.
			The output include the word and the kana.
 * IN:		sWordList - output filename
			pDict - pointer to the dictionary
			nEntries - number of entries in the dictionary
 * OUT:		output is written to the file 
 * RETURN:	none
 * NOTES:	
 *END_HEADER***************************/
void WriteDictionaryWordList(char *sWordList, JPDICTPTR pDict, unsigned int nEntries, bool bWriteBitFlag)
{
	wchar_t 
		wc,
		wsWord[128],
		wsKana[128];
	unsigned int 
		i;
	FILE 
		*fp;

	if( (fp = fopen(sWordList, "wb")) == NULL )
		return;

	// Write list in Unicode
	wc = 0xfeff;
	fwrite(&wc, sizeof(wchar_t), 1, fp);

	for(i=0; i<nEntries; i++)
	{
		if( pDict[i].wsKana == NULL )
			continue;

		wcscpy(wsWord, pDict[i].wsWord);
		e2u(wsWord);
		wcscpy(wsKana, pDict[i].wsKana);
		e2u(wsKana);

		if( bWriteBitFlag )
			fwprintf(fp, L"%s;%s;0x%x\n", wsWord, wsKana, pDict[i].iBitFlags);
		else
			fwprintf(fp, L"%s;%s;(%S)\n", wsWord, wsKana, pDict[i].sWordTypes);

	}
}

/*FUNCTION_HEADER**********************
 * NAME:	;ContainsKnownKanji
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int ContainsKnownKanji(wchar_t *wsWord, KANJIDIC **ppKanji, unsigned int nKanji )
{
	wchar_t *pwc;

	for( pwc=wsWord; pwc && *pwc; pwc++ )
	{
		if( UNI_HIRAGANA_MIN <= *pwc && *pwc <= UNI_HIRAGANA_MAX )			// character is hiragana
			continue;
		else if( UNI_KATAKANA_MIN <= *pwc && *pwc <= UNI_KATAKANA_MAX )		// character is katakana
			continue;
		else if( UNI_STDCJK_MIN <= *pwc && *pwc <= UNI_STDCJK_MAX &&		// character is valid kanji
				 FindKanji(*pwc, ppKanji, nKanji) )
			continue;
		else
			return 0;

	}
	return 1;
}

/*FUNCTION_HEADER**********************
 * NAME:	;WritePrunedDictionaryWordList
 * DESC: 	Output a list of Japanese words from the dictionary.
			The output include the word and the kana.
 * IN:		sWordList - output filename
			pDict - pointer to the dictionary
			nEntries - number of entries in the dictionary
 * OUT:		output is written to the file 
 * RETURN:	none
 * NOTES:	
 *END_HEADER***************************/
void WritePrunedDictionaryWordList(char *sWordList, JPDICTPTR pDict, unsigned int nEntries, 
								   KANJIDIC **ppKanji, unsigned int nKanji, bool bWriteBitFlag)
{
	wchar_t 
		wc,
		wsWord[128],
		wsKana[128];
	unsigned int 
		i,
		nWords=0;
	FILE 
		*fp;

	if( (fp = fopen(sWordList, "wb")) == NULL )
		return;

	// Write list in Unicode
	wc = 0xfeff;
	fwrite(&wc, sizeof(wchar_t), 1, fp);

	for(i=0; i<nEntries; i++)
	{
		if( pDict[i].wsKana == NULL )
			continue;

		wcscpy(wsWord, pDict[i].wsWord);
		e2u(wsWord);
		wcscpy(wsKana, pDict[i].wsKana);
		e2u(wsKana);

		if( ContainsKnownKanji(wsWord, ppKanji, nKanji) )
		{
			nWords++;
			if( bWriteBitFlag )
				fwprintf(fp, L"%s;%s;0x%x\n", wsWord, wsKana, pDict[i].iBitFlags);
			else
				fwprintf(fp, L"%s;%s;(%S)\n", wsWord, wsKana, pDict[i].sWordTypes);
		}

	}
	printf("Words written to pruned list: %d\n", nWords);
	fclose(fp);
}


/*FUNCTION_HEADER**********************
 * NAME:	;main
 * DESC: 	
 * IN:		
 * OUT:		
 * RETURN:	
 * NOTES:	
 *END_HEADER***************************/
int main(int argc, char **argv)
{
	char 
		*sInFile=NULL,
		*sOutFile=NULL,
		*sKanjiDic=NULL,
		*sWordList=NULL,
		*sKanjiList=NULL;
	bool 
		bTestDict=false,
		bBigEndian=false,
		bKoreanDict=false,
		bPruneToKanjiList=false;
	unsigned int
		i, 
		nEntries, 
		nKanji;
	JPDICTPTR 
		pDict=NULL;
	BRANCHPTR 
		pLTree=NULL;
	KANJIDIC 
		**ppKanji=NULL;

	// Parse the command line arguments to get the input and output filenames
	for( i=1; i<(unsigned int)argc; i++)
	{
		if( argv[i][0] == '-' )
		{
			switch( argv[i][1] )
			{
				case 'B':
				case 'b':
					bBigEndian = true;
					break;

				case 'c':
					sInFile = argv[++i];
					sOutFile = argv[++i];
					ConvertKanjiToList(sInFile, sOutFile);
					exit(0);

				case 'I':
				case 'i':	
					sInFile = argv[++i];		// Get the input (raw text dictionary) filename
					break;

				case 'K':
					bKoreanDict = true;
					break;

				case 'k':
					sKanjiDic = argv[++i];		// Get the input kanji dictionary
					break;

				case 'l':
					sKanjiList = argv[++i];
					break;

				case 'O':
				case 'o':	
					sOutFile = argv[++i];		// Get the output (file system letter tree) filename
					break;

				case 'p':
					sKanjiList = argv[++i];
					bPruneToKanjiList = true;
					break;

				case 'W':
				case 'w':
					sWordList = argv[++i];		// Output list of words from dictionary
					break;

				case 'T':
				case 't':
					bTestDict = true;
					break;

			}
		}
	}

	if( sInFile == NULL )
	{
		printf("Usage: %s -i <infile> -o <outfile> [-t] [-b]\n", argv[0]);
		printf("\t<infile> - input raw text dictionary file\n");
		printf("\t<output> - output file system letter-tree format dictionary\n");
		printf("\t-t - test dictionary\n");
		printf("\t-b - output dictionary in big-endian format\n");
		exit(0);
	}

	// Read Kanji dictionary
	if( sKanjiDic )
	{
		ppKanji = ReadKanjiDic(sKanjiDic, &nKanji);
		if( bKoreanDict && ppKanji )
		{
			WriteHanjaTable("..\\..\\Korean\\KoreanHanjaReadings.utxt", ppKanji, nKanji);
			exit(0);
		}
	}

	// Read the raw text dictionary file
	printf("Reading: %s ...", sInFile);
	if( (pDict = LoadDictionary(sInFile, &nEntries)) == NULL )
	{
		exit(0);
	}

#if 0
	AlignKanjiAndKana(pDict, nEntries);

	// Match words with Kanji
	if( ppKanji && pDict )
	{
		MatchWordsWithKanji(pDict, nEntries, ppKanji, nKanji);
		WriteKanjiTable(".\\KanjiTable.utxt", ppKanji, nKanji);
		FreeKanjiDic(ppKanji, nKanji);
	}
#endif

	if( bPruneToKanjiList && sKanjiList && sWordList )
	{
		ppKanji = (KANJIDIC **)ReadKanjiList(sKanjiList, &nKanji);
		WritePrunedDictionaryWordList(sWordList, pDict, nEntries, ppKanji, nKanji, 0);
	}
	else if( ppKanji && sKanjiList )
	{
		WriteKanjiList(sKanjiList, ppKanji, nKanji);
	}
	else if( sWordList )
	{
		WriteDictionaryWordList(sWordList, pDict, nEntries, 0);
	}

	// Sort the in-memory dictionary
//	SortDictionary(pDict, nEntries);

	printf(" Done. %d entries read\n", nEntries);

#if 0
	// Build an in-memory letter-tree version of the dictionary
	printf("Building LTree in memory.\n");
	if( (pLTree = BuildLetterTree(pDict, nEntries)) != NULL )
	{
		if( bTestDict )
		{
			TestLTree(pLTree);
		}
	}

	if( pLTree )
		FreeLTree(pLTree);
#endif

	FreeDictionary(pDict, nEntries);

}

