/*C_HEADER_FILE****************************************************************
FILE			:	S1C33SpeakerTts.c
DESC			:	wave playing functions for Epson S1C33 using vox software
TABS			:	4
OWNER			:	fonix
DATE CREATED:	

(C) Copyright 2005 All rights reserved.

*END_HEADER*******************************************************************/
#include "S1C33Audio.h"
#include "S1C33Memory.h"
#include "S1C33ManageInterrupts.h"

#include "Speak.h"

#define FNX_AUDIO_OUT_SOURCE_10bit       5

//#define CHANNEL SPK_9_MONO	//use only with SND33 library & SpkIntr4 interrupt
#define CHANNEL SPK_10_MONO	//use only with SND33 library & SpkIntr0 interrupt
//#define CHANNEL SPK_15_MONO		//use only with SND33 library & SpkIntr1 interrupt
//#define CHANNEL SPK_15_STEREO	//use only with SND33 library & SpkIntr2 interrupt
//#define CHANNEL 1				//use only with VOX33 library & SpkIntr0 interrupt
#define CHANNEL_PIN CHANNEL+2

#if !defined SPK_10_MONO 
	// linking with VOX33 Software
	// input to SpkAppend unsigned 10 bit mono -> output unsigned 10 bit mono
	// output pin by Channel: 0 - TM0, 1 - TM1, 2 - TM2
	#define OFFSET 0x200
	#define SPEAKER_UPPER_LIMIT 0x3FE
	#define REQUIRED_SHIFT_FROM_16B 6
	#define ROUND_UP_16B 8
	#define REQUIRED_SHIFT_FROM_10B 0
	#define ROUND_UP_10B 0
#else
	//linking with SND33 Software
	#if CHANNEL == SPK_10_MONO 
		// input to SpkAppend unsigned 10 bit mono -> output unsigned 10 bit mono
		// output pin: TM1
		#define OFFSET 0x200
		#define SPEAKER_UPPER_LIMIT 0x3FE
		#define REQUIRED_SHIFT_FROM_16B 6
		#define ROUND_UP_16B 32
//		#define REQUIRED_SHIFT_FROM_16B 5	// this is technically wrong and could cause clipping but does increase SNR to mask hardware noise
//		#define ROUND_UP_16B 16
		#define REQUIRED_SHIFT_FROM_10B 0
		#define ROUND_UP_10B 0
	#else
		// CHANNEL == SPK_15_MONO
		// input to SpkAppend unsigned 16 bit mono -> output unsigned 15 bit mono
		// output pins: TM1 & TM2
		// CHANNEL == SPK_15_STEREO
		// input to SpkAppend unsigned 16 bit stereo -> output unsigned 15 bit stereo
		// output pins: TM1, TM2, TM3, & TM4
		// CHANNEL == SPK_9_MONO
		// input to SpkAppend unsigned 16 bit mono -> output unsigned 9 bit mono
		// output pin: TM1
		#define OFFSET 0x8000
		#define SPEAKER_UPPER_LIMIT 0xFFFF
		#define REQUIRED_SHIFT_FROM_16B 0
		#define ROUND_UP_16B 0
		#define REQUIRED_SHIFT_FROM_10B 0
		#define ROUND_UP_10B 0
	#endif
#endif

typedef struct tagSpkLList
{
	short * pData;
	struct tagSpkLList * pNext;
	int	UsedFlag;
} SpkLList;

SpkLList * gSpkLLHead = NULL;
SpkLList * gSpkLLTail = NULL;

unsigned char *SpkParams = NULL;
short *SpkBuffer;
volatile int spkrDone = 1;
volatile int gQueueDifference;
volatile int gAudioOutHalted;
int gSource;

void SpkQueueDone(unsigned char *SpkParams, short *Buffer, int Length);
void SpeakerFinished(unsigned char *SpkParams);
SpkLList * NewSpkLListMember( short * pData );
int FormatAudioData16bTo10b(short* Src, short* Dst, int Length, unsigned int source);
void AudioOutInitalize( int * samplesEachPlay, int sampleRate, int source );
void AudioOutHalt();
void AudioOutClearHalt();

/**************************************************************************************
//	SpkLList * NewSpkLListMember( short * pData )
//  audio output link list management 
//		add new member to link list
//**************************************************************************************/
SpkLList * NewSpkLListMember( short * pData )
{
	SpkLList * llist = S1C33Malloc( sizeof( SpkLList ) );
	if( llist )
	{
		llist->pData = pData;
		llist->pNext = NULL;
		llist->UsedFlag = 0;
	}
	return llist;
}

/**************************************************************************************
//int CopyAndFormatAudioData(short* Src, short* Dst, int Length, unsigned int source)
//	move the audio data to a buffer owned by the audio driver and format the data according
//  to the input data and the resolution of the speaker channel
//		source:							source data:
//			DECtalk TTS						16b	0 centered
//				
//**************************************************************************************/
int CopyAndFormatAudioData(short* Src, short* Dst, int Length, unsigned int source)
{
	int i;
	int temp;
	short *pDst = Dst;
	short *pSrc = Src;
	int offset;
	
	// this is technically wrong and could cause clipping but does increase SNR to mask hardware noise
	//data is 16bit zero centered
	offset = OFFSET;
	for (i = 0; i < Length;i++)
	{
		temp = (int)*pSrc++;
#if REQUIRED_SHIFT_FROM_16B == 0
		//temp += ROUND_UP_16B;		//don't need to round up
		temp = temp << 2;	//shift up to have more volume
#else
//		temp += ROUND_UP_16B - 2^3;						//change the bit that is rounded
//		temp = temp >> (REQUIRED_SHIFT_FROM_16B - 2);	//don't shift down as much so we have higher volume
		temp += ROUND_UP_16B;
		temp = temp >> (REQUIRED_SHIFT_FROM_16B);
#endif
		temp += OFFSET;
		if( temp > SPEAKER_UPPER_LIMIT)
			temp = SPEAKER_UPPER_LIMIT;
		else if( temp < 0 )
			temp = 0;
		*pDst = (short)temp;
		pDst++;
	}
	return 0;
}

/**************************************************************************************
//	void SpkQueueDone(unsigned char *SpkParams, short *Buffer, int Length)
//  call back function for Vox software 
//	
//	used when buffer in the output que is finished
//**************************************************************************************/
void SpkQueueDone(unsigned char *SpkParams, short *Buffer, int Length)
{
	SpkLList * SpkLListTemp = gSpkLLHead;

	gQueueDifference--;

	//find the item in the linked list that contains the Buffer that just finished
	while( SpkLListTemp && SpkLListTemp->pData != Buffer )
	{
		SpkLListTemp = SpkLListTemp->pNext;
	}
	//mark Buffer as used in the link list
	if( SpkLListTemp )
	{
		SpkLListTemp->UsedFlag = 1;
	}
}

/**************************************************************************************
//	void SpeakerFinished(unsigned char *SpkParams)
//  call back function for Vox software 
//	
//	used when output que is emptied
//**************************************************************************************/
void SpeakerFinished(unsigned char *SpkParams)
{
	spkrDone = 1;
	SpkHalt(SpkParams);		// stop audio out
}

/**************************************************************************************
//	void AudioOutCreate()
//	contains any speaker initialization that occurs only once during a program.
//**************************************************************************************/
void AudioOutCreate()
{
	unsigned char *P2_IO_DATA = (char *) 0x000402d9;	// I/O Data
	unsigned char *P2_IO_CONTROL = (char *) 0x000402da; //I/O direction control
	unsigned char PinSetting;

#if !defined SPK_10_MONO
	PinSetting = 0x01 << CHANNEL_PIN;
#else 
	if( CHANNEL == SPK_10_MONO || CHANNEL == SPK_9_MONO )
		PinSetting = 0x08;
	else if( CHANNEL == SPK_15_MONO )
		PinSetting = 0x18;
	else	//CHANNEL == SPK_15_STEREO
		PinSetting = 0x78;
#endif
	*P2_IO_CONTROL = *P2_IO_CONTROL | PinSetting;	//if only one bit is set assembly uses bset
	*P2_IO_DATA = *P2_IO_DATA & (~PinSetting);		//if only one bit is cleared assembly uses bclr
	return;
}

/**************************************************************************************
//	void AudioOutInit( int * samplesEachPlay, int sampleRate, int source, int bytesPerSample, int format )
//	contains all speaker initialization that must occur each time the 
//	speaker is open
//**************************************************************************************/
void AudioOutInit( int * samplesEachPlay, int sampleRate, int source, int bytesPerSample, int format )
{
	gSpkLLHead = gSpkLLTail = NULL;

	spkrDone = 0;
	gQueueDifference = 0;
	gAudioOutHalted = 0;
	gSource = source;

	SpkInit();
	SpkSoftening(1);
	SpkParams = SpkOpen(CHANNEL, SPK_SAMPLING(CPU_CLK_RATE, sampleRate));
	SpkOnDone(SpkParams, SpkQueueDone);
	SpkOnEmpty(SpkParams, SpeakerFinished);

	*samplesEachPlay = TTS_BUFFER_SIZE;
}

/**************************************************************************************
//	int AudioOutPlay( short * pData, int samples )
//	is called each time there is additional data to send to the speaker
//**************************************************************************************/
int AudioOutPlay( short * pData, int samples )
{
	short	*pBuffer;
	SpkLList * SpkLListNext = NULL;
	int IntStatus;

	// Critical section - if interrupt stops the audio out here you never free this buffer
	IntStatus = BlockInterrupts( );

	//clean up segment that have already been played
	while( gSpkLLHead && gSpkLLHead->pData && gSpkLLHead->UsedFlag )
	{
		S1C33Free( gSpkLLHead->pData );
		SpkLListNext = gSpkLLHead->pNext;
		S1C33Free( gSpkLLHead );
		gSpkLLHead = SpkLListNext;
	}

	pBuffer = (short *)S1C33Malloc( samples*sizeof(short) );
	if (pBuffer)
	{
		if( !gSpkLLHead )
		{
			gSpkLLHead = gSpkLLTail = NewSpkLListMember( pBuffer );
			if( !gSpkLLHead )
			{
				S1C33Free( pBuffer );
				return 0;
			}
				
		}
		else
		{
			gSpkLLTail->pNext = NewSpkLListMember( pBuffer );
			if( !gSpkLLTail->pNext )
			{
				S1C33Free( pBuffer );
				return 0;
			}
			gSpkLLTail = gSpkLLTail->pNext;
		}

		UnBlockInterrupts( IntStatus );
		if( CopyAndFormatAudioData( pData, pBuffer, samples, gSource ) )
			return 0;

		if(( SpkRoom(SpkParams) > 0 ) && ( gAudioOutHalted != 1 ))
		{
			gQueueDifference++;
			SpkAppend(SpkParams, pBuffer, samples); 
			spkrDone = 0;
			
			if( !SpkIsRunning(SpkParams) )
			{
				SpkStart(SpkParams);		/* start speaking */
			}
			while( 1 )
			{
				// if buffered more than 5 segments of TTS don't exit wait loop (will limit memory required)
				if( gQueueDifference < 5 )
					break;
				// if output has been halted (premature exit from TTS) exit wait loop
				if( gAudioOutHalted == 1 )
					break;
			}
		}
	}
	else
		UnBlockInterrupts( IntStatus );
	return 0;
}

/**************************************************************************************
//	void AudioOutQuit()
//	called to close the speaker & free all memory used for temp data storage
//**************************************************************************************/
void AudioOutQuit()
{
	SpkLList * llist = NULL;
	SpkLList * llistNext = NULL;

	if( SpkParams && gSpkLLHead )		//make sure the channel has been open and their is data to play
	{
		if( gAudioOutHalted != 1 )
		{
			if( !SpkIsRunning(SpkParams) )
			{
				SpkStart(SpkParams);		/* start speaking */
			}
			while ( !spkrDone )	{}			// wait for all data to be played, spkrDone set by empty callback
		}
		SpkClose(SpkParams);
	}

	//delete audio ouput linked list
	llist = gSpkLLHead;
	while( llist )
	{
		gQueueDifference--;
		S1C33Free( llist->pData );
		llistNext = llist->pNext;
		S1C33Free( llist );
		llist = llistNext;
	}

	gSpkLLHead = gSpkLLTail = NULL;
}

/**************************************************************************************
//	void AudioOutDestroy()
//	called only once at the end of program and closes any portion of the speaker not
//	created by AudioOutCreate
//**************************************************************************************/
void AudioOutDestroy()
{
	return;
}

/**************************************************************************************
//	void AudioOutInitalize( int * samplesEachPlay, int sampleRate, int source )
//	used to call standard initialization from non-ASR code
//**************************************************************************************/
// used by microDECtalk
void AudioOutInitalize( int * samplesEachPlay, int sampleRate, int source )
{
	AudioOutInit( samplesEachPlay, sampleRate, source, 2, 2);
}

/**************************************************************************************
//	void AudioOutHalt()
//	Stops the speaker and sets a flag to stop speaker before all appended data is played
//	AudioOutQuit is then called to flush all remaining appended data
//**************************************************************************************/
void AudioOutHalt()
{
	if( spkrDone == FALSE )
	{
		SpkHalt(SpkParams);		// stop audio out
		spkrDone = TRUE;
	}
	
	gAudioOutHalted = 1;
}

/**************************************************************************************
//	void AudioOutClearHalt()
//	resets the halt flag - also done in AudioOutInit
//**************************************************************************************/
void AudioOutClearHalt()
{
	gAudioOutHalted = 0;
}
