// Mikropuhe so-library cpp-wrapper and MINIMAL test program
// Petteri 1/2004
//
// NOTE: compile with: gcc -o test -ldl -lpthread libmplinux_testmini.cpp
//

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <dlfcn.h>
#include <unistd.h>

#include "mpwrfile.h"



/////////////////////////////////////////////////////////////////////////////
//  CMPChannel - The wrapper class for one channel

// All functions must be called from only on thread or calling must be synchronized other way
// Speech parameter are maintained by instance - instances are completely isolated.
// Internally libmplinux.so creates voice file cache when first InitEx is called.
// The cache is cleaned when its reference count drops to zero
class CMPChannel
{
// Creating & destroying
public:
	CMPChannel()
	{
		m_pLib = 0;
		m_pChannel = 0;
		m_pfnChannelInitEx = 0;
		m_pfnChannelUseSettings = 0;
		m_pfnChannelExit = 0;
		m_pfnChannelSpeakFile = 0;
	}
		

	~CMPChannel()
	{
		FreeAll();
	}

	// Frees the library, synthesizer MUST NOT be active when this is called.
	// (All running SpeakFile-calls must have been completed).
	void FreeAll();

	/*
	*	Loads given library and obtains all needed symbols from it
	*
	*	pSoFileName	-> Full path name for the libmplinux.so
	*	pErr			-> English error message is saved here in case of failure
	*	nErrLen		-> Length of previous
	*
	*	Return value: 0=Ok and pErr="", others=failed and pErr is set
	*/
	int Load( const char *pSoFileName, char *pErr, int nErrLen );


// Functions, all return 0 if successful and negative if error is from Mikropuhe
// Please see mpwrfile.h for more information
public:
	// You MUST call this before anything else
	int InitEx( const char *pKey, const char *pSettingsStr, void *pReserved=0 )
	{
		if ( !m_pfnChannelInitEx )
			return( -1 );

		return( m_pfnChannelInitEx(&m_pChannel, pKey, pSettingsStr, pReserved) );
	}


	int UseSettings( const char *pSettingsStr )
	{
		if ( !m_pfnChannelUseSettings )
			return( -1 );

		return( m_pfnChannelUseSettings(m_pChannel, pSettingsStr) );
	}


	int Exit(char *pSettingsStr, int nSettingsLen)
	{
		int	nRes;

		if ( !m_pfnChannelExit )
			return( -1 );

		nRes = m_pfnChannelExit(m_pChannel, pSettingsStr, nSettingsLen);
		m_pChannel = 0;

		return( nRes );
	}

	int SpeakFile( const char *pText, const char *pFileName, MPINT_SpeakFileParams *prParams )
	{
		if ( !m_pfnChannelSpeakFile )
			return( -1 );

		return( m_pfnChannelSpeakFile(m_pChannel, pText, pFileName, prParams) );
	}


protected:
	void                          *m_pLib;			// dlopen-handle
	void                          *m_pChannel;	// Channel "handle"


	// Function pointers
	MPINT_ChannelInitExType       m_pfnChannelInitEx;
	MPINT_ChannelUseSettingsType  m_pfnChannelUseSettings;
	MPINT_ChannelExitType         m_pfnChannelExit;
	MPINT_ChannelSpeakFileType    m_pfnChannelSpeakFile;
};



void CMPChannel::FreeAll()
{
	// Uninitialize Mikropuhe
	Exit( NULL, 0 );

	if ( m_pLib ) {
		dlclose( m_pLib );
		m_pLib = 0;
	}

	m_pfnChannelInitEx = 0;
	m_pfnChannelUseSettings = 0;
	m_pfnChannelExit = 0;
	m_pfnChannelSpeakFile = 0;
}


int CMPChannel::Load( const char *pSoFileName, char *pErr, int nErrLen )
{
   char	sNameTemp[4096];		// Because dlopen does not take const char *
	int	nIndex;

	*pErr = 0;
	FreeAll();

	memset( sNameTemp, 0, sizeof(sNameTemp) );

	// Load the so
   strncpy( sNameTemp, pSoFileName, sizeof(sNameTemp)-1 );
	m_pLib = dlopen( sNameTemp, RTLD_NOW );
	if ( !m_pLib ) {
		snprintf( pErr, nErrLen, "Loading library %s failed, error message=%s", sNameTemp, dlerror() );
		return( -1 );
	}

	// Get func addresses
	const char	*pFunc;

	if (	(m_pfnChannelInitEx      = (MPINT_ChannelInitExType)      dlsym(m_pLib, pFunc = "MPINT_ChannelInitEx")) == 0  ||
			(m_pfnChannelUseSettings = (MPINT_ChannelUseSettingsType) dlsym(m_pLib, pFunc = "MPINT_ChannelUseSettings")) == 0  ||
			(m_pfnChannelExit        = (MPINT_ChannelExitType)        dlsym(m_pLib, pFunc = "MPINT_ChannelExit")) == 0  ||
			(m_pfnChannelSpeakFile   = (MPINT_ChannelSpeakFileType)   dlsym(m_pLib, pFunc = "MPINT_ChannelSpeakFile")) == 0 ) {

		snprintf( pErr, nErrLen, "Could not get function %s from library %s, error message=%s", pFunc, pSoFileName, dlerror() );
		return( -2 );
	}

	// All ok
	return( 0 );
}

//  CMPChannel - The wrapper class
/////////////////////////////////////////////////////////////////////////////




/*
*	The signal writer
*
*	pData			-> The signal
*	uBytes		-> How many bytes to write
*	pWriteData	-> User data
*
*	Returns: 0=Ok, positive value=error code that will SpeakFile function will return, negative=illegal (reserved for Mikropuhe error values)
*/
static int SyntWrite(const void * /*pData*/, unsigned uBytes, void *pWriteData, void * /*pReserved*/)
{
	printf( "SyntWrite %u bytes\n", uBytes );

	return( 0 );
}



int main(int argc, char* argv[])
{
	char           sFileName[1024];
	char           sErr[1024], *pPos;
	int            nStatus;

	CMPChannel     rMP;		// One instance for main program

	const char		*pLibName = "libmplinux.so";		// Or possibly "libmplinux-arm.so" for Arm processor


	// libmplinux.so from program's folder
	strcpy( sFileName, argv[0] );
	pPos = strrchr(sFileName, '/');
	if ( pPos )
		pPos++;
	else
		pPos = sFileName;

	strcpy( pPos, pLibName );


	if ( rMP.Load(sFileName, sErr, sizeof(sErr)) ) {
		printf( "ERROR:\n%s\n", sErr );
		return( 1 );
	}

	if ( (nStatus = rMP.InitEx(NULL, NULL)) != 0 ) {
		printf( "ERROR:\n%d\n", nStatus );
		return( 1 );
	}

	/*
	*	Speak tagged text to user defined call-back signal writer
	*/

	MPINT_SpeakFileParams	rParams;

	memset( &rParams, 0, sizeof(rParams) );

	rParams.nTags = MPINT_TAGS_SAPI5;

	rParams.nSampleFreq = 22050;	// Here is a little design flaw: Some voices are 22050 (saga, petteri) and some 44100 (all others). Mikropuhe does the conversion on-fly, this takes some processing power and slightly affects voice quality.
	rParams.nBits = 16;
	rParams.nChannels = 1;
	rParams.nWriteWavHeader = 0;

	rParams.pfnWrite = SyntWrite;
	rParams.pWriteData = NULL;		// Add here any data you want to pass to SyntWrite-function

	nStatus = rMP.SpeakFile( "<beep freq=\"400\" length=\"1000\" volume=\"100\"/>Hei, mit kuuluu?", NULL, &rParams );
	if ( nStatus ) {
		printf( "SpeakFile-call failed [%d]\n", nStatus );
		return( 1 );
	}

   printf( "Everything ok, goodbye.\n" );
	return 0;
}
