/* lpca.c -- LPC Autocorrelation routine */

/*
 * (c) 1994 Mark Huckvale
 * Search Systems Ltd
 * 52-54 High Holborn
 * London WC1E 6BT
 * Tel. 071-430 1685
 */

#include <stdio.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"
#include "lpca.h"

int LPCAutocorrelation(x,n,order,ltis,pe)
float *x;		/* windowed input signal */
int n;			/* # samples in x */
int order;		/* predictor order required */
LTIState *ltis;		/* returned predictor coefficients */
double *pe;		/* returned predictor error */
			/* returns 0=OK, 1=zero power, 2=fail */
{
	double	r[MAXORDER+1];		/* autocorrelations */
	double	pc[MAXORDER+1];		/* predictor coefficients */
	double	ai,aj,akk;		/* temporary coefficient values */
	double	sum;			/* accumulator */
	int	i,k;			/* loop counters */
	int	retcode=0;

	/* check */
	if (order > MAXORDER) {
		order = MAXORDER;
		retcode=2;
	}

	/* compute autocorrelations */
	for (i=0;i<=order;i++) {
		sum = 0;
		for (k=0;k<(n-i);k++)
			sum += x[k] * x[k+i];
		r[i] = sum;
	}

	/* compute predictor coefficients */
	if (r[0] == 0) 
		/* no signal ! */
		retcode = 1;
	else {
		*pe = r[0];
		pc[0] = 1.0;
		for (k=1;k<=order;k++) {
			sum = 0;
			for (i=1;i<=k;i++)
				sum -= pc[k-i] * r[i];
			akk = sum/(*pe);
			/* new predictor coefficients */
			pc[k] = akk;
			for (i=1;i<=(k/2);i++) {
				ai = pc[i];
				aj = pc[k-i];
				pc[i] = ai + akk * aj;
				pc[k-i] = aj + akk * ai;
			}
			/* new prediction error */
			*pe = *pe * (1.0 - akk*akk);
			if (*pe <= 0)
				/* negative/zero error ! */
				retcode = 2;
		}
	}

	/* copy into LTI System */
	LTISystemClear(ltis);
	ltis->na = 0;
	ltis->a[0] = 1.0;
	ltis->nb = order;
	for (i=1;i<=order;i++)
		ltis->b[i] = pc[i];

	return(retcode);
}

#ifdef EMO
#define WINDOWSIZE 32
#define NCOEFF 4
#define C1	0.16
#define C2	-0.12
#define C3	0.08
#define C4	-0.04
main()
{
	float	iwin[WINDOWSIZE+1];
	LTIState osys;
	double	res;
	int	i;

	/* make a test signal from recursive filter */
	iwin[1] = 0.75;		/* put in pulse of power sqr(0.75) */
	iwin[2] = -C1 * iwin[1];
	iwin[3] = -C1 * iwin[2] - C2 * iwin[1];
	iwin[4] = -C1 * iwin[3] - C2 * iwin[2] - C3 * iwin[1];
	for (i=5;i<=WINDOWSIZE;i++)
		iwin[i] = -C1 * iwin[i-1] - C2 * iwin[i-2] -
				C3 * iwin[i-3] - C4 * iwin[i-4];

	/* do LPC analysis */
	if (LPCAutocorrelation(iwin+1,WINDOWSIZE,NCOEFF,&osys,&res)==0) {
		printf("Predictor coefficients:\n");
		for (i=1;i<=NCOEFF;i++)
			printf("%d = %g\n",i,osys.b[i]);
		printf("Residual Power=%g\n",res);
	}

}
#endif
