/* itspec -- decode item specifier */

/* version 1 */
/* version 1.1 - December 1986
	- allow %d.%d rather than %d.%02d
*/
/* allow up to MAXDATATYPE numbers */
/* SFS VERSION 2 - with character string return */

/*--------------------------------------------------------------------------*/
/**MAN
.TH ITSPEC SFS3 UCL SFS
.SH NAME
itspec - decode item specification
.SH SYNOPSIS
.nf

int     itspec(string,type,match)
char    *string;        /\* item type specification *\/
int32    *datatype;      /\* returned type *\/
char    **subtype;      /\* returned subtype match *\/

.fi
.SH DESCRIPTION
An item specification is composed of two parts: an item "datatype" selector and an
item "subtype" selector; the routine
.I itspec
decodes an item specification into a "datatype" number and a "subtype" match string.
The "datatype" number can be specified as a digit string or
as a two-character "label"
(SP, LX, etc: lower case is automatically converted).  The "subtype" 
match string can consist of one of three alternatives:
.TP 11
.B ".number"
the number is taken to be the "subtype" number of the item - usually the occurrence
count of the given type in the file.
.TP 11
.B "^string", "^string^"
the string is taken to be a direct history match on the item required.
If no trailing "^" is present, a final "*" is assumed, if the match
string ends in a "^", the expression must match the whole of the 
target expanded history.  Matching facilities are described in histmatch(SFS3).
.TP 11
.B "string"
(datatype part=sfslabel, only)
the entire specification (with the first two letters in capitals) is taken to be
a label code.  This code is looked up in the
labels file(s) to return a history match string which may then be used to locate an item
in the file. See labels(SFS5).
.PP
When the "subtype" part of the item specification is not present, 
.I itspec
returns suitable strings according to the entries in the summary table below:
.PP
.nf

Input String          Example       Datatype  Subtype Match String

<digit(s)>            5             5         "0"
<digit(s)>.           5.            5         "*"
<digit(s)>.<digit(s)> 5.03          5         "03"
<digit(s)>.^string    5^pitch       5         "pitch*"
<digit(s)>.^string^   5^pitch^      5         "pitch"

<label>               LP            6          "0"
<label>.              LP.           6          "*"
<label>.<digit(s)>    LP.05         6          "05"
<label>^<string>      LP^pitch      6          "pitch*"
<label>^<string>^     LP^pitch^     6          "pitch"
<label><string>       LPpitch       6          "PSapply(*rules=pitch*)" (say)

.fi
.SH DIAGNOSTICS
.I itspec
returns 0 on success, 1 on failure.
.SH WARNING
.I itspec
modifies its
.I string
argument when the subtype part of the specification is a history match.
.SH NOTE
There is no need to copy the decoded
.I subtype
string into another character array following a call to
.I itspec.
.SH VERSION/AUTHOR
2.0 - Mark Huckvale.
.SH SEE ALSO
histmatch(SFS3) labels(SFS5)
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include "sfs.h"

extern char	*sfslabel[];

int itspec(spec,it,ty)
char	*spec;		/* option string */
int32	*it;		/* data item number */
char	**ty;		/* data sub-type number */
{
	int	i,num;
	static char	*zero="0";
	static char	*star="*";
	register char	*s=spec;
	char		*p;

	/* get data type part */
	if (isdigit(*s)) {
		/* normal mode */
		num = atoi(s);
		if ((num < 0) || (num > MAXDATATYPE))
			return(1);
		*it = num;
		while (isdigit(*s)) s++;
	}
	else {
		if (*s=='*') {
			*it=0;
			s++;
		}
		else {
			/* label mode */
			num = -1;
			for (i=1;i<=MAXDATATYPE;i++) 
				if ((toupper(s[0])==sfslabel[i][0]) &&
				    (toupper(s[1])==sfslabel[i][1])) {
					num=i;
					break;
				}
			if (num==-1) return(1);
			*it=num;
			s += 2;
		}
	}

	/* get subtype part */
	if (!*s) {
		/* nothing else */
		*ty = zero;
		return(0);
	}
	else if (*s=='.') {
		/* check for just a dot */
		if (!*(++s)) {
			*ty = star;
			return(0);
		}
		/* otherwise should be a number */
		else if (isdigit(*s)) {
			*ty = s;
			return(0);
		}
		/* ??? error */
		return(1);
	}
	else if (*s=='^') {
		/* explicit history match */
		*ty = s++;
		while (*s && (*s != '^')) {
			*(s-1) = *s;		/* shift back one */
			s++;
		}
		if (*s != '^') {
			*(s-1)='*';
			*s='\0';
		}
		else
			*(s-1)='\0';
		return(0);
	}
	else if (isalpha(spec[0])) {
		/* label code expected */
		spec[0]=toupper(spec[0]);
		spec[1]=toupper(spec[1]);
		p = lab_getmatch(spec);
		if (!p) return(1);
		*ty = malloc(strlen(p)+1);
		strcpy(*ty,p);
		return(0);
	}
	return(1);
}

#ifdef IAG
main ()
{
	int	it;
	char	*ty;
	char	line[80];

	for (;;) {
		printf("Enter spec : ");
		fflush(stdout);
		gets(line);
		if (itspec(line,&it,&ty)==0) 
			printf("it=%d ty=%s\n",it,ty);
		else
			printf("error\n");
	}
}
#endif
