/* sbp16dma -- SoundBlaster Compatible 16-bit replay by DMA */

/* Mark Huckvale - University College London */

/* version 1.0 - April 1995 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <dos.h>
#include <conio.h>
#include <go32.h>
#include <dpmi.h>

/* routines for Protected Mode replay of 16-bit data */
int	SB16Init(void);
int	SB16Play(short *buff,int numf,int srate,int nchan,int nbits);
void	SB16Close(void);

/* defines for SoundBlaster */
#define SB_DSP_WRITEMIXER	0x04
#define SB_DSP_READMIXER	0x05
#define SB_DSP_RESET		0x06
#define SB_DSP_READ_DATA	0x0A
#define SB_DSP_WRITE_DATA	0x0C
#define SB_DSP_WRITE_STATUS	0x0C
#define SB_DSP_DATA_AVAIL	0x0E
#define SB_DSP_DATA_AVAIL16	0x0F

/* DSP Commands */
#define SB_DMA_16_BIT_DAC	0xB6
#define SB_TIME_16BIT		0x41
#define SB_SPEAKER_ON		0xD1
#define SB_SPEAKER_OFF		0xD3
#define SB_PAUSE		0xD5

/* Card parameters */
unsigned int	sb_ioaddr;
unsigned int	sb_irq;
unsigned int	sb_dma16;
unsigned int	sb_type;

/* write a DSP command to SoundBlaster board */
#define sb_writedsp(x) {						\
	while (inportb(sb_ioaddr + SB_DSP_WRITE_STATUS) & 0x80) /* wait */;	\
	outportb(sb_ioaddr + SB_DSP_WRITE_DATA, (x));			\
}

/* Defines for 16-bit DMA Controller IO addresses */
#define SB_DMA16_BASE		(0xC0 + 4*(sb_dma16-4))
#define SB_DMA16_COUNT		(0xC2 + 4*(sb_dma16-4))
#define SB_DMA16_MASK		(0xD4)
#define SB_DMA16_MODE		(0xD6)
#define SB_DMA16_FF		(0xD8)
#define SB_DMA16_STOP_MASK	(sb_dma16 - 4 + 4)
#define SB_DMA16_START_MASK	(sb_dma16 - 4 + 0)
#define SB_DMA16_OUT_MODE	(sb_dma16 - 4 + 0x58)
static unsigned SB_DMA16_PAGE[8]={0x87,0x83,0x81,0x82,0,0x8B,0x89,0x8A};

/*  GO32 DPMI structures for accessing DOS memory. */
static _go32_dpmi_seginfo dosmem;   /* DOS (conventional) memory buffer */

/* Is a sound currently playing or recorded ? */
volatile int		sb_dma_active;

/* Conventional memory buffers for DMA. */
static char		*sb_buf[2];
static unsigned int	sb_buflen[2];

/* store current settings */
static int	sb_sample_rate=8000;
static int	sb_stereo=0;
static int	sb_nbits;

/* DMA chunk size, in bytes, must be power of 2 */
#define DMA_CHUNK (0x4000)	/* NOTE: I have found that small values
					may be allocated above 640k, and
					that this leads to problems */

/* read a value returned from DSP */
static int sb_read_dsp(int iobase)
{
	int	i;

	for (i=0;i<10000;i++)
	        if (inportb(iobase + SB_DSP_DATA_AVAIL) & 0x080)
			break;
	return(inportb(iobase+SB_DSP_READ_DATA));
}

/* reset SoundBlaster card */
static void sb_reset(void)
{
	int j;

	outportb(sb_ioaddr + SB_DSP_RESET,1);

	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
    
	outportb(sb_ioaddr + SB_DSP_RESET, 0);
	for (j=0;j<100;j++)
		if (sb_read_dsp(sb_ioaddr) == 0xAA)
			 break;
}

#define SB_NUM_BASES	6
static int sbbase[SB_NUM_BASES]={0x210,0x220,0x230,0x240,0x250,0x260};

/* try and find card if port address not given */
static int sb_findcard(void)
{
	int i,j;

	/* search possible base addresses */
	for (i=0;i<SB_NUM_BASES;i++) {

		/* send a reset */
	        outportb(sbbase[i] + SB_DSP_RESET,1);

	        /* wait for a little while */
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);

		/* request acknowledgement */
		outportb(sbbase[i] + SB_DSP_RESET, 0);

		/* look for reply */
		for(j=0;j<100;j++)
			if (sb_read_dsp(sbbase[i]) == 0xAA)
				/* got it ! */
				return(sbbase[i]);
	}
	/* nope */
	return(0);
}

/* Read soundblaster card parameters from BLASTER enivronment variable .*/
static int sb_getparams()
{
	char *t, *blaster;

	/* set Interrupt and DMA channel to usual defaults */
	sb_irq = 7;
	sb_dma16 = 5;

	/* assume port address unknown */
	sb_ioaddr = 0;

	/* see if environment variable set */
	if ((t=getenv("BLASTER"))) {

		/* take a copy */
		blaster = strdup(t);

		/* Parse the BLASTER variable */
		t = strtok(blaster, " \t");
		while (t && *t) {
			switch (t[0]) {
			case 'A':
			case 'a':
				/* I/O address */
				sscanf(t + 1, "%x", &sb_ioaddr);
				break;
			case 'I':
			case 'i':
				/* IRQ */
				sb_irq = atoi(t + 1);
				break;
			case 'H': 
			case 'h':
				/* 16bit DMA channel */
				sb_dma16 = atoi(t + 1);
				break;
			default:
				/* ignore everything else */
				break;
			}
			t = strtok(NULL," \t");
		}
		free(blaster);
        }

#ifdef EMO
printf("BLASTER: port=0x%X dma16=%d int=%d\n",sb_ioaddr,sb_dma16,sb_irq);
#endif

        if (sb_ioaddr==0)
        	return(sb_findcard());
        else
        	return(sb_ioaddr);
}

/* get DOS memory buffers */
static int sb_DOS_alloc()
{
	/* get a block in low memory big enough to hold
	   two buffers aligned on multiple of DMA_CHUNK */
	dosmem.size = (DMA_CHUNK*3)/16;
	if (_go32_dpmi_allocate_dos_memory(&dosmem)) {
		dosmem.size = -1;
		return(0);
	}

	(unsigned long) sb_buf[0] = dosmem.rm_segment * 16;
	(unsigned long) sb_buf[0] += (DMA_CHUNK-1);
	(unsigned long) sb_buf[0] &= ~(long)(DMA_CHUNK-1);
	(unsigned long) sb_buf[1] = (unsigned long) sb_buf[0] + DMA_CHUNK;

#ifdef EMO
	printf("DOS buffers at %08lX, %08lX\n",(unsigned long)(sb_buf[0]),
						(unsigned long)(sb_buf[1]));
#endif
	return(1);
}

/* free DOS memory */
static void sb_DOS_free()
{
	if (dosmem.size != -1)
		_go32_dpmi_free_dos_memory(&dosmem);
	dosmem.size = -1;
}

/* set SoundBlaster sample rate */
static void sb_set_sample_rate(unsigned int rate)
{
	if (rate < 5000)
		sb_sample_rate = 5000;
	else if (rate < 44100)
		sb_sample_rate = rate;
	else
		sb_sample_rate = 44100;

	sb_writedsp(SB_TIME_16BIT);
	sb_writedsp((sb_sample_rate >> 8) & 0xFF);
	sb_writedsp(sb_sample_rate & 0xFF);
}

/* SoundBlaster output switch */
static void sb_voice(int state)
{
	sb_writedsp(state ? SB_SPEAKER_ON : SB_SPEAKER_OFF);
}

/* fill a DOS buffer */
static int sb_fill_buffer(int bno,short *buff,int numf)
{
	int	i;
	int	count;
	unsigned char	xbuf[DMA_CHUNK];

	/* copy a chunk into an unsigned byte buffer */
	if (numf<=0) {
		sb_buflen[bno]=0;
		return(0);
	}
	count = (2*numf <= DMA_CHUNK) ? 2*numf : DMA_CHUNK;
	if (sb_nbits==16) {
		for (i=0;i<count;i+=2,buff++) {
			xbuf[i] = (*buff & 0xFF);
			xbuf[i+1] = (*buff >> 8);
		}
	}
	else {
		for (i=0;i<count;i+=2,buff++) {
			xbuf[i] = (*buff << 4) & 0xF0;
			xbuf[i+1] = (*buff >> 4);
		}
	}
	for (;i<DMA_CHUNK;i++) xbuf[i]=0;

	/* copy down into DOS memory for DMA */
	sb_buflen[bno] = count;
        dosmemput(xbuf, DMA_CHUNK, (unsigned long) sb_buf[bno]);
	return(count/2);
}

/* close down card and DMA */
static void sb_halt_play()
{
	if (sb_dma_active) {
		sb_writedsp(SB_PAUSE);
		outportb(SB_DMA16_MASK, SB_DMA16_STOP_MASK);
	}
	sb_dma_active=0;
}
	
/* set DOS buffer playing repetitively */
static void sb_play_buffer()
{
	int     t;

	/* set DMA mode to read memory */
	outportb(SB_DMA16_MASK, SB_DMA16_STOP_MASK);
	outportb(SB_DMA16_FF, 0);
	outportb(SB_DMA16_MODE, SB_DMA16_OUT_MODE);

	/* set DMA transfer address */
	t = (int) (((unsigned long) sb_buf[0] & 0x1FFFE) >> 1);
	outportb(SB_DMA16_BASE, t & 0xFF);
	outportb(SB_DMA16_BASE, t >> 8);
	t = (int) (((unsigned long) sb_buf[0] & 0xE0000) >> 16);
	outportb(SB_DMA16_PAGE[sb_dma16], t);

	/* set transfer length */
	outportb(SB_DMA16_COUNT, (DMA_CHUNK-1) & 0xFF);
	outportb(SB_DMA16_COUNT, (DMA_CHUNK-1) >> 8);

	/* OK set it going */
	outportb(SB_DMA16_MASK, SB_DMA16_START_MASK);

	/* set SoundBlaster to receive data */
	sb_writedsp(SB_DMA_16_BIT_DAC);
	sb_writedsp((sb_stereo)?0x30:0x10);

	sb_writedsp((DMA_CHUNK/2-1) & 0xFF);   /* write length */
	sb_writedsp((DMA_CHUNK/2-1) >> 8);

	/* a sound is playing now. */
	sb_dma_active = 1;

	atexit(sb_halt_play);
}

/* wait for DMA counter to indicate half buffer done */
static int sb_wait_counter(int bno)
{
	int	len;
	int	done;

	len = inportb(SB_DMA16_COUNT);
	len |= inportb(SB_DMA16_COUNT) << 8;
	done = DMA_CHUNK - len - 1;

	if ((bno==1) && (done >= DMA_CHUNK/2)) {
		/* OK done buffer 0 - refill it */
		inportb(sb_ioaddr + SB_DSP_DATA_AVAIL16);
		return(0);
	}
	else if ((bno==0) && (done < DMA_CHUNK/2)) {
		/* OK done buffer 1 - refill it */
		inportb(sb_ioaddr + SB_DSP_DATA_AVAIL16);
		return(0);
	}
	return(1);
}

/* SoundBlaster initialise - interface routine */
int	SB16Init(void)
{
	if (sb_getparams()==0)
		return(0);
	if (sb_DOS_alloc()==0)
		return(0);
	return(1);
}

/* play signed 16-bit data buffer */
int	SB16Play(short *buff,int numf,int srate,int nchan,int nbits)
{
	int	count,len;
	int	bno=0;
	int	nbuff=0;

	sb_reset();
	sb_voice(1);
	sb_set_sample_rate(srate);
	sb_stereo = (nchan>1);
	sb_nbits = (nbits==12) ? 12 : 16;

	count = numf;
	nbuff=0;
	if (count > 0) {
		len = sb_fill_buffer(0,buff,count);
		count -= len;
		buff += len;
		nbuff++;
		bno = 1;	/* pretend to have two buffers worth */
	}
	if (count > 0) {
		len = sb_fill_buffer(1,buff,count);
		count -= len;
		buff += len;
		nbuff++;
		bno = 1;
	}

	/* start off replay */
	if (nbuff > 0) sb_play_buffer();

	/* while buffers waiting */
	while (nbuff > 0) {
#ifdef EMO
		printf("playing buffer %d, samples left %d\n",1-bno,count);
#endif
		while (sb_wait_counter(bno)) /* loop */;
		nbuff--;		/* done a buffer */
		bno = 1 - bno;		/* swap to next buffer */
		if (count > 0) {
			len = sb_fill_buffer(bno,buff,count);
			count -= len;
			buff += len;
			nbuff++;	/* filled a buffer */
		}
#ifdef EMO
		if (kbhit())
			if (getch() == 27) 
				break;
#endif
	}
	sb_halt_play();
	sb_voice(0);
	sb_reset();
	return(numf-count);
}

void	SB16Close(void)
{
	sb_DOS_free();
}

#ifdef EMO
#include <math.h>
short	sbuf[100000];
void	main(int argc,char *argv[])
{
	int	i,j;
	double	omega,t;
	FILE	*ip;
	int	l;

	omega = 500.0*8.0*atan(1.0)/20000.0;
	for (i=0,t=0;i<100000;i++,t+=omega)
		sbuf[i] = 20000*sin(t);
	for (i=0;i<20;i++) printf("%d,",sbuf[i]); printf("\n");
	l = 100000;
/*
	ip = fopen("speech","rb");
	l = fread(sbuf,2,100000,ip);
	for (i=0;i<l;i++) sbuf[i] <<= 4;
	fclose(ip);
*/	
	if (SB16Init()) {
		printf("Mono - "); fflush(stdout);
		i = SB16Play(sbuf,40000,20000,1,16);
		printf("played %d samples\n",i);
		printf("Stereo - "); fflush(stdout);
		i = SB16Play(sbuf,40000,20000,2,16);
		printf("played %d samples\n",i);
		
		printf("Single Channel - "); fflush(stdout);
		for (j=1;j<40000;j+=2) sbuf[j]=0;
		i = SB16Play(sbuf,40000,20000,2,16);
		i = SB16Play(sbuf+1,40000,20000,2,16);
		i = SB16Play(sbuf,40000,20000,2,16);
		i = SB16Play(sbuf+1,40000,20000,2,16);
		printf("played %d samples\n",i);
		SB16Close();
	}
	exit(0);
}
#endif
