/* genfilt -- general purpose SFS filtering program */

/* M.A.Huckvale - University College London */

/* version 1.0 - September 1992 */

#define PROGNAME "genfilt"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH GENFILT UCL1 UCL SFS
.SH NAME
genfilt - general purpose filtering 
.SH SYNOPSIS
.B genfilt
(-i item) (-l lowfreq) (-h highfreq) (-n order) file
.SH DESCRIPTION
.I genfilt
is a program to design and apply general purpose IIR filters to speech
and lx signals.
.I genfilt
can design low-pass, high-pass, band-pass and band-stop filters.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -l lowfreq
Select frequency of low-pass band edge.  Default 0.5xsampling frequency.
.TP 11
.BI -h highfreq
Select frequency of high-pass band edge.  Default 0Hz.
.TP 11
.BI -n order
Select filter order.  Default low-pass and high-pass: 4, band-pass and band-stop: 4,
maximum 20.
.SH INPUT ITEMS
.IP SP
Any speech item
.IP LX
(optional) Any Lx item
.SH OUTPUT ITEMS
.IP SP
Filtered speech
.IP
(optional) filtered lx
.SH HISTORY
.IP lowfreq=
low-pass band edge
.IP highfreq=
High-pass band edge
.IP order=
filter order.
.SH FILES
.IP <filename>
<description>
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"
#include "filter.h"

/* global variables */
struct item_header spitem;
#define SPBUFSIZE 4096
short sp[SPBUFSIZE];
struct item_header opitem;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it=SP_TYPE;	/* item selection */
	char		*ty="0";	/* item sub type */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;		/* output file descriptor */

	/* processing variables */
	FILTER		*flt;
	int		fil_type=FILTER_LOW_PASS;
	int		fil_order=4;
	int		fil_low=50000;
	int		fil_high=0;
	int		half_freq;
	int		len,pos;
	char		fil_messg[50];
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:l:h:o:n:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: General-purpose filtering V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) || (it== LX_TYPE))
					/* ok */;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'l' :	/* low band edge */
			fil_low = atoi(optarg);
			break;
		case 'h' :	/* high band edge */
			fil_high = atoi(optarg);
			break;
		case 'n' :	/* order */
			fil_order = atoi(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-l lowedge) (-h highedge) (-n order) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* find input item */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,it,ty,&spitem))
		error("could not find input item in '%s'",filename);

	/* choose filter type */
	half_freq = (int)(0.5+0.5/spitem.frameduration);
	if (fil_low > half_freq) fil_low=half_freq;
	if (fil_low <= 0) fil_low=half_freq;
	if (fil_high < 0) fil_high=0;
	if ((fil_high==0) && (fil_low < half_freq))
		fil_type = FILTER_LOW_PASS;
	else if ((fil_high < half_freq) && (fil_low==half_freq))
		fil_type = FILTER_HIGH_PASS;
	else if ((fil_high < half_freq) && (fil_low < half_freq) && (fil_high < fil_low))
		fil_type = FILTER_BAND_PASS;
	else if ((fil_high < half_freq) && (fil_low < half_freq) && (fil_high > fil_low))
		fil_type = FILTER_BAND_STOP;
	else
		error("no/illegal filter specification: lo=%d hi=%d hf=%d",fil_low,fil_high,half_freq);

	/* design filter */
	if (fil_type==FILTER_HIGH_PASS) {
		if ((flt = filter_design(fil_type,fil_order,fil_high,fil_high,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}
	else if (fil_type==FILTER_LOW_PASS) {
		if ((flt = filter_design(fil_type,fil_order,fil_low,fil_low,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}
	else if (fil_type==FILTER_BAND_PASS) {
		if ((flt = filter_design(fil_type,fil_order,fil_high,fil_low,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}
	else {
		if ((flt = filter_design(fil_type,fil_order,fil_low,fil_high,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}

	/* create output item header */
	sfsheader(&opitem,spitem.datatype,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	if (fil_type==FILTER_LOW_PASS)
		sprintf(fil_messg,"low-pass=%d",fil_low);
	else if (fil_type==FILTER_HIGH_PASS)
		sprintf(fil_messg,"high-pass=%d",fil_high);
	else if (fil_type==FILTER_BAND_PASS)
		sprintf(fil_messg,"band-pass=%d,%d",fil_high,fil_low);
	else if (fil_type==FILTER_BAND_STOP)
		sprintf(fil_messg,"band-stop=%d,%d",fil_low,fil_high);
	sprintf(opitem.history,"%s(%d.%02d;%s,order=%d)",
		PROGNAME,
		spitem.datatype,
		spitem.subtype,
		fil_messg,
		fil_order);

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel");

	/* process the signal */
	pos=0;
	while ((len=sfsread(fid,pos,SPBUFSIZE,sp)) > 0) {
		filter_signal(flt,sp,sp,len);
		if (sfswrite(ofid,len,sp)!=len)
			error("write error on output");
		pos += len;
		if (ttytest()) {
			printf("%d/%d frames\r",pos+1,spitem.numframes);
			fflush(stdout);
		}
	}
	if (ttytest()) {
		printf("                        \r");
		fflush(stdout);
	}

	/* update and exit */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}


