/* resamp -- resample waveform to new sampling frequency */

/* version 2.0 - January 1991
	- SFS version from GEC dbase
*/

/* version 2.1 - May 1992
	- correct mistake about filter fractions 10, 11 and 12
*/

/**MAN
.TH RESAMP 1 GEC SFS
.SH NAME
resamp - resample waveform to new rate
.SH SYNOPSIS
.B resamp
(-I) (-i item) (-F) (-f new_sampfreq) file
.SH DESCRIPTION
.I resamp
returns a new item sampled at a different frequency.
The new and old sampling rates should be related by a simple rational number.
The sampling rate change is done in stages, each stage involves going up 
n times in frequency  and down d times. n and d are determined automatically
at each stage. Filtering may be done using 
.sp
1) IIR elliptic 
filters with passband ripple and stopband attenuation of 0.2db and approx 80db
respectively.(default)
.sp
2) or FIR 
filters with passband ripple and stopband attenuation of 0.2db and approx 70db
respectively. (-F flag has to be used)
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item type and subtype.
.TP 11
.BI -f new_sampfreq
Specify integer sampling rate in Hz.
.TP 11
.BI -F
Use FIR filters instead of IIR.
.SH NOTE
The new sampling rate should always be different from the old.
If the new sampling rate results in very large interpolation or decimation 
factors, the filters designed by this program will be almost unstable if IIR.
Arithmetic errors would then cause problems.
.sp
If FIR these filters might not have been designed.
.SH INPUT ITEMS
.IP 1.xx 11
2.xx   Any waveform item
.SH OUTPUT ITEMS
.IP 1. 11
2.     The same item as input
.SH VERSION/AUTHOR
1.1 -  K Frimpong-Ansah (17/3/87)
.br
2.0 - Mark Huckvale
*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "sfs.h"

#define PROGNAME "resamp"
#define PROGVERS "2.1"

#define sign(a)  ((a)< 0)? (-1):(1))
#define max(a,b) ((a)>(b)? (a): (b))
#define min(a,b) ((a)<(b)? (a): (b))

/* Global file variables */
char	*progname = PROGNAME;
char	filename[SFSMAXFILENAME];

struct stage_rec {
	int	up;
	double	cutoff;
	int	down;
};

struct	iir_rec {
	double	a0,a1,a2;
	double	b0,b1,b2;
	double	h0;
};
	
#ifndef PI
#define	PI 3.1415926536
#endif
#define	ln10 2.302585093
#define STAGES 6

struct	ser_iir {
	double	a0,a1,a2;
	double	b0,b1,b2;
	double	h0;
	};

void normanaedges(wndp,wnda,T,k)
double	wndp; 	/* digital filter passband edge	*/
double	wnda;	/* digital filter stopband edge	*/
double	*T;	/* analogue sampling rate	*/
double	*k;	/* analogue filter selectivity	*/
{
	double	ta,tp;

	ta = tan(wnda/2.0);
	tp = tan(wndp/2.0);
	*T = 2.0 * sqrt(ta * tp);
	*k = tp/ta;
}

void elliptan(k,ap,aa,order,coeffs,rpass,rstop)
double	k;	/*	selectivity factor			*/
double	ap,aa;	/*	passband ripple and stopband attenuation	*/
int	*order;	/*	returned order of the filter		*/
struct	ser_iir coeffs[];	/* array of filter coeffs for 2nd order sections*/
double	*rpass,*rstop;	/*	actual pass and stop bands	*/
{
	int	m,n,i,r;
	double	k1,uu,q0,q2,q4,q8,q12,q,D,xn,xxn,xm,gama,qfourth;
	double	numer,denom,mult,mnumer,mdenom,sigma0,ssigma,W,V;
	double	h0,mu,omega,somega;
	double	poww();
		
	k1 = sqrt(1.0-k*k);
	uu = sqrt(k1);
	q0 = 0.5 * (1.0-uu)/(1.0+uu);
	q2 = q0 * q0;
	q4 = q2 * q2;
	q8 = q4 * q4;
	q12 = q8 * q4;
	q = q0 * (1.0 + 2.0*q4 + 15.0*q8 + 150.0*q12);
	D = (exp(0.1*aa*ln10)-1.0)/(exp(0.1*ap*ln10)-1.0);
	xn = -log(16.0*D)/log(q);
	
	for (n=2; ((double) n) < xn; n += 2)
		;
	*rpass = ap;
	*rstop = 10.0*log((exp(0.1*ap*ln10)-1.0)/(16.0*poww(q,n))+1.0)/ln10;
	xxn = 1.0/((double) n);
	gama = exp(0.05*ap*ln10);
	gama = 0.5 * xxn * log((gama+1.0)/(gama-1.0));
	qfourth = exp(0.25 * log(q));

	denom = 0.0;
	numer = 0.0;
	mult = -1.0;
	for (m=0; m<STAGES; m++){
		mult = (mult<0.0)? 1.0:(-1.0);
		xm = (double) m;
		mnumer = poww(q,m*(m+1));
		mdenom = poww(q,m*m);
		numer = numer + mult * mnumer * sinh((2.0*xm+1.0) * gama);
		if (m>0) denom = denom + mult * mdenom * cosh(2.0*xm*gama);
		}
	sigma0 = 2.0 * qfourth * numer / (1.0 + 2.0 * denom);
	sigma0 = fabs(sigma0);
	ssigma = sigma0 * sigma0;
	W = sqrt((1.0 + ssigma * k) * (1.0 + ssigma/k));
	r = n/2;
	h0 = exp( -0.05 * ln10 * ap);
	for ( i=0; i<r; i++) {
		mu = ((double) i)+0.5;
		numer = 0.0;
		denom = 0.0;
		mult = -1.0;
		for (m=0; m<STAGES; m++) {
			mult = (mult<0.0)? 1.0:(-1.0);
			mnumer = poww(q,m*(m+1));
			mdenom = poww(q,m*m);
			xm = (double) m;
			numer = numer + mult * mnumer * sin((2.0*xm+1.0)*PI*mu*xxn);
			if (m>0) denom = denom + mult * mdenom * cos(2.0*xm*PI*mu*xxn);
			}
		omega = 2.0 * qfourth * numer / (1.0 + 2.0*denom);
		somega = omega * omega;
		V = sqrt((1.0 - k*somega) * (1.0 - somega/k));
		coeffs[i].a0 = 1.0/somega;
		coeffs[i].a1 = 0.0;
		coeffs[i].a2 = 1.0;
		
		denom = (1.0 + ssigma*somega);
		coeffs[i].b2 = 1.0;
		coeffs[i].b1 = 2.0 * sigma0 * V/denom;
		coeffs[i].b0 = (poww((sigma0*V),2) + poww((omega*W),2))/(denom*denom);
		h0 = h0 * coeffs[i].b0/coeffs[i].a0;
		}	
	*order = n;
}

double	poww(x,n)
double	x;
int	n;
{
	int	nn;
	double	ans;

	ans = 1.0;
	nn = abs(n);
	while (nn>0){
		ans = x * ans;
		nn--;
		}
	if (n >= 0) return(ans);
	else	return (1.0/ans);
}

void s_to_z(T,order,analco,digco)
double	T;
int	order;
struct	ser_iir analco[],digco[];
{
	int	n,r;
	double	sum; 
	struct	ser_iir	*ap,*dp;
	
	r = order/2;
	ap = analco;
	dp = digco;
	for (n=0; n<r; n++) {
		dp->a0 = ap->a0 * T*T + ap->a1 * 2.0*T + ap->a2 * 4.0;
		dp->a1 = ap->a0 * 2.0*T*T - ap->a2 * 8.0;
		dp->a2 = ap->a0 * T*T - ap->a1 * 2.0*T + ap->a2 * 4.0;
		dp->b0 = ap->b0 * T*T + ap->b1 * 2.0*T + ap->b2 * 4.0;
		dp->b1 = ap->b0 * 2.0*T*T - ap->b2 * 8.0;
		dp->b2 = ap->b0 * T*T - ap->b1 * 2.0*T + ap->b2 * 4.0;
		dp->b1 = dp->b1/dp->b0;
		dp->b2 = dp->b2/dp->b0;
		dp->b0 = 1.0;
		ap++;
		dp++;
	}

	dp = digco;
	for (n=0; n<r; n++) {
		sum = (dp->b0+dp->b1+dp->b2) / (dp->a0+dp->a1+dp->a2); 
		dp->a0 = dp->a0 * sum;
		dp->a1 = dp->a1 * sum;
		dp->a2 = dp->a2 * sum;
		dp++;
	}
}	

void expand(order,coeffs,adircoeffs,bdircoeffs)
int	order;
struct	ser_iir coeffs[];
double	adircoeffs[],bdircoeffs[];
{
	int	i,ii,j,r;
	double	*a,aa[32];
	double	*b,bb[32];
	struct	ser_iir *cp;

	r = order/2;
	cp = coeffs;
	b = bdircoeffs;
	b[0]=1.0;
	b[1]=cp->b1;
	b[2]=cp->b2;
	ii=2;
	for (i=1; i<r; i++) {
		cp++;
		bb[0] = 1.0;
		bb[1] = b[1] + cp->b1;
		bb[ii+1] = b[ii]*cp->b1 + b[ii-1]*cp->b2;
		bb[ii+2] = b[ii]*cp->b2;
		for (j=2; j<=ii; j++)
			bb[j] = b[j] + b[j-1]*cp->b1 + b[j-2]*cp->b2;
		ii += 2;
		for (j=1; j<=ii; j++) b[j] = bb[j];
	}

	cp = coeffs;
	a = adircoeffs;
	a[0]=cp->a0;
	a[1]=cp->a1;
	a[2]=cp->a2;
	ii=2;
	for (i=1; i<r; i++) {
		cp++;
		aa[0] = a[0]*cp->a0;
		aa[1] = a[1]*cp->a0 + a[0]*cp->a1;
		aa[ii+1] = a[ii]*cp->a1 + a[ii-1]*cp->a2;
		aa[ii+2] = a[ii]*cp->a2;
		for (j=2; j<=ii; j++)
			aa[j] = a[j]*cp->a0 + a[j-1]*cp->a1 + a[j-2]*cp->a2;
		ii += 2;
		for (j=0; j<=ii; j++) a[j] = aa[j];
	}
}

void freqresp(order,a,b)
int	order;
double	a[],b[];
{
	int	i,j,NPTS=128;
	double	fnr,fni,fdr,fdi,omega,expon,c,s,f[256];

	for (i=0; i<NPTS; i++){
		if (i!=0) {
			fnr = 0.0;
			fni = 0.0;
			fdr = 0.0;
			fdi = 0.0;
			omega = PI * ((double) i)/((double) NPTS);
			for (j=0; j<=order; j++) {
				expon = omega * ((double) j);
				c = cos(expon);
				s = sin(expon);
				fnr = a[j]*c + fnr;
				fni = a[j]*s + fni;
				fdr = b[j]*c + fdr;
				fdi = b[j]*s + fdi;
			}
		}
		else {
			fnr = 0.0;
			fdr = 0.0;
			for (j=0; j<=order; j++) {
				fnr = a[j] + fnr;
				fdr = b[j] + fdr;
				}
			fni = fdi = 0.0;
			printf(" %f  %f\n",fnr,fdr);
		}
		f[i] = 10.0*log((fnr*fnr+fni*fni)/(fdr*fdr+fdi*fdi) + 1.0e-12)/ln10;
		printf(" %7.2f",f[i]);
		if (i%8 == 7)printf("\n");
	}
}

double coeff02[] = {
           0.44569632E-03, 0.17496775E-02, 0.22892826E-02, 0.10849013E-02,
           -0.89626352E-03, -0.10779329E-02, 0.58612094E-03, 0.12281805E-02,
           -0.38811407E-03, -0.14684915E-02, 0.12588300E-03, 0.17352034E-02,
            0.24301541E-03, -0.19765324E-02, -0.73254155E-03, 0.21463025E-02,
            0.13388793E-02, -0.22032061E-02, -0.20495055E-02, 0.21056875E-02,
            0.28398510E-02, -0.18161826E-02, -0.36764652E-02, 0.12997772E-02,
            0.45140679E-02, -0.52863214E-03, -0.52973791E-02, -0.51628920E-03,
            0.59629874E-02, 0.18445370E-02, -0.64389911E-02, -0.34526575E-02,
            0.66497545E-02, 0.53293235E-02, -0.65086209E-02, -0.74461582E-02,
            0.59263613E-02, 0.97641321E-02, -0.48035844E-02, -0.12226844E-01,
            0.30377465E-02, 0.14774894E-01, -0.50342717E-03, -0.17345920E-01,
           -0.29761968E-02, 0.19853041E-01, 0.76344702E-02, -0.22228833E-01,
           -0.13883973E-01, 0.24391372E-01, 0.22482079E-01, -0.26272809E-01,
           -0.35082888E-01, 0.27808316E-01, 0.56038052E-01, -0.28944574E-01,
           -0.10144627E+00, 0.29642584E-01, 0.31674376E+00, 0.47012237E+00,
            0.31674376E+00, 0.29642584E-01, -0.10144627E+00, -0.28944574E-01,
            0.56038052E-01, 0.27808316E-01, -0.35082888E-01, -0.26272809E-01,
            0.22482079E-01, 0.24391372E-01, -0.13883973E-01, -0.22228833E-01,
            0.76344702E-02, 0.19853041E-01, -0.29761968E-02, -0.17345920E-01,
           -0.50342717E-03, 0.14774894E-01, 0.30377465E-02, -0.12226844E-01,
           -0.48035844E-02, 0.97641321E-02, 0.59263613E-02, -0.74461582E-02,
           -0.65086209E-02, 0.53293235E-02, 0.66497545E-02, -0.34526575E-02,
           -0.64389911E-02, 0.18445370E-02, 0.59629874E-02, -0.51628920E-03,
           -0.52973791E-02, -0.52863214E-03, 0.45140679E-02, 0.12997772E-02,
           -0.36764652E-02, -0.18161826E-02, 0.28398510E-02, 0.21056875E-02,
           -0.20495055E-02, -0.22032061E-02, 0.13388793E-02, 0.21463025E-02,
           -0.73254155E-03, -0.19765324E-02, 0.24301541E-03, 0.17352034E-02,
            0.12588300E-03, -0.14684915E-02, -0.38811407E-03, 0.12281805E-02,
            0.58612094E-03, -0.10779329E-02, -0.89626352E-03, 0.10849013E-02,
            0.22892826E-02, 0.17496775E-02, 0.44569632E-03
		};

double coeff03[] = {
            0.14383845E-03, 0.73517178E-03, 0.12585344E-02, 0.16416742E-02,
            0.14479148E-02, 0.68084232E-03, -0.29275761E-03, -0.88313041E-03,
           -0.71757997E-03, 0.39959887E-04, 0.76022011E-03, 0.82402071E-03,
            0.14430097E-03, -0.71708817E-03, -0.99161221E-03, -0.37527716E-03,
            0.65036654E-03, 0.11791110E-02, 0.66916738E-03, -0.51733432E-03,
           -0.13517141E-02, -0.10203765E-02, 0.29556709E-03, 0.14791318E-02,
            0.14147897E-02, 0.28013159E-04, -0.15321471E-02, -0.18322016E-02,
           -0.46005778E-03, 0.14821441E-02, 0.22458662E-02, 0.99908258E-03,
           -0.13028032E-02, -0.26242451E-02, -0.16370149E-02, 0.97012619E-03,
            0.29308086E-02, 0.23568477E-02, -0.46453782E-03, -0.31252604E-02,
           -0.31334211E-02, -0.22722025E-03, 0.31659473E-02, 0.39338409E-02,
            0.11132977E-02, -0.30081568E-02, -0.47152736E-02, -0.21915971E-02,
            0.26092585E-02, 0.54289866E-02, 0.34544817E-02, -0.19246273E-02,
           -0.60152127E-02, -0.48821447E-02, 0.91317680E-03, 0.64082080E-02,
            0.64458419E-02, 0.46374919E-03, -0.65367762E-02, -0.81127463E-02,
           -0.22496891E-02, 0.63174283E-02, 0.98432377E-02, 0.44980985E-02,
           -0.56463405E-02, -0.11586713E-01, -0.72749220E-02, 0.43955562E-02,
            0.13288680E-01, 0.10680182E-01, -0.23937270E-02, -0.14904696E-01,
           -0.14909236E-01, -0.65114140E-03, 0.16375361E-01, 0.20322265E-01,
            0.52539222E-02, -0.17655769E-01, -0.27726799E-01, -0.12542081E-01,
            0.18700657E-01, 0.39200433E-01, 0.25566002E-01, -0.19474661E-01,
           -0.61937064E-01, -0.56568332E-01, 0.19950423E-01, 0.14662725E+00,
            0.26500466E+00, 0.31322247E+00, 0.26500466E+00, 0.14662725E+00,
            0.19950423E-01, -0.56568332E-01, -0.61937064E-01, -0.19474661E-01,
            0.25566002E-01, 0.39200433E-01, 0.18700657E-01, -0.12542081E-01,
           -0.27726799E-01, -0.17655769E-01, 0.52539222E-02, 0.20322265E-01,
            0.16375361E-01, -0.65114140E-03, -0.14909236E-01, -0.14904696E-01,
           -0.23937270E-02, 0.10680182E-01, 0.13288680E-01, 0.43955562E-02,
           -0.72749220E-02, -0.11586713E-01, -0.56463405E-02, 0.44980985E-02,
            0.98432377E-02, 0.63174283E-02, -0.22496891E-02, -0.81127463E-02,
           -0.65367762E-02, 0.46374919E-03, 0.64458419E-02, 0.64082080E-02,
            0.91317680E-03, -0.48821447E-02, -0.60152127E-02, -0.19246273E-02,
            0.34544817E-02, 0.54289866E-02, 0.26092585E-02, -0.21915971E-02,
           -0.47152736E-02, -0.30081568E-02, 0.11132977E-02, 0.39338409E-02,
            0.31659473E-02, -0.22722025E-03, -0.31334211E-02, -0.31252604E-02,
           -0.46453782E-03, 0.23568477E-02, 0.29308086E-02, 0.97012619E-03,
           -0.16370149E-02, -0.26242451E-02, -0.13028032E-02, 0.99908258E-03,
            0.22458662E-02, 0.14821441E-02, -0.46005778E-03, -0.18322016E-02,
           -0.15321471E-02, 0.28013159E-04, 0.14147897E-02, 0.14791318E-02,
            0.29556709E-03, -0.10203765E-02, -0.13517141E-02, -0.51733432E-03,
            0.66916738E-03, 0.11791110E-02, 0.65036654E-03, -0.37527716E-03,
           -0.99161221E-03, -0.71708817E-03, 0.14430097E-03, 0.82402071E-03,
            0.76022011E-03, 0.39959887E-04, -0.71757997E-03, -0.88313041E-03,
           -0.29275761E-03, 0.68084232E-03, 0.14479148E-02, 0.16416742E-02,
            0.12585344E-02, 0.73517178E-03, 0.14383845E-03
		};

double coeff04[] = {
           -0.28874114E-03, -0.70327835E-03, -0.10047526E-02, -0.13977777E-02,
           -0.14465820E-02, -0.12775303E-02, -0.75086963E-03, -0.11853060E-03,
            0.48427097E-03, 0.78408042E-03, 0.71302982E-03, 0.28871809E-03,
           -0.26350687E-03, -0.68999606E-03, -0.76434255E-03, -0.44559519E-03,
            0.12466764E-03, 0.65838196E-03, 0.87669160E-03, 0.64228650E-03,
            0.50065290E-04, -0.61256380E-03, -0.99784276E-03, -0.88020065E-03,
           -0.28338542E-03, 0.51116775E-03, 0.10951379E-02, 0.11398902E-02,
            0.57895039E-03, -0.33564281E-03, -0.11428747E-02, -0.14025477E-02,
           -0.93249790E-03, 0.72810770E-04, 0.11171760E-02, 0.16447476E-02,
            0.13330787E-02, 0.28448878E-03, -0.99582737E-03, -0.18401388E-02,
           -0.17632011E-02, -0.73698093E-03, 0.75981300E-03, 0.19604983E-02,
            0.21998785E-02, 0.12788170E-02, -0.39406677E-03, -0.19773324E-02,
           -0.26157389E-02, -0.18988720E-02, -0.11329425E-03, 0.18606235E-02,
            0.29779910E-02, 0.25787046E-02, 0.76893653E-03, -0.15809869E-02,
           -0.32488988E-02, -0.32920265E-02, -0.15725236E-02, 0.11114411E-02,
            0.33875224E-02, 0.40056217E-02, 0.25169558E-02, -0.42779336E-03,
           -0.33504968E-02, -0.46800035E-02, -0.35891656E-02, -0.49184309E-03,
            0.30910699E-02, 0.52687339E-02, 0.47691893E-02, 0.16671509E-02,
           -0.25588167E-02, -0.57175318E-02, -0.60301474E-02, -0.31160647E-02,
            0.16985084E-02, 0.59642764E-02, 0.73401937E-02, 0.48591611E-02,
           -0.44478817E-03, -0.59354040E-02, -0.86631831E-02, -0.69260732E-02,
           -0.12872985E-02, 0.55379951E-02, 0.99593056E-02, 0.93664862E-02,
            0.36207640E-02, -0.46453704E-02, -0.11186537E-01, -0.12274655E-01,
           -0.67562354E-02, 0.30652683E-02, 0.12306274E-01, 0.15850380E-01,
            0.11074070E-01, -0.44982400E-03, -0.13282768E-01, -0.20543387E-01,
           -0.17401578E-01, -0.39546634E-02, 0.14081622E-01, 0.27509946E-01,
            0.27915355E-01, 0.12219222E-01, -0.14670337E-01, -0.40639818E-01,
           -0.50656956E-01, -0.33017348E-01, 0.15035498E-01, 0.84843114E-01,
            0.15835018E+00, 0.21408217E+00, 0.23484306E+00, 0.21408217E+00,
            0.15835018E+00, 0.84843114E-01, 0.15035498E-01, -0.33017348E-01,
           -0.50656956E-01, -0.40639818E-01, -0.14670337E-01, 0.12219222E-01,
            0.27915355E-01, 0.27509946E-01, 0.14081622E-01, -0.39546634E-02,
           -0.17401578E-01, -0.20543387E-01, -0.13282768E-01, -0.44982400E-03,
            0.11074070E-01, 0.15850380E-01, 0.12306274E-01, 0.30652683E-02,
           -0.67562354E-02, -0.12274655E-01, -0.11186537E-01, -0.46453704E-02,
            0.36207640E-02, 0.93664862E-02, 0.99593056E-02, 0.55379951E-02,
           -0.12872985E-02, -0.69260732E-02, -0.86631831E-02, -0.59354040E-02,
           -0.44478817E-03, 0.48591611E-02, 0.73401937E-02, 0.59642764E-02,
            0.16985084E-02, -0.31160647E-02, -0.60301474E-02, -0.57175318E-02,
           -0.25588167E-02, 0.16671509E-02, 0.47691893E-02, 0.52687339E-02,
            0.30910699E-02, -0.49184309E-03, -0.35891656E-02, -0.46800035E-02,
           -0.33504968E-02, -0.42779336E-03, 0.25169558E-02, 0.40056217E-02,
            0.33875224E-02, 0.11114411E-02, -0.15725236E-02, -0.32920265E-02,
           -0.32488988E-02, -0.15809869E-02, 0.76893653E-03, 0.25787046E-02,
            0.29779910E-02, 0.18606235E-02, -0.11329425E-03, -0.18988720E-02,
           -0.26157389E-02, -0.19773324E-02, -0.39406677E-03, 0.12788170E-02,
            0.21998785E-02, 0.19604983E-02, 0.75981300E-03, -0.73698093E-03,
           -0.17632011E-02, -0.18401388E-02, -0.99582737E-03, 0.28448878E-03,
            0.13330787E-02, 0.16447476E-02, 0.11171760E-02, 0.72810770E-04,
           -0.93249790E-03, -0.14025477E-02, -0.11428747E-02, -0.33564281E-03,
            0.57895039E-03, 0.11398902E-02, 0.10951379E-02, 0.51116775E-03,
           -0.28338542E-03, -0.88020065E-03, -0.99784276E-03, -0.61256380E-03,
            0.50065290E-04, 0.64228650E-03, 0.87669160E-03, 0.65838196E-03,
            0.12466764E-03, -0.44559519E-03, -0.76434255E-03, -0.68999606E-03,
           -0.26350687E-03, 0.28871809E-03, 0.71302982E-03, 0.78408042E-03,
            0.48427097E-03, -0.11853060E-03, -0.75086963E-03, -0.12775303E-02,
           -0.14465820E-02, -0.13977777E-02, -0.10047526E-02, -0.70327835E-03,
           -0.28874114E-03
		};

double coeff05[] = { 
	    -0.31056290E-03, -0.42265721E-03, -0.60850632E-03, -0.74726529E-03,
           -0.78608841E-03, -0.68550411E-03, -0.43453460E-03, -0.60563791E-04,
            0.37220801E-03, 0.77531504E-03, 0.10580175E-02, 0.11523211E-02,
            0.10357517E-02, 0.74167759E-03, 0.35356946E-03, -0.16706728E-04,
           -0.25992520E-03, -0.30453206E-03, -0.13923268E-03, 0.18026053E-03,
            0.54544129E-03, 0.82942628E-03, 0.92629623E-03, 0.78882714E-03,
            0.44660276E-03, 0.87808229E-06, -0.40605821E-03, -0.63652615E-03,
           -0.60488965E-03, -0.30999823E-03, 0.15927578E-03, 0.64886670E-03,
            0.98826212E-03, 0.10483125E-02, 0.78824756E-03, 0.27515509E-03,
           -0.33191705E-03, -0.83160214E-03, -0.10466469E-02, -0.88690862E-03,
           -0.38601540E-03, 0.30260487E-03, 0.95108937E-03, 0.13314696E-02,
            0.12943693E-02, 0.82578749E-03, 0.60135204E-04, -0.75705908E-03,
           -0.13473838E-02, -0.14940283E-02, -0.11205958E-02, -0.32600045E-03,
            0.63912506E-03, 0.14502872E-02, 0.18171221E-02, 0.15859085E-02,
            0.80113549E-03, -0.29847471E-03, -0.13510790E-02, -0.19900817E-02,
           -0.19708110E-02, -0.12625086E-02, -0.71317692E-04, 0.12175161E-02,
            0.21628507E-02, 0.24174466E-02, 0.18536691E-02, 0.62022335E-03,
           -0.89361594E-03, -0.21793824E-02, -0.27789401E-02, -0.24461264E-02,
           -0.12441571E-02, 0.45888807E-03, 0.21005247E-02, 0.31079028E-02,
            0.30959805E-02, 0.20115145E-02, 0.17033156E-03, -0.18315547E-02,
           -0.33087065E-02, -0.37190730E-02, -0.28600092E-02, -0.95780846E-03,
            0.13864313E-02, 0.33839683E-02, 0.43213307E-02, 0.38120132E-02,
            0.19494293E-02, -0.69786969E-03, -0.32558495E-02, -0.48303818E-02,
           -0.48162513E-02, -0.31241353E-02, -0.24217449E-03, 0.28988549E-02,
            0.52218577E-02, 0.58681639E-02, 0.45089223E-02, 0.14917873E-02,
           -0.22373330E-02, -0.54248599E-02, -0.69247829E-02, -0.61032535E-02,
           -0.30919707E-02, 0.12066980E-02, 0.53810105E-02, 0.79646977E-02,
            0.79419063E-02, 0.51313075E-02, 0.30907925E-03, -0.49900804E-02,
           -0.89482730E-02, -0.10072971E-01, -0.77397288E-02, -0.24780901E-02,
	    0.41170344E-02, 0.98475609E-02, 0.12620548E-01, 0.11181167E-01,
            0.56237211E-02, -0.25123011E-02, -0.10630105E-01, -0.15861154E-01,
           -0.16021814E-01, -0.10438225E-01, -0.35198921E-03, 0.11269534E-01,
            0.20518808E-01, 0.23738435E-01, 0.18779704E-01, 0.59402501E-02,
           -0.11743145E-01, -0.29002026E-01, -0.39570808E-01, -0.37853234E-01,
           -0.20577002E-01, 0.12034111E-01, 0.55933259E-01, 0.10391253E+00,
            0.14713004E+00, 0.17713384E+00, 0.18786754E+00, 0.17713384E+00,
            0.14713004E+00, 0.10391253E+00, 0.55933259E-01, 0.12034111E-01,
           -0.20577002E-01, -0.37853234E-01, -0.39570808E-01, -0.29002026E-01,
           -0.11743145E-01, 0.59402501E-02, 0.18779704E-01, 0.23738435E-01,
            0.20518808E-01, 0.11269534E-01, -0.35198921E-03, -0.10438225E-01,
           -0.16021814E-01, -0.15861154E-01, -0.10630105E-01, -0.25123011E-02,
            0.56237211E-02, 0.11181167E-01, 0.12620548E-01, 0.98475609E-02,
            0.41170344E-02, -0.24780901E-02, -0.77397288E-02, -0.10072971E-01,
           -0.89482730E-02, -0.49900804E-02, 0.30907925E-03, 0.51313075E-02,
            0.79419063E-02, 0.79646977E-02, 0.53810105E-02, 0.12066980E-02,
           -0.30919707E-02, -0.61032535E-02, -0.69247829E-02, -0.54248599E-02,
           -0.22373330E-02, 0.14917873E-02, 0.45089223E-02, 0.58681639E-02,
            0.52218577E-02, 0.28988549E-02, -0.24217449E-03, -0.31241353E-02,
           -0.48162513E-02, -0.48303818E-02, -0.32558495E-02, -0.69786969E-03,
  	    0.19494293E-02, 0.38120132E-02, 0.43213307E-02, 0.33839683E-02,
            0.13864313E-02, -0.95780846E-03, -0.28600092E-02, -0.37190730E-02,
           -0.33087065E-02, -0.18315547E-02, 0.17033156E-03, 0.20115145E-02,
            0.30959805E-02, 0.31079028E-02, 0.21005247E-02, 0.45888807E-03,
           -0.12441571E-02, -0.24461264E-02, -0.27789401E-02, -0.21793824E-02,
           -0.89361594E-03, 0.62022335E-03, 0.18536691E-02, 0.24174466E-02,
            0.21628507E-02, 0.12175161E-02, -0.71317692E-04, -0.12625086E-02,
           -0.19708110E-02, -0.19900817E-02, -0.13510790E-02, -0.29847471E-03,
            0.80113549E-03, 0.15859085E-02, 0.18171221E-02, 0.14502872E-02,
            0.63912506E-03, -0.32600045E-03, -0.11205958E-02, -0.14940283E-02,
           -0.13473838E-02, -0.75705908E-03, 0.60135204E-04, 0.82578749E-03,
            0.12943693E-02, 0.13314696E-02, 0.95108937E-03, 0.30260487E-03,
           -0.38601540E-03, -0.88690862E-03, -0.10466469E-02, -0.83160214E-03,
           -0.33191705E-03, 0.27515509E-03, 0.78824756E-03, 0.10483125E-02,
            0.98826212E-03, 0.64886670E-03, 0.15927578E-03, -0.30999823E-03,
           -0.60488965E-03, -0.63652615E-03, -0.40605821E-03, 0.87808229E-06,
            0.44660276E-03, 0.78882714E-03, 0.92629623E-03, 0.82942628E-03,
            0.54544129E-03, 0.18026053E-03, -0.13923268E-03, -0.30453206E-03,
           -0.25992520E-03, -0.16706728E-04, 0.35356946E-03, 0.74167759E-03,
            0.10357517E-02, 0.11523211E-02, 0.10580175E-02, 0.77531504E-03,
            0.37220801E-03, -0.60563791E-04, -0.43453460E-03, -0.68550411E-03,
           -0.78608841E-03, -0.74726529E-03, -0.60850632E-03, -0.42265721E-03,
           -0.31056290E-03
		};

double coeff06[] = {
	    0.14598534E-04, -0.26360134E-03, -0.34754284E-03,
           -0.51836052E-03, -0.69649174E-03, -0.85160404E-03, -0.95443579E-03,
           -0.97977230E-03, -0.91213261E-03, -0.75046939E-03, -0.51041646E-03,
           -0.22337424E-03, 0.68040004E-04, 0.31730597E-03, 0.48300854E-03,
            0.53746888E-03, 0.47333125E-03, 0.30620518E-03, 0.72769362E-04,
           -0.17593931E-03, -0.38466873E-03, -0.50580275E-03, -0.51005941E-03,
           -0.39379176E-03, -0.18070247E-03, 0.82385690E-04, 0.33538428E-03,
            0.51812787E-03, 0.58445567E-03, 0.51358948E-03, 0.31634609E-03,
            0.33980356E-04, -0.27004091E-03, -0.52411848E-03, -0.66495378E-03,
           -0.65313943E-03, -0.48371739E-03, -0.18938546E-03, 0.16557699E-03,
            0.49902702E-03, 0.72986615E-03, 0.79720537E-03, 0.67629805E-03,
            0.38639444E-03, -0.11357049E-04, -0.42702028E-03, -0.76158030E-03,
           -0.93025772E-03, -0.88356924E-03, -0.62222307E-03, -0.19747285E-03,
            0.29647810E-03, 0.74414781E-03, 0.10355659E-02, 0.10922613E-02,
            0.88890654E-03, 0.46115299E-03, -0.99074088E-04, -0.66348317E-03,
           -0.10961168E-02, -0.12861703E-02, -0.11756894E-02, -0.77651109E-03,
           -0.17061208E-03, 0.50596212E-03, 0.10934926E-02, 0.14458160E-02,
            0.14665192E-02, 0.11344854E-02, 0.51284232E-03, -0.26236713E-03,
           -0.10113625E-02, -0.15520903E-02, -0.17439902E-02, -0.15241257E-02,
           -0.92599576E-03, -0.76137927E-04, 0.83214272E-03, 0.15821893E-02,
            0.19853690E-02, 0.19279473E-02, 0.14022400E-02, 0.51333051E-03,
           -0.54095360E-03, -0.15135630E-02, -0.21655676E-02, -0.23244543E-02,
           -0.19288698E-02, -0.10486781E-02, 0.12587580E-03, 0.13244168E-02,
            0.22582135E-02, 0.26891474E-02, 0.24896399E-02, 0.16785704E-02,
            0.42373172E-03, -0.99189195E-03, -0.22333686E-02, -0.29920668E-02,
           -0.30618005E-02, -0.23935353E-02, -0.11142454E-02, 0.49492036E-03,
            0.20600406E-02, 0.31999357E-02, 0.36184366E-02, 0.31798135E-02,
            0.19493952E-02, 0.18725857E-03, -0.17052089E-02, -0.32757313E-02,
           -0.41279681E-02, -0.40202159E-02, -0.29313336E-02, -0.10767784E-02,
            0.11319059E-02, 0.31767543E-02, 0.45534507E-02, 0.48939087E-02,
            0.40622959E-02, 0.21996775E-02, -0.29570010E-03, -0.28513921E-02,
           -0.48499568E-02, -0.57759578E-02, -0.53459485E-02, -0.35896620E-02,
           -0.85991441E-03, 0.22346615E-02, 0.49624545E-02, 0.66385930E-02,
            0.67931763E-02, 0.52973102E-02, 0.24145923E-02, -0.12389446E-02,
           -0.48218840E-02, -0.74565001E-02, -0.84381439E-02, -0.74165324E-02,
           -0.45035500E-02, -0.27748483E-03, 0.43241563E-02, 0.82049211E-02,
            0.10361831E-01, 0.10133359E-01, 0.73841829E-02, 0.25788422E-02,
           -0.32814604E-02, -0.88541806E-02, -0.12746756E-01, -0.13844689E-01,
           -0.11601629E-01, -0.62230672E-02, 0.13086611E-02, 0.93864826E-02,
            0.16089004E-01, 0.19593595E-01, 0.18609727E-01, 0.12739453E-01,
            0.26811794E-02, -0.97802738E-02, -0.22012753E-01, -0.30973891E-01,
           -0.33772022E-01, -0.28247325E-01, -0.13458985E-01, 0.10022545E-01,
            0.40085915E-01, 0.73330753E-01, 0.10556649E+00, 0.13247453E+00,
            0.15031683E+00, 0.15656243E+00, 0.15031683E+00, 0.13247453E+00,
            0.10556649E+00, 0.73330753E-01, 0.40085915E-01, 0.10022545E-01,
           -0.13458985E-01, -0.28247325E-01, -0.33772022E-01, -0.30973891E-01,
           -0.22012753E-01, -0.97802738E-02, 0.26811794E-02, 0.12739453E-01,
            0.18609727E-01, 0.19593595E-01, 0.16089004E-01, 0.93864826E-02,
            0.13086611E-02, -0.62230672E-02, -0.11601629E-01, -0.13844689E-01,
           -0.12746756E-01, -0.88541806E-02, -0.32814604E-02, 0.25788422E-02,
            0.73841829E-02, 0.10133359E-01, 0.10361831E-01, 0.82049211E-02,
            0.43241563E-02, -0.27748483E-03, -0.45035500E-02, -0.74165324E-02,
           -0.84381439E-02, -0.74565001E-02, -0.48218840E-02, -0.12389446E-02,
            0.24145923E-02, 0.52973102E-02, 0.67931763E-02, 0.66385930E-02,
            0.49624545E-02, 0.22346615E-02, -0.85991441E-03, -0.35896620E-02,
           -0.53459485E-02, -0.57759578E-02, -0.48499568E-02, -0.28513921E-02,
           -0.29570010E-03, 0.21996775E-02, 0.40622959E-02, 0.48939087E-02,
            0.45534507E-02, 0.31767543E-02, 0.11319059E-02, -0.10767784E-02,
           -0.29313336E-02, -0.40202159E-02, -0.41279681E-02, -0.32757313E-02,
           -0.17052089E-02, 0.18725857E-03, 0.19493952E-02, 0.31798135E-02,
            0.36184366E-02, 0.31999357E-02, 0.20600406E-02, 0.49492036E-03,
           -0.11142454E-02, -0.23935353E-02, -0.30618005E-02, -0.29920668E-02,
           -0.22333686E-02, -0.99189195E-03, 0.42373172E-03, 0.16785704E-02,
            0.24896399E-02, 0.26891474E-02, 0.22582135E-02, 0.13244168E-02,
            0.12587580E-03, -0.10486781E-02, -0.19288698E-02, -0.23244543E-02,
           -0.21655676E-02, -0.15135630E-02, -0.54095360E-03, 0.51333051E-03,
            0.14022400E-02, 0.19279473E-02, 0.19853690E-02, 0.15821893E-02,
            0.83214272E-03, -0.76137927E-04, -0.92599576E-03, -0.15241257E-02,
           -0.17439902E-02, -0.15520903E-02, -0.10113625E-02, -0.26236713E-03,
            0.51284232E-03, 0.11344854E-02, 0.14665192E-02, 0.14458160E-02,
            0.10934926E-02, 0.50596212E-03, -0.17061208E-03, -0.77651109E-03,
           -0.11756894E-02, -0.12861703E-02, -0.10961168E-02, -0.66348317E-03,
           -0.99074088E-04, 0.46115299E-03, 0.88890654E-03, 0.10922613E-02,
            0.10355659E-02, 0.74414781E-03, 0.29647810E-03, -0.19747285E-03,
           -0.62222307E-03, -0.88356924E-03, -0.93025772E-03, -0.76158030E-03,
           -0.42702028E-03, -0.11357049E-04, 0.38639444E-03, 0.67629805E-03,
            0.79720537E-03, 0.72986615E-03, 0.49902702E-03, 0.16557699E-03,
           -0.18938546E-03, -0.48371739E-03, -0.65313943E-03, -0.66495378E-03,
           -0.52411848E-03, -0.27004091E-03, 0.33980356E-04, 0.31634609E-03,
            0.51358948E-03, 0.58445567E-03, 0.51812787E-03, 0.33538428E-03,
            0.82385690E-04, -0.18070247E-03, -0.39379176E-03, -0.51005941E-03,
           -0.50580275E-03, -0.38466873E-03, -0.17593931E-03, 0.72769362E-04,
            0.30620518E-03, 0.47333125E-03, 0.53746888E-03, 0.48300854E-03,
            0.31730597E-03, 0.68040004E-04, -0.22337424E-03, -0.51041646E-03,
           -0.75046939E-03, -0.91213261E-03, -0.97977230E-03, -0.95443579E-03,
           -0.85160404E-03, -0.69649174E-03, -0.51836052E-03, -0.34754284E-03,
           -0.26360134E-03, 0.14598534E-04 
		};

double coeff07[] = {
            0.12214734E-03, -0.60547220E-04, -0.11636934E-03, -0.20546843E-03,
           -0.32029767E-03, -0.45073929E-03, -0.58361021E-03, -0.70331554E-03,
           -0.79353037E-03, -0.83932845E-03, -0.82961662E-03, -0.75917499E-03,
           -0.63022366E-03, -0.45283514E-03, -0.24437343E-03, -0.27564631E-04,
            0.17226036E-03, 0.33078628E-03, 0.42827736E-03, 0.45291355E-03,
            0.40289093E-03, 0.28729177E-03, 0.12503950E-03, -0.57501307E-04,
           -0.23048975E-03, -0.36503296E-03, -0.43801026E-03, -0.43589596E-03,
           -0.35754833E-03, -0.21456314E-03, -0.29955087E-04, 0.16567663E-03,
            0.33860767E-03, 0.45797369E-03, 0.50048000E-03, 0.45661698E-03,
            0.33021756E-03, 0.14032755E-03, -0.82030965E-04, -0.29920577E-03,
           -0.47259816E-03, -0.56984887E-03, -0.57046145E-03, -0.47020358E-03,
           -0.28233288E-03, -0.36258243E-04, 0.22712689E-03, 0.46197293E-03,
            0.62551565E-03, 0.68562198E-03, 0.62708522E-03, 0.45494176E-03,
            0.19462843E-03, -0.11171203E-03, -0.41183271E-03, -0.65231544E-03,
           -0.78788784E-03, -0.78992260E-03, -0.65212895E-03, -0.39277750E-03,
           -0.52460884E-04, 0.31198867E-03, 0.63702633E-03, 0.86325983E-03,
            0.94633328E-03, 0.86525112E-03, 0.62731269E-03, 0.26790815E-03,
           -0.15425554E-03, -0.56713464E-03, -0.89698465E-03, -0.10816692E-02,
           -0.10823342E-02, -0.89099590E-03, -0.53356955E-03, -0.66504792E-04, 
 	    0.43182817E-03, 0.87425474E-03, 0.11798354E-02, 0.12886100E-02,
            0.11730705E-02, 0.84448280E-03, 0.35243959E-03, -0.22232867E-03,
           -0.78108313E-03, -0.12238518E-02, -0.14672056E-02, -0.14597467E-02,
           -0.11926142E-02, -0.70276490E-03, -0.68297268E-04, 0.60382625E-03,
            0.11955975E-02, 0.15985516E-02, 0.17334195E-02, 0.15654808E-02,
            0.11124774E-02, 0.44362442E-03, -0.33072784E-03, -0.10769996E-02,
           -0.16612811E-02, -0.19731328E-02, -0.19463006E-02, -0.15720746E-02,
           -0.90327655E-03, -0.47346872E-04, 0.85081253E-03, 0.16329143E-02,
            0.21551205E-02, 0.23141238E-02, 0.20672320E-02, 0.14423153E-02,
            0.53570577E-03, -0.50228363E-03, -0.14919277E-02, -0.22549648E-02,
           -0.26463741E-02, -0.25815645E-02, -0.20538217E-02, -0.11386510E-02,
            0.16392310E-04, 0.12150598E-02, 0.22454220E-02, 0.29169598E-02,
            0.30953877E-02, 0.27288168E-02, 0.18603748E-02, 0.62438590E-03,
           -0.77378360E-03, -0.20914217E-02, -0.30901122E-02, -0.35784086E-02,
           -0.34478267E-02, -0.26956555E-02, -0.14298976E-02, 0.14530717E-03,
            0.17619909E-02, 0.31332793E-02, 0.40036533E-02, 0.41959002E-02,
            0.36458860E-02, 0.24190694E-02, 0.70446625E-03, -0.12134596E-02,
           -0.30011875E-02, -0.43332484E-02, -0.49505294E-02, -0.47093877E-02,
           -0.36125698E-02, -0.18155578E-02, 0.39386918E-03, 0.26405051E-02,
            0.45241993E-02, 0.56897774E-02, 0.58923811E-02, 0.50465018E-02,
            0.32492343E-02, 0.77240553E-03, -0.19766190E-02, -0.45200749E-02,
           -0.63908193E-02, -0.72157723E-02, -0.67872959E-02, -0.51092813E-02,
           -0.24085906E-02, 0.89213951E-03, 0.42376537E-02, 0.70302137E-02,
            0.87326756E-02, 0.89671211E-02, 0.75918203E-02, 0.47418200E-02,
            0.82394923E-03, -0.35359818E-02, -0.75873351E-02, -0.10577184E-01,
           -0.11881824E-01, -0.11125664E-01, -0.82664220E-02, -0.36298740E-02,
            0.21152359E-02, 0.80423318E-02, 0.13102532E-01, 0.16292129E-01,
            0.16827174E-01, 0.14297723E-01, 0.87757166E-02, 0.85616665E-03,
           -0.83794417E-02, -0.17477380E-01, -0.24791395E-01, -0.28705278E-01,
           -0.27867969E-01, -0.21407615E-01, -0.90924632E-02, 0.85870465E-02,
            0.30430358E-01, 0.54635253E-01, 0.78991398E-01, 0.10113514E+00,
            0.11883266E+00, 0.13025388E+00, 0.13419999E+00, 0.13025388E+00,
            0.11883266E+00, 0.10113514E+00, 0.78991398E-01, 0.54635253E-01,
            0.30430358E-01, 0.85870465E-02, -0.90924632E-02, -0.21407615E-01, 
	   -0.27867969E-01, -0.28705278E-01, -0.24791395E-01, -0.17477380E-01, 
	   -0.83794417E-02, 0.85616665E-03, 0.87757166E-02, 0.14297723E-01,
            0.16827174E-01, 0.16292129E-01, 0.13102532E-01, 0.80423318E-02,
            0.21152359E-02, -0.36298740E-02, -0.82664220E-02, -0.11125664E-01,
           -0.11881824E-01, -0.10577184E-01, -0.75873351E-02, -0.35359818E-02,
            0.82394923E-03, 0.47418200E-02, 0.75918203E-02, 0.89671211E-02,
            0.87326756E-02, 0.70302137E-02, 0.42376537E-02, 0.89213951E-03,
           -0.24085906E-02, -0.51092813E-02, -0.67872959E-02, -0.72157723E-02,
           -0.63908193E-02, -0.45200749E-02, -0.19766190E-02, 0.77240553E-03,
            0.32492343E-02, 0.50465018E-02, 0.58923811E-02, 0.56897774E-02,
            0.45241993E-02, 0.26405051E-02, 0.39386918E-03, -0.18155578E-02,
           -0.36125698E-02, -0.47093877E-02, -0.49505294E-02, -0.43332484E-02,
           -0.30011875E-02, -0.12134596E-02, 0.70446625E-03, 0.24190694E-02,
            0.36458860E-02, 0.41959002E-02, 0.40036533E-02, 0.31332793E-02,
            0.17619909E-02, 0.14530717E-03, -0.14298976E-02, -0.26956555E-02,
           -0.34478267E-02, -0.35784086E-02, -0.30901122E-02, -0.20914217E-02,
           -0.77378360E-03, 0.62438590E-03, 0.18603748E-02, 0.27288168E-02,
            0.30953877E-02, 0.29169598E-02, 0.22454220E-02, 0.12150598E-02,
            0.16392310E-04, -0.11386510E-02, -0.20538217E-02, -0.25815645E-02,
           -0.26463741E-02, -0.22549648E-02, -0.14919277E-02, -0.50228363E-03,
            0.53570577E-03, 0.14423153E-02, 0.20672320E-02, 0.23141238E-02,
            0.21551205E-02, 0.16329143E-02, 0.85081253E-03, -0.47346872E-04,
           -0.90327655E-03, -0.15720746E-02, -0.19463006E-02, -0.19731328E-02,
           -0.16612811E-02, -0.10769996E-02, -0.33072784E-03, 0.44362442E-03,
            0.11124774E-02, 0.15654808E-02, 0.17334195E-02, 0.15985516E-02,
            0.11955975E-02, 0.60382625E-03, -0.68297268E-04, -0.70276490E-03,
           -0.11926142E-02, -0.14597467E-02, -0.14672056E-02, -0.12238518E-02,
           -0.78108313E-03, -0.22232867E-03, 0.35243959E-03, 0.84448280E-03,
            0.11730705E-02, 0.12886100E-02, 0.11798354E-02, 0.87425474E-03,
            0.43182817E-03, -0.66504792E-04, -0.53356955E-03, -0.89099590E-03,
           -0.10823342E-02, -0.10816692E-02, -0.89698465E-03, -0.56713464E-03,
           -0.15425554E-03, 0.26790815E-03, 0.62731269E-03, 0.86525112E-03,
            0.94633328E-03, 0.86325983E-03, 0.63702633E-03, 0.31198867E-03,
           -0.52460884E-04, -0.39277750E-03, -0.65212895E-03, -0.78992260E-03,
           -0.78788784E-03, -0.65231544E-03, -0.41183271E-03, -0.11171203E-03,
            0.19462843E-03, 0.45494176E-03, 0.62708522E-03, 0.68562198E-03,
            0.62551565E-03, 0.46197293E-03, 0.22712689E-03, -0.36258243E-04,
           -0.28233288E-03, -0.47020358E-03, -0.57046145E-03, -0.56984887E-03,
           -0.47259816E-03, -0.29920577E-03, -0.82030965E-04, 0.14032755E-03,
            0.33021756E-03, 0.45661698E-03, 0.50048000E-03, 0.45797369E-03,
            0.33860767E-03, 0.16567663E-03, -0.29955087E-04, -0.21456314E-03,
           -0.35754833E-03, -0.43589596E-03, -0.43801026E-03, -0.36503296E-03, 
	   -0.23048975E-03, -0.57501307E-04, 0.12503950E-03, 0.28729177E-03,
            0.40289093E-03, 0.45291355E-03, 0.42827736E-03, 0.33078628E-03,
            0.17226036E-03, -0.27564631E-04, -0.24437343E-03, -0.45283514E-03,
           -0.63022366E-03, -0.75917499E-03, -0.82961662E-03, -0.83932845E-03,
           -0.79353037E-03, -0.70331554E-03, -0.58361021E-03, -0.45073929E-03,
           -0.32029767E-03, -0.20546843E-03, -0.11636934E-03, -0.60547220E-04,
            0.12214734E-03 
		};

double coeff08[] = { 
	   0.82103797E-05, -0.23067401E-03, -0.23108313E-03, -0.32221549E-03, 
	   -0.42275951E-03, -0.52250561E-03, -0.61248150E-03, -0.68334991E-03,
           -0.72634802E-03, -0.73435332E-03, -0.70292165E-03, -0.63106482E-03,
           -0.52175077E-03, -0.38194182E-03, -0.22219167E-03, -0.55754543E-04,
            0.10265913E-03, 0.23844595E-03, 0.33873491E-03, 0.39397061E-03,
            0.39916812E-03, 0.35471123E-03, 0.26652322E-03, 0.14560395E-03,
            0.68782415E-05, -0.13240085E-03, -0.25468261E-03, -0.34419811E-03,
           -0.38900832E-03, -0.38260277E-03, -0.32490634E-03, -0.22247669E-03,
           -0.87825953E-04, 0.62011553E-04, 0.20755727E-03, 0.32930012E-03,
            0.41023662E-03, 0.43823785E-03, 0.40784420E-03, 0.32125591E-03,
            0.18824211E-03, 0.25075649E-04, -0.14738075E-03, -0.30619587E-03,
           -0.42937472E-03, -0.49892085E-03, -0.50351734E-03, -0.44016063E-03,
           -0.31476055E-03, -0.14180694E-03, 0.56958346E-04, 0.25549514E-03,
            0.42713166E-03, 0.54766884E-03, 0.59825013E-03, 0.57004864E-03,
            0.46324756E-03, 0.28957624E-03, 0.69898109E-04, -0.16776180E-03,
           -0.39184059E-03, -0.57142659E-03, -0.68041455E-03, -0.70130342E-03,
           -0.62784378E-03, -0.46637602E-03, -0.23542883E-03, 0.36289312E-04,
            0.31343414E-03, 0.55855454E-03, 0.73714246E-03, 0.82240190E-03,
            0.79921866E-03, 0.66663628E-03, 0.43854979E-03, 0.14236629E-03,
	   -0.18418914E-03, -0.49786316E-03, -0.75552310E-03, -0.92006841E-03,
           -0.96565409E-03, -0.88167767E-03, -0.67477918E-03, -0.36859201E-03,
           -0.11396108E-05, 0.37983980E-03, 0.72299998E-03, 0.98026625E-03,
            0.11134810E-02, 0.11000098E-02, 0.93640707E-03, 0.63953700E-03,
            0.24501598E-03, -0.19695557E-03, -0.62787527E-03, -0.98864117E-03,
           -0.12276262E-02, -0.13079685E-02, -0.12130543E-02, -0.94958447E-03,
           -0.54753030E-03, -0.56793000E-04, 0.45894444E-03, 0.93025574E-03,
            0.12912938E-02, 0.14890410E-02, 0.14908402E-02, 0.12894622E-02,
            0.90544234E-03, 0.38460011E-03, -0.20672077E-03, -0.79039205E-03,
           -0.12865391E-02, -0.16244546E-02, -0.17525668E-02, -0.16461092E-02,
           -0.13114762E-02, -0.78639481E-03, -0.13590157E-03, 0.55560359E-03,
            0.11951807E-02, 0.16937570E-02, 0.19784095E-02, 0.20028895E-02,
            0.17548320E-02, 0.12586436E-02, 0.57344581E-03, -0.21377402E-03,
           -0.99908642E-03, -0.16752025E-02, -0.21460149E-02, -0.23400697E-02,
           -0.22212155E-02, -0.17947418E-02, -0.11081401E-02, -0.24607324E-03,
            0.67987246E-03, 0.15454496E-02, 0.22302822E-02, 0.26343910E-02,
            0.26926280E-02, 0.23847369E-02, 0.17397390E-02, 0.83373138E-03,
           -0.21881415E-03, -0.12792865E-02, -0.22030121E-02, -0.28587638E-02,
           -0.31471101E-02, -0.30151904E-02, -0.24656393E-02, -0.15582053E-02,
           -0.40371725E-03, 0.84909645E-03, 0.20325300E-02, 0.29821028E-02,
            0.35590357E-02, 0.36701679E-02, 0.32824725E-02, 0.24297740E-02,
            0.12106723E-02, -0.22221189E-03, -0.16812229E-02, -0.29675981E-02,
           -0.38983319E-02, -0.43316172E-02, -0.41878889E-02, -0.34631698E-02,
           -0.22326659E-02, -0.64381049E-03, 0.11008445E-02, 0.27688749E-02,
            0.41283993E-02, 0.49796537E-02, 0.51835710E-02, 0.46833428E-02,
            0.35157956E-02, 0.18105427E-02, -0.22343063E-03, -0.23225623E-02,
           -0.42019901E-02, -0.55932603E-02, -0.62815514E-02, -0.61374400E-02,
           -0.51386259E-02, -0.33783074E-02, -0.10583488E-02, 0.15324566E-02,
            0.40529678E-02, 0.61530690E-02, 0.75205211E-02, 0.79252617E-02,
            0.72550066E-02, 0.55372273E-02, 0.29435449E-02, -0.22468859E-03,
           -0.35690069E-02, -0.66412268E-02, -0.90002716E-02, -0.10271665E-01,
           -0.10201983E-01, -0.87009016E-02, -0.58647757E-02, -0.19775054E-02,
            0.25129209E-02, 0.70406594E-02, 0.10986656E-01, 0.13754920E-01,
            0.14851678E-01, 0.13957756E-01, 0.10985126E-01, 0.61097592E-02,
           -0.22478361E-03, -0.73350258E-02, -0.14355149E-01, -0.20319844E-01,
           -0.24263874E-01, -0.25328536E-01, -0.22862909E-01, -0.16508818E-01,
           -0.62591969E-02, 0.75175683E-02, 0.24091266E-01, 0.42421442E-01,
            0.61242584E-01, 0.79175092E-01, 0.94850905E-01, 0.10704119E+00,
            0.11477282E+00, 0.11742169E+00, 0.11477282E+00, 0.10704119E+00,
            0.94850905E-01, 0.79175092E-01, 0.61242584E-01, 0.42421442E-01,
            0.24091266E-01, 0.75175683E-02, -0.62591969E-02, -0.16508818E-01,
           -0.22862909E-01, -0.25328536E-01, -0.24263874E-01, -0.20319844E-01,
           -0.14355149E-01, -0.73350258E-02, -0.22478361E-03, 0.61097592E-02,
            0.10985126E-01, 0.13957756E-01, 0.14851678E-01, 0.13754920E-01,
            0.10986656E-01, 0.70406594E-02, 0.25129209E-02, -0.19775054E-02,
           -0.58647757E-02, -0.87009016E-02, -0.10201983E-01, -0.10271665E-01,
           -0.90002716E-02, -0.66412268E-02, -0.35690069E-02, -0.22468859E-03,
            0.29435449E-02, 0.55372273E-02, 0.72550066E-02, 0.79252617E-02,
            0.75205211E-02, 0.61530690E-02, 0.40529678E-02, 0.15324566E-02,
           -0.10583488E-02, -0.33783074E-02, -0.51386259E-02, -0.61374400E-02,
           -0.62815514E-02, -0.55932603E-02, -0.42019901E-02, -0.23225623E-02,
           -0.22343063E-03, 0.18105427E-02, 0.35157956E-02, 0.46833428E-02,
            0.51835710E-02, 0.49796537E-02, 0.41283993E-02, 0.27688749E-02,
            0.11008445E-02, -0.64381049E-03, -0.22326659E-02, -0.34631698E-02,
           -0.41878889E-02, -0.43316172E-02, -0.38983319E-02, -0.29675981E-02,
           -0.16812229E-02, -0.22221189E-03, 0.12106723E-02, 0.24297740E-02,
            0.32824725E-02, 0.36701679E-02, 0.35590357E-02, 0.29821028E-02,
            0.20325300E-02, 0.84909645E-03, -0.40371725E-03, -0.15582053E-02,
           -0.24656393E-02, -0.30151904E-02, -0.31471101E-02, -0.28587638E-02,
           -0.22030121E-02, -0.12792865E-02, -0.21881415E-03, 0.83373138E-03,
            0.17397390E-02, 0.23847369E-02, 0.26926280E-02, 0.26343910E-02,
            0.22302822E-02, 0.15454496E-02, 0.67987246E-03, -0.24607324E-03,
           -0.11081401E-02, -0.17947418E-02, -0.22212155E-02, -0.23400697E-02,
           -0.21460149E-02, -0.16752025E-02, -0.99908642E-03, -0.21377402E-03,
            0.57344581E-03, 0.12586436E-02, 0.17548320E-02, 0.20028895E-02,
            0.19784095E-02, 0.16937570E-02, 0.11951807E-02, 0.55560359E-03,
           -0.13590157E-03, -0.78639481E-03, -0.13114762E-02, -0.16461092E-02,
           -0.17525668E-02, -0.16244546E-02, -0.12865391E-02, -0.79039205E-03,
           -0.20672077E-03, 0.38460011E-03, 0.90544234E-03, 0.12894622E-02,
            0.14908402E-02, 0.14890410E-02, 0.12912938E-02, 0.93025574E-03,
            0.45894444E-03, -0.56793000E-04, -0.54753030E-03, -0.94958447E-03,
           -0.12130543E-02, -0.13079685E-02, -0.12276262E-02, -0.98864117E-03,
           -0.62787527E-03, -0.19695557E-03, 0.24501598E-03, 0.63953700E-03,
            0.93640707E-03, 0.11000098E-02, 0.11134810E-02, 0.98026625E-03,
            0.72299998E-03, 0.37983980E-03, -0.11396108E-05, -0.36859201E-03,
           -0.67477918E-03, -0.88167767E-03, -0.96565409E-03, -0.92006841E-03,
           -0.75552310E-03, -0.49786316E-03, -0.18418914E-03, 0.14236629E-03,
            0.43854979E-03, 0.66663628E-03, 0.79921866E-03, 0.82240190E-03,
            0.73714246E-03, 0.55855454E-03, 0.31343414E-03, 0.36289312E-04,
           -0.23542883E-03, -0.46637602E-03, -0.62784378E-03, -0.70130342E-03,
           -0.68041455E-03, -0.57142659E-03, -0.39184059E-03, -0.16776180E-03,
            0.69898109E-04, 0.28957624E-03, 0.46324756E-03, 0.57004864E-03,
            0.59825013E-03, 0.54766884E-03, 0.42713166E-03, 0.25549514E-03,
            0.56958346E-04, -0.14180694E-03, -0.31476055E-03, -0.44016063E-03,
           -0.50351734E-03, -0.49892085E-03, -0.42937472E-03, -0.30619587E-03,
           -0.14738075E-03, 0.25075649E-04, 0.18824211E-03, 0.32125591E-03,
            0.40784420E-03, 0.43823785E-03, 0.41023662E-03, 0.32930012E-03,
            0.20755727E-03, 0.62011553E-04, -0.87825953E-04, -0.22247669E-03,
           -0.32490634E-03, -0.38260277E-03, -0.38900832E-03, -0.34419811E-03,
           -0.25468261E-03, -0.13240085E-03, 0.68782415E-05, 0.14560395E-03,
            0.26652322E-03, 0.35471123E-03, 0.39916812E-03, 0.39397061E-03,
            0.33873491E-03, 0.23844595E-03, 0.10265913E-03, -0.55754543E-04,
           -0.22219167E-03, -0.38194182E-03, -0.52175077E-03, -0.63106482E-03,
           -0.70292165E-03, -0.73435332E-03, -0.72634802E-03, -0.68334991E-03,
           -0.61248150E-03, -0.52250561E-03, -0.42275951E-03, -0.32221549E-03,
           -0.23108313E-03, -0.23067401E-03, 0.82103797E-05
		};

double coeff09[] = {
            0.10334683E-03, -0.86830500E-04, -0.11309417E-03, -0.16211296E-03,
           -0.22659537E-03, -0.30118640E-03, -0.38070773E-03, -0.45970571E-03,
           -0.53227443E-03, -0.59244421E-03, -0.63447969E-03, -0.65350812E-03,
           -0.64589776E-03, -0.60984417E-03, -0.54555916E-03, -0.45557879E-03,
           -0.34458062E-03, -0.21929364E-03, -0.87891836E-04, 0.40459221E-04,
            0.15651913E-03, 0.25160259E-03, 0.31856893E-03, 0.35234439E-03,
            0.35068300E-03, 0.31427765E-03, 0.24703503E-03, 0.15557429E-03,
            0.48952628E-04, -0.62407962E-04, -0.16735152E-03, -0.25538626E-03,
           -0.31728926E-03, -0.34653116E-03, -0.33942491E-03, -0.29635328E-03,
           -0.22081575E-03, -0.12052052E-03, -0.52067521E-05, 0.11374974E-03,
            0.22325249E-03, 0.31187609E-03, 0.36955086E-03, 0.38942581E-03,
            0.36817839E-03, 0.30695475E-03, 0.21097164E-03, 0.89457069E-04,
           -0.45512024E-04, -0.17991000E-03, -0.29943729E-03, -0.39079876E-03,
           -0.44336138E-03, -0.45017933E-03, -0.40903798E-03, -0.32267929E-03,
           -0.19884510E-03, -0.49470611E-04, 0.11029350E-03, 0.26373097E-03,
            0.39415859E-03, 0.48684792E-03, 0.53054572E-03, 0.51891909E-03,
            0.45128353E-03, 0.33298458E-03, 0.17492464E-03, -0.73020005E-05,
           -0.19501391E-03, -0.36823680E-03, -0.50792220E-03, -0.59790630E-03,
           -0.62689139E-03, -0.58959745E-03, -0.48790514E-03, -0.33029242E-03,
           -0.13197312E-03, 0.87192966E-04, 0.30443328E-03, 0.49634854E-03,
            0.64145186E-03, 0.72259828E-03, 0.72891382E-03, 0.65721612E-03,
            0.51258289E-03, 0.30812886E-03, 0.63816005E-04, -0.19541323E-03,
           -0.44220843E-03, -0.64962462E-03, -0.79408364E-03, -0.85797149E-03,
           -0.83178846E-03, -0.71538176E-03, -0.51833474E-03, -0.25919278E-03,
            0.36177713E-04, 0.33719951E-03, 0.61166321E-03, 0.82920428E-03,
            0.96456910E-03, 0.10005676E-02, 0.93011744E-03, 0.75740938E-03,
            0.49774197E-03, 0.17631814E-03, -0.17426009E-03, -0.51715533E-03,
           -0.81525062E-03, -0.10349898E-02, -0.11502819E-02, -0.11453861E-02,
           -0.10170409E-02, -0.77526714E-03, -0.44246501E-03, -0.51658822E-04,
            0.35685976E-03, 0.73963474E-03, 0.10546373E-02, 0.12658136E-02,
            0.13471795E-02, 0.12859404E-02, 0.10842727E-02, 0.75956603E-03,
            0.34300814E-03, -0.12333745E-03, -0.59074350E-03, -0.10088523E-02,
           -0.13310716E-02, -0.15197047E-02, -0.15503041E-02, -0.14147066E-02,
           -0.11224281E-02, -0.70016965E-03, -0.18947947E-03, 0.35728040E-03,
            0.88223885E-03, 0.13280993E-02, 0.16443284E-02, 0.17928299E-02,
            0.17524241E-02, 0.15216689E-02, 0.11196053E-02, 0.58434450E-03,
           -0.30447221E-04, -0.66087063E-03, -0.12393472E-02, -0.17018598E-02,
           -0.19948573E-02, -0.20815125E-02, -0.19460535E-02, -0.15961953E-02,
           -0.10630871E-02, -0.39857399E-03, 0.32968217E-03, 0.10450371E-02,
            0.16698187E-02, 0.21337250E-02, 0.23815925E-02, 0.23798570E-02,
            0.21208248E-02, 0.16243445E-02, 0.93651499E-03, 0.12556644E-03,
           -0.72481605E-03, -0.15240074E-02, -0.21840157E-02, -0.26290340E-02,
           -0.28040307E-02, -0.26813296E-02, -0.22645064E-02, -0.15889953E-02,
           -0.71933249E-03, 0.25687224E-03, 0.12376034E-02, 0.21169803E-02,
            0.27965275E-02, 0.31960499E-02, 0.32628782E-02, 0.29784779E-02,
            0.23616101E-02, 0.14675737E-02, 0.38350106E-03, -0.77994767E-03,
           -0.18996744E-02, -0.28529640E-02, -0.35307747E-02, -0.38498417E-02,
           -0.37627306E-02, -0.32640856E-02, -0.23927984E-02, -0.12294052E-02,
            0.11101994E-03, 0.14900197E-02, 0.27597412E-02, 0.37785100E-02,
            0.44262833E-02, 0.46184114E-02, 0.43160855E-02, 0.35323263E-02,
            0.23326038E-02, 0.82992419E-03, -0.82541956E-03, -0.24604306E-02,
           -0.38970730E-02, -0.49711303E-02, -0.55504381E-02, -0.55503962E-02,
           -0.49451254E-02, -0.37727980E-02, -0.21344258E-02, -0.18586659E-03,
            0.18763011E-02, 0.38339228E-02, 0.54695224E-02, 0.65895766E-02,
            0.70463009E-02, 0.67557096E-02, 0.57097333E-02, 0.39810408E-02,
            0.17194953E-02, -0.85954223E-03, -0.34945770E-02, -0.59029749E-02,
           -0.78097181E-02, -0.89768339E-02, -0.92306277E-02, -0.84838737E-02,
           -0.67503345E-02, -0.41497448E-02, -0.90225454E-03, 0.26877085E-02,
            0.62571536E-02, 0.94178319E-02, 0.11794215E-01, 0.13062808E-01,
            0.12988933E-01, 0.11457441E-01, 0.84939487E-02, 0.42741848E-02,
           -0.88017370E-03, -0.65194429E-02, -0.12096788E-01, -0.17009767E-01,
           -0.20649420E-01, -0.22452800E-01, -0.21954497E-01, -0.18832555E-01,
           -0.12944577E-01, -0.43504583E-02, 0.66805948E-02, 0.19680019E-01,
            0.34006387E-01, 0.48886821E-01, 0.63470125E-01, 0.76887675E-01,
            0.88317029E-01, 0.97043283E-01, 0.10251296E+00, 0.10437608E+00,
            0.10251296E+00, 0.97043283E-01, 0.88317029E-01, 0.76887675E-01,
            0.63470125E-01, 0.48886821E-01, 0.34006387E-01, 0.19680019E-01,
            0.66805948E-02, -0.43504583E-02, -0.12944577E-01, -0.18832555E-01,
           -0.21954497E-01, -0.22452800E-01, -0.20649420E-01, -0.17009767E-01,
           -0.12096788E-01, -0.65194429E-02, -0.88017370E-03, 0.42741848E-02,
            0.84939487E-02, 0.11457441E-01, 0.12988933E-01, 0.13062808E-01,
            0.11794215E-01, 0.94178319E-02, 0.62571536E-02, 0.26877085E-02,
           -0.90225454E-03, -0.41497448E-02, -0.67503345E-02, -0.84838737E-02,
           -0.92306277E-02, -0.89768339E-02, -0.78097181E-02, -0.59029749E-02,
           -0.34945770E-02, -0.85954223E-03, 0.17194953E-02, 0.39810408E-02,
            0.57097333E-02, 0.67557096E-02, 0.70463009E-02, 0.65895766E-02,
            0.54695224E-02, 0.38339228E-02, 0.18763011E-02, -0.18586659E-03,
           -0.21344258E-02, -0.37727980E-02, -0.49451254E-02, -0.55503962E-02,
           -0.55504381E-02, -0.49711303E-02, -0.38970730E-02, -0.24604306E-02,
           -0.82541956E-03, 0.82992419E-03, 0.23326038E-02, 0.35323263E-02,
            0.43160855E-02, 0.46184114E-02, 0.44262833E-02, 0.37785100E-02,
            0.27597412E-02, 0.14900197E-02, 0.11101994E-03, -0.12294052E-02,
           -0.23927984E-02, -0.32640856E-02, -0.37627306E-02, -0.38498417E-02,
           -0.35307747E-02, -0.28529640E-02, -0.18996744E-02, -0.77994767E-03,
            0.38350106E-03, 0.14675737E-02, 0.23616101E-02, 0.29784779E-02,
            0.32628782E-02, 0.31960499E-02, 0.27965275E-02, 0.21169803E-02,
            0.12376034E-02, 0.25687224E-03, -0.71933249E-03, -0.15889953E-02,
           -0.22645064E-02, -0.26813296E-02, -0.28040307E-02, -0.26290340E-02,
           -0.21840157E-02, -0.15240074E-02, -0.72481605E-03, 0.12556644E-03,
            0.93651499E-03, 0.16243445E-02, 0.21208248E-02, 0.23798570E-02,
            0.23815925E-02, 0.21337250E-02, 0.16698187E-02, 0.10450371E-02,
            0.32968217E-03, -0.39857399E-03, -0.10630871E-02, -0.15961953E-02,
           -0.19460535E-02, -0.20815125E-02, -0.19948573E-02, -0.17018598E-02,
           -0.12393472E-02, -0.66087063E-03, -0.30447221E-04, 0.58434450E-03,
            0.11196053E-02, 0.15216689E-02, 0.17524241E-02, 0.17928299E-02,
            0.16443284E-02, 0.13280993E-02, 0.88223885E-03, 0.35728040E-03,
           -0.18947947E-03, -0.70016965E-03, -0.11224281E-02, -0.14147066E-02,
           -0.15503041E-02, -0.15197047E-02, -0.13310716E-02, -0.10088523E-02,
           -0.59074350E-03, -0.12333745E-03, 0.34300814E-03, 0.75956603E-03,
            0.10842727E-02, 0.12859404E-02, 0.13471795E-02, 0.12658136E-02,
            0.10546373E-02, 0.73963474E-03, 0.35685976E-03, -0.51658822E-04,
           -0.44246501E-03, -0.77526714E-03, -0.10170409E-02, -0.11453861E-02,
           -0.11502819E-02, -0.10349898E-02, -0.81525062E-03, -0.51715533E-03,
           -0.17426009E-03, 0.17631814E-03, 0.49774197E-03, 0.75740938E-03,
            0.93011744E-03, 0.10005676E-02, 0.96456910E-03, 0.82920428E-03,
            0.61166321E-03, 0.33719951E-03, 0.36177713E-04, -0.25919278E-03,
           -0.51833474E-03, -0.71538176E-03, -0.83178846E-03, -0.85797149E-03,
           -0.79408364E-03, -0.64962462E-03, -0.44220843E-03, -0.19541323E-03,
            0.63816005E-04, 0.30812886E-03, 0.51258289E-03, 0.65721612E-03,
            0.72891382E-03, 0.72259828E-03, 0.64145186E-03, 0.49634854E-03,
            0.30443328E-03, 0.87192966E-04, -0.13197312E-03, -0.33029242E-03,
           -0.48790514E-03, -0.58959745E-03, -0.62689139E-03, -0.59790630E-03,
           -0.50792220E-03, -0.36823680E-03, -0.19501391E-03, -0.73020005E-05,
            0.17492464E-03, 0.33298458E-03, 0.45128353E-03, 0.51891909E-03,
            0.53054572E-03, 0.48684792E-03, 0.39415859E-03, 0.26373097E-03,
            0.11029350E-03, -0.49470611E-04, -0.19884510E-03, -0.32267929E-03,
           -0.40903798E-03, -0.45017933E-03, -0.44336138E-03, -0.39079876E-03,
           -0.29943729E-03, -0.17991000E-03, -0.45512024E-04, 0.89457069E-04,
            0.21097164E-03, 0.30695475E-03, 0.36817839E-03, 0.38942581E-03,
            0.36955086E-03, 0.31187609E-03, 0.22325249E-03, 0.11374974E-03,
           -0.52067521E-05, -0.12052052E-03, -0.22081575E-03, -0.29635328E-03,
           -0.33942491E-03, -0.34653116E-03, -0.31728926E-03, -0.25538626E-03,
           -0.16735152E-03, -0.62407962E-04, 0.48952628E-04, 0.15557429E-03,
            0.24703503E-03, 0.31427765E-03, 0.35068300E-03, 0.35234439E-03,
            0.31856893E-03, 0.25160259E-03, 0.15651913E-03, 0.40459221E-04,
           -0.87891836E-04, -0.21929364E-03, -0.34458062E-03, -0.45557879E-03,
           -0.54555916E-03, -0.60984417E-03, -0.64589776E-03, -0.65350812E-03,
           -0.63447969E-03, -0.59244421E-03, -0.53227443E-03, -0.45970571E-03,
           -0.38070773E-03, -0.30118640E-03, -0.22659537E-03, -0.16211296E-03,
           -0.11309417E-03, -0.86830500E-04, 0.10334683E-03
		};


void remezdig(stopedge,order,adircoeffs)
double	stopedge; /* stopband edge frequency as a fraction of sampling rate */
int	*order; /* filter order of filter which met specs (always odd) */
double	*(*adircoeffs);		/* filter coeffs */
{
	int	cutoff;

	cutoff = (int) (0.5/stopedge + 1.0E-06);
	switch (cutoff) {
		case 2:
			*order = 119;
			*adircoeffs = coeff02;
			break;
		case 3:
			*order = 179;
			*adircoeffs = coeff03;
			break;
		case 4:
			*order = 229;
			*adircoeffs = coeff04;
			break;
		case 5:
			*order = 285;
			*adircoeffs = coeff05;
			break;
		case 6:
			*order = 345;
			*adircoeffs = coeff06;
			break;
		case 7:
			*order = 405;
			*adircoeffs = coeff07;
			break;
		case 8:
			*order = 459;
			*adircoeffs = coeff08;
			break;
		case 9:
			*order = 519;
			*adircoeffs = coeff09;
			break;
		default:
			printf("FIR filter has not been designed for this cutoff!!\n");
			exit(0);
			break;
	}
}


void elliptdig(k,stopedge,ap,aa,rap,raa,order,digco,adircoeffs,bdircoeffs)
double	k;	/* selectivity, ie (passband edge freq)/(stopband edge freq) */
double	stopedge; /* stopband edge frequency as a fraction of sampling rate */
double	ap;	/* required passband ripple in db (eg 0.1)	*/
double	aa;	/* required stopband attenuation in db (eg 60.0) */
double	*rap;	/* actual ripple achieved	*/
double	*raa;	/* actual attenuation achieved	*/
int	*order; /* filter order of filter which met specs, (always even) */
struct	ser_iir digco[];	/* coeffs of series second order sub-filters */
double	adircoeffs[],bdircoeffs[];	/* numerator and denominator coeffs of
				      filter polynomial if expanded out	*/
{
	double	wa,wp;
	double	T,kan;
	double	k1,k2,gain;
	int	n,ii,stpedg;
	struct	ser_iir analco[64];

	stpedg = (int) (0.5/stopedge + 1.0e-6);
	wa = 2.0 * PI * stopedge;
	wp = k * wa;
	normanaedges(wp,wa,&T,&kan);	/* normalized analogue filter
						   edges		*/
	/*
	printf("T=%f  k=%f\n",T,kan);
	*/

	switch (stpedg) {
		case 2: ;
		case 3: ;
		case 4: ;
		case 5: ;
		case 6: ;
		case 7: ;
		case 8: ;
		case 9: ;
		case 10: ;
		case 11: ;
		case 12: ;
		case 13: ;
		case 14: ;
		case 15: ;
		case 16: ;
		case 17: ;
		case 18: ;
		case 19: 
		elliptan(kan,ap,aa,order,analco,rap,raa); /* design analogue filter */
/*
		printf("Filter order=%d passripple=%.2fdB attenuation=%.1fdB\n",*order,*rap,*raa);
*/
		/*
		for (n=0; n<(*order)/2; n++) {
			printf("AA= %8.4f  %8.4f  %8.4f\n",analco[n].a0,analco[n].a1,analco[n].a2);
			printf("BA= %8.4f  %8.4f  %8.4f\n",analco[n].b0,analco[n].b1,analco[n].b2);
			}
		printf("\n");
		*/
		break;
		case 20: ;
		case 21: ;
		case 22: ;
		case 23: ;
		case 24: ;
		case 25: ;
		case 26: ;
		case 27: ;
		case 28: ;
		case 29: 
		elliptan(kan,(ap/2.0),(aa/2.0),order,analco,rap,raa); 
					/* design analogue filter */
/*
		printf("Filter order=%d passripple=%.2fdB attenuation=%.1fdB\n",
		(2* *order),(2* *rap),(2* *raa));
*/
		for (n=0; n<(*order/2); n++) {
			analco[n + *order/2].a0 = analco[n].a0;
			analco[n + *order/2].a1 = analco[n].a1;
			analco[n + *order/2].a2 = analco[n].a2;
			analco[n + *order/2].b0 = analco[n].b0;
			analco[n + *order/2].b1 = analco[n].b1;
			analco[n + *order/2].b2 = analco[n].b2;
			}
		*order = (int)(2.0 * *order);
		/*
		for (n=0; n<(*order)/2; n++) {
			printf("AA= %8.4f  %8.4f  %8.4f\n",analco[n].a0,analco[n].a1,analco[n].a2);
			printf("BA= %8.4f  %8.4f  %8.4f\n",analco[n].b0,analco[n].b1,analco[n].b2);
			}
		*/
		break;
		case 30: ; 
		case 31: ; 
		case 32: ; 
		case 33: ; 
		case 34: ; 
		case 35: ; 
		case 36: ; 
		case 37: ; 
		case 38: ; 
		case 39: ; 
		case 40: ;
		case 41: 
		elliptan(kan,(ap/3.0),(aa/3.0),order,analco,rap,raa); 
					/* design analogue filter */
/*
		printf("Filter order=%d passripple=%.2fdB attenuation=%.1fdB\n",
		(3* *order),(3* *rap),(3* *raa));
*/
		for (n=0; n<(*order/2); n++) 
			for (ii=1; ii<3; ii++) {
			analco[n + (ii* *order)/2].a0 = analco[n].a0;
			analco[n + (ii* *order)/2].a1 = analco[n].a1;
			analco[n + (ii* *order)/2].a2 = analco[n].a2;
			analco[n + (ii* *order)/2].b0 = analco[n].b0;
			analco[n + (ii* *order)/2].b1 = analco[n].b1;
			analco[n + (ii* *order)/2].b2 = analco[n].b2;
			}
		*order = (int)(3.0 * *order);
		/*
		for (n=0; n<(*order)/2; n++) {
			printf("AA= %8.4f  %8.4f  %8.4f\n",analco[n].a0,analco[n].a1,analco[n].a2);
			printf("BA= %8.4f  %8.4f  %8.4f\n",analco[n].b0,analco[n].b1,analco[n].b2);
			}
		*/
		break;
		default: printf("Filter design too difficult! Stopedge=%d\n",stpedg);
			exit(1);
		
		}
		
	
	s_to_z(T,*order,analco,digco);		/* s to z transformation  */
	/*
	for (n=0; n<(*order)/2; n++) {
		printf("AD= %8.4f  %8.4f  %8.4f\n",digco[n].a0,digco[n].a1,digco[n].a2);
		printf("BD= %8.4f  %8.4f  %8.4f\n",digco[n].b0,digco[n].b1,digco[n].b2);
		}
	*/
	for (n=0; n<(*order)/2; n++) {
		k2 = -digco[n].b2;
		k1 = -digco[n].b1/(1.0-k2);
		gain = (1.0-k1*k1) * (1.0-k2*k2);
		/*
		printf("Refl coeffs =%8.5f  %8.5f, Gain = %8.6f\n",k1,k2,gain);
		*/
		if (gain < 0.00001) {
			printf("Filter will suffer from limit cycles...\n");
			exit(1);
			}
		}

	expand(*order,digco,adircoeffs,bdircoeffs);
						/* transform 2nd order structure
						   coeffs to long direct
						   implementation coeffs    */
	/*
	for (n=0; n<= *order; n++)
		printf("%8.4f  %8.4f\n",adircoeffs[n],bdircoeffs[n]);
	freqresp(*order,adircoeffs,bdircoeffs);
	*/
}

#define EXTEND 1024

void intdecdo2(x,xlen,yaddpt,ylen,procede,order,a,b,coeffs,nstage,nstages)
float	x[],*(*yaddpt);
int	xlen,*ylen;
int	order;
double	a[],b[];
int	nstage,nstages;
struct	stage_rec *procede;
struct	ser_iir coeffs[];
{
	char	*mc;
	int	n,m,r,up,down,npos,dpos,ncount,accncount,lmt;
	long	intlen,inttlen;
	float	*y,*yp,*xp;
	double	u0[64],u1[64],u2[64];
	double	mult;
	unsigned allocate;

	register double	inp,*p0,*p1,*p2;
	register struct	ser_iir *c;

	r = order/2;
	up = procede->up;
	down = procede->down;
	*ylen = (xlen * up)/down;
	intlen = xlen * up;
	inttlen = intlen + EXTEND;
	lmt = inttlen/down;
	allocate = sizeof(float) * (*ylen + EXTEND);
	if ( (mc = malloc(allocate)) == NULL) {
		printf("cannot allocate space for interpolated data\n");
		exit(0);
	}
	y = (float *) mc;
	*yaddpt = y;

	printf("Resamp Stage %d/%d: Up %d Down %d\n",nstage+1,nstages,up,down);

	xp = x;
	npos = up;
	dpos = down;
	mult = (double) up;
	yp = y;
	ncount = 0;
	accncount = 0;
	for (m=0; m<r; m++) {
		u0[m] = 0.0;
		u1[m] = 0.0;
		u2[m] = 0.0;
		}

	for (n=0; n<inttlen; n++){
		inp = 0.0;
		if (npos == up) {
			npos = 0;
			if (n < intlen) inp = *xp++;
			else		inp = 0.0;
			/* printf("%f ",inp); */
			}
		npos++;
		
		c = coeffs;
		p0 = u0;
		p1 = u1;
		p2 = u2;
		for (m=0; m<r; m++){
			*p0 = inp - (c->b1* *p1 + c->b2* *p2);
			inp = c->a0 * (*p0 + *p2) + c->a1 * *p1;
			*p2++ = *p1;
			*p1++ = *p0++;
			c++;
			}

		if (dpos == down){
			*yp++ = (float)inp;
			/* printf("%f\n",inp); */
			dpos = 0;
			if (ncount == 200){
				accncount += ncount;
				ncount = 0;
				if (ttytest()) {
					printf("Forward %d/%d\r",accncount,lmt);
					fflush(stdout);
					}
				}
			ncount++;
			}
		dpos++;
		}

	up = down;
	mult = mult * ((double) up);
	xp = --yp;
	npos = up;
	dpos = down;
	ncount = 0;
	accncount = 0;
	for (m=0; m<r; m++) {
		u0[m] = 0.0;
		u1[m] = 0.0;
		u2[m] = 0.0;
	}

	for (n=0; n<inttlen; n++){
		inp = 0.0;
		if (npos == up) {
			npos = 0;
			inp = *xp--;
		}
		npos++;

		c = coeffs;
		p0 = u0;
		p1 = u1;
		p2 = u2;
		for (m=0; m<r; m++){
			*p0 = inp - (c->b1* *p1 + c->b2* *p2);
			inp = c->a0 * (*p0 + *p2) + c->a1 * *p1;
			*p2++ = *p1;
			*p1++ = *p0++;
			c++;
		}

		if (dpos == down){
			*yp-- = (float)inp;
			/* printf("%f\n",inp); */
			dpos = 0;
			if (ncount == 200){
				accncount += ncount;
				ncount = 0;
				if (ttytest()) {
					printf("Backward: %d/%d\r",accncount,lmt);
					fflush(stdout);
					}
				}
			ncount++;
		}
		dpos++;
	}

	yp = y;
	for (n=0; n<*ylen; n++) {
		*yp = (float)(mult * *yp);
		yp++;
	}
	if (ttytest()) printf("\r                                 \r");
}

void intdecdo3(x,xlen,yaddpt,ylen,procede,order,a,nstage,nstages)
float	x[],*(*yaddpt);
int	xlen,*ylen;
int	order;
double	a[];
int	nstage,nstages;
struct stage_rec *procede;
{
	char	*mc;
	int	n,r,up,down,npos,dpos,norder,ncount,accncount,lmt;
	long	intlen,inttlen;
	float	w[1024];
	float	*y,*yp,*xp,*wpp=NULL,*wp;
	double	mult;
	register int m;
	register double	nsum,*ap;
	unsigned allocate;

	r = 1 + (order/2);
	up = procede->up;
	down = procede->down;
	*ylen = (xlen * up)/down;
	intlen = xlen * up;
	inttlen = intlen + r;
	lmt = inttlen/down;
	allocate = sizeof(float) * (*ylen + r);
	if ( (mc = malloc(allocate)) == NULL) {
		printf("cannot allocate space for interpolated data\n");
		exit(0);
	}
	y = (float *) mc;
	*yaddpt = y;

	printf("Resamp Stage %d/%d: Up %d Down %d\n",nstage+1,nstages,up,down);

	xp = x;
	npos = up;
	dpos = 1-r;
	mult = (double) up;
	if (up > 1) 
		for (m=0; m< xlen; m++) {
			*xp = (float)(*xp * mult);
			xp++;
		}
	xp = x;
	yp = y;
	norder = 0;
	ncount = 0;
	accncount = 0;
	for (m=0; m<=order; m++) w[m]=(float)0.0;

	for (n=0; n<inttlen; n++){
		nsum = 0.0;
		if (npos == up) {
			npos = 0;
			wpp = w+norder;
			if (n < intlen) *wpp = *xp++;
			else		*wpp = (float)0.0;
		}
		if (dpos == down || dpos == 0){
			ap = a+npos;
			wp = wpp;
			for (m=npos; m<=norder; m += up) {
				nsum = nsum + *ap * *wp;
				ap += up;
				wp -= up;
			}
			wp = w+norder + order - m + 1;
			for (  ; m<=order; m += up) {
				nsum = nsum + *ap * *wp;
				ap += up;
				wp -= up;
			}
			*yp++ = (float)nsum;
			if (ncount == 200){
				accncount += ncount;
				ncount = 0;
				if (ttytest()) {
					printf("Filter %d/%d\r",accncount,lmt);
					fflush(stdout);
				}
			}
			ncount++;
			dpos = 0;
		}
		npos++;
		dpos++;

		norder++;
		if (norder > order) norder = 0;
	}
}

#define PRIMES 175
int 	primearr[PRIMES] = {2,3,5,7,11,13,17,19,23,29,31,
			37,41,43,47,53,59,61,67,71,73,
			79,83,89,97,101,103,107,109,113,
			127,131,137,139,149,151,157,163,
			167,173,179,181,191,193,197,199,
			211,223,227,229,233,239,241,251,
			257,263,269,271,277,281,283,293,
			307,311,313,317,331,337,347,349,
			353,359,367,373,379,383,389,397,
			401,409,419,421,431,433,439,443,
			449,457,561,463,467,479,487,491,
			499,503,509,521,523,541,547,557,
			563,569,571,577,587,593,599,601,
			607,613,617,619,631,641,643,647,
			653,659,661,673,677,683,691,701,
			709,719,727,733,739,743,751,757,
			761,769,773,787,797,809,811,821,
			823,827,829,839,853,857,859,863,
			877,881,883,887,907,911,919,929,
			937,941,947,953,967,971,977,983,
			991,997,1009,1013,1019,1021,1031,
			1033,1039};
		/* 	array of prime numbers		*/

#define SMALLFLOAT 1.0e-6
#define BIGINT 1000

void rationalize(x,numer,denom)
double	x;
int	*numer,*denom;
{
	int	n,flag,nxn;
	double	xn,dif;

	flag = 0;
	for (n=1; n<BIGINT; n++) {
		xn = (double)n * x;
		nxn = (int)(0.5 + xn);
		dif = x - (double)nxn/(double)n;
		if (dif < 0) dif = -dif;
		if (dif <= SMALLFLOAT) {
/* printf("target=%g denominator=%d xn=%g numerator=%d ratio=%g\n",x,n,xn,nxn,(double)nxn/(double)n); */
			*denom = n;
			*numer = nxn;
			flag = 1;
		}
		if (flag) break;
	}
	if (flag == 0) {
		fprintf(stderr,"No simple factors for ratio %g\n",x);
		exit(0);
	}
}

void factor(numer,denom,nnum,num,nden,den)
int	numer,denom;
int	*nnum,num[],*nden,den[];
{
	int	i,n;

	*nnum = *nden = 0;

	n = numer;
	if (n != 1)
		for (i=0; i<PRIMES; i++){
			while (n%primearr[i]==0){
				n = n/primearr[i];
				num[(*nnum)++] = primearr[i];
			}
			if (n==1) break;
		}
	else
		num[(*nnum)++] = 1;

	n = denom;
	if (n != 1)
		for (i=0; i<PRIMES; i++){
			while (n%primearr[i]==0){
				n = n/primearr[i];
				den[(*nden)++] = primearr[i];
			}
			if (n==1) break;
		}
	else
		den[(*nden)++] = 1;
}

void order(nnum,num,nden,den,nstages,procede)
int	nnum,num[32],nden,den[32];
int	*nstages;
struct stage_rec procede[];
{
	int	dcount,ncount,up,down=0,flag1,flag2=0,cc;
	double	sratio;

	*nstages = 0;
	sratio = 1.0;
	up = 1;
	dcount = 0;
	for (ncount=0; ncount<nnum; ncount++){
		up = num[ncount] * up;
		sratio = num[ncount] * sratio;
		down = 1;
		flag1 = 0;
		flag2 = 0;
		if (sratio/((double) den[dcount]) > 0.999999) flag1 = 1;
		if (dcount < nden) flag2 = 1;
		while (flag1 && flag2){
			down = down * den[dcount];
			sratio = sratio / ((double) den[dcount]);
			if ((dcount+1) == nden) flag2 = 0;
			else	{
				dcount++;
				if (sratio/((double) den[dcount]) < 0.999999) flag1 = 0;
			}
		}
		if (flag2 == 1 && down != 1){
			procede[*nstages].up = up;
			cc = max(up,down);
			procede[*nstages].cutoff = 0.5/cc;
			procede[*nstages].down = down;
			(*nstages)++;
			up = 1;
		}
	}
	
	/*
	if (*nstages > 0 && up == 1){
		(*nstages)--;
		procede[*nstages].down *= den[dcount];
		cc=max(procede[*nstages].down,procede[*nstages].up);
		procede[*nstages].cutoff = 0.5/cc;
		(*nstages)++;
		dcount++;
		}
	*/
	for ( ; dcount < nden; dcount++){
		if (flag2 == 0) 
			flag2 = 1;
		else	down = den[dcount];
		procede[*nstages].up = up;
		cc = max(down,up);
		procede[*nstages].cutoff = 0.5/cc;
		procede[*nstages].down = down;
		(*nstages)++;
		up = 1;
	}
}

void intdecseq(x,nstages,procede)
double	x;
int	*nstages;
struct stage_rec procede[];
{
	int	denom,numer;
	int	nnum,num[32],nden,den[32];

	rationalize(x,&numer,&denom);
/*
	printf("%f = %d/%d\n",x,numer,denom);
*/
	factor(numer,denom,&nnum,num,&nden,den);
	order(nnum,num,nden,den,nstages,procede);
/*
	for (i=0; i<(nnum-1); i++) printf("%d.",num[i]);
	printf("%d",num[i]);
	printf("/");
	for (i=0; i<(nden-1); i++) printf("%d.",den[i]);
	printf("%d",den[i]);
	printf("\n");
*/
/*
	for (i=0; i<*nstages; i++)
		printf("Resamp Stage %d: Up %d Down %d\n",i,
		procede[i].up,procede[i].down,procede[i].cutoff);
*/
}

void intdecE(x,xlen,fx,yptpt,ylen,fy)
float	x[],*(*yptpt);	/*	input and output data arrays	*/
int	xlen,*ylen;	/*	input and output data lengths	*/
int	fx,fy;		/*	input and output sampling rates	*/
{
	int	n,order,nstages;
	int	zlen;
	double	k = 0.9;
	double	ap = 0.2;
	double	aa = 80.0;
	double	adircoeffs[64],bdircoeffs[64];
	double	rap,raa;
	float	*z,*y;

	struct	stage_rec procede[32];

	struct	iir_rec coeffs[32];

	intdecseq((((double) fy)/((double) fx)),&nstages,procede);
	for (n=0; n<nstages; n++) 
		if ( ((int) (0.5/procede[n].cutoff +1.0e-6)) > 41 ){
			printf("\nFilter design for this sampling rate is too\n");
			printf("difficult. Try another frequency so that the\n");
			printf("relationship between the required and \n");
			printf("initial sampling rates is simpler\n");
			exit(1);
		}
	for (n=0; n<nstages; n++) {
		elliptdig(k,procede[n].cutoff,ap,aa,&rap,&raa,&order,coeffs,
		       adircoeffs,bdircoeffs);
		if (n==0)
			intdecdo2(x,xlen,&y,ylen,(procede+n),order,
				 adircoeffs,bdircoeffs,coeffs,n,nstages);
		else	{
			intdecdo2(y,*ylen,&z,&zlen,(procede+n),order,
				 adircoeffs,bdircoeffs,coeffs,n,nstages);
			free((char *) y);
			y=z;
			*ylen = zlen;
		}
	}	
	*yptpt = y;
}

void intdecF(x,xlen,fx,yptpt,ylen,fy)
float	x[],*(*yptpt);	/*	input and output data arrays	*/
int	xlen,*ylen;	/*	input and output data lengths	*/
int	fx,fy;		/*	input and output sampling rates	*/
{
	int	n,order,nstages;
	int	zlen;
	double	*adircoeffs;
	float	*z,*y;

	struct	stage_rec procede[32];

	intdecseq((((double) fy)/((double) fx)),&nstages,procede);
	for (n=0; n<nstages; n++) 
		if ( ((int) (0.5/procede[n].cutoff +1.0e-6)) > 9 ){
			printf("\nFilter design for this sampling rate is too\n");
			printf("difficult. Try another frequency so that \n");
			printf("the relationship between the required and \n");
			printf("initial sampling rates is simpler\n");
			printf("Otherwise try with IIR filter\n");
			exit(1);
		}
	for (n=0; n<nstages; n++) {
		remezdig(procede[n].cutoff,&order,&adircoeffs);
		if (n==0)
			intdecdo3(x,xlen,&y,ylen,(procede+n),order,adircoeffs,n,nstages);
		else	{
			intdecdo3(y,*ylen,&z,&zlen,(procede+n),order,adircoeffs,n,nstages);
			free((char *) y);
			y=z;
			*ylen = zlen;
		}
	}	
	*yptpt = y;
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item/sub-type specifiers */
	char		*ty;
	int		INP_TYPE = SP_TYPE;
	char		*inptype = "0";

	/* process data */
	struct item_header	inpitem,outitem;

	short			*inpdata,*outdata;
	
	float			*x,*y;		/* input and output waveforms */
	int			xlen,ylen;	/* length of data 	*/
	int			SAMP1,SAMP2 = 0; /* sampling frequencies*/

	/* data variables */

	int		n;
	char		filtype = 'E';

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f:F")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Resample waveform V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				INP_TYPE = it;
				inptype = ty;
				if(INP_TYPE != SP_TYPE &&
				   INP_TYPE != LX_TYPE )
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :
			sscanf(optarg,"%d",&SAMP2);
			break;
		case 'F': 	/* FIR linear phase filter selected */
			filtype = 'F';
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-fnew_sampl_freq) (-F) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check new sampling rate given */
	if (SAMP2 == 0) 
		error("No or inappropriate new sampling rate",NULL);

	/* read in the input waveform item */
	getitem(filename,INP_TYPE,inptype,&inpitem,(void **)&inpdata);

	/* get float speech buffers */
	xlen=inpitem.numframes;
	SAMP1=(int) (0.5+1.0/inpitem.frameduration);
	if ((x = (float *) calloc(xlen,sizeof(float)))==NULL)
		error("could not get memory for buffer");
	for (n=0; n<xlen; n++) 
		x[n] = (float) inpdata[n];
	free(inpdata);

	/* filter and decimate */
	if (filtype == 'E')
		intdecE(x,xlen,SAMP1,&y,&ylen,SAMP2);
	else
		intdecF(x,xlen,SAMP1,&y,&ylen,SAMP2);

	/* get output data */
	if ((outdata = (short *) calloc(ylen,sizeof(short))) == NULL)
		error("could not get memory buffer");
	for (n=0; n<ylen; n++)
		outdata[n] = (short) y[n];
	
	/* produce new item */
	sfsheader(&outitem,INP_TYPE,0,2,1,(1.0/SAMP2),inpitem.offset,0,0,0);
	sprintf(outitem.history,"%s(%d.%02d;rate=%d,filt=%s)",
		PROGNAME,INP_TYPE,inpitem.subtype,SAMP2,(filtype=='F')?"FIR":"IIR");
	putitem(filename,&outitem,ylen,outdata);
	exit(0);
}


