/* svg -- DIG routines for Scalable Vector Graphics output */

/* Mark Huckvale - University College London */

/* version 1.0 - June 2004 */

#include "SFSCONFG.h"
#ifdef	EVg

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <malloc.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"

/* default image size */
#define DEFWIDTH	600
#define DEFHEIGHT	450
#define DEFCHARWIDTH	6
#define DEFCHARHEIGHT	10

/* operating modes */
#define SVG_INITIALISED 1

/* local storage of parameters */
struct {
	int	mode;		/* bitmap modes */
	int	subtype;	/* subtype as supplied from digmap */
	int	col;
	int	width,height;
} digDEVg;

/* bitmap colours */
#define NUM_COLOUR	32
static struct rgb_rec {
	unsigned char	r,g,b;
} colmap[NUM_COLOUR]={
	{ 255,255,255},	/* 0 = background */
	{ 200, 50, 50},	/* 1 */
	{ 100, 75,100},	/* 2 */
	{ 100, 75, 50},	/* 3 */
	{ 150,200,  0},	/* 4 */
	{  25, 25,100},	/* 5 = was red - mapped to dark grey/blue */
	{  50,200,100},	/* 6 */
	{  50,100,200},	/* 7 */
	{ 150,150,  0},	/* 8 */
	{  50, 50, 50},	/* 9 = was white - mapped to dark grey */
	{ 200,100,  0},	/* 10 */
	{ 100,  0,100},	/* 11 */
	{   0,100,100},	/* 12 */
	{ 200,100,100},	/* 13 */
	{  50,100,200},	/* 14 */
	{   0,  0,  0},	/* 15 = foreground */
	{ 250,250,250},	/* 16 */
	{ 248,248,248},	/* 17 */
	{ 244,244,244},	/* 18 */
	{ 238,238,238},	/* 19 */
	{ 230,230,230},	/* 20 */
	{ 220,220,220},	/* 21 */
 	{ 208,208,208},	/* 22 */
	{ 194,194,194},	/* 23 */
	{ 178,178,178},	/* 24 */
	{ 160,160,160},	/* 25 */
	{ 140,140,140},	/* 26 */
	{ 118,118,118},	/* 27 */
	{  94, 94, 94},	/* 28 */
	{  68, 68, 68},	/* 29 */
	{  40, 40, 40},	/* 30 */
	{  10, 10, 10},	/* 31 */
};

/* get graphics parameters */
int svg_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	char	*p;
	char	cbuf[32];

	/* record operating mode */
	digDEVg.subtype = subtype;

	/* set up image dimensions */
	if ((p=getenv("GSIZE"))!=NULL) {
		strncpy(cbuf,p,31);
		digDEVg.width = atoi(strtok(cbuf,"x"));
		digDEVg.height = atoi(strtok(NULL,""));
	}
	else {
		digDEVg.width = DEFWIDTH;
		digDEVg.height = DEFHEIGHT;
	}
	if (digDEVg.width > 10)
		digdev->nhoriz = digDEVg.width;
	else
		digdev->nhoriz = digDEVg.width = DEFWIDTH;
	if (digDEVg.height > 10)
		digdev->nvert = digDEVg.height;
	else
		digdev->nvert = digDEVg.height = DEFHEIGHT;

	/* return other image characteristics */
	digdev->aspect = (float)1.0;
 	digdev->chwidth = DEFCHARWIDTH;
	digdev->chheight = DEFCHARHEIGHT;
	digdev->greylevels = 16;
	digdev->greypixwidth = 1;
	digdev->greypixheight = 1;

	return(0);
}

/* open output file */
int svg_outfile(fp)
FILE	**fp;
{
	char	*p,ofilename[256];

	if ((p=getenv("GFILE"))!=NULL)
		strncpy(ofilename,p,255);
	else
		strcpy(ofilename,"dig.svg");
	if ((*fp = fopen(ofilename,"w"))==NULL)
		error("could not open '%s'",ofilename);
	return(0);
}

/* open output */
int svg_open()
{
	int	i;

	if (digDEVg.mode & SVG_INITIALISED) return(0);

	fprintf(digdata.outfile,"<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"no\"?>\n");
	fprintf(digdata.outfile,"<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20010904//EN\" ");
	fprintf(digdata.outfile,"\"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">\n");
	fprintf(digdata.outfile,"<svg xmlns=\"http://www.w3.org/2000/svg\" xmlns:xlink=\"http://www.w3.org/1999/xlink\"\n");
	fprintf(digdata.outfile,"width=\"%d\" height=\"%d\">\n",digDEVg.width,digDEVg.height);
	fprintf(digdata.outfile,"<style type=\"text/css\">\n");
	fprintf(digdata.outfile,"text { font-family:\"Courier New\"; font-size:10; }\n");
	fprintf(digdata.outfile,"path { fill:none; stroke-width:1; }\n");
	fprintf(digdata.outfile,"</style>\n");


	digDEVg.mode |= SVG_INITIALISED;
	return(0);
}

/* interrupt */
void svg_inter()
{
	int	i;
	if (digDEVg.mode & SVG_INITIALISED) {
	}
}

/* close */
void svg_close()
{
	if (digDEVg.mode & SVG_INITIALISED) {

		fprintf(digdata.outfile,"</svg>\n");
		/* shut down output stream */
		fclose(digdata.outfile);
	}
	digDEVg.mode = 0;
}

/* clear screen */
int svg_clear()
{
	return(0);
}

/* set up colour from bundle */
static void svg_colour(bundle)
int	bundle;
{
	if (bundle==0)
		digDEVg.col=0;	/* background */
	else if ((bundle >= 100) && (bundle < 115))
		digDEVg.col=bundle-84;	/* grey scale */
	else {
		bundle = bundle % 15;
		if (bundle==0)
			digDEVg.col=15;
		else
			digDEVg.col=bundle;
	}
}

/* plot poly-line */
void svg_polyline(bundle,buff,len)
int bundle;
short *buff;
int32	len;
{
	int     x1,y1,x2,y2;

	/* set up plot colour */
	svg_colour(bundle);

	/* move to first point */
	x1 = *buff++;
	y1 = *buff++;
	len-=(int32)2;
	fprintf(digdata.outfile,"<path d=\"M%d %d",x1,digDEVg.height-y1-1);

	/* plot lines */
	while (len > (int32)0) {
		x2 = *buff++;
		y2 = *buff++;
		len -= (int32)2;
		fprintf(digdata.outfile,"\nL%d %d %d %d",x1,digDEVg.height-y1-1,x2,digDEVg.height-y2-1);
		x1 = x2;
		y1 = y2;
	}

	fprintf(digdata.outfile,"\"\nstyle=\"stroke:rgb(%d,%d,%d)\" />\n",
		colmap[digDEVg.col].r,colmap[digDEVg.col].g,colmap[digDEVg.col].b);
}

/* draw text */
void svg_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	/* set up plot colour */
	svg_colour(bundle);

	fprintf(digdata.outfile,"<text x=\"%d\" y=\"%d\" fill=\"rgb(%d,%d,%d)\">",
		x,digDEVg.height-y-1,
		colmap[digDEVg.col].r,colmap[digDEVg.col].g,colmap[digDEVg.col].b);

	while (*str) {
		if (*str=='<')
			fprintf(digdata.outfile,"&lt;");
		else
			fputc(*str,digdata.outfile);
		str++;
	}
	fprintf(digdata.outfile,"</text>\n",str);
}

/* draw a filled rectangle */
void svg_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	int	t;

	/* set up plot colour */
	svg_colour(bundle);

	y1 = digDEVg.height - y1 -1;
	y2 = digDEVg.height - y2 -1;
	if (x1 > x2) { t = x1; x1=x2; x2=t; }
	if (y1 > y2) { t = y1; y1=y2; y2=t; }

	fprintf(digdata.outfile,"<rect x=\"%d\" y=\"%d\" width=\"%d\" height=\"%d\" ",
		x1,y1,x2-x1+1,y2-y1+1);
	fprintf(digdata.outfile,"fill=\"rgb(%d,%d,%d)\" />\n",
		colmap[digDEVg.col].r,colmap[digDEVg.col].g,colmap[digDEVg.col].b);
}

/* grey-level display */
void svg_gscale(x,y,height,buff)
int x;
int y;
int height;
unsigned char *buff;
{
	int		i,lasty;
	unsigned char 	*p,lastc;

	/* replace greyscales by colours */
	p=buff;
	lastc=*p++;
	y = digdata.nvert - y -1;
	y -= height - 1;
	lasty = y;
	for (i = 1; i < height; i++, p++) {
		if (*p != lastc) {
			fprintf(digdata.outfile,"<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" stroke=\"rgb(%d,%d,%d)\" />\n",
				x,lasty,x,y+i,
				colmap[16+lastc].r,colmap[16+lastc].g,colmap[16+lastc].b);
			lasty = y+i;
			lastc = *p;
		}
	}
	fprintf(digdata.outfile,"<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" stroke=\"rgb(%d,%d,%d)\" />\n",
		x,lasty,x,y+height-1,
		colmap[16+lastc].r,colmap[16+lastc].g,colmap[16+lastc].b);
}

/* wait for key routine - NULL don't wait */
void svg_pause()
{
}

int svg_vtext(int bundle,int x1,int y1,int x2,int y2,char *str,int height,int prop)
{
	static int	pathno=0;

	/* check for length only */
	if (bundle >= 0) {

		/* set up plot colour */
		svg_colour(bundle);

		/* create a path and draw text on path */
		y1 = digDEVg.height - y1 - 1;
		y2 = digDEVg.height - y2 - 1;

		fprintf(digdata.outfile,"<defs><path id=\"path%d\" d=\"M%d %d L%d %d\" /></defs>\n",
			++pathno, x1,y1,x2,y2);
		fprintf(digdata.outfile,"<text style=\"font-family:%s; font-size:%d; fill:rgb(%d,%d,%d);\">",
			(prop)?"Verdana":"Courier New",
			(prop)?(int)(1.35*height):(int)(1.7*height),
			colmap[digDEVg.col].r,colmap[digDEVg.col].g,colmap[digDEVg.col].b);
		fprintf(digdata.outfile,"<textPath xlink:href=\"#path%d\" xml:space=\"default\">%s</textPath></text>\n",
			pathno,str);
	}
	if (prop)
		return((int)(strlen(str)*1.35*height*0.65));
	else
		return((int)(strlen(str)*1.7*height*0.57));
}
#endif
